/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53domains;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.route53domains.model.AcceptDomainTransferFromAnotherAwsAccountRequest;
import software.amazon.awssdk.services.route53domains.model.AcceptDomainTransferFromAnotherAwsAccountResponse;
import software.amazon.awssdk.services.route53domains.model.AssociateDelegationSignerToDomainRequest;
import software.amazon.awssdk.services.route53domains.model.AssociateDelegationSignerToDomainResponse;
import software.amazon.awssdk.services.route53domains.model.CancelDomainTransferToAnotherAwsAccountRequest;
import software.amazon.awssdk.services.route53domains.model.CancelDomainTransferToAnotherAwsAccountResponse;
import software.amazon.awssdk.services.route53domains.model.CheckDomainAvailabilityRequest;
import software.amazon.awssdk.services.route53domains.model.CheckDomainAvailabilityResponse;
import software.amazon.awssdk.services.route53domains.model.CheckDomainTransferabilityRequest;
import software.amazon.awssdk.services.route53domains.model.CheckDomainTransferabilityResponse;
import software.amazon.awssdk.services.route53domains.model.DeleteDomainRequest;
import software.amazon.awssdk.services.route53domains.model.DeleteDomainResponse;
import software.amazon.awssdk.services.route53domains.model.DeleteTagsForDomainRequest;
import software.amazon.awssdk.services.route53domains.model.DeleteTagsForDomainResponse;
import software.amazon.awssdk.services.route53domains.model.DisableDomainAutoRenewRequest;
import software.amazon.awssdk.services.route53domains.model.DisableDomainAutoRenewResponse;
import software.amazon.awssdk.services.route53domains.model.DisableDomainTransferLockRequest;
import software.amazon.awssdk.services.route53domains.model.DisableDomainTransferLockResponse;
import software.amazon.awssdk.services.route53domains.model.DisassociateDelegationSignerFromDomainRequest;
import software.amazon.awssdk.services.route53domains.model.DisassociateDelegationSignerFromDomainResponse;
import software.amazon.awssdk.services.route53domains.model.DnssecLimitExceededException;
import software.amazon.awssdk.services.route53domains.model.DomainLimitExceededException;
import software.amazon.awssdk.services.route53domains.model.DuplicateRequestException;
import software.amazon.awssdk.services.route53domains.model.EnableDomainAutoRenewRequest;
import software.amazon.awssdk.services.route53domains.model.EnableDomainAutoRenewResponse;
import software.amazon.awssdk.services.route53domains.model.EnableDomainTransferLockRequest;
import software.amazon.awssdk.services.route53domains.model.EnableDomainTransferLockResponse;
import software.amazon.awssdk.services.route53domains.model.GetContactReachabilityStatusRequest;
import software.amazon.awssdk.services.route53domains.model.GetContactReachabilityStatusResponse;
import software.amazon.awssdk.services.route53domains.model.GetDomainDetailRequest;
import software.amazon.awssdk.services.route53domains.model.GetDomainDetailResponse;
import software.amazon.awssdk.services.route53domains.model.GetDomainSuggestionsRequest;
import software.amazon.awssdk.services.route53domains.model.GetDomainSuggestionsResponse;
import software.amazon.awssdk.services.route53domains.model.GetOperationDetailRequest;
import software.amazon.awssdk.services.route53domains.model.GetOperationDetailResponse;
import software.amazon.awssdk.services.route53domains.model.InvalidInputException;
import software.amazon.awssdk.services.route53domains.model.ListDomainsRequest;
import software.amazon.awssdk.services.route53domains.model.ListDomainsResponse;
import software.amazon.awssdk.services.route53domains.model.ListOperationsRequest;
import software.amazon.awssdk.services.route53domains.model.ListOperationsResponse;
import software.amazon.awssdk.services.route53domains.model.ListPricesRequest;
import software.amazon.awssdk.services.route53domains.model.ListPricesResponse;
import software.amazon.awssdk.services.route53domains.model.ListTagsForDomainRequest;
import software.amazon.awssdk.services.route53domains.model.ListTagsForDomainResponse;
import software.amazon.awssdk.services.route53domains.model.OperationLimitExceededException;
import software.amazon.awssdk.services.route53domains.model.PushDomainRequest;
import software.amazon.awssdk.services.route53domains.model.PushDomainResponse;
import software.amazon.awssdk.services.route53domains.model.RegisterDomainRequest;
import software.amazon.awssdk.services.route53domains.model.RegisterDomainResponse;
import software.amazon.awssdk.services.route53domains.model.RejectDomainTransferFromAnotherAwsAccountRequest;
import software.amazon.awssdk.services.route53domains.model.RejectDomainTransferFromAnotherAwsAccountResponse;
import software.amazon.awssdk.services.route53domains.model.RenewDomainRequest;
import software.amazon.awssdk.services.route53domains.model.RenewDomainResponse;
import software.amazon.awssdk.services.route53domains.model.ResendContactReachabilityEmailRequest;
import software.amazon.awssdk.services.route53domains.model.ResendContactReachabilityEmailResponse;
import software.amazon.awssdk.services.route53domains.model.ResendOperationAuthorizationRequest;
import software.amazon.awssdk.services.route53domains.model.ResendOperationAuthorizationResponse;
import software.amazon.awssdk.services.route53domains.model.RetrieveDomainAuthCodeRequest;
import software.amazon.awssdk.services.route53domains.model.RetrieveDomainAuthCodeResponse;
import software.amazon.awssdk.services.route53domains.model.Route53DomainsException;
import software.amazon.awssdk.services.route53domains.model.TldRulesViolationException;
import software.amazon.awssdk.services.route53domains.model.TransferDomainRequest;
import software.amazon.awssdk.services.route53domains.model.TransferDomainResponse;
import software.amazon.awssdk.services.route53domains.model.TransferDomainToAnotherAwsAccountRequest;
import software.amazon.awssdk.services.route53domains.model.TransferDomainToAnotherAwsAccountResponse;
import software.amazon.awssdk.services.route53domains.model.UnsupportedTldException;
import software.amazon.awssdk.services.route53domains.model.UpdateDomainContactPrivacyRequest;
import software.amazon.awssdk.services.route53domains.model.UpdateDomainContactPrivacyResponse;
import software.amazon.awssdk.services.route53domains.model.UpdateDomainContactRequest;
import software.amazon.awssdk.services.route53domains.model.UpdateDomainContactResponse;
import software.amazon.awssdk.services.route53domains.model.UpdateDomainNameserversRequest;
import software.amazon.awssdk.services.route53domains.model.UpdateDomainNameserversResponse;
import software.amazon.awssdk.services.route53domains.model.UpdateTagsForDomainRequest;
import software.amazon.awssdk.services.route53domains.model.UpdateTagsForDomainResponse;
import software.amazon.awssdk.services.route53domains.model.ViewBillingRequest;
import software.amazon.awssdk.services.route53domains.model.ViewBillingResponse;
import software.amazon.awssdk.services.route53domains.transform.AcceptDomainTransferFromAnotherAwsAccountRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.AssociateDelegationSignerToDomainRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.CancelDomainTransferToAnotherAwsAccountRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.CheckDomainAvailabilityRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.CheckDomainTransferabilityRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.DeleteDomainRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.DeleteTagsForDomainRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.DisableDomainAutoRenewRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.DisableDomainTransferLockRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.DisassociateDelegationSignerFromDomainRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.EnableDomainAutoRenewRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.EnableDomainTransferLockRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.GetContactReachabilityStatusRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.GetDomainDetailRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.GetDomainSuggestionsRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.GetOperationDetailRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.ListDomainsRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.ListOperationsRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.ListPricesRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.ListTagsForDomainRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.PushDomainRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.RegisterDomainRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.RejectDomainTransferFromAnotherAwsAccountRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.RenewDomainRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.ResendContactReachabilityEmailRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.ResendOperationAuthorizationRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.RetrieveDomainAuthCodeRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.TransferDomainRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.TransferDomainToAnotherAwsAccountRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.UpdateDomainContactPrivacyRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.UpdateDomainContactRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.UpdateDomainNameserversRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.UpdateTagsForDomainRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.ViewBillingRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link Route53DomainsClient}.
 *
 * @see Route53DomainsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRoute53DomainsClient implements Route53DomainsClient {
    private static final Logger log = Logger.loggerFor(DefaultRoute53DomainsClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final Route53DomainsServiceClientConfiguration serviceClientConfiguration;

    protected DefaultRoute53DomainsClient(Route53DomainsServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Accepts the transfer of a domain from another Amazon Web Services account to the currentAmazon Web Services
     * account. You initiate a transfer between Amazon Web Services accounts using <a href=
     * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_TransferDomainToAnotherAwsAccount.html"
     * >TransferDomainToAnotherAwsAccount</a>.
     * </p>
     * <p>
     * If you use the CLI command at <a href=
     * "https://docs.aws.amazon.com/cli/latest/reference/route53domains/accept-domain-transfer-from-another-aws-account.html"
     * >accept-domain-transfer-from-another-aws-account</a>, use JSON format as input instead of text because otherwise
     * CLI will throw an error from domain transfer input that includes single quotes.
     * </p>
     * <p>
     * Use either <a
     * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_ListOperations.html">ListOperations</a>
     * or <a href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_GetOperationDetail.html">
     * GetOperationDetail</a> to determine whether the operation succeeded. <a
     * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_GetOperationDetail.html"
     * >GetOperationDetail</a> provides additional information, for example,
     * <code>Domain Transfer from Aws Account 111122223333 has been cancelled</code>.
     * </p>
     *
     * @param acceptDomainTransferFromAnotherAwsAccountRequest
     *        The AcceptDomainTransferFromAnotherAwsAccount request includes the following elements.
     * @return Result of the AcceptDomainTransferFromAnotherAwsAccount operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws DomainLimitExceededException
     *         The number of domains has exceeded the allowed threshold for the account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.AcceptDomainTransferFromAnotherAwsAccount
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/AcceptDomainTransferFromAnotherAwsAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AcceptDomainTransferFromAnotherAwsAccountResponse acceptDomainTransferFromAnotherAwsAccount(
            AcceptDomainTransferFromAnotherAwsAccountRequest acceptDomainTransferFromAnotherAwsAccountRequest)
            throws InvalidInputException, OperationLimitExceededException, DomainLimitExceededException, UnsupportedTldException,
            AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AcceptDomainTransferFromAnotherAwsAccountResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, AcceptDomainTransferFromAnotherAwsAccountResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                acceptDomainTransferFromAnotherAwsAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcceptDomainTransferFromAnotherAwsAccount");

            return clientHandler
                    .execute(new ClientExecutionParams<AcceptDomainTransferFromAnotherAwsAccountRequest, AcceptDomainTransferFromAnotherAwsAccountResponse>()
                            .withOperationName("AcceptDomainTransferFromAnotherAwsAccount").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withInput(acceptDomainTransferFromAnotherAwsAccountRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AcceptDomainTransferFromAnotherAwsAccountRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a delegation signer (DS) record in the registry zone for this domain name.
     * </p>
     * <p>
     * Note that creating DS record at the registry impacts DNSSEC validation of your DNS records. This action may
     * render your domain name unavailable on the internet if the steps are completed in the wrong order, or with
     * incorrect timing. For more information about DNSSEC signing, see <a
     * href="https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/dns-configuring-dnssec.html">Configuring DNSSEC
     * signing</a> in the <i>Route 53 developer guide</i>.
     * </p>
     *
     * @param associateDelegationSignerToDomainRequest
     * @return Result of the AssociateDelegationSignerToDomain operation returned by the service.
     * @throws DuplicateRequestException
     *         The request is already in progress for the domain.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws TldRulesViolationException
     *         The top-level domain does not support this operation.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws DnssecLimitExceededException
     *         This error is returned if you call <code>AssociateDelegationSignerToDomain</code> when the specified
     *         domain has reached the maximum number of DS records. You can't add any additional DS records unless you
     *         delete an existing one first.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.AssociateDelegationSignerToDomain
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/AssociateDelegationSignerToDomain"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateDelegationSignerToDomainResponse associateDelegationSignerToDomain(
            AssociateDelegationSignerToDomainRequest associateDelegationSignerToDomainRequest) throws DuplicateRequestException,
            InvalidInputException, OperationLimitExceededException, TldRulesViolationException, UnsupportedTldException,
            DnssecLimitExceededException, AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateDelegationSignerToDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateDelegationSignerToDomainResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                associateDelegationSignerToDomainRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateDelegationSignerToDomain");

            return clientHandler
                    .execute(new ClientExecutionParams<AssociateDelegationSignerToDomainRequest, AssociateDelegationSignerToDomainResponse>()
                            .withOperationName("AssociateDelegationSignerToDomain").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(associateDelegationSignerToDomainRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AssociateDelegationSignerToDomainRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Cancels the transfer of a domain from the current Amazon Web Services account to another Amazon Web Services
     * account. You initiate a transfer betweenAmazon Web Services accounts using <a href=
     * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_TransferDomainToAnotherAwsAccount.html"
     * >TransferDomainToAnotherAwsAccount</a>.
     * </p>
     * <important>
     * <p>
     * You must cancel the transfer before the other Amazon Web Services account accepts the transfer using <a href=
     * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_AcceptDomainTransferFromAnotherAwsAccount.html"
     * >AcceptDomainTransferFromAnotherAwsAccount</a>.
     * </p>
     * </important>
     * <p>
     * Use either <a
     * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_ListOperations.html">ListOperations</a>
     * or <a href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_GetOperationDetail.html">
     * GetOperationDetail</a> to determine whether the operation succeeded. <a
     * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_GetOperationDetail.html"
     * >GetOperationDetail</a> provides additional information, for example,
     * <code>Domain Transfer from Aws Account 111122223333 has been cancelled</code>.
     * </p>
     *
     * @param cancelDomainTransferToAnotherAwsAccountRequest
     *        The CancelDomainTransferToAnotherAwsAccount request includes the following element.
     * @return Result of the CancelDomainTransferToAnotherAwsAccount operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.CancelDomainTransferToAnotherAwsAccount
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/CancelDomainTransferToAnotherAwsAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CancelDomainTransferToAnotherAwsAccountResponse cancelDomainTransferToAnotherAwsAccount(
            CancelDomainTransferToAnotherAwsAccountRequest cancelDomainTransferToAnotherAwsAccountRequest)
            throws InvalidInputException, OperationLimitExceededException, UnsupportedTldException, AwsServiceException,
            SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelDomainTransferToAnotherAwsAccountResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, CancelDomainTransferToAnotherAwsAccountResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                cancelDomainTransferToAnotherAwsAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelDomainTransferToAnotherAwsAccount");

            return clientHandler
                    .execute(new ClientExecutionParams<CancelDomainTransferToAnotherAwsAccountRequest, CancelDomainTransferToAnotherAwsAccountResponse>()
                            .withOperationName("CancelDomainTransferToAnotherAwsAccount").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withInput(cancelDomainTransferToAnotherAwsAccountRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CancelDomainTransferToAnotherAwsAccountRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation checks the availability of one domain name. Note that if the availability status of a domain is
     * pending, you must submit another request to determine the availability of the domain name.
     * </p>
     *
     * @param checkDomainAvailabilityRequest
     *        The CheckDomainAvailability request contains the following elements.
     * @return Result of the CheckDomainAvailability operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.CheckDomainAvailability
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/CheckDomainAvailability"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CheckDomainAvailabilityResponse checkDomainAvailability(CheckDomainAvailabilityRequest checkDomainAvailabilityRequest)
            throws InvalidInputException, UnsupportedTldException, AwsServiceException, SdkClientException,
            Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CheckDomainAvailabilityResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CheckDomainAvailabilityResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, checkDomainAvailabilityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CheckDomainAvailability");

            return clientHandler
                    .execute(new ClientExecutionParams<CheckDomainAvailabilityRequest, CheckDomainAvailabilityResponse>()
                            .withOperationName("CheckDomainAvailability").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(checkDomainAvailabilityRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CheckDomainAvailabilityRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Checks whether a domain name can be transferred to Amazon Route 53.
     * </p>
     *
     * @param checkDomainTransferabilityRequest
     *        The CheckDomainTransferability request contains the following elements.
     * @return Result of the CheckDomainTransferability operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.CheckDomainTransferability
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/CheckDomainTransferability"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CheckDomainTransferabilityResponse checkDomainTransferability(
            CheckDomainTransferabilityRequest checkDomainTransferabilityRequest) throws InvalidInputException,
            UnsupportedTldException, AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CheckDomainTransferabilityResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CheckDomainTransferabilityResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, checkDomainTransferabilityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CheckDomainTransferability");

            return clientHandler
                    .execute(new ClientExecutionParams<CheckDomainTransferabilityRequest, CheckDomainTransferabilityResponse>()
                            .withOperationName("CheckDomainTransferability").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(checkDomainTransferabilityRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CheckDomainTransferabilityRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation deletes the specified domain. This action is permanent. For more information, see <a
     * href="https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/domain-delete.html">Deleting a domain name
     * registration</a>.
     * </p>
     * <p>
     * To transfer the domain registration to another registrar, use the transfer process that’s provided by the
     * registrar to which you want to transfer the registration. Otherwise, the following apply:
     * </p>
     * <ol>
     * <li>
     * <p>
     * You can’t get a refund for the cost of a deleted domain registration.
     * </p>
     * </li>
     * <li>
     * <p>
     * The registry for the top-level domain might hold the domain name for a brief time before releasing it for other
     * users to register (varies by registry).
     * </p>
     * </li>
     * <li>
     * <p>
     * When the registration has been deleted, we'll send you a confirmation to the registrant contact. The email will
     * come from <code>noreply@domainnameverification.net</code> or <code>noreply@registrar.amazon.com</code>.
     * </p>
     * </li>
     * </ol>
     *
     * @param deleteDomainRequest
     * @return Result of the DeleteDomain operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws DuplicateRequestException
     *         The request is already in progress for the domain.
     * @throws TldRulesViolationException
     *         The top-level domain does not support this operation.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.DeleteDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/DeleteDomain" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteDomainResponse deleteDomain(DeleteDomainRequest deleteDomainRequest) throws InvalidInputException,
            DuplicateRequestException, TldRulesViolationException, UnsupportedTldException, AwsServiceException,
            SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteDomainResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDomain");

            return clientHandler.execute(new ClientExecutionParams<DeleteDomainRequest, DeleteDomainResponse>()
                    .withOperationName("DeleteDomain").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteDomainRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteDomainRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation deletes the specified tags for a domain.
     * </p>
     * <p>
     * All tag operations are eventually consistent; subsequent operations might not immediately represent all issued
     * operations.
     * </p>
     *
     * @param deleteTagsForDomainRequest
     *        The DeleteTagsForDomainRequest includes the following elements.
     * @return Result of the DeleteTagsForDomain operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.DeleteTagsForDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/DeleteTagsForDomain"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteTagsForDomainResponse deleteTagsForDomain(DeleteTagsForDomainRequest deleteTagsForDomainRequest)
            throws InvalidInputException, OperationLimitExceededException, UnsupportedTldException, AwsServiceException,
            SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteTagsForDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteTagsForDomainResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTagsForDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTagsForDomain");

            return clientHandler.execute(new ClientExecutionParams<DeleteTagsForDomainRequest, DeleteTagsForDomainResponse>()
                    .withOperationName("DeleteTagsForDomain").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteTagsForDomainRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteTagsForDomainRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation disables automatic renewal of domain registration for the specified domain.
     * </p>
     *
     * @param disableDomainAutoRenewRequest
     * @return Result of the DisableDomainAutoRenew operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.DisableDomainAutoRenew
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/DisableDomainAutoRenew"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisableDomainAutoRenewResponse disableDomainAutoRenew(DisableDomainAutoRenewRequest disableDomainAutoRenewRequest)
            throws InvalidInputException, UnsupportedTldException, AwsServiceException, SdkClientException,
            Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisableDomainAutoRenewResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisableDomainAutoRenewResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disableDomainAutoRenewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableDomainAutoRenew");

            return clientHandler
                    .execute(new ClientExecutionParams<DisableDomainAutoRenewRequest, DisableDomainAutoRenewResponse>()
                            .withOperationName("DisableDomainAutoRenew").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(disableDomainAutoRenewRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisableDomainAutoRenewRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation removes the transfer lock on the domain (specifically the <code>clientTransferProhibited</code>
     * status) to allow domain transfers. We recommend you refrain from performing this action unless you intend to
     * transfer the domain to a different registrar. Successful submission returns an operation ID that you can use to
     * track the progress and completion of the action. If the request is not completed successfully, the domain
     * registrant will be notified by email.
     * </p>
     *
     * @param disableDomainTransferLockRequest
     *        The DisableDomainTransferLock request includes the following element.
     * @return Result of the DisableDomainTransferLock operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws DuplicateRequestException
     *         The request is already in progress for the domain.
     * @throws TldRulesViolationException
     *         The top-level domain does not support this operation.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.DisableDomainTransferLock
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/DisableDomainTransferLock"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisableDomainTransferLockResponse disableDomainTransferLock(
            DisableDomainTransferLockRequest disableDomainTransferLockRequest) throws InvalidInputException,
            DuplicateRequestException, TldRulesViolationException, OperationLimitExceededException, UnsupportedTldException,
            AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisableDomainTransferLockResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisableDomainTransferLockResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disableDomainTransferLockRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableDomainTransferLock");

            return clientHandler
                    .execute(new ClientExecutionParams<DisableDomainTransferLockRequest, DisableDomainTransferLockResponse>()
                            .withOperationName("DisableDomainTransferLock").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(disableDomainTransferLockRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisableDomainTransferLockRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a delegation signer (DS) record in the registry zone for this domain name.
     * </p>
     *
     * @param disassociateDelegationSignerFromDomainRequest
     * @return Result of the DisassociateDelegationSignerFromDomain operation returned by the service.
     * @throws DuplicateRequestException
     *         The request is already in progress for the domain.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws TldRulesViolationException
     *         The top-level domain does not support this operation.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.DisassociateDelegationSignerFromDomain
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/DisassociateDelegationSignerFromDomain"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateDelegationSignerFromDomainResponse disassociateDelegationSignerFromDomain(
            DisassociateDelegationSignerFromDomainRequest disassociateDelegationSignerFromDomainRequest)
            throws DuplicateRequestException, InvalidInputException, OperationLimitExceededException, TldRulesViolationException,
            UnsupportedTldException, AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateDelegationSignerFromDomainResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, DisassociateDelegationSignerFromDomainResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateDelegationSignerFromDomainRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateDelegationSignerFromDomain");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateDelegationSignerFromDomainRequest, DisassociateDelegationSignerFromDomainResponse>()
                            .withOperationName("DisassociateDelegationSignerFromDomain").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withInput(disassociateDelegationSignerFromDomainRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateDelegationSignerFromDomainRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation configures Amazon Route 53 to automatically renew the specified domain before the domain
     * registration expires. The cost of renewing your domain registration is billed to your Amazon Web Services
     * account.
     * </p>
     * <p>
     * The period during which you can renew a domain name varies by TLD. For a list of TLDs and their renewal policies,
     * see <a href="https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/registrar-tld-list.html">Domains That You
     * Can Register with Amazon Route 53</a> in the <i>Amazon Route 53 Developer Guide</i>. Route 53 requires that you
     * renew before the end of the renewal period so we can complete processing before the deadline.
     * </p>
     *
     * @param enableDomainAutoRenewRequest
     * @return Result of the EnableDomainAutoRenew operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws TldRulesViolationException
     *         The top-level domain does not support this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.EnableDomainAutoRenew
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/EnableDomainAutoRenew"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public EnableDomainAutoRenewResponse enableDomainAutoRenew(EnableDomainAutoRenewRequest enableDomainAutoRenewRequest)
            throws InvalidInputException, UnsupportedTldException, TldRulesViolationException, AwsServiceException,
            SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<EnableDomainAutoRenewResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, EnableDomainAutoRenewResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enableDomainAutoRenewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableDomainAutoRenew");

            return clientHandler.execute(new ClientExecutionParams<EnableDomainAutoRenewRequest, EnableDomainAutoRenewResponse>()
                    .withOperationName("EnableDomainAutoRenew").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(enableDomainAutoRenewRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new EnableDomainAutoRenewRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation sets the transfer lock on the domain (specifically the <code>clientTransferProhibited</code>
     * status) to prevent domain transfers. Successful submission returns an operation ID that you can use to track the
     * progress and completion of the action. If the request is not completed successfully, the domain registrant will
     * be notified by email.
     * </p>
     *
     * @param enableDomainTransferLockRequest
     *        A request to set the transfer lock for the specified domain.
     * @return Result of the EnableDomainTransferLock operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws DuplicateRequestException
     *         The request is already in progress for the domain.
     * @throws TldRulesViolationException
     *         The top-level domain does not support this operation.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.EnableDomainTransferLock
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/EnableDomainTransferLock"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public EnableDomainTransferLockResponse enableDomainTransferLock(
            EnableDomainTransferLockRequest enableDomainTransferLockRequest) throws InvalidInputException,
            DuplicateRequestException, TldRulesViolationException, OperationLimitExceededException, UnsupportedTldException,
            AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<EnableDomainTransferLockResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, EnableDomainTransferLockResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enableDomainTransferLockRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableDomainTransferLock");

            return clientHandler
                    .execute(new ClientExecutionParams<EnableDomainTransferLockRequest, EnableDomainTransferLockResponse>()
                            .withOperationName("EnableDomainTransferLock").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(enableDomainTransferLockRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new EnableDomainTransferLockRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * For operations that require confirmation that the email address for the registrant contact is valid, such as
     * registering a new domain, this operation returns information about whether the registrant contact has responded.
     * </p>
     * <p>
     * If you want us to resend the email, use the <code>ResendContactReachabilityEmail</code> operation.
     * </p>
     *
     * @param getContactReachabilityStatusRequest
     * @return Result of the GetContactReachabilityStatus operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.GetContactReachabilityStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/GetContactReachabilityStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetContactReachabilityStatusResponse getContactReachabilityStatus(
            GetContactReachabilityStatusRequest getContactReachabilityStatusRequest) throws InvalidInputException,
            OperationLimitExceededException, UnsupportedTldException, AwsServiceException, SdkClientException,
            Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetContactReachabilityStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetContactReachabilityStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getContactReachabilityStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetContactReachabilityStatus");

            return clientHandler
                    .execute(new ClientExecutionParams<GetContactReachabilityStatusRequest, GetContactReachabilityStatusResponse>()
                            .withOperationName("GetContactReachabilityStatus").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getContactReachabilityStatusRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetContactReachabilityStatusRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation returns detailed information about a specified domain that is associated with the current Amazon
     * Web Services account. Contact information for the domain is also returned as part of the output.
     * </p>
     *
     * @param getDomainDetailRequest
     *        The GetDomainDetail request includes the following element.
     * @return Result of the GetDomainDetail operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.GetDomainDetail
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/GetDomainDetail"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetDomainDetailResponse getDomainDetail(GetDomainDetailRequest getDomainDetailRequest) throws InvalidInputException,
            UnsupportedTldException, AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDomainDetailResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetDomainDetailResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDomainDetailRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDomainDetail");

            return clientHandler.execute(new ClientExecutionParams<GetDomainDetailRequest, GetDomainDetailResponse>()
                    .withOperationName("GetDomainDetail").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getDomainDetailRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetDomainDetailRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The GetDomainSuggestions operation returns a list of suggested domain names.
     * </p>
     *
     * @param getDomainSuggestionsRequest
     * @return Result of the GetDomainSuggestions operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.GetDomainSuggestions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/GetDomainSuggestions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetDomainSuggestionsResponse getDomainSuggestions(GetDomainSuggestionsRequest getDomainSuggestionsRequest)
            throws InvalidInputException, UnsupportedTldException, AwsServiceException, SdkClientException,
            Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDomainSuggestionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetDomainSuggestionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDomainSuggestionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDomainSuggestions");

            return clientHandler.execute(new ClientExecutionParams<GetDomainSuggestionsRequest, GetDomainSuggestionsResponse>()
                    .withOperationName("GetDomainSuggestions").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getDomainSuggestionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetDomainSuggestionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation returns the current status of an operation that is not completed.
     * </p>
     *
     * @param getOperationDetailRequest
     *        The <a href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_GetOperationDetail.html">
     *        GetOperationDetail</a> request includes the following element.
     * @return Result of the GetOperationDetail operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.GetOperationDetail
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/GetOperationDetail"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetOperationDetailResponse getOperationDetail(GetOperationDetailRequest getOperationDetailRequest)
            throws InvalidInputException, AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetOperationDetailResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetOperationDetailResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getOperationDetailRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetOperationDetail");

            return clientHandler.execute(new ClientExecutionParams<GetOperationDetailRequest, GetOperationDetailResponse>()
                    .withOperationName("GetOperationDetail").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getOperationDetailRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetOperationDetailRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation returns all the domain names registered with Amazon Route 53 for the current Amazon Web Services
     * account if no filtering conditions are used.
     * </p>
     *
     * @param listDomainsRequest
     *        The ListDomains request includes the following elements.
     * @return Result of the ListDomains operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.ListDomains
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/ListDomains" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListDomainsResponse listDomains(ListDomainsRequest listDomainsRequest) throws InvalidInputException,
            AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDomainsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListDomainsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDomainsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDomains");

            return clientHandler.execute(new ClientExecutionParams<ListDomainsRequest, ListDomainsResponse>()
                    .withOperationName("ListDomains").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listDomainsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDomainsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about all of the operations that return an operation ID and that have ever been performed on
     * domains that were registered by the current account.
     * </p>
     * <p>
     * This command runs only in the us-east-1 Region.
     * </p>
     *
     * @param listOperationsRequest
     *        The ListOperations request includes the following elements.
     * @return Result of the ListOperations operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.ListOperations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/ListOperations" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListOperationsResponse listOperations(ListOperationsRequest listOperationsRequest) throws InvalidInputException,
            AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListOperationsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListOperationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listOperationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOperations");

            return clientHandler.execute(new ClientExecutionParams<ListOperationsRequest, ListOperationsResponse>()
                    .withOperationName("ListOperations").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listOperationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListOperationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the following prices for either all the TLDs supported by Route 53, or the specified TLD:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Registration
     * </p>
     * </li>
     * <li>
     * <p>
     * Transfer
     * </p>
     * </li>
     * <li>
     * <p>
     * Owner change
     * </p>
     * </li>
     * <li>
     * <p>
     * Domain renewal
     * </p>
     * </li>
     * <li>
     * <p>
     * Domain restoration
     * </p>
     * </li>
     * </ul>
     *
     * @param listPricesRequest
     * @return Result of the ListPrices operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.ListPrices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/ListPrices" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListPricesResponse listPrices(ListPricesRequest listPricesRequest) throws InvalidInputException,
            UnsupportedTldException, AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPricesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListPricesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPricesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPrices");

            return clientHandler
                    .execute(new ClientExecutionParams<ListPricesRequest, ListPricesResponse>().withOperationName("ListPrices")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listPricesRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListPricesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation returns all of the tags that are associated with the specified domain.
     * </p>
     * <p>
     * All tag operations are eventually consistent; subsequent operations might not immediately represent all issued
     * operations.
     * </p>
     *
     * @param listTagsForDomainRequest
     *        The ListTagsForDomainRequest includes the following elements.
     * @return Result of the ListTagsForDomain operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.ListTagsForDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/ListTagsForDomain"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForDomainResponse listTagsForDomain(ListTagsForDomainRequest listTagsForDomainRequest)
            throws InvalidInputException, OperationLimitExceededException, UnsupportedTldException, AwsServiceException,
            SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTagsForDomainResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForDomain");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForDomainRequest, ListTagsForDomainResponse>()
                    .withOperationName("ListTagsForDomain").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForDomainRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForDomainRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Moves a domain from Amazon Web Services to another registrar.
     * </p>
     * <p>
     * Supported actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Changes the IPS tags of a .uk domain, and pushes it to transit. Transit means that the domain is ready to be
     * transferred to another registrar.
     * </p>
     * </li>
     * </ul>
     *
     * @param pushDomainRequest
     * @return Result of the PushDomain operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.PushDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/PushDomain" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PushDomainResponse pushDomain(PushDomainRequest pushDomainRequest) throws InvalidInputException,
            OperationLimitExceededException, UnsupportedTldException, AwsServiceException, SdkClientException,
            Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PushDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PushDomainResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, pushDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PushDomain");

            return clientHandler
                    .execute(new ClientExecutionParams<PushDomainRequest, PushDomainResponse>().withOperationName("PushDomain")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(pushDomainRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PushDomainRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation registers a domain. For some top-level domains (TLDs), this operation requires extra parameters.
     * </p>
     * <p>
     * When you register a domain, Amazon Route 53 does the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a Route 53 hosted zone that has the same name as the domain. Route 53 assigns four name servers to your
     * hosted zone and automatically updates your domain registration with the names of these name servers.
     * </p>
     * </li>
     * <li>
     * <p>
     * Enables auto renew, so your domain registration will renew automatically each year. We'll notify you in advance
     * of the renewal date so you can choose whether to renew the registration.
     * </p>
     * </li>
     * <li>
     * <p>
     * Optionally enables privacy protection, so WHOIS queries return contact for the registrar or the phrase
     * "REDACTED FOR PRIVACY", or "On behalf of &lt;domain name&gt; owner." If you don't enable privacy protection,
     * WHOIS queries return the information that you entered for the administrative, registrant, and technical contacts.
     * </p>
     * <note>
     * <p>
     * While some domains may allow different privacy settings per contact, we recommend specifying the same privacy
     * setting for all contacts.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * If registration is successful, returns an operation ID that you can use to track the progress and completion of
     * the action. If the request is not completed successfully, the domain registrant is notified by email.
     * </p>
     * </li>
     * <li>
     * <p>
     * Charges your Amazon Web Services account an amount based on the top-level domain. For more information, see <a
     * href="http://aws.amazon.com/route53/pricing/">Amazon Route 53 Pricing</a>.
     * </p>
     * </li>
     * </ul>
     *
     * @param registerDomainRequest
     *        The RegisterDomain request includes the following elements.
     * @return Result of the RegisterDomain operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws DuplicateRequestException
     *         The request is already in progress for the domain.
     * @throws TldRulesViolationException
     *         The top-level domain does not support this operation.
     * @throws DomainLimitExceededException
     *         The number of domains has exceeded the allowed threshold for the account.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.RegisterDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/RegisterDomain" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public RegisterDomainResponse registerDomain(RegisterDomainRequest registerDomainRequest) throws InvalidInputException,
            UnsupportedTldException, DuplicateRequestException, TldRulesViolationException, DomainLimitExceededException,
            OperationLimitExceededException, AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RegisterDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RegisterDomainResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterDomain");

            return clientHandler.execute(new ClientExecutionParams<RegisterDomainRequest, RegisterDomainResponse>()
                    .withOperationName("RegisterDomain").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(registerDomainRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RegisterDomainRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Rejects the transfer of a domain from another Amazon Web Services account to the current Amazon Web Services
     * account. You initiate a transfer betweenAmazon Web Services accounts using <a href=
     * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_TransferDomainToAnotherAwsAccount.html"
     * >TransferDomainToAnotherAwsAccount</a>.
     * </p>
     * <p>
     * Use either <a
     * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_ListOperations.html">ListOperations</a>
     * or <a href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_GetOperationDetail.html">
     * GetOperationDetail</a> to determine whether the operation succeeded. <a
     * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_GetOperationDetail.html"
     * >GetOperationDetail</a> provides additional information, for example,
     * <code>Domain Transfer from Aws Account 111122223333 has been cancelled</code>.
     * </p>
     *
     * @param rejectDomainTransferFromAnotherAwsAccountRequest
     *        The RejectDomainTransferFromAnotherAwsAccount request includes the following element.
     * @return Result of the RejectDomainTransferFromAnotherAwsAccount operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.RejectDomainTransferFromAnotherAwsAccount
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/RejectDomainTransferFromAnotherAwsAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RejectDomainTransferFromAnotherAwsAccountResponse rejectDomainTransferFromAnotherAwsAccount(
            RejectDomainTransferFromAnotherAwsAccountRequest rejectDomainTransferFromAnotherAwsAccountRequest)
            throws InvalidInputException, OperationLimitExceededException, UnsupportedTldException, AwsServiceException,
            SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RejectDomainTransferFromAnotherAwsAccountResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, RejectDomainTransferFromAnotherAwsAccountResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                rejectDomainTransferFromAnotherAwsAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RejectDomainTransferFromAnotherAwsAccount");

            return clientHandler
                    .execute(new ClientExecutionParams<RejectDomainTransferFromAnotherAwsAccountRequest, RejectDomainTransferFromAnotherAwsAccountResponse>()
                            .withOperationName("RejectDomainTransferFromAnotherAwsAccount").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withInput(rejectDomainTransferFromAnotherAwsAccountRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RejectDomainTransferFromAnotherAwsAccountRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation renews a domain for the specified number of years. The cost of renewing your domain is billed to
     * your Amazon Web Services account.
     * </p>
     * <p>
     * We recommend that you renew your domain several weeks before the expiration date. Some TLD registries delete
     * domains before the expiration date if you haven't renewed far enough in advance. For more information about
     * renewing domain registration, see <a
     * href="https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/domain-renew.html">Renewing Registration for a
     * Domain</a> in the <i>Amazon Route 53 Developer Guide</i>.
     * </p>
     *
     * @param renewDomainRequest
     *        A <code>RenewDomain</code> request includes the number of years that you want to renew for and the current
     *        expiration year.
     * @return Result of the RenewDomain operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws DuplicateRequestException
     *         The request is already in progress for the domain.
     * @throws TldRulesViolationException
     *         The top-level domain does not support this operation.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.RenewDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/RenewDomain" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public RenewDomainResponse renewDomain(RenewDomainRequest renewDomainRequest) throws InvalidInputException,
            UnsupportedTldException, DuplicateRequestException, TldRulesViolationException, OperationLimitExceededException,
            AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RenewDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RenewDomainResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, renewDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RenewDomain");

            return clientHandler.execute(new ClientExecutionParams<RenewDomainRequest, RenewDomainResponse>()
                    .withOperationName("RenewDomain").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(renewDomainRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RenewDomainRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * For operations that require confirmation that the email address for the registrant contact is valid, such as
     * registering a new domain, this operation resends the confirmation email to the current email address for the
     * registrant contact.
     * </p>
     *
     * @param resendContactReachabilityEmailRequest
     * @return Result of the ResendContactReachabilityEmail operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.ResendContactReachabilityEmail
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/ResendContactReachabilityEmail"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ResendContactReachabilityEmailResponse resendContactReachabilityEmail(
            ResendContactReachabilityEmailRequest resendContactReachabilityEmailRequest) throws InvalidInputException,
            OperationLimitExceededException, UnsupportedTldException, AwsServiceException, SdkClientException,
            Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ResendContactReachabilityEmailResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ResendContactReachabilityEmailResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                resendContactReachabilityEmailRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResendContactReachabilityEmail");

            return clientHandler
                    .execute(new ClientExecutionParams<ResendContactReachabilityEmailRequest, ResendContactReachabilityEmailResponse>()
                            .withOperationName("ResendContactReachabilityEmail").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(resendContactReachabilityEmailRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ResendContactReachabilityEmailRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Resend the form of authorization email for this operation.
     * </p>
     *
     * @param resendOperationAuthorizationRequest
     * @return Result of the ResendOperationAuthorization operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.ResendOperationAuthorization
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/ResendOperationAuthorization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ResendOperationAuthorizationResponse resendOperationAuthorization(
            ResendOperationAuthorizationRequest resendOperationAuthorizationRequest) throws InvalidInputException,
            AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ResendOperationAuthorizationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ResendOperationAuthorizationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, resendOperationAuthorizationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResendOperationAuthorization");

            return clientHandler
                    .execute(new ClientExecutionParams<ResendOperationAuthorizationRequest, ResendOperationAuthorizationResponse>()
                            .withOperationName("ResendOperationAuthorization").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(resendOperationAuthorizationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ResendOperationAuthorizationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation returns the authorization code for the domain. To transfer a domain to another registrar, you
     * provide this value to the new registrar.
     * </p>
     *
     * @param retrieveDomainAuthCodeRequest
     *        A request for the authorization code for the specified domain. To transfer a domain to another registrar,
     *        you provide this value to the new registrar.
     * @return Result of the RetrieveDomainAuthCode operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.RetrieveDomainAuthCode
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/RetrieveDomainAuthCode"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RetrieveDomainAuthCodeResponse retrieveDomainAuthCode(RetrieveDomainAuthCodeRequest retrieveDomainAuthCodeRequest)
            throws InvalidInputException, UnsupportedTldException, AwsServiceException, SdkClientException,
            Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RetrieveDomainAuthCodeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RetrieveDomainAuthCodeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, retrieveDomainAuthCodeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RetrieveDomainAuthCode");

            return clientHandler
                    .execute(new ClientExecutionParams<RetrieveDomainAuthCodeRequest, RetrieveDomainAuthCodeResponse>()
                            .withOperationName("RetrieveDomainAuthCode").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(retrieveDomainAuthCodeRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RetrieveDomainAuthCodeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Transfers a domain from another registrar to Amazon Route 53.
     * </p>
     * <p>
     * For more information about transferring domains, see the following topics:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For transfer requirements, a detailed procedure, and information about viewing the status of a domain that you're
     * transferring to Route 53, see <a
     * href="https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/domain-transfer-to-route-53.html">Transferring
     * Registration for a Domain to Amazon Route 53</a> in the <i>Amazon Route 53 Developer Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For information about how to transfer a domain from one Amazon Web Services account to another, see <a href=
     * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_TransferDomainToAnotherAwsAccount.html"
     * >TransferDomainToAnotherAwsAccount</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For information about how to transfer a domain to another domain registrar, see <a
     * href="https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/domain-transfer-from-route-53.html">Transferring
     * a Domain from Amazon Route 53 to Another Registrar</a> in the <i>Amazon Route 53 Developer Guide</i>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the registrar for your domain is also the DNS service provider for the domain, we highly recommend that you
     * transfer your DNS service to Route 53 or to another DNS service provider before you transfer your registration.
     * Some registrars provide free DNS service when you purchase a domain registration. When you transfer the
     * registration, the previous registrar will not renew your domain registration and could end your DNS service at
     * any time.
     * </p>
     * <important>
     * <p>
     * If the registrar for your domain is also the DNS service provider for the domain and you don't transfer DNS
     * service to another provider, your website, email, and the web applications associated with the domain might
     * become unavailable.
     * </p>
     * </important>
     * <p>
     * If the transfer is successful, this method returns an operation ID that you can use to track the progress and
     * completion of the action. If the transfer doesn't complete successfully, the domain registrant will be notified
     * by email.
     * </p>
     *
     * @param transferDomainRequest
     *        The TransferDomain request includes the following elements.
     * @return Result of the TransferDomain operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws DuplicateRequestException
     *         The request is already in progress for the domain.
     * @throws TldRulesViolationException
     *         The top-level domain does not support this operation.
     * @throws DomainLimitExceededException
     *         The number of domains has exceeded the allowed threshold for the account.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.TransferDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/TransferDomain" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TransferDomainResponse transferDomain(TransferDomainRequest transferDomainRequest) throws InvalidInputException,
            UnsupportedTldException, DuplicateRequestException, TldRulesViolationException, DomainLimitExceededException,
            OperationLimitExceededException, AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TransferDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TransferDomainResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, transferDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TransferDomain");

            return clientHandler.execute(new ClientExecutionParams<TransferDomainRequest, TransferDomainResponse>()
                    .withOperationName("TransferDomain").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(transferDomainRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TransferDomainRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Transfers a domain from the current Amazon Web Services account to another Amazon Web Services account. Note the
     * following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The Amazon Web Services account that you're transferring the domain to must accept the transfer. If the other
     * account doesn't accept the transfer within 3 days, we cancel the transfer. See <a href=
     * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_AcceptDomainTransferFromAnotherAwsAccount.html"
     * >AcceptDomainTransferFromAnotherAwsAccount</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * You can cancel the transfer before the other account accepts it. See <a href=
     * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_CancelDomainTransferToAnotherAwsAccount.html"
     * >CancelDomainTransferToAnotherAwsAccount</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * The other account can reject the transfer. See <a href=
     * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_RejectDomainTransferFromAnotherAwsAccount.html"
     * >RejectDomainTransferFromAnotherAwsAccount</a>.
     * </p>
     * </li>
     * </ul>
     * <important>
     * <p>
     * When you transfer a domain from one Amazon Web Services account to another, Route 53 doesn't transfer the hosted
     * zone that is associated with the domain. DNS resolution isn't affected if the domain and the hosted zone are
     * owned by separate accounts, so transferring the hosted zone is optional. For information about transferring the
     * hosted zone to another Amazon Web Services account, see <a
     * href="https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/hosted-zones-migrating.html">Migrating a Hosted
     * Zone to a Different Amazon Web Services Account</a> in the <i>Amazon Route 53 Developer Guide</i>.
     * </p>
     * </important>
     * <p>
     * Use either <a
     * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_ListOperations.html">ListOperations</a>
     * or <a href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_GetOperationDetail.html">
     * GetOperationDetail</a> to determine whether the operation succeeded. <a
     * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_GetOperationDetail.html"
     * >GetOperationDetail</a> provides additional information, for example,
     * <code>Domain Transfer from Aws Account 111122223333 has been cancelled</code>.
     * </p>
     *
     * @param transferDomainToAnotherAwsAccountRequest
     *        The TransferDomainToAnotherAwsAccount request includes the following elements.
     * @return Result of the TransferDomainToAnotherAwsAccount operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws DuplicateRequestException
     *         The request is already in progress for the domain.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.TransferDomainToAnotherAwsAccount
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/TransferDomainToAnotherAwsAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public TransferDomainToAnotherAwsAccountResponse transferDomainToAnotherAwsAccount(
            TransferDomainToAnotherAwsAccountRequest transferDomainToAnotherAwsAccountRequest) throws InvalidInputException,
            OperationLimitExceededException, DuplicateRequestException, UnsupportedTldException, AwsServiceException,
            SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TransferDomainToAnotherAwsAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, TransferDomainToAnotherAwsAccountResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                transferDomainToAnotherAwsAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TransferDomainToAnotherAwsAccount");

            return clientHandler
                    .execute(new ClientExecutionParams<TransferDomainToAnotherAwsAccountRequest, TransferDomainToAnotherAwsAccountResponse>()
                            .withOperationName("TransferDomainToAnotherAwsAccount").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(transferDomainToAnotherAwsAccountRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new TransferDomainToAnotherAwsAccountRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation updates the contact information for a particular domain. You must specify information for at least
     * one contact: registrant, administrator, or technical.
     * </p>
     * <p>
     * If the update is successful, this method returns an operation ID that you can use to track the progress and
     * completion of the operation. If the request is not completed successfully, the domain registrant will be notified
     * by email.
     * </p>
     *
     * @param updateDomainContactRequest
     *        The UpdateDomainContact request includes the following elements.
     * @return Result of the UpdateDomainContact operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws DuplicateRequestException
     *         The request is already in progress for the domain.
     * @throws TldRulesViolationException
     *         The top-level domain does not support this operation.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.UpdateDomainContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/UpdateDomainContact"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateDomainContactResponse updateDomainContact(UpdateDomainContactRequest updateDomainContactRequest)
            throws InvalidInputException, DuplicateRequestException, TldRulesViolationException, OperationLimitExceededException,
            UnsupportedTldException, AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDomainContactResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateDomainContactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDomainContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDomainContact");

            return clientHandler.execute(new ClientExecutionParams<UpdateDomainContactRequest, UpdateDomainContactResponse>()
                    .withOperationName("UpdateDomainContact").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateDomainContactRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateDomainContactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation updates the specified domain contact's privacy setting. When privacy protection is enabled, your
     * contact information is replaced with contact information for the registrar or with the phrase
     * "REDACTED FOR PRIVACY", or "On behalf of &lt;domain name&gt; owner."
     * </p>
     * <note>
     * <p>
     * While some domains may allow different privacy settings per contact, we recommend specifying the same privacy
     * setting for all contacts.
     * </p>
     * </note>
     * <p>
     * This operation affects only the contact information for the specified contact type (administrative, registrant,
     * or technical). If the request succeeds, Amazon Route 53 returns an operation ID that you can use with <a
     * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_GetOperationDetail.html"
     * >GetOperationDetail</a> to track the progress and completion of the action. If the request doesn't complete
     * successfully, the domain registrant will be notified by email.
     * </p>
     * <important>
     * <p>
     * By disabling the privacy service via API, you consent to the publication of the contact information provided for
     * this domain via the public WHOIS database. You certify that you are the registrant of this domain name and have
     * the authority to make this decision. You may withdraw your consent at any time by enabling privacy protection
     * using either <code>UpdateDomainContactPrivacy</code> or the Route 53 console. Enabling privacy protection removes
     * the contact information provided for this domain from the WHOIS database. For more information on our privacy
     * practices, see <a href="https://aws.amazon.com/privacy/">https://aws.amazon.com/privacy/</a>.
     * </p>
     * </important>
     *
     * @param updateDomainContactPrivacyRequest
     *        The UpdateDomainContactPrivacy request includes the following elements.
     * @return Result of the UpdateDomainContactPrivacy operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws DuplicateRequestException
     *         The request is already in progress for the domain.
     * @throws TldRulesViolationException
     *         The top-level domain does not support this operation.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.UpdateDomainContactPrivacy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/UpdateDomainContactPrivacy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateDomainContactPrivacyResponse updateDomainContactPrivacy(
            UpdateDomainContactPrivacyRequest updateDomainContactPrivacyRequest) throws InvalidInputException,
            DuplicateRequestException, TldRulesViolationException, OperationLimitExceededException, UnsupportedTldException,
            AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDomainContactPrivacyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateDomainContactPrivacyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDomainContactPrivacyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDomainContactPrivacy");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateDomainContactPrivacyRequest, UpdateDomainContactPrivacyResponse>()
                            .withOperationName("UpdateDomainContactPrivacy").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateDomainContactPrivacyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateDomainContactPrivacyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation replaces the current set of name servers for the domain with the specified set of name servers. If
     * you use Amazon Route 53 as your DNS service, specify the four name servers in the delegation set for the hosted
     * zone for the domain.
     * </p>
     * <p>
     * If successful, this operation returns an operation ID that you can use to track the progress and completion of
     * the action. If the request is not completed successfully, the domain registrant will be notified by email.
     * </p>
     *
     * @param updateDomainNameserversRequest
     *        Replaces the current set of name servers for the domain with the specified set of name servers. If you use
     *        Amazon Route 53 as your DNS service, specify the four name servers in the delegation set for the hosted
     *        zone for the domain.</p>
     *        <p>
     *        If successful, this operation returns an operation ID that you can use to track the progress and
     *        completion of the action. If the request is not completed successfully, the domain registrant will be
     *        notified by email.
     * @return Result of the UpdateDomainNameservers operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws DuplicateRequestException
     *         The request is already in progress for the domain.
     * @throws TldRulesViolationException
     *         The top-level domain does not support this operation.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.UpdateDomainNameservers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/UpdateDomainNameservers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateDomainNameserversResponse updateDomainNameservers(UpdateDomainNameserversRequest updateDomainNameserversRequest)
            throws InvalidInputException, DuplicateRequestException, TldRulesViolationException, OperationLimitExceededException,
            UnsupportedTldException, AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDomainNameserversResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateDomainNameserversResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDomainNameserversRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDomainNameservers");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateDomainNameserversRequest, UpdateDomainNameserversResponse>()
                            .withOperationName("UpdateDomainNameservers").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateDomainNameserversRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateDomainNameserversRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation adds or updates tags for a specified domain.
     * </p>
     * <p>
     * All tag operations are eventually consistent; subsequent operations might not immediately represent all issued
     * operations.
     * </p>
     *
     * @param updateTagsForDomainRequest
     *        The UpdateTagsForDomainRequest includes the following elements.
     * @return Result of the UpdateTagsForDomain operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.UpdateTagsForDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/UpdateTagsForDomain"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateTagsForDomainResponse updateTagsForDomain(UpdateTagsForDomainRequest updateTagsForDomainRequest)
            throws InvalidInputException, OperationLimitExceededException, UnsupportedTldException, AwsServiceException,
            SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateTagsForDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateTagsForDomainResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTagsForDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTagsForDomain");

            return clientHandler.execute(new ClientExecutionParams<UpdateTagsForDomainRequest, UpdateTagsForDomainResponse>()
                    .withOperationName("UpdateTagsForDomain").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateTagsForDomainRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateTagsForDomainRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns all the domain-related billing records for the current Amazon Web Services account for a specified period
     * </p>
     *
     * @param viewBillingRequest
     *        The ViewBilling request includes the following elements.
     * @return Result of the ViewBilling operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for APIs that accept a domain name, the request might
     *         specify a domain name that doesn't belong to the account that submitted the request. For
     *         <code>AcceptDomainTransferFromAnotherAwsAccount</code>, the password might be invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.ViewBilling
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/ViewBilling" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ViewBillingResponse viewBilling(ViewBillingRequest viewBillingRequest) throws InvalidInputException,
            AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ViewBillingResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ViewBillingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, viewBillingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route 53 Domains");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ViewBilling");

            return clientHandler.execute(new ClientExecutionParams<ViewBillingRequest, ViewBillingResponse>()
                    .withOperationName("ViewBilling").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(viewBillingRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ViewBillingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(Route53DomainsException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DuplicateRequest")
                                .exceptionBuilderSupplier(DuplicateRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OperationLimitExceeded")
                                .exceptionBuilderSupplier(OperationLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedTLD")
                                .exceptionBuilderSupplier(UnsupportedTldException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DnssecLimitExceeded")
                                .exceptionBuilderSupplier(DnssecLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInput")
                                .exceptionBuilderSupplier(InvalidInputException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DomainLimitExceeded")
                                .exceptionBuilderSupplier(DomainLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TLDRulesViolation")
                                .exceptionBuilderSupplier(TldRulesViolationException::builder).httpStatusCode(400).build());
    }

    @Override
    public final Route53DomainsServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
