/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pipes;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.pipes.internal.PipesServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.pipes.model.ConflictException;
import software.amazon.awssdk.services.pipes.model.CreatePipeRequest;
import software.amazon.awssdk.services.pipes.model.CreatePipeResponse;
import software.amazon.awssdk.services.pipes.model.DeletePipeRequest;
import software.amazon.awssdk.services.pipes.model.DeletePipeResponse;
import software.amazon.awssdk.services.pipes.model.DescribePipeRequest;
import software.amazon.awssdk.services.pipes.model.DescribePipeResponse;
import software.amazon.awssdk.services.pipes.model.InternalException;
import software.amazon.awssdk.services.pipes.model.ListPipesRequest;
import software.amazon.awssdk.services.pipes.model.ListPipesResponse;
import software.amazon.awssdk.services.pipes.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.pipes.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.pipes.model.NotFoundException;
import software.amazon.awssdk.services.pipes.model.PipesException;
import software.amazon.awssdk.services.pipes.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.pipes.model.StartPipeRequest;
import software.amazon.awssdk.services.pipes.model.StartPipeResponse;
import software.amazon.awssdk.services.pipes.model.StopPipeRequest;
import software.amazon.awssdk.services.pipes.model.StopPipeResponse;
import software.amazon.awssdk.services.pipes.model.TagResourceRequest;
import software.amazon.awssdk.services.pipes.model.TagResourceResponse;
import software.amazon.awssdk.services.pipes.model.ThrottlingException;
import software.amazon.awssdk.services.pipes.model.UntagResourceRequest;
import software.amazon.awssdk.services.pipes.model.UntagResourceResponse;
import software.amazon.awssdk.services.pipes.model.UpdatePipeRequest;
import software.amazon.awssdk.services.pipes.model.UpdatePipeResponse;
import software.amazon.awssdk.services.pipes.model.ValidationException;
import software.amazon.awssdk.services.pipes.transform.CreatePipeRequestMarshaller;
import software.amazon.awssdk.services.pipes.transform.DeletePipeRequestMarshaller;
import software.amazon.awssdk.services.pipes.transform.DescribePipeRequestMarshaller;
import software.amazon.awssdk.services.pipes.transform.ListPipesRequestMarshaller;
import software.amazon.awssdk.services.pipes.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.pipes.transform.StartPipeRequestMarshaller;
import software.amazon.awssdk.services.pipes.transform.StopPipeRequestMarshaller;
import software.amazon.awssdk.services.pipes.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.pipes.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.pipes.transform.UpdatePipeRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link PipesClient}.
 *
 * @see PipesClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultPipesClient implements PipesClient {
    private static final Logger log = Logger.loggerFor(DefaultPipesClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final PipesServiceClientConfiguration serviceClientConfiguration;

    protected DefaultPipesClient(PipesServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Create a pipe. Amazon EventBridge Pipes connect event sources to targets and reduces the need for specialized
     * knowledge and integration code.
     * </p>
     *
     * @param createPipeRequest
     * @return Result of the CreatePipe operation returned by the service.
     * @throws InternalException
     *         This exception occurs due to unexpected causes.
     * @throws ValidationException
     *         Indicates that an error has occurred while performing a validate operation.
     * @throws ThrottlingException
     *         An action was throttled.
     * @throws NotFoundException
     *         An entity that you specified does not exist.
     * @throws ConflictException
     *         An action you attempted resulted in an exception.
     * @throws ServiceQuotaExceededException
     *         A quota has been exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PipesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PipesClient.CreatePipe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/CreatePipe" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreatePipeResponse createPipe(CreatePipeRequest createPipeRequest) throws InternalException, ValidationException,
            ThrottlingException, NotFoundException, ConflictException, ServiceQuotaExceededException, AwsServiceException,
            SdkClientException, PipesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreatePipeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreatePipeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPipeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPipeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pipes");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePipe");

            return clientHandler.execute(new ClientExecutionParams<CreatePipeRequest, CreatePipeResponse>()
                    .withOperationName("CreatePipe").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createPipeRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreatePipeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Delete an existing pipe. For more information about pipes, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes.html">Amazon EventBridge Pipes</a> in the
     * Amazon EventBridge User Guide.
     * </p>
     *
     * @param deletePipeRequest
     * @return Result of the DeletePipe operation returned by the service.
     * @throws InternalException
     *         This exception occurs due to unexpected causes.
     * @throws ValidationException
     *         Indicates that an error has occurred while performing a validate operation.
     * @throws ThrottlingException
     *         An action was throttled.
     * @throws NotFoundException
     *         An entity that you specified does not exist.
     * @throws ConflictException
     *         An action you attempted resulted in an exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PipesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PipesClient.DeletePipe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/DeletePipe" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeletePipeResponse deletePipe(DeletePipeRequest deletePipeRequest) throws InternalException, ValidationException,
            ThrottlingException, NotFoundException, ConflictException, AwsServiceException, SdkClientException, PipesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeletePipeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeletePipeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePipeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePipeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pipes");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePipe");

            return clientHandler.execute(new ClientExecutionParams<DeletePipeRequest, DeletePipeResponse>()
                    .withOperationName("DeletePipe").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deletePipeRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeletePipeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get the information about an existing pipe. For more information about pipes, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes.html">Amazon EventBridge Pipes</a> in the
     * Amazon EventBridge User Guide.
     * </p>
     *
     * @param describePipeRequest
     * @return Result of the DescribePipe operation returned by the service.
     * @throws InternalException
     *         This exception occurs due to unexpected causes.
     * @throws ValidationException
     *         Indicates that an error has occurred while performing a validate operation.
     * @throws ThrottlingException
     *         An action was throttled.
     * @throws NotFoundException
     *         An entity that you specified does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PipesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PipesClient.DescribePipe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/DescribePipe" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribePipeResponse describePipe(DescribePipeRequest describePipeRequest) throws InternalException,
            ValidationException, ThrottlingException, NotFoundException, AwsServiceException, SdkClientException, PipesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribePipeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribePipeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describePipeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describePipeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pipes");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePipe");

            return clientHandler.execute(new ClientExecutionParams<DescribePipeRequest, DescribePipeResponse>()
                    .withOperationName("DescribePipe").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describePipeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribePipeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get the pipes associated with this account. For more information about pipes, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes.html">Amazon EventBridge Pipes</a> in the
     * Amazon EventBridge User Guide.
     * </p>
     *
     * @param listPipesRequest
     * @return Result of the ListPipes operation returned by the service.
     * @throws InternalException
     *         This exception occurs due to unexpected causes.
     * @throws ValidationException
     *         Indicates that an error has occurred while performing a validate operation.
     * @throws ThrottlingException
     *         An action was throttled.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PipesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PipesClient.ListPipes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/ListPipes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListPipesResponse listPipes(ListPipesRequest listPipesRequest) throws InternalException, ValidationException,
            ThrottlingException, AwsServiceException, SdkClientException, PipesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPipesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListPipesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPipesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPipesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pipes");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPipes");

            return clientHandler.execute(new ClientExecutionParams<ListPipesRequest, ListPipesResponse>()
                    .withOperationName("ListPipes").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listPipesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPipesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Displays the tags associated with a pipe.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InternalException
     *         This exception occurs due to unexpected causes.
     * @throws ValidationException
     *         Indicates that an error has occurred while performing a validate operation.
     * @throws NotFoundException
     *         An entity that you specified does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PipesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PipesClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InternalException, ValidationException, NotFoundException, AwsServiceException, SdkClientException,
            PipesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pipes");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Start an existing pipe.
     * </p>
     *
     * @param startPipeRequest
     * @return Result of the StartPipe operation returned by the service.
     * @throws InternalException
     *         This exception occurs due to unexpected causes.
     * @throws ValidationException
     *         Indicates that an error has occurred while performing a validate operation.
     * @throws ThrottlingException
     *         An action was throttled.
     * @throws NotFoundException
     *         An entity that you specified does not exist.
     * @throws ConflictException
     *         An action you attempted resulted in an exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PipesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PipesClient.StartPipe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/StartPipe" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartPipeResponse startPipe(StartPipeRequest startPipeRequest) throws InternalException, ValidationException,
            ThrottlingException, NotFoundException, ConflictException, AwsServiceException, SdkClientException, PipesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartPipeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartPipeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startPipeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startPipeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pipes");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartPipe");

            return clientHandler.execute(new ClientExecutionParams<StartPipeRequest, StartPipeResponse>()
                    .withOperationName("StartPipe").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(startPipeRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartPipeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stop an existing pipe.
     * </p>
     *
     * @param stopPipeRequest
     * @return Result of the StopPipe operation returned by the service.
     * @throws InternalException
     *         This exception occurs due to unexpected causes.
     * @throws ValidationException
     *         Indicates that an error has occurred while performing a validate operation.
     * @throws ThrottlingException
     *         An action was throttled.
     * @throws NotFoundException
     *         An entity that you specified does not exist.
     * @throws ConflictException
     *         An action you attempted resulted in an exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PipesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PipesClient.StopPipe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/StopPipe" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopPipeResponse stopPipe(StopPipeRequest stopPipeRequest) throws InternalException, ValidationException,
            ThrottlingException, NotFoundException, ConflictException, AwsServiceException, SdkClientException, PipesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopPipeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopPipeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopPipeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopPipeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pipes");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopPipe");

            return clientHandler.execute(new ClientExecutionParams<StopPipeRequest, StopPipeResponse>()
                    .withOperationName("StopPipe").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(stopPipeRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopPipeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified pipe. Tags can help you organize and categorize your
     * resources. You can also use them to scope user permissions by granting a user permission to access or change only
     * resources with certain tag values.
     * </p>
     * <p>
     * Tags don't have any semantic meaning to Amazon Web Services and are interpreted strictly as strings of
     * characters.
     * </p>
     * <p>
     * You can use the <code>TagResource</code> action with a pipe that already has tags. If you specify a new tag key,
     * this tag is appended to the list of tags associated with the pipe. If you specify a tag key that is already
     * associated with the pipe, the new tag value that you specify replaces the previous value for that tag.
     * </p>
     * <p>
     * You can associate as many as 50 tags with a pipe.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InternalException
     *         This exception occurs due to unexpected causes.
     * @throws ValidationException
     *         Indicates that an error has occurred while performing a validate operation.
     * @throws NotFoundException
     *         An entity that you specified does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PipesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PipesClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InternalException, ValidationException,
            NotFoundException, AwsServiceException, SdkClientException, PipesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pipes");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes one or more tags from the specified pipes.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InternalException
     *         This exception occurs due to unexpected causes.
     * @throws ValidationException
     *         Indicates that an error has occurred while performing a validate operation.
     * @throws NotFoundException
     *         An entity that you specified does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PipesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PipesClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InternalException,
            ValidationException, NotFoundException, AwsServiceException, SdkClientException, PipesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pipes");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Update an existing pipe. When you call <code>UpdatePipe</code>, only the fields that are included in the request
     * are changed, the rest are unchanged. The exception to this is if you modify any Amazon Web Services-service
     * specific fields in the <code>SourceParameters</code>, <code>EnrichmentParameters</code>, or
     * <code>TargetParameters</code> objects. The fields in these objects are updated atomically as one and override
     * existing values. This is by design and means that if you don't specify an optional field in one of these
     * Parameters objects, that field will be set to its system-default value after the update.
     * </p>
     * <p>
     * For more information about pipes, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes.html"> Amazon EventBridge Pipes</a> in
     * the Amazon EventBridge User Guide.
     * </p>
     *
     * @param updatePipeRequest
     * @return Result of the UpdatePipe operation returned by the service.
     * @throws InternalException
     *         This exception occurs due to unexpected causes.
     * @throws ValidationException
     *         Indicates that an error has occurred while performing a validate operation.
     * @throws ThrottlingException
     *         An action was throttled.
     * @throws NotFoundException
     *         An entity that you specified does not exist.
     * @throws ConflictException
     *         An action you attempted resulted in an exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PipesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PipesClient.UpdatePipe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/UpdatePipe" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdatePipeResponse updatePipe(UpdatePipeRequest updatePipeRequest) throws InternalException, ValidationException,
            ThrottlingException, NotFoundException, ConflictException, AwsServiceException, SdkClientException, PipesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdatePipeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdatePipeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePipeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePipeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pipes");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePipe");

            return clientHandler.execute(new ClientExecutionParams<UpdatePipeRequest, UpdatePipeResponse>()
                    .withOperationName("UpdatePipe").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(updatePipeRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdatePipeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        PipesServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = PipesServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(PipesException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalException")
                                .exceptionBuilderSupplier(InternalException::builder).httpStatusCode(500).build());
    }

    @Override
    public final PipesServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
