/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.paymentcryptographydata.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Parameters requried to encrypt plaintext data using symmetric keys.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SymmetricEncryptionAttributes implements SdkPojo, Serializable,
        ToCopyableBuilder<SymmetricEncryptionAttributes.Builder, SymmetricEncryptionAttributes> {
    private static final SdkField<String> INITIALIZATION_VECTOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InitializationVector").getter(getter(SymmetricEncryptionAttributes::initializationVector))
            .setter(setter(Builder::initializationVector))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InitializationVector").build())
            .build();

    private static final SdkField<String> MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Mode")
            .getter(getter(SymmetricEncryptionAttributes::modeAsString)).setter(setter(Builder::mode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Mode").build()).build();

    private static final SdkField<String> PADDING_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PaddingType").getter(getter(SymmetricEncryptionAttributes::paddingTypeAsString))
            .setter(setter(Builder::paddingType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PaddingType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INITIALIZATION_VECTOR_FIELD,
            MODE_FIELD, PADDING_TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String initializationVector;

    private final String mode;

    private final String paddingType;

    private SymmetricEncryptionAttributes(BuilderImpl builder) {
        this.initializationVector = builder.initializationVector;
        this.mode = builder.mode;
        this.paddingType = builder.paddingType;
    }

    /**
     * <p>
     * An input to cryptographic primitive used to provide the intial state. The <code>InitializationVector</code> is
     * typically required have a random or psuedo-random value, but sometimes it only needs to be unpredictable or
     * unique. If a value is not provided, Amazon Web Services Payment Cryptography generates a random value.
     * </p>
     * 
     * @return An input to cryptographic primitive used to provide the intial state. The
     *         <code>InitializationVector</code> is typically required have a random or psuedo-random value, but
     *         sometimes it only needs to be unpredictable or unique. If a value is not provided, Amazon Web Services
     *         Payment Cryptography generates a random value.
     */
    public final String initializationVector() {
        return initializationVector;
    }

    /**
     * <p>
     * The block cipher mode of operation. Block ciphers are designed to encrypt a block of data of fixed size (for
     * example, 128 bits). The size of the input block is usually same as the size of the encrypted output block, while
     * the key length can be different. A mode of operation describes how to repeatedly apply a cipher's single-block
     * operation to securely transform amounts of data larger than a block.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mode} will return
     * {@link EncryptionMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #modeAsString}.
     * </p>
     * 
     * @return The block cipher mode of operation. Block ciphers are designed to encrypt a block of data of fixed size
     *         (for example, 128 bits). The size of the input block is usually same as the size of the encrypted output
     *         block, while the key length can be different. A mode of operation describes how to repeatedly apply a
     *         cipher's single-block operation to securely transform amounts of data larger than a block.
     * @see EncryptionMode
     */
    public final EncryptionMode mode() {
        return EncryptionMode.fromValue(mode);
    }

    /**
     * <p>
     * The block cipher mode of operation. Block ciphers are designed to encrypt a block of data of fixed size (for
     * example, 128 bits). The size of the input block is usually same as the size of the encrypted output block, while
     * the key length can be different. A mode of operation describes how to repeatedly apply a cipher's single-block
     * operation to securely transform amounts of data larger than a block.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mode} will return
     * {@link EncryptionMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #modeAsString}.
     * </p>
     * 
     * @return The block cipher mode of operation. Block ciphers are designed to encrypt a block of data of fixed size
     *         (for example, 128 bits). The size of the input block is usually same as the size of the encrypted output
     *         block, while the key length can be different. A mode of operation describes how to repeatedly apply a
     *         cipher's single-block operation to securely transform amounts of data larger than a block.
     * @see EncryptionMode
     */
    public final String modeAsString() {
        return mode;
    }

    /**
     * <p>
     * The padding to be included with the data.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #paddingType} will
     * return {@link PaddingType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #paddingTypeAsString}.
     * </p>
     * 
     * @return The padding to be included with the data.
     * @see PaddingType
     */
    public final PaddingType paddingType() {
        return PaddingType.fromValue(paddingType);
    }

    /**
     * <p>
     * The padding to be included with the data.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #paddingType} will
     * return {@link PaddingType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #paddingTypeAsString}.
     * </p>
     * 
     * @return The padding to be included with the data.
     * @see PaddingType
     */
    public final String paddingTypeAsString() {
        return paddingType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(initializationVector());
        hashCode = 31 * hashCode + Objects.hashCode(modeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(paddingTypeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SymmetricEncryptionAttributes)) {
            return false;
        }
        SymmetricEncryptionAttributes other = (SymmetricEncryptionAttributes) obj;
        return Objects.equals(initializationVector(), other.initializationVector())
                && Objects.equals(modeAsString(), other.modeAsString())
                && Objects.equals(paddingTypeAsString(), other.paddingTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SymmetricEncryptionAttributes")
                .add("InitializationVector", initializationVector() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Mode", modeAsString()).add("PaddingType", paddingTypeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "InitializationVector":
            return Optional.ofNullable(clazz.cast(initializationVector()));
        case "Mode":
            return Optional.ofNullable(clazz.cast(modeAsString()));
        case "PaddingType":
            return Optional.ofNullable(clazz.cast(paddingTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SymmetricEncryptionAttributes, T> g) {
        return obj -> g.apply((SymmetricEncryptionAttributes) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SymmetricEncryptionAttributes> {
        /**
         * <p>
         * An input to cryptographic primitive used to provide the intial state. The <code>InitializationVector</code>
         * is typically required have a random or psuedo-random value, but sometimes it only needs to be unpredictable
         * or unique. If a value is not provided, Amazon Web Services Payment Cryptography generates a random value.
         * </p>
         * 
         * @param initializationVector
         *        An input to cryptographic primitive used to provide the intial state. The
         *        <code>InitializationVector</code> is typically required have a random or psuedo-random value, but
         *        sometimes it only needs to be unpredictable or unique. If a value is not provided, Amazon Web Services
         *        Payment Cryptography generates a random value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder initializationVector(String initializationVector);

        /**
         * <p>
         * The block cipher mode of operation. Block ciphers are designed to encrypt a block of data of fixed size (for
         * example, 128 bits). The size of the input block is usually same as the size of the encrypted output block,
         * while the key length can be different. A mode of operation describes how to repeatedly apply a cipher's
         * single-block operation to securely transform amounts of data larger than a block.
         * </p>
         * 
         * @param mode
         *        The block cipher mode of operation. Block ciphers are designed to encrypt a block of data of fixed
         *        size (for example, 128 bits). The size of the input block is usually same as the size of the encrypted
         *        output block, while the key length can be different. A mode of operation describes how to repeatedly
         *        apply a cipher's single-block operation to securely transform amounts of data larger than a block.
         * @see EncryptionMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EncryptionMode
         */
        Builder mode(String mode);

        /**
         * <p>
         * The block cipher mode of operation. Block ciphers are designed to encrypt a block of data of fixed size (for
         * example, 128 bits). The size of the input block is usually same as the size of the encrypted output block,
         * while the key length can be different. A mode of operation describes how to repeatedly apply a cipher's
         * single-block operation to securely transform amounts of data larger than a block.
         * </p>
         * 
         * @param mode
         *        The block cipher mode of operation. Block ciphers are designed to encrypt a block of data of fixed
         *        size (for example, 128 bits). The size of the input block is usually same as the size of the encrypted
         *        output block, while the key length can be different. A mode of operation describes how to repeatedly
         *        apply a cipher's single-block operation to securely transform amounts of data larger than a block.
         * @see EncryptionMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EncryptionMode
         */
        Builder mode(EncryptionMode mode);

        /**
         * <p>
         * The padding to be included with the data.
         * </p>
         * 
         * @param paddingType
         *        The padding to be included with the data.
         * @see PaddingType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PaddingType
         */
        Builder paddingType(String paddingType);

        /**
         * <p>
         * The padding to be included with the data.
         * </p>
         * 
         * @param paddingType
         *        The padding to be included with the data.
         * @see PaddingType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PaddingType
         */
        Builder paddingType(PaddingType paddingType);
    }

    static final class BuilderImpl implements Builder {
        private String initializationVector;

        private String mode;

        private String paddingType;

        private BuilderImpl() {
        }

        private BuilderImpl(SymmetricEncryptionAttributes model) {
            initializationVector(model.initializationVector);
            mode(model.mode);
            paddingType(model.paddingType);
        }

        public final String getInitializationVector() {
            return initializationVector;
        }

        public final void setInitializationVector(String initializationVector) {
            this.initializationVector = initializationVector;
        }

        @Override
        public final Builder initializationVector(String initializationVector) {
            this.initializationVector = initializationVector;
            return this;
        }

        public final String getMode() {
            return mode;
        }

        public final void setMode(String mode) {
            this.mode = mode;
        }

        @Override
        public final Builder mode(String mode) {
            this.mode = mode;
            return this;
        }

        @Override
        public final Builder mode(EncryptionMode mode) {
            this.mode(mode == null ? null : mode.toString());
            return this;
        }

        public final String getPaddingType() {
            return paddingType;
        }

        public final void setPaddingType(String paddingType) {
            this.paddingType = paddingType;
        }

        @Override
        public final Builder paddingType(String paddingType) {
            this.paddingType = paddingType;
            return this;
        }

        @Override
        public final Builder paddingType(PaddingType paddingType) {
            this.paddingType(paddingType == null ? null : paddingType.toString());
            return this;
        }

        @Override
        public SymmetricEncryptionAttributes build() {
            return new SymmetricEncryptionAttributes(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
