/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.omics.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The metadata of a single part of a file that was added to a multipart upload. A list of these parts is returned in
 * the response to the ListReadSetUploadParts API.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ReadSetUploadPartListItem implements SdkPojo, Serializable,
        ToCopyableBuilder<ReadSetUploadPartListItem.Builder, ReadSetUploadPartListItem> {
    private static final SdkField<Integer> PART_NUMBER_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("partNumber").getter(getter(ReadSetUploadPartListItem::partNumber)).setter(setter(Builder::partNumber))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("partNumber").build()).build();

    private static final SdkField<Long> PART_SIZE_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("partSize")
            .getter(getter(ReadSetUploadPartListItem::partSize)).setter(setter(Builder::partSize))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("partSize").build()).build();

    private static final SdkField<String> PART_SOURCE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("partSource").getter(getter(ReadSetUploadPartListItem::partSourceAsString))
            .setter(setter(Builder::partSource))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("partSource").build()).build();

    private static final SdkField<String> CHECKSUM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("checksum").getter(getter(ReadSetUploadPartListItem::checksum)).setter(setter(Builder::checksum))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("checksum").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("creationTime")
            .getter(getter(ReadSetUploadPartListItem::creationTime))
            .setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("creationTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> LAST_UPDATED_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastUpdatedTime")
            .getter(getter(ReadSetUploadPartListItem::lastUpdatedTime))
            .setter(setter(Builder::lastUpdatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastUpdatedTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PART_NUMBER_FIELD,
            PART_SIZE_FIELD, PART_SOURCE_FIELD, CHECKSUM_FIELD, CREATION_TIME_FIELD, LAST_UPDATED_TIME_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer partNumber;

    private final Long partSize;

    private final String partSource;

    private final String checksum;

    private final Instant creationTime;

    private final Instant lastUpdatedTime;

    private ReadSetUploadPartListItem(BuilderImpl builder) {
        this.partNumber = builder.partNumber;
        this.partSize = builder.partSize;
        this.partSource = builder.partSource;
        this.checksum = builder.checksum;
        this.creationTime = builder.creationTime;
        this.lastUpdatedTime = builder.lastUpdatedTime;
    }

    /**
     * <p>
     * The number identifying the part in an upload.
     * </p>
     * 
     * @return The number identifying the part in an upload.
     */
    public final Integer partNumber() {
        return partNumber;
    }

    /**
     * <p>
     * The size of the the part in an upload.
     * </p>
     * 
     * @return The size of the the part in an upload.
     */
    public final Long partSize() {
        return partSize;
    }

    /**
     * <p>
     * The origin of the part being direct uploaded.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #partSource} will
     * return {@link ReadSetPartSource#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #partSourceAsString}.
     * </p>
     * 
     * @return The origin of the part being direct uploaded.
     * @see ReadSetPartSource
     */
    public final ReadSetPartSource partSource() {
        return ReadSetPartSource.fromValue(partSource);
    }

    /**
     * <p>
     * The origin of the part being direct uploaded.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #partSource} will
     * return {@link ReadSetPartSource#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #partSourceAsString}.
     * </p>
     * 
     * @return The origin of the part being direct uploaded.
     * @see ReadSetPartSource
     */
    public final String partSourceAsString() {
        return partSource;
    }

    /**
     * <p>
     * A unique identifier used to confirm that parts are being added to the correct upload.
     * </p>
     * 
     * @return A unique identifier used to confirm that parts are being added to the correct upload.
     */
    public final String checksum() {
        return checksum;
    }

    /**
     * <p>
     * The time stamp for when a direct upload was created.
     * </p>
     * 
     * @return The time stamp for when a direct upload was created.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The time stamp for the most recent update to an uploaded part.
     * </p>
     * 
     * @return The time stamp for the most recent update to an uploaded part.
     */
    public final Instant lastUpdatedTime() {
        return lastUpdatedTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(partNumber());
        hashCode = 31 * hashCode + Objects.hashCode(partSize());
        hashCode = 31 * hashCode + Objects.hashCode(partSourceAsString());
        hashCode = 31 * hashCode + Objects.hashCode(checksum());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ReadSetUploadPartListItem)) {
            return false;
        }
        ReadSetUploadPartListItem other = (ReadSetUploadPartListItem) obj;
        return Objects.equals(partNumber(), other.partNumber()) && Objects.equals(partSize(), other.partSize())
                && Objects.equals(partSourceAsString(), other.partSourceAsString())
                && Objects.equals(checksum(), other.checksum()) && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(lastUpdatedTime(), other.lastUpdatedTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ReadSetUploadPartListItem").add("PartNumber", partNumber()).add("PartSize", partSize())
                .add("PartSource", partSourceAsString()).add("Checksum", checksum()).add("CreationTime", creationTime())
                .add("LastUpdatedTime", lastUpdatedTime()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "partNumber":
            return Optional.ofNullable(clazz.cast(partNumber()));
        case "partSize":
            return Optional.ofNullable(clazz.cast(partSize()));
        case "partSource":
            return Optional.ofNullable(clazz.cast(partSourceAsString()));
        case "checksum":
            return Optional.ofNullable(clazz.cast(checksum()));
        case "creationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "lastUpdatedTime":
            return Optional.ofNullable(clazz.cast(lastUpdatedTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ReadSetUploadPartListItem, T> g) {
        return obj -> g.apply((ReadSetUploadPartListItem) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ReadSetUploadPartListItem> {
        /**
         * <p>
         * The number identifying the part in an upload.
         * </p>
         * 
         * @param partNumber
         *        The number identifying the part in an upload.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder partNumber(Integer partNumber);

        /**
         * <p>
         * The size of the the part in an upload.
         * </p>
         * 
         * @param partSize
         *        The size of the the part in an upload.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder partSize(Long partSize);

        /**
         * <p>
         * The origin of the part being direct uploaded.
         * </p>
         * 
         * @param partSource
         *        The origin of the part being direct uploaded.
         * @see ReadSetPartSource
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReadSetPartSource
         */
        Builder partSource(String partSource);

        /**
         * <p>
         * The origin of the part being direct uploaded.
         * </p>
         * 
         * @param partSource
         *        The origin of the part being direct uploaded.
         * @see ReadSetPartSource
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReadSetPartSource
         */
        Builder partSource(ReadSetPartSource partSource);

        /**
         * <p>
         * A unique identifier used to confirm that parts are being added to the correct upload.
         * </p>
         * 
         * @param checksum
         *        A unique identifier used to confirm that parts are being added to the correct upload.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder checksum(String checksum);

        /**
         * <p>
         * The time stamp for when a direct upload was created.
         * </p>
         * 
         * @param creationTime
         *        The time stamp for when a direct upload was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The time stamp for the most recent update to an uploaded part.
         * </p>
         * 
         * @param lastUpdatedTime
         *        The time stamp for the most recent update to an uploaded part.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedTime(Instant lastUpdatedTime);
    }

    static final class BuilderImpl implements Builder {
        private Integer partNumber;

        private Long partSize;

        private String partSource;

        private String checksum;

        private Instant creationTime;

        private Instant lastUpdatedTime;

        private BuilderImpl() {
        }

        private BuilderImpl(ReadSetUploadPartListItem model) {
            partNumber(model.partNumber);
            partSize(model.partSize);
            partSource(model.partSource);
            checksum(model.checksum);
            creationTime(model.creationTime);
            lastUpdatedTime(model.lastUpdatedTime);
        }

        public final Integer getPartNumber() {
            return partNumber;
        }

        public final void setPartNumber(Integer partNumber) {
            this.partNumber = partNumber;
        }

        @Override
        public final Builder partNumber(Integer partNumber) {
            this.partNumber = partNumber;
            return this;
        }

        public final Long getPartSize() {
            return partSize;
        }

        public final void setPartSize(Long partSize) {
            this.partSize = partSize;
        }

        @Override
        public final Builder partSize(Long partSize) {
            this.partSize = partSize;
            return this;
        }

        public final String getPartSource() {
            return partSource;
        }

        public final void setPartSource(String partSource) {
            this.partSource = partSource;
        }

        @Override
        public final Builder partSource(String partSource) {
            this.partSource = partSource;
            return this;
        }

        @Override
        public final Builder partSource(ReadSetPartSource partSource) {
            this.partSource(partSource == null ? null : partSource.toString());
            return this;
        }

        public final String getChecksum() {
            return checksum;
        }

        public final void setChecksum(String checksum) {
            this.checksum = checksum;
        }

        @Override
        public final Builder checksum(String checksum) {
            this.checksum = checksum;
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final Instant getLastUpdatedTime() {
            return lastUpdatedTime;
        }

        public final void setLastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
        }

        @Override
        public final Builder lastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
            return this;
        }

        @Override
        public ReadSetUploadPartListItem build() {
            return new ReadSetUploadPartListItem(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
