/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.migrationhubstrategy.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains detailed information about a recommendation report.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RecommendationReportDetails implements SdkPojo, Serializable,
        ToCopyableBuilder<RecommendationReportDetails.Builder, RecommendationReportDetails> {
    private static final SdkField<Instant> COMPLETION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("completionTime").getter(getter(RecommendationReportDetails::completionTime))
            .setter(setter(Builder::completionTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("completionTime").build()).build();

    private static final SdkField<String> S3_BUCKET_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("s3Bucket").getter(getter(RecommendationReportDetails::s3Bucket)).setter(setter(Builder::s3Bucket))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("s3Bucket").build()).build();

    private static final SdkField<List<String>> S3_KEYS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("s3Keys")
            .getter(getter(RecommendationReportDetails::s3Keys))
            .setter(setter(Builder::s3Keys))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("s3Keys").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("startTime").getter(getter(RecommendationReportDetails::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startTime").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(RecommendationReportDetails::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<String> STATUS_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("statusMessage").getter(getter(RecommendationReportDetails::statusMessage))
            .setter(setter(Builder::statusMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusMessage").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(COMPLETION_TIME_FIELD,
            S3_BUCKET_FIELD, S3_KEYS_FIELD, START_TIME_FIELD, STATUS_FIELD, STATUS_MESSAGE_FIELD));

    private static final long serialVersionUID = 1L;

    private final Instant completionTime;

    private final String s3Bucket;

    private final List<String> s3Keys;

    private final Instant startTime;

    private final String status;

    private final String statusMessage;

    private RecommendationReportDetails(BuilderImpl builder) {
        this.completionTime = builder.completionTime;
        this.s3Bucket = builder.s3Bucket;
        this.s3Keys = builder.s3Keys;
        this.startTime = builder.startTime;
        this.status = builder.status;
        this.statusMessage = builder.statusMessage;
    }

    /**
     * <p>
     * The time that the recommendation report generation task completes.
     * </p>
     * 
     * @return The time that the recommendation report generation task completes.
     */
    public final Instant completionTime() {
        return completionTime;
    }

    /**
     * <p>
     * The S3 bucket where the report file is located.
     * </p>
     * 
     * @return The S3 bucket where the report file is located.
     */
    public final String s3Bucket() {
        return s3Bucket;
    }

    /**
     * For responses, this returns true if the service returned a value for the S3Keys property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasS3Keys() {
        return s3Keys != null && !(s3Keys instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon S3 key name of the report file.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasS3Keys} method.
     * </p>
     * 
     * @return The Amazon S3 key name of the report file.
     */
    public final List<String> s3Keys() {
        return s3Keys;
    }

    /**
     * <p>
     * The time that the recommendation report generation task starts.
     * </p>
     * 
     * @return The time that the recommendation report generation task starts.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The status of the recommendation report generation task.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link RecommendationReportStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the recommendation report generation task.
     * @see RecommendationReportStatus
     */
    public final RecommendationReportStatus status() {
        return RecommendationReportStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the recommendation report generation task.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link RecommendationReportStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the recommendation report generation task.
     * @see RecommendationReportStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The status message for recommendation report generation.
     * </p>
     * 
     * @return The status message for recommendation report generation.
     */
    public final String statusMessage() {
        return statusMessage;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(completionTime());
        hashCode = 31 * hashCode + Objects.hashCode(s3Bucket());
        hashCode = 31 * hashCode + Objects.hashCode(hasS3Keys() ? s3Keys() : null);
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusMessage());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RecommendationReportDetails)) {
            return false;
        }
        RecommendationReportDetails other = (RecommendationReportDetails) obj;
        return Objects.equals(completionTime(), other.completionTime()) && Objects.equals(s3Bucket(), other.s3Bucket())
                && hasS3Keys() == other.hasS3Keys() && Objects.equals(s3Keys(), other.s3Keys())
                && Objects.equals(startTime(), other.startTime()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusMessage(), other.statusMessage());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RecommendationReportDetails").add("CompletionTime", completionTime())
                .add("S3Bucket", s3Bucket()).add("S3Keys", hasS3Keys() ? s3Keys() : null).add("StartTime", startTime())
                .add("Status", statusAsString()).add("StatusMessage", statusMessage()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "completionTime":
            return Optional.ofNullable(clazz.cast(completionTime()));
        case "s3Bucket":
            return Optional.ofNullable(clazz.cast(s3Bucket()));
        case "s3Keys":
            return Optional.ofNullable(clazz.cast(s3Keys()));
        case "startTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "statusMessage":
            return Optional.ofNullable(clazz.cast(statusMessage()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RecommendationReportDetails, T> g) {
        return obj -> g.apply((RecommendationReportDetails) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RecommendationReportDetails> {
        /**
         * <p>
         * The time that the recommendation report generation task completes.
         * </p>
         * 
         * @param completionTime
         *        The time that the recommendation report generation task completes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completionTime(Instant completionTime);

        /**
         * <p>
         * The S3 bucket where the report file is located.
         * </p>
         * 
         * @param s3Bucket
         *        The S3 bucket where the report file is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Bucket(String s3Bucket);

        /**
         * <p>
         * The Amazon S3 key name of the report file.
         * </p>
         * 
         * @param s3Keys
         *        The Amazon S3 key name of the report file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Keys(Collection<String> s3Keys);

        /**
         * <p>
         * The Amazon S3 key name of the report file.
         * </p>
         * 
         * @param s3Keys
         *        The Amazon S3 key name of the report file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Keys(String... s3Keys);

        /**
         * <p>
         * The time that the recommendation report generation task starts.
         * </p>
         * 
         * @param startTime
         *        The time that the recommendation report generation task starts.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The status of the recommendation report generation task.
         * </p>
         * 
         * @param status
         *        The status of the recommendation report generation task.
         * @see RecommendationReportStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RecommendationReportStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the recommendation report generation task.
         * </p>
         * 
         * @param status
         *        The status of the recommendation report generation task.
         * @see RecommendationReportStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RecommendationReportStatus
         */
        Builder status(RecommendationReportStatus status);

        /**
         * <p>
         * The status message for recommendation report generation.
         * </p>
         * 
         * @param statusMessage
         *        The status message for recommendation report generation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusMessage(String statusMessage);
    }

    static final class BuilderImpl implements Builder {
        private Instant completionTime;

        private String s3Bucket;

        private List<String> s3Keys = DefaultSdkAutoConstructList.getInstance();

        private Instant startTime;

        private String status;

        private String statusMessage;

        private BuilderImpl() {
        }

        private BuilderImpl(RecommendationReportDetails model) {
            completionTime(model.completionTime);
            s3Bucket(model.s3Bucket);
            s3Keys(model.s3Keys);
            startTime(model.startTime);
            status(model.status);
            statusMessage(model.statusMessage);
        }

        public final Instant getCompletionTime() {
            return completionTime;
        }

        public final void setCompletionTime(Instant completionTime) {
            this.completionTime = completionTime;
        }

        @Override
        public final Builder completionTime(Instant completionTime) {
            this.completionTime = completionTime;
            return this;
        }

        public final String getS3Bucket() {
            return s3Bucket;
        }

        public final void setS3Bucket(String s3Bucket) {
            this.s3Bucket = s3Bucket;
        }

        @Override
        public final Builder s3Bucket(String s3Bucket) {
            this.s3Bucket = s3Bucket;
            return this;
        }

        public final Collection<String> getS3Keys() {
            if (s3Keys instanceof SdkAutoConstructList) {
                return null;
            }
            return s3Keys;
        }

        public final void setS3Keys(Collection<String> s3Keys) {
            this.s3Keys = S3KeysCopier.copy(s3Keys);
        }

        @Override
        public final Builder s3Keys(Collection<String> s3Keys) {
            this.s3Keys = S3KeysCopier.copy(s3Keys);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder s3Keys(String... s3Keys) {
            s3Keys(Arrays.asList(s3Keys));
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(RecommendationReportStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getStatusMessage() {
            return statusMessage;
        }

        public final void setStatusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
        }

        @Override
        public final Builder statusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
            return this;
        }

        @Override
        public RecommendationReportDetails build() {
            return new RecommendationReportDetails(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
