/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.medicalimaging;

import java.nio.file.Path;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ResponseBytes;
import software.amazon.awssdk.core.ResponseInputStream;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.sync.ResponseTransformer;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.medicalimaging.model.AccessDeniedException;
import software.amazon.awssdk.services.medicalimaging.model.ConflictException;
import software.amazon.awssdk.services.medicalimaging.model.CopyImageSetRequest;
import software.amazon.awssdk.services.medicalimaging.model.CopyImageSetResponse;
import software.amazon.awssdk.services.medicalimaging.model.CreateDatastoreRequest;
import software.amazon.awssdk.services.medicalimaging.model.CreateDatastoreResponse;
import software.amazon.awssdk.services.medicalimaging.model.DeleteDatastoreRequest;
import software.amazon.awssdk.services.medicalimaging.model.DeleteDatastoreResponse;
import software.amazon.awssdk.services.medicalimaging.model.DeleteImageSetRequest;
import software.amazon.awssdk.services.medicalimaging.model.DeleteImageSetResponse;
import software.amazon.awssdk.services.medicalimaging.model.GetDatastoreRequest;
import software.amazon.awssdk.services.medicalimaging.model.GetDatastoreResponse;
import software.amazon.awssdk.services.medicalimaging.model.GetDicomImportJobRequest;
import software.amazon.awssdk.services.medicalimaging.model.GetDicomImportJobResponse;
import software.amazon.awssdk.services.medicalimaging.model.GetImageFrameRequest;
import software.amazon.awssdk.services.medicalimaging.model.GetImageFrameResponse;
import software.amazon.awssdk.services.medicalimaging.model.GetImageSetMetadataRequest;
import software.amazon.awssdk.services.medicalimaging.model.GetImageSetMetadataResponse;
import software.amazon.awssdk.services.medicalimaging.model.GetImageSetRequest;
import software.amazon.awssdk.services.medicalimaging.model.GetImageSetResponse;
import software.amazon.awssdk.services.medicalimaging.model.InternalServerException;
import software.amazon.awssdk.services.medicalimaging.model.ListDatastoresRequest;
import software.amazon.awssdk.services.medicalimaging.model.ListDatastoresResponse;
import software.amazon.awssdk.services.medicalimaging.model.ListDicomImportJobsRequest;
import software.amazon.awssdk.services.medicalimaging.model.ListDicomImportJobsResponse;
import software.amazon.awssdk.services.medicalimaging.model.ListImageSetVersionsRequest;
import software.amazon.awssdk.services.medicalimaging.model.ListImageSetVersionsResponse;
import software.amazon.awssdk.services.medicalimaging.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.medicalimaging.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.medicalimaging.model.MedicalImagingException;
import software.amazon.awssdk.services.medicalimaging.model.ResourceNotFoundException;
import software.amazon.awssdk.services.medicalimaging.model.SearchImageSetsRequest;
import software.amazon.awssdk.services.medicalimaging.model.SearchImageSetsResponse;
import software.amazon.awssdk.services.medicalimaging.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.medicalimaging.model.StartDicomImportJobRequest;
import software.amazon.awssdk.services.medicalimaging.model.StartDicomImportJobResponse;
import software.amazon.awssdk.services.medicalimaging.model.TagResourceRequest;
import software.amazon.awssdk.services.medicalimaging.model.TagResourceResponse;
import software.amazon.awssdk.services.medicalimaging.model.ThrottlingException;
import software.amazon.awssdk.services.medicalimaging.model.UntagResourceRequest;
import software.amazon.awssdk.services.medicalimaging.model.UntagResourceResponse;
import software.amazon.awssdk.services.medicalimaging.model.UpdateImageSetMetadataRequest;
import software.amazon.awssdk.services.medicalimaging.model.UpdateImageSetMetadataResponse;
import software.amazon.awssdk.services.medicalimaging.model.ValidationException;
import software.amazon.awssdk.services.medicalimaging.paginators.ListDICOMImportJobsIterable;
import software.amazon.awssdk.services.medicalimaging.paginators.ListDatastoresIterable;
import software.amazon.awssdk.services.medicalimaging.paginators.ListImageSetVersionsIterable;
import software.amazon.awssdk.services.medicalimaging.paginators.SearchImageSetsIterable;

/**
 * Service client for accessing AWS Health Imaging. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * This is the <i>AWS HealthImaging API Reference</i>. AWS HealthImaging is an AWS service for storing, accessing, and
 * analyzing medical images. For an introduction to the service, see the <a
 * href="https://docs.aws.amazon.com/medical-imaging/latest/devguide"> <i>AWS HealthImaging Developer Guide</i> </a>.
 * </p>
 * <note>
 * <p>
 * We recommend using one of the AWS Software Development Kits (SDKs) for your programming language, as they take care
 * of request authentication, serialization, and connection management. For more information, see <a
 * href="http://aws.amazon.com/developer/tools">Tools to build on AWS</a>.
 * </p>
 * <p>
 * For information about using AWS HealthImaging API actions in one of the language-specific AWS SDKs, refer to the
 * <i>See Also</i> link at the end of each section that describes an API action or data type.
 * </p>
 * </note>
 * <p>
 * The following sections list AWS HealthImaging API actions categorized according to functionality. Links are provided
 * to actions within this Reference, along with links back to corresponding sections in the <i>AWS HealthImaging
 * Developer Guide</i> so you can view console procedures and CLI/SDK code examples.
 * </p>
 * <p class="title">
 * <b>Data store actions</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a
 * href="https://docs.aws.amazon.com/medical-imaging/latest/APIReference/API_CreateDatastore.html">CreateDatastore</a> –
 * See <a href="https://docs.aws.amazon.com/medical-imaging/latest/devguide/create-data-store.html">Creating a data
 * store</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/medical-imaging/latest/APIReference/API_GetDatastore.html">GetDatastore</a> –
 * See <a href="https://docs.aws.amazon.com/medical-imaging/latest/devguide/get-data-store.html">Getting data store
 * properties</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/medical-imaging/latest/APIReference/API_ListDatastores.html">ListDatastores</a>
 * – See <a href="https://docs.aws.amazon.com/medical-imaging/latest/devguide/list-data-stores.html">Listing data
 * stores</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a
 * href="https://docs.aws.amazon.com/medical-imaging/latest/APIReference/API_DeleteDatastore.html">DeleteDatastore</a> –
 * See <a href="https://docs.aws.amazon.com/medical-imaging/latest/devguide/delete-data-store.html">Deleting a data
 * store</a>.
 * </p>
 * </li>
 * </ul>
 * <p class="title">
 * <b>Import job actions</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/medical-imaging/latest/APIReference/API_StartDICOMImportJob.html">
 * StartDICOMImportJob</a> – See <a
 * href="https://docs.aws.amazon.com/medical-imaging/latest/devguide/start-dicom-import-job.html">Starting an import
 * job</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a
 * href="https://docs.aws.amazon.com/medical-imaging/latest/APIReference/API_GetDICOMImportJob.html">GetDICOMImportJob
 * </a> – See <a href="https://docs.aws.amazon.com/medical-imaging/latest/devguide/get-dicom-import-job.html">Getting
 * import job properties</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/medical-imaging/latest/APIReference/API_ListDICOMImportJobs.html">
 * ListDICOMImportJobs</a> – See <a
 * href="https://docs.aws.amazon.com/medical-imaging/latest/devguide/list-dicom-import-jobs.html">Listing import
 * jobs</a>.
 * </p>
 * </li>
 * </ul>
 * <p class="title">
 * <b>Image set access actions</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a
 * href="https://docs.aws.amazon.com/medical-imaging/latest/APIReference/API_SearchImageSets.html">SearchImageSets</a> –
 * See <a href="https://docs.aws.amazon.com/medical-imaging/latest/devguide/search-image-sets.html">Searching image
 * sets</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/medical-imaging/latest/APIReference/API_GetImageSet.html">GetImageSet</a> – See
 * <a href="https://docs.aws.amazon.com/medical-imaging/latest/devguide/get-image-set-properties.html">Getting image set
 * properties</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/medical-imaging/latest/APIReference/API_GetImageSetMetadata.html">
 * GetImageSetMetadata</a> – See <a
 * href="https://docs.aws.amazon.com/medical-imaging/latest/devguide/get-image-set-metadata.html">Getting image set
 * metadata</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/medical-imaging/latest/APIReference/API_GetImageFrame.html">GetImageFrame</a> –
 * See <a href="https://docs.aws.amazon.com/medical-imaging/latest/devguide/get-image-frame.html">Getting image set
 * pixel data</a>.
 * </p>
 * </li>
 * </ul>
 * <p class="title">
 * <b>Image set modification actions</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/medical-imaging/latest/APIReference/API_ListImageSetVersions.html">
 * ListImageSetVersions</a> – See <a
 * href="https://docs.aws.amazon.com/medical-imaging/latest/devguide/list-image-set-versions.html">Listing image set
 * versions</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/medical-imaging/latest/APIReference/API_UpdateImageSetMetadata.html">
 * UpdateImageSetMetadata</a> – See <a
 * href="https://docs.aws.amazon.com/medical-imaging/latest/devguide/update-image-set-metadata.html">Updating image set
 * metadata</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/medical-imaging/latest/APIReference/API_CopyImageSet.html">CopyImageSet</a> –
 * See <a href="https://docs.aws.amazon.com/medical-imaging/latest/devguide/copy-image-set.html">Copying an image
 * set</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/medical-imaging/latest/APIReference/API_DeleteImageSet.html">DeleteImageSet</a>
 * – See <a href="https://docs.aws.amazon.com/medical-imaging/latest/devguide/delete-image-set.html">Deleting an image
 * set</a>.
 * </p>
 * </li>
 * </ul>
 * <p class="title">
 * <b>Tagging actions</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/medical-imaging/latest/APIReference/API_TagResource.html">TagResource</a> – See
 * <a href="https://docs.aws.amazon.com/medical-imaging/latest/devguide/tag-list-untag-data-store.html">Tagging a data
 * store</a> and <a
 * href="https://docs.aws.amazon.com/medical-imaging/latest/devguide/tag-list-untag-image-set.html">Tagging an image
 * set</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/medical-imaging/latest/APIReference/API_ListTagsForResource.html">
 * ListTagsForResource</a> – See <a
 * href="https://docs.aws.amazon.com/medical-imaging/latest/devguide/tag-list-untag-data-store.html">Tagging a data
 * store</a> and <a
 * href="https://docs.aws.amazon.com/medical-imaging/latest/devguide/tag-list-untag-image-set.html">Tagging an image
 * set</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/medical-imaging/latest/APIReference/API_UntagResource.html">UntagResource</a> –
 * See <a href="https://docs.aws.amazon.com/medical-imaging/latest/devguide/tag-list-untag-data-store.html">Tagging a
 * data store</a> and <a
 * href="https://docs.aws.amazon.com/medical-imaging/latest/devguide/tag-list-untag-image-set.html">Tagging an image
 * set</a>.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface MedicalImagingClient extends AwsClient {
    String SERVICE_NAME = "medical-imaging";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "medical-imaging";

    /**
     * <p>
     * Copy an image set.
     * </p>
     *
     * @param copyImageSetRequest
     * @return Result of the CopyImageSet operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ServiceQuotaExceededException
     *         The request caused a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.CopyImageSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/CopyImageSet" target="_top">AWS
     *      API Documentation</a>
     */
    default CopyImageSetResponse copyImageSet(CopyImageSetRequest copyImageSetRequest) throws ThrottlingException,
            ConflictException, AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, MedicalImagingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Copy an image set.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CopyImageSetRequest.Builder} avoiding the need to
     * create one manually via {@link CopyImageSetRequest#builder()}
     * </p>
     *
     * @param copyImageSetRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.CopyImageSetRequest.Builder} to create a
     *        request.
     * @return Result of the CopyImageSet operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ServiceQuotaExceededException
     *         The request caused a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.CopyImageSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/CopyImageSet" target="_top">AWS
     *      API Documentation</a>
     */
    default CopyImageSetResponse copyImageSet(Consumer<CopyImageSetRequest.Builder> copyImageSetRequest)
            throws ThrottlingException, ConflictException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            MedicalImagingException {
        return copyImageSet(CopyImageSetRequest.builder().applyMutation(copyImageSetRequest).build());
    }

    /**
     * <p>
     * Create a data store.
     * </p>
     *
     * @param createDatastoreRequest
     * @return Result of the CreateDatastore operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ServiceQuotaExceededException
     *         The request caused a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.CreateDatastore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/CreateDatastore"
     *      target="_top">AWS API Documentation</a>
     */
    default CreateDatastoreResponse createDatastore(CreateDatastoreRequest createDatastoreRequest) throws ThrottlingException,
            ConflictException, AccessDeniedException, ValidationException, InternalServerException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, MedicalImagingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Create a data store.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateDatastoreRequest.Builder} avoiding the need
     * to create one manually via {@link CreateDatastoreRequest#builder()}
     * </p>
     *
     * @param createDatastoreRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.CreateDatastoreRequest.Builder} to create a
     *        request.
     * @return Result of the CreateDatastore operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ServiceQuotaExceededException
     *         The request caused a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.CreateDatastore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/CreateDatastore"
     *      target="_top">AWS API Documentation</a>
     */
    default CreateDatastoreResponse createDatastore(Consumer<CreateDatastoreRequest.Builder> createDatastoreRequest)
            throws ThrottlingException, ConflictException, AccessDeniedException, ValidationException, InternalServerException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, MedicalImagingException {
        return createDatastore(CreateDatastoreRequest.builder().applyMutation(createDatastoreRequest).build());
    }

    /**
     * <p>
     * Delete a data store.
     * </p>
     * <note>
     * <p>
     * Before a data store can be deleted, you must first delete all image sets within it.
     * </p>
     * </note>
     *
     * @param deleteDatastoreRequest
     * @return Result of the DeleteDatastore operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.DeleteDatastore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/DeleteDatastore"
     *      target="_top">AWS API Documentation</a>
     */
    default DeleteDatastoreResponse deleteDatastore(DeleteDatastoreRequest deleteDatastoreRequest) throws ThrottlingException,
            ConflictException, AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, MedicalImagingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Delete a data store.
     * </p>
     * <note>
     * <p>
     * Before a data store can be deleted, you must first delete all image sets within it.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteDatastoreRequest.Builder} avoiding the need
     * to create one manually via {@link DeleteDatastoreRequest#builder()}
     * </p>
     *
     * @param deleteDatastoreRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.DeleteDatastoreRequest.Builder} to create a
     *        request.
     * @return Result of the DeleteDatastore operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.DeleteDatastore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/DeleteDatastore"
     *      target="_top">AWS API Documentation</a>
     */
    default DeleteDatastoreResponse deleteDatastore(Consumer<DeleteDatastoreRequest.Builder> deleteDatastoreRequest)
            throws ThrottlingException, ConflictException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, MedicalImagingException {
        return deleteDatastore(DeleteDatastoreRequest.builder().applyMutation(deleteDatastoreRequest).build());
    }

    /**
     * <p>
     * Delete an image set.
     * </p>
     *
     * @param deleteImageSetRequest
     * @return Result of the DeleteImageSet operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.DeleteImageSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/DeleteImageSet"
     *      target="_top">AWS API Documentation</a>
     */
    default DeleteImageSetResponse deleteImageSet(DeleteImageSetRequest deleteImageSetRequest) throws ThrottlingException,
            ConflictException, AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, MedicalImagingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Delete an image set.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteImageSetRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteImageSetRequest#builder()}
     * </p>
     *
     * @param deleteImageSetRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.DeleteImageSetRequest.Builder} to create a
     *        request.
     * @return Result of the DeleteImageSet operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.DeleteImageSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/DeleteImageSet"
     *      target="_top">AWS API Documentation</a>
     */
    default DeleteImageSetResponse deleteImageSet(Consumer<DeleteImageSetRequest.Builder> deleteImageSetRequest)
            throws ThrottlingException, ConflictException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, MedicalImagingException {
        return deleteImageSet(DeleteImageSetRequest.builder().applyMutation(deleteImageSetRequest).build());
    }

    /**
     * <p>
     * Get the import job properties to learn more about the job or job progress.
     * </p>
     *
     * @param getDicomImportJobRequest
     * @return Result of the GetDICOMImportJob operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.GetDICOMImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetDICOMImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    default GetDicomImportJobResponse getDICOMImportJob(GetDicomImportJobRequest getDicomImportJobRequest)
            throws ThrottlingException, ConflictException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, MedicalImagingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Get the import job properties to learn more about the job or job progress.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetDicomImportJobRequest.Builder} avoiding the need
     * to create one manually via {@link GetDicomImportJobRequest#builder()}
     * </p>
     *
     * @param getDicomImportJobRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.GetDicomImportJobRequest.Builder} to create a
     *        request.
     * @return Result of the GetDICOMImportJob operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.GetDICOMImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetDICOMImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    default GetDicomImportJobResponse getDICOMImportJob(Consumer<GetDicomImportJobRequest.Builder> getDicomImportJobRequest)
            throws ThrottlingException, ConflictException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, MedicalImagingException {
        return getDICOMImportJob(GetDicomImportJobRequest.builder().applyMutation(getDicomImportJobRequest).build());
    }

    /**
     * <p>
     * Get data store properties.
     * </p>
     *
     * @param getDatastoreRequest
     * @return Result of the GetDatastore operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.GetDatastore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetDatastore" target="_top">AWS
     *      API Documentation</a>
     */
    default GetDatastoreResponse getDatastore(GetDatastoreRequest getDatastoreRequest) throws ThrottlingException,
            AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, MedicalImagingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Get data store properties.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetDatastoreRequest.Builder} avoiding the need to
     * create one manually via {@link GetDatastoreRequest#builder()}
     * </p>
     *
     * @param getDatastoreRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.GetDatastoreRequest.Builder} to create a
     *        request.
     * @return Result of the GetDatastore operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.GetDatastore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetDatastore" target="_top">AWS
     *      API Documentation</a>
     */
    default GetDatastoreResponse getDatastore(Consumer<GetDatastoreRequest.Builder> getDatastoreRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, MedicalImagingException {
        return getDatastore(GetDatastoreRequest.builder().applyMutation(getDatastoreRequest).build());
    }

    /**
     * <p>
     * Get an image frame (pixel data) for an image set.
     * </p>
     *
     * @param getImageFrameRequest
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled GetImageFrameResponse
     *        and an InputStream to the response content are provided as parameters to the callback. The callback may
     *        return a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows '
     *        <p>
     *        The blob containing the aggregated image frame information.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.GetImageFrame
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageFrame" target="_top">AWS
     *      API Documentation</a>
     */
    default <ReturnT> ReturnT getImageFrame(GetImageFrameRequest getImageFrameRequest,
            ResponseTransformer<GetImageFrameResponse, ReturnT> responseTransformer) throws ThrottlingException,
            ConflictException, AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, MedicalImagingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Get an image frame (pixel data) for an image set.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetImageFrameRequest.Builder} avoiding the need to
     * create one manually via {@link GetImageFrameRequest#builder()}
     * </p>
     *
     * @param getImageFrameRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.GetImageFrameRequest.Builder} to create a
     *        request.
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled GetImageFrameResponse
     *        and an InputStream to the response content are provided as parameters to the callback. The callback may
     *        return a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows '
     *        <p>
     *        The blob containing the aggregated image frame information.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.GetImageFrame
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageFrame" target="_top">AWS
     *      API Documentation</a>
     */
    default <ReturnT> ReturnT getImageFrame(Consumer<GetImageFrameRequest.Builder> getImageFrameRequest,
            ResponseTransformer<GetImageFrameResponse, ReturnT> responseTransformer) throws ThrottlingException,
            ConflictException, AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, MedicalImagingException {
        return getImageFrame(GetImageFrameRequest.builder().applyMutation(getImageFrameRequest).build(), responseTransformer);
    }

    /**
     * <p>
     * Get an image frame (pixel data) for an image set.
     * </p>
     *
     * @param getImageFrameRequest
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows '
     *        <p>
     *        The blob containing the aggregated image frame information.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.GetImageFrame
     * @see #getImageFrame(GetImageFrameRequest, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageFrame" target="_top">AWS
     *      API Documentation</a>
     */
    default GetImageFrameResponse getImageFrame(GetImageFrameRequest getImageFrameRequest, Path destinationPath)
            throws ThrottlingException, ConflictException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, MedicalImagingException {
        return getImageFrame(getImageFrameRequest, ResponseTransformer.toFile(destinationPath));
    }

    /**
     * <p>
     * Get an image frame (pixel data) for an image set.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetImageFrameRequest.Builder} avoiding the need to
     * create one manually via {@link GetImageFrameRequest#builder()}
     * </p>
     *
     * @param getImageFrameRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.GetImageFrameRequest.Builder} to create a
     *        request.
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows '
     *        <p>
     *        The blob containing the aggregated image frame information.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.GetImageFrame
     * @see #getImageFrame(GetImageFrameRequest, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageFrame" target="_top">AWS
     *      API Documentation</a>
     */
    default GetImageFrameResponse getImageFrame(Consumer<GetImageFrameRequest.Builder> getImageFrameRequest, Path destinationPath)
            throws ThrottlingException, ConflictException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, MedicalImagingException {
        return getImageFrame(GetImageFrameRequest.builder().applyMutation(getImageFrameRequest).build(), destinationPath);
    }

    /**
     * <p>
     * Get an image frame (pixel data) for an image set.
     * </p>
     *
     * @param getImageFrameRequest
     * @return A {@link ResponseInputStream} containing data streamed from service. Note that this is an unmanaged
     *         reference to the underlying HTTP connection so great care must be taken to ensure all data if fully read
     *         from the input stream and that it is properly closed. Failure to do so may result in sub-optimal behavior
     *         and exhausting connections in the connection pool. The unmarshalled response object can be obtained via
     *         {@link ResponseInputStream#response()}. The service documentation for the response content is as follows
     *         '
     *         <p>
     *         The blob containing the aggregated image frame information.
     *         </p>
     *         '.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.GetImageFrame
     * @see #getObject(getImageFrame, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageFrame" target="_top">AWS
     *      API Documentation</a>
     */
    default ResponseInputStream<GetImageFrameResponse> getImageFrame(GetImageFrameRequest getImageFrameRequest)
            throws ThrottlingException, ConflictException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, MedicalImagingException {
        return getImageFrame(getImageFrameRequest, ResponseTransformer.toInputStream());
    }

    /**
     * <p>
     * Get an image frame (pixel data) for an image set.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetImageFrameRequest.Builder} avoiding the need to
     * create one manually via {@link GetImageFrameRequest#builder()}
     * </p>
     *
     * @param getImageFrameRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.GetImageFrameRequest.Builder} to create a
     *        request.
     * @return A {@link ResponseInputStream} containing data streamed from service. Note that this is an unmanaged
     *         reference to the underlying HTTP connection so great care must be taken to ensure all data if fully read
     *         from the input stream and that it is properly closed. Failure to do so may result in sub-optimal behavior
     *         and exhausting connections in the connection pool. The unmarshalled response object can be obtained via
     *         {@link ResponseInputStream#response()}. The service documentation for the response content is as follows
     *         '
     *         <p>
     *         The blob containing the aggregated image frame information.
     *         </p>
     *         '.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.GetImageFrame
     * @see #getObject(getImageFrame, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageFrame" target="_top">AWS
     *      API Documentation</a>
     */
    default ResponseInputStream<GetImageFrameResponse> getImageFrame(Consumer<GetImageFrameRequest.Builder> getImageFrameRequest)
            throws ThrottlingException, ConflictException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, MedicalImagingException {
        return getImageFrame(GetImageFrameRequest.builder().applyMutation(getImageFrameRequest).build());
    }

    /**
     * <p>
     * Get an image frame (pixel data) for an image set.
     * </p>
     *
     * @param getImageFrameRequest
     * @return A {@link ResponseBytes} that loads the data streamed from the service into memory and exposes it in
     *         convenient in-memory representations like a byte buffer or string. The unmarshalled response object can
     *         be obtained via {@link ResponseBytes#response()}. The service documentation for the response content is
     *         as follows '
     *         <p>
     *         The blob containing the aggregated image frame information.
     *         </p>
     *         '.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.GetImageFrame
     * @see #getObject(getImageFrame, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageFrame" target="_top">AWS
     *      API Documentation</a>
     */
    default ResponseBytes<GetImageFrameResponse> getImageFrameAsBytes(GetImageFrameRequest getImageFrameRequest)
            throws ThrottlingException, ConflictException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, MedicalImagingException {
        return getImageFrame(getImageFrameRequest, ResponseTransformer.toBytes());
    }

    /**
     * <p>
     * Get an image frame (pixel data) for an image set.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetImageFrameRequest.Builder} avoiding the need to
     * create one manually via {@link GetImageFrameRequest#builder()}
     * </p>
     *
     * @param getImageFrameRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.GetImageFrameRequest.Builder} to create a
     *        request.
     * @return A {@link ResponseBytes} that loads the data streamed from the service into memory and exposes it in
     *         convenient in-memory representations like a byte buffer or string. The unmarshalled response object can
     *         be obtained via {@link ResponseBytes#response()}. The service documentation for the response content is
     *         as follows '
     *         <p>
     *         The blob containing the aggregated image frame information.
     *         </p>
     *         '.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.GetImageFrame
     * @see #getObject(getImageFrame, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageFrame" target="_top">AWS
     *      API Documentation</a>
     */
    default ResponseBytes<GetImageFrameResponse> getImageFrameAsBytes(Consumer<GetImageFrameRequest.Builder> getImageFrameRequest)
            throws ThrottlingException, ConflictException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, MedicalImagingException {
        return getImageFrameAsBytes(GetImageFrameRequest.builder().applyMutation(getImageFrameRequest).build());
    }

    /**
     * <p>
     * Get image set properties.
     * </p>
     *
     * @param getImageSetRequest
     * @return Result of the GetImageSet operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.GetImageSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageSet" target="_top">AWS
     *      API Documentation</a>
     */
    default GetImageSetResponse getImageSet(GetImageSetRequest getImageSetRequest) throws ThrottlingException, ConflictException,
            AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, MedicalImagingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Get image set properties.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetImageSetRequest.Builder} avoiding the need to
     * create one manually via {@link GetImageSetRequest#builder()}
     * </p>
     *
     * @param getImageSetRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.GetImageSetRequest.Builder} to create a
     *        request.
     * @return Result of the GetImageSet operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.GetImageSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageSet" target="_top">AWS
     *      API Documentation</a>
     */
    default GetImageSetResponse getImageSet(Consumer<GetImageSetRequest.Builder> getImageSetRequest) throws ThrottlingException,
            ConflictException, AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, MedicalImagingException {
        return getImageSet(GetImageSetRequest.builder().applyMutation(getImageSetRequest).build());
    }

    /**
     * <p>
     * Get metadata attributes for an image set.
     * </p>
     *
     * @param getImageSetMetadataRequest
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled
     *        GetImageSetMetadataResponse and an InputStream to the response content are provided as parameters to the
     *        callback. The callback may return a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows '
     *        <p>
     *        The blob containing the aggregated metadata information for the image set.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.GetImageSetMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageSetMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default <ReturnT> ReturnT getImageSetMetadata(GetImageSetMetadataRequest getImageSetMetadataRequest,
            ResponseTransformer<GetImageSetMetadataResponse, ReturnT> responseTransformer) throws ThrottlingException,
            ConflictException, AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, MedicalImagingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Get metadata attributes for an image set.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetImageSetMetadataRequest.Builder} avoiding the
     * need to create one manually via {@link GetImageSetMetadataRequest#builder()}
     * </p>
     *
     * @param getImageSetMetadataRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.GetImageSetMetadataRequest.Builder} to create
     *        a request.
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled
     *        GetImageSetMetadataResponse and an InputStream to the response content are provided as parameters to the
     *        callback. The callback may return a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows '
     *        <p>
     *        The blob containing the aggregated metadata information for the image set.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.GetImageSetMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageSetMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default <ReturnT> ReturnT getImageSetMetadata(Consumer<GetImageSetMetadataRequest.Builder> getImageSetMetadataRequest,
            ResponseTransformer<GetImageSetMetadataResponse, ReturnT> responseTransformer) throws ThrottlingException,
            ConflictException, AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, MedicalImagingException {
        return getImageSetMetadata(GetImageSetMetadataRequest.builder().applyMutation(getImageSetMetadataRequest).build(),
                responseTransformer);
    }

    /**
     * <p>
     * Get metadata attributes for an image set.
     * </p>
     *
     * @param getImageSetMetadataRequest
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows '
     *        <p>
     *        The blob containing the aggregated metadata information for the image set.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.GetImageSetMetadata
     * @see #getImageSetMetadata(GetImageSetMetadataRequest, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageSetMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default GetImageSetMetadataResponse getImageSetMetadata(GetImageSetMetadataRequest getImageSetMetadataRequest,
            Path destinationPath) throws ThrottlingException, ConflictException, AccessDeniedException, ValidationException,
            InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException, MedicalImagingException {
        return getImageSetMetadata(getImageSetMetadataRequest, ResponseTransformer.toFile(destinationPath));
    }

    /**
     * <p>
     * Get metadata attributes for an image set.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetImageSetMetadataRequest.Builder} avoiding the
     * need to create one manually via {@link GetImageSetMetadataRequest#builder()}
     * </p>
     *
     * @param getImageSetMetadataRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.GetImageSetMetadataRequest.Builder} to create
     *        a request.
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows '
     *        <p>
     *        The blob containing the aggregated metadata information for the image set.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.GetImageSetMetadata
     * @see #getImageSetMetadata(GetImageSetMetadataRequest, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageSetMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default GetImageSetMetadataResponse getImageSetMetadata(
            Consumer<GetImageSetMetadataRequest.Builder> getImageSetMetadataRequest, Path destinationPath)
            throws ThrottlingException, ConflictException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, MedicalImagingException {
        return getImageSetMetadata(GetImageSetMetadataRequest.builder().applyMutation(getImageSetMetadataRequest).build(),
                destinationPath);
    }

    /**
     * <p>
     * Get metadata attributes for an image set.
     * </p>
     *
     * @param getImageSetMetadataRequest
     * @return A {@link ResponseInputStream} containing data streamed from service. Note that this is an unmanaged
     *         reference to the underlying HTTP connection so great care must be taken to ensure all data if fully read
     *         from the input stream and that it is properly closed. Failure to do so may result in sub-optimal behavior
     *         and exhausting connections in the connection pool. The unmarshalled response object can be obtained via
     *         {@link ResponseInputStream#response()}. The service documentation for the response content is as follows
     *         '
     *         <p>
     *         The blob containing the aggregated metadata information for the image set.
     *         </p>
     *         '.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.GetImageSetMetadata
     * @see #getObject(getImageSetMetadata, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageSetMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default ResponseInputStream<GetImageSetMetadataResponse> getImageSetMetadata(
            GetImageSetMetadataRequest getImageSetMetadataRequest) throws ThrottlingException, ConflictException,
            AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, MedicalImagingException {
        return getImageSetMetadata(getImageSetMetadataRequest, ResponseTransformer.toInputStream());
    }

    /**
     * <p>
     * Get metadata attributes for an image set.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetImageSetMetadataRequest.Builder} avoiding the
     * need to create one manually via {@link GetImageSetMetadataRequest#builder()}
     * </p>
     *
     * @param getImageSetMetadataRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.GetImageSetMetadataRequest.Builder} to create
     *        a request.
     * @return A {@link ResponseInputStream} containing data streamed from service. Note that this is an unmanaged
     *         reference to the underlying HTTP connection so great care must be taken to ensure all data if fully read
     *         from the input stream and that it is properly closed. Failure to do so may result in sub-optimal behavior
     *         and exhausting connections in the connection pool. The unmarshalled response object can be obtained via
     *         {@link ResponseInputStream#response()}. The service documentation for the response content is as follows
     *         '
     *         <p>
     *         The blob containing the aggregated metadata information for the image set.
     *         </p>
     *         '.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.GetImageSetMetadata
     * @see #getObject(getImageSetMetadata, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageSetMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default ResponseInputStream<GetImageSetMetadataResponse> getImageSetMetadata(
            Consumer<GetImageSetMetadataRequest.Builder> getImageSetMetadataRequest) throws ThrottlingException,
            ConflictException, AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, MedicalImagingException {
        return getImageSetMetadata(GetImageSetMetadataRequest.builder().applyMutation(getImageSetMetadataRequest).build());
    }

    /**
     * <p>
     * Get metadata attributes for an image set.
     * </p>
     *
     * @param getImageSetMetadataRequest
     * @return A {@link ResponseBytes} that loads the data streamed from the service into memory and exposes it in
     *         convenient in-memory representations like a byte buffer or string. The unmarshalled response object can
     *         be obtained via {@link ResponseBytes#response()}. The service documentation for the response content is
     *         as follows '
     *         <p>
     *         The blob containing the aggregated metadata information for the image set.
     *         </p>
     *         '.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.GetImageSetMetadata
     * @see #getObject(getImageSetMetadata, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageSetMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default ResponseBytes<GetImageSetMetadataResponse> getImageSetMetadataAsBytes(
            GetImageSetMetadataRequest getImageSetMetadataRequest) throws ThrottlingException, ConflictException,
            AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, MedicalImagingException {
        return getImageSetMetadata(getImageSetMetadataRequest, ResponseTransformer.toBytes());
    }

    /**
     * <p>
     * Get metadata attributes for an image set.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetImageSetMetadataRequest.Builder} avoiding the
     * need to create one manually via {@link GetImageSetMetadataRequest#builder()}
     * </p>
     *
     * @param getImageSetMetadataRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.GetImageSetMetadataRequest.Builder} to create
     *        a request.
     * @return A {@link ResponseBytes} that loads the data streamed from the service into memory and exposes it in
     *         convenient in-memory representations like a byte buffer or string. The unmarshalled response object can
     *         be obtained via {@link ResponseBytes#response()}. The service documentation for the response content is
     *         as follows '
     *         <p>
     *         The blob containing the aggregated metadata information for the image set.
     *         </p>
     *         '.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.GetImageSetMetadata
     * @see #getObject(getImageSetMetadata, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageSetMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default ResponseBytes<GetImageSetMetadataResponse> getImageSetMetadataAsBytes(
            Consumer<GetImageSetMetadataRequest.Builder> getImageSetMetadataRequest) throws ThrottlingException,
            ConflictException, AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, MedicalImagingException {
        return getImageSetMetadataAsBytes(GetImageSetMetadataRequest.builder().applyMutation(getImageSetMetadataRequest).build());
    }

    /**
     * <p>
     * List import jobs created by this AWS account for a specific data store.
     * </p>
     *
     * @param listDicomImportJobsRequest
     * @return Result of the ListDICOMImportJobs operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.ListDICOMImportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListDICOMImportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    default ListDicomImportJobsResponse listDICOMImportJobs(ListDicomImportJobsRequest listDicomImportJobsRequest)
            throws ThrottlingException, ConflictException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, MedicalImagingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * List import jobs created by this AWS account for a specific data store.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListDicomImportJobsRequest.Builder} avoiding the
     * need to create one manually via {@link ListDicomImportJobsRequest#builder()}
     * </p>
     *
     * @param listDicomImportJobsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.ListDicomImportJobsRequest.Builder} to create
     *        a request.
     * @return Result of the ListDICOMImportJobs operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.ListDICOMImportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListDICOMImportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    default ListDicomImportJobsResponse listDICOMImportJobs(
            Consumer<ListDicomImportJobsRequest.Builder> listDicomImportJobsRequest) throws ThrottlingException,
            ConflictException, AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, MedicalImagingException {
        return listDICOMImportJobs(ListDicomImportJobsRequest.builder().applyMutation(listDicomImportJobsRequest).build());
    }

    /**
     * <p>
     * List import jobs created by this AWS account for a specific data store.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDICOMImportJobs(software.amazon.awssdk.services.medicalimaging.model.ListDicomImportJobsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.ListDICOMImportJobsIterable responses = client.listDICOMImportJobsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.medicalimaging.paginators.ListDICOMImportJobsIterable responses = client
     *             .listDICOMImportJobsPaginator(request);
     *     for (software.amazon.awssdk.services.medicalimaging.model.ListDicomImportJobsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.ListDICOMImportJobsIterable responses = client.listDICOMImportJobsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDICOMImportJobs(software.amazon.awssdk.services.medicalimaging.model.ListDicomImportJobsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listDicomImportJobsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.ListDICOMImportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListDICOMImportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    default ListDICOMImportJobsIterable listDICOMImportJobsPaginator(ListDicomImportJobsRequest listDicomImportJobsRequest)
            throws ThrottlingException, ConflictException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, MedicalImagingException {
        return new ListDICOMImportJobsIterable(this, listDicomImportJobsRequest);
    }

    /**
     * <p>
     * List import jobs created by this AWS account for a specific data store.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDICOMImportJobs(software.amazon.awssdk.services.medicalimaging.model.ListDicomImportJobsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.ListDICOMImportJobsIterable responses = client.listDICOMImportJobsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.medicalimaging.paginators.ListDICOMImportJobsIterable responses = client
     *             .listDICOMImportJobsPaginator(request);
     *     for (software.amazon.awssdk.services.medicalimaging.model.ListDicomImportJobsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.ListDICOMImportJobsIterable responses = client.listDICOMImportJobsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDICOMImportJobs(software.amazon.awssdk.services.medicalimaging.model.ListDicomImportJobsRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListDicomImportJobsRequest.Builder} avoiding the
     * need to create one manually via {@link ListDicomImportJobsRequest#builder()}
     * </p>
     *
     * @param listDicomImportJobsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.ListDicomImportJobsRequest.Builder} to create
     *        a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.ListDICOMImportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListDICOMImportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    default ListDICOMImportJobsIterable listDICOMImportJobsPaginator(
            Consumer<ListDicomImportJobsRequest.Builder> listDicomImportJobsRequest) throws ThrottlingException,
            ConflictException, AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, MedicalImagingException {
        return listDICOMImportJobsPaginator(ListDicomImportJobsRequest.builder().applyMutation(listDicomImportJobsRequest)
                .build());
    }

    /**
     * <p>
     * List data stores created by this AWS account.
     * </p>
     *
     * @param listDatastoresRequest
     * @return Result of the ListDatastores operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.ListDatastores
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListDatastores"
     *      target="_top">AWS API Documentation</a>
     */
    default ListDatastoresResponse listDatastores(ListDatastoresRequest listDatastoresRequest) throws ThrottlingException,
            AccessDeniedException, ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            MedicalImagingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * List data stores created by this AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListDatastoresRequest.Builder} avoiding the need to
     * create one manually via {@link ListDatastoresRequest#builder()}
     * </p>
     *
     * @param listDatastoresRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.ListDatastoresRequest.Builder} to create a
     *        request.
     * @return Result of the ListDatastores operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.ListDatastores
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListDatastores"
     *      target="_top">AWS API Documentation</a>
     */
    default ListDatastoresResponse listDatastores(Consumer<ListDatastoresRequest.Builder> listDatastoresRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, MedicalImagingException {
        return listDatastores(ListDatastoresRequest.builder().applyMutation(listDatastoresRequest).build());
    }

    /**
     * <p>
     * List data stores created by this AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDatastores(software.amazon.awssdk.services.medicalimaging.model.ListDatastoresRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.ListDatastoresIterable responses = client.listDatastoresPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.medicalimaging.paginators.ListDatastoresIterable responses = client
     *             .listDatastoresPaginator(request);
     *     for (software.amazon.awssdk.services.medicalimaging.model.ListDatastoresResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.ListDatastoresIterable responses = client.listDatastoresPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDatastores(software.amazon.awssdk.services.medicalimaging.model.ListDatastoresRequest)}
     * operation.</b>
     * </p>
     *
     * @param listDatastoresRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.ListDatastores
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListDatastores"
     *      target="_top">AWS API Documentation</a>
     */
    default ListDatastoresIterable listDatastoresPaginator(ListDatastoresRequest listDatastoresRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, MedicalImagingException {
        return new ListDatastoresIterable(this, listDatastoresRequest);
    }

    /**
     * <p>
     * List data stores created by this AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDatastores(software.amazon.awssdk.services.medicalimaging.model.ListDatastoresRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.ListDatastoresIterable responses = client.listDatastoresPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.medicalimaging.paginators.ListDatastoresIterable responses = client
     *             .listDatastoresPaginator(request);
     *     for (software.amazon.awssdk.services.medicalimaging.model.ListDatastoresResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.ListDatastoresIterable responses = client.listDatastoresPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDatastores(software.amazon.awssdk.services.medicalimaging.model.ListDatastoresRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListDatastoresRequest.Builder} avoiding the need to
     * create one manually via {@link ListDatastoresRequest#builder()}
     * </p>
     *
     * @param listDatastoresRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.ListDatastoresRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.ListDatastores
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListDatastores"
     *      target="_top">AWS API Documentation</a>
     */
    default ListDatastoresIterable listDatastoresPaginator(Consumer<ListDatastoresRequest.Builder> listDatastoresRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, MedicalImagingException {
        return listDatastoresPaginator(ListDatastoresRequest.builder().applyMutation(listDatastoresRequest).build());
    }

    /**
     * <p>
     * List image set versions.
     * </p>
     *
     * @param listImageSetVersionsRequest
     * @return Result of the ListImageSetVersions operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.ListImageSetVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListImageSetVersions"
     *      target="_top">AWS API Documentation</a>
     */
    default ListImageSetVersionsResponse listImageSetVersions(ListImageSetVersionsRequest listImageSetVersionsRequest)
            throws ThrottlingException, ConflictException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, MedicalImagingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * List image set versions.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListImageSetVersionsRequest.Builder} avoiding the
     * need to create one manually via {@link ListImageSetVersionsRequest#builder()}
     * </p>
     *
     * @param listImageSetVersionsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.ListImageSetVersionsRequest.Builder} to create
     *        a request.
     * @return Result of the ListImageSetVersions operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.ListImageSetVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListImageSetVersions"
     *      target="_top">AWS API Documentation</a>
     */
    default ListImageSetVersionsResponse listImageSetVersions(
            Consumer<ListImageSetVersionsRequest.Builder> listImageSetVersionsRequest) throws ThrottlingException,
            ConflictException, AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, MedicalImagingException {
        return listImageSetVersions(ListImageSetVersionsRequest.builder().applyMutation(listImageSetVersionsRequest).build());
    }

    /**
     * <p>
     * List image set versions.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listImageSetVersions(software.amazon.awssdk.services.medicalimaging.model.ListImageSetVersionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.ListImageSetVersionsIterable responses = client.listImageSetVersionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.medicalimaging.paginators.ListImageSetVersionsIterable responses = client
     *             .listImageSetVersionsPaginator(request);
     *     for (software.amazon.awssdk.services.medicalimaging.model.ListImageSetVersionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.ListImageSetVersionsIterable responses = client.listImageSetVersionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listImageSetVersions(software.amazon.awssdk.services.medicalimaging.model.ListImageSetVersionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listImageSetVersionsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.ListImageSetVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListImageSetVersions"
     *      target="_top">AWS API Documentation</a>
     */
    default ListImageSetVersionsIterable listImageSetVersionsPaginator(ListImageSetVersionsRequest listImageSetVersionsRequest)
            throws ThrottlingException, ConflictException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, MedicalImagingException {
        return new ListImageSetVersionsIterable(this, listImageSetVersionsRequest);
    }

    /**
     * <p>
     * List image set versions.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listImageSetVersions(software.amazon.awssdk.services.medicalimaging.model.ListImageSetVersionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.ListImageSetVersionsIterable responses = client.listImageSetVersionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.medicalimaging.paginators.ListImageSetVersionsIterable responses = client
     *             .listImageSetVersionsPaginator(request);
     *     for (software.amazon.awssdk.services.medicalimaging.model.ListImageSetVersionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.ListImageSetVersionsIterable responses = client.listImageSetVersionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listImageSetVersions(software.amazon.awssdk.services.medicalimaging.model.ListImageSetVersionsRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListImageSetVersionsRequest.Builder} avoiding the
     * need to create one manually via {@link ListImageSetVersionsRequest#builder()}
     * </p>
     *
     * @param listImageSetVersionsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.ListImageSetVersionsRequest.Builder} to create
     *        a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.ListImageSetVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListImageSetVersions"
     *      target="_top">AWS API Documentation</a>
     */
    default ListImageSetVersionsIterable listImageSetVersionsPaginator(
            Consumer<ListImageSetVersionsRequest.Builder> listImageSetVersionsRequest) throws ThrottlingException,
            ConflictException, AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, MedicalImagingException {
        return listImageSetVersionsPaginator(ListImageSetVersionsRequest.builder().applyMutation(listImageSetVersionsRequest)
                .build());
    }

    /**
     * <p>
     * Lists all tags associated with a medical imaging resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, MedicalImagingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists all tags associated with a medical imaging resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.ListTagsForResourceRequest.Builder} to create
     *        a request.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) throws ThrottlingException,
            AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, MedicalImagingException {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Search image sets based on defined input attributes.
     * </p>
     *
     * @param searchImageSetsRequest
     * @return Result of the SearchImageSets operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.SearchImageSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/SearchImageSets"
     *      target="_top">AWS API Documentation</a>
     */
    default SearchImageSetsResponse searchImageSets(SearchImageSetsRequest searchImageSetsRequest) throws ThrottlingException,
            ConflictException, AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, MedicalImagingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Search image sets based on defined input attributes.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link SearchImageSetsRequest.Builder} avoiding the need
     * to create one manually via {@link SearchImageSetsRequest#builder()}
     * </p>
     *
     * @param searchImageSetsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.SearchImageSetsRequest.Builder} to create a
     *        request.
     * @return Result of the SearchImageSets operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.SearchImageSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/SearchImageSets"
     *      target="_top">AWS API Documentation</a>
     */
    default SearchImageSetsResponse searchImageSets(Consumer<SearchImageSetsRequest.Builder> searchImageSetsRequest)
            throws ThrottlingException, ConflictException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, MedicalImagingException {
        return searchImageSets(SearchImageSetsRequest.builder().applyMutation(searchImageSetsRequest).build());
    }

    /**
     * <p>
     * Search image sets based on defined input attributes.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #searchImageSets(software.amazon.awssdk.services.medicalimaging.model.SearchImageSetsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.SearchImageSetsIterable responses = client.searchImageSetsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.medicalimaging.paginators.SearchImageSetsIterable responses = client
     *             .searchImageSetsPaginator(request);
     *     for (software.amazon.awssdk.services.medicalimaging.model.SearchImageSetsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.SearchImageSetsIterable responses = client.searchImageSetsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #searchImageSets(software.amazon.awssdk.services.medicalimaging.model.SearchImageSetsRequest)}
     * operation.</b>
     * </p>
     *
     * @param searchImageSetsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.SearchImageSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/SearchImageSets"
     *      target="_top">AWS API Documentation</a>
     */
    default SearchImageSetsIterable searchImageSetsPaginator(SearchImageSetsRequest searchImageSetsRequest)
            throws ThrottlingException, ConflictException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, MedicalImagingException {
        return new SearchImageSetsIterable(this, searchImageSetsRequest);
    }

    /**
     * <p>
     * Search image sets based on defined input attributes.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #searchImageSets(software.amazon.awssdk.services.medicalimaging.model.SearchImageSetsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.SearchImageSetsIterable responses = client.searchImageSetsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.medicalimaging.paginators.SearchImageSetsIterable responses = client
     *             .searchImageSetsPaginator(request);
     *     for (software.amazon.awssdk.services.medicalimaging.model.SearchImageSetsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.SearchImageSetsIterable responses = client.searchImageSetsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #searchImageSets(software.amazon.awssdk.services.medicalimaging.model.SearchImageSetsRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link SearchImageSetsRequest.Builder} avoiding the need
     * to create one manually via {@link SearchImageSetsRequest#builder()}
     * </p>
     *
     * @param searchImageSetsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.SearchImageSetsRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.SearchImageSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/SearchImageSets"
     *      target="_top">AWS API Documentation</a>
     */
    default SearchImageSetsIterable searchImageSetsPaginator(Consumer<SearchImageSetsRequest.Builder> searchImageSetsRequest)
            throws ThrottlingException, ConflictException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, MedicalImagingException {
        return searchImageSetsPaginator(SearchImageSetsRequest.builder().applyMutation(searchImageSetsRequest).build());
    }

    /**
     * <p>
     * Start importing bulk data into an <code>ACTIVE</code> data store. The import job imports DICOM P10 files found in
     * the S3 prefix specified by the <code>inputS3Uri</code> parameter. The import job stores processing results in the
     * file specified by the <code>outputS3Uri</code> parameter.
     * </p>
     *
     * @param startDicomImportJobRequest
     * @return Result of the StartDICOMImportJob operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ServiceQuotaExceededException
     *         The request caused a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.StartDICOMImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/StartDICOMImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    default StartDicomImportJobResponse startDICOMImportJob(StartDicomImportJobRequest startDicomImportJobRequest)
            throws ThrottlingException, ConflictException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            MedicalImagingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Start importing bulk data into an <code>ACTIVE</code> data store. The import job imports DICOM P10 files found in
     * the S3 prefix specified by the <code>inputS3Uri</code> parameter. The import job stores processing results in the
     * file specified by the <code>outputS3Uri</code> parameter.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartDicomImportJobRequest.Builder} avoiding the
     * need to create one manually via {@link StartDicomImportJobRequest#builder()}
     * </p>
     *
     * @param startDicomImportJobRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.StartDicomImportJobRequest.Builder} to create
     *        a request.
     * @return Result of the StartDICOMImportJob operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ServiceQuotaExceededException
     *         The request caused a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.StartDICOMImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/StartDICOMImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    default StartDicomImportJobResponse startDICOMImportJob(
            Consumer<StartDicomImportJobRequest.Builder> startDicomImportJobRequest) throws ThrottlingException,
            ConflictException, AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, MedicalImagingException {
        return startDICOMImportJob(StartDicomImportJobRequest.builder().applyMutation(startDicomImportJobRequest).build());
    }

    /**
     * <p>
     * Adds a user-specifed key and value tag to a medical imaging resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    default TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ThrottlingException,
            AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, MedicalImagingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Adds a user-specifed key and value tag to a medical imaging resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.TagResourceRequest.Builder} to create a
     *        request.
     * @return Result of the TagResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    default TagResourceResponse tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest) throws ThrottlingException,
            AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, MedicalImagingException {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Removes tags from a medical imaging resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    default UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ThrottlingException,
            AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, MedicalImagingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes tags from a medical imaging resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.UntagResourceRequest.Builder} to create a
     *        request.
     * @return Result of the UntagResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    default UntagResourceResponse untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, MedicalImagingException {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * <p>
     * Update image set metadata attributes.
     * </p>
     *
     * @param updateImageSetMetadataRequest
     * @return Result of the UpdateImageSetMetadata operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ServiceQuotaExceededException
     *         The request caused a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.UpdateImageSetMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/UpdateImageSetMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default UpdateImageSetMetadataResponse updateImageSetMetadata(UpdateImageSetMetadataRequest updateImageSetMetadataRequest)
            throws ThrottlingException, ConflictException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            MedicalImagingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Update image set metadata attributes.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateImageSetMetadataRequest.Builder} avoiding the
     * need to create one manually via {@link UpdateImageSetMetadataRequest#builder()}
     * </p>
     *
     * @param updateImageSetMetadataRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.UpdateImageSetMetadataRequest.Builder} to
     *        create a request.
     * @return Result of the UpdateImageSetMetadata operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ServiceQuotaExceededException
     *         The request caused a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.UpdateImageSetMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/UpdateImageSetMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default UpdateImageSetMetadataResponse updateImageSetMetadata(
            Consumer<UpdateImageSetMetadataRequest.Builder> updateImageSetMetadataRequest) throws ThrottlingException,
            ConflictException, AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, MedicalImagingException {
        return updateImageSetMetadata(UpdateImageSetMetadataRequest.builder().applyMutation(updateImageSetMetadataRequest)
                .build());
    }

    /**
     * Create a {@link MedicalImagingClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static MedicalImagingClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link MedicalImagingClient}.
     */
    static MedicalImagingClientBuilder builder() {
        return new DefaultMedicalImagingClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    default MedicalImagingServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }
}
