/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.medicalimaging;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.sync.ResponseTransformer;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.medicalimaging.model.AccessDeniedException;
import software.amazon.awssdk.services.medicalimaging.model.ConflictException;
import software.amazon.awssdk.services.medicalimaging.model.CopyImageSetRequest;
import software.amazon.awssdk.services.medicalimaging.model.CopyImageSetResponse;
import software.amazon.awssdk.services.medicalimaging.model.CreateDatastoreRequest;
import software.amazon.awssdk.services.medicalimaging.model.CreateDatastoreResponse;
import software.amazon.awssdk.services.medicalimaging.model.DeleteDatastoreRequest;
import software.amazon.awssdk.services.medicalimaging.model.DeleteDatastoreResponse;
import software.amazon.awssdk.services.medicalimaging.model.DeleteImageSetRequest;
import software.amazon.awssdk.services.medicalimaging.model.DeleteImageSetResponse;
import software.amazon.awssdk.services.medicalimaging.model.GetDatastoreRequest;
import software.amazon.awssdk.services.medicalimaging.model.GetDatastoreResponse;
import software.amazon.awssdk.services.medicalimaging.model.GetDicomImportJobRequest;
import software.amazon.awssdk.services.medicalimaging.model.GetDicomImportJobResponse;
import software.amazon.awssdk.services.medicalimaging.model.GetImageFrameRequest;
import software.amazon.awssdk.services.medicalimaging.model.GetImageFrameResponse;
import software.amazon.awssdk.services.medicalimaging.model.GetImageSetMetadataRequest;
import software.amazon.awssdk.services.medicalimaging.model.GetImageSetMetadataResponse;
import software.amazon.awssdk.services.medicalimaging.model.GetImageSetRequest;
import software.amazon.awssdk.services.medicalimaging.model.GetImageSetResponse;
import software.amazon.awssdk.services.medicalimaging.model.InternalServerException;
import software.amazon.awssdk.services.medicalimaging.model.ListDatastoresRequest;
import software.amazon.awssdk.services.medicalimaging.model.ListDatastoresResponse;
import software.amazon.awssdk.services.medicalimaging.model.ListDicomImportJobsRequest;
import software.amazon.awssdk.services.medicalimaging.model.ListDicomImportJobsResponse;
import software.amazon.awssdk.services.medicalimaging.model.ListImageSetVersionsRequest;
import software.amazon.awssdk.services.medicalimaging.model.ListImageSetVersionsResponse;
import software.amazon.awssdk.services.medicalimaging.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.medicalimaging.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.medicalimaging.model.MedicalImagingException;
import software.amazon.awssdk.services.medicalimaging.model.ResourceNotFoundException;
import software.amazon.awssdk.services.medicalimaging.model.SearchImageSetsRequest;
import software.amazon.awssdk.services.medicalimaging.model.SearchImageSetsResponse;
import software.amazon.awssdk.services.medicalimaging.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.medicalimaging.model.StartDicomImportJobRequest;
import software.amazon.awssdk.services.medicalimaging.model.StartDicomImportJobResponse;
import software.amazon.awssdk.services.medicalimaging.model.TagResourceRequest;
import software.amazon.awssdk.services.medicalimaging.model.TagResourceResponse;
import software.amazon.awssdk.services.medicalimaging.model.ThrottlingException;
import software.amazon.awssdk.services.medicalimaging.model.UntagResourceRequest;
import software.amazon.awssdk.services.medicalimaging.model.UntagResourceResponse;
import software.amazon.awssdk.services.medicalimaging.model.UpdateImageSetMetadataRequest;
import software.amazon.awssdk.services.medicalimaging.model.UpdateImageSetMetadataResponse;
import software.amazon.awssdk.services.medicalimaging.model.ValidationException;
import software.amazon.awssdk.services.medicalimaging.transform.CopyImageSetRequestMarshaller;
import software.amazon.awssdk.services.medicalimaging.transform.CreateDatastoreRequestMarshaller;
import software.amazon.awssdk.services.medicalimaging.transform.DeleteDatastoreRequestMarshaller;
import software.amazon.awssdk.services.medicalimaging.transform.DeleteImageSetRequestMarshaller;
import software.amazon.awssdk.services.medicalimaging.transform.GetDatastoreRequestMarshaller;
import software.amazon.awssdk.services.medicalimaging.transform.GetDicomImportJobRequestMarshaller;
import software.amazon.awssdk.services.medicalimaging.transform.GetImageFrameRequestMarshaller;
import software.amazon.awssdk.services.medicalimaging.transform.GetImageSetMetadataRequestMarshaller;
import software.amazon.awssdk.services.medicalimaging.transform.GetImageSetRequestMarshaller;
import software.amazon.awssdk.services.medicalimaging.transform.ListDatastoresRequestMarshaller;
import software.amazon.awssdk.services.medicalimaging.transform.ListDicomImportJobsRequestMarshaller;
import software.amazon.awssdk.services.medicalimaging.transform.ListImageSetVersionsRequestMarshaller;
import software.amazon.awssdk.services.medicalimaging.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.medicalimaging.transform.SearchImageSetsRequestMarshaller;
import software.amazon.awssdk.services.medicalimaging.transform.StartDicomImportJobRequestMarshaller;
import software.amazon.awssdk.services.medicalimaging.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.medicalimaging.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.medicalimaging.transform.UpdateImageSetMetadataRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link MedicalImagingClient}.
 *
 * @see MedicalImagingClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMedicalImagingClient implements MedicalImagingClient {
    private static final Logger log = Logger.loggerFor(DefaultMedicalImagingClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final MedicalImagingServiceClientConfiguration serviceClientConfiguration;

    protected DefaultMedicalImagingClient(MedicalImagingServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Copy an image set.
     * </p>
     *
     * @param copyImageSetRequest
     * @return Result of the CopyImageSet operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ServiceQuotaExceededException
     *         The request caused a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.CopyImageSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/CopyImageSet" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CopyImageSetResponse copyImageSet(CopyImageSetRequest copyImageSetRequest) throws ThrottlingException,
            ConflictException, AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, MedicalImagingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CopyImageSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CopyImageSetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, copyImageSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Medical Imaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CopyImageSet");
            String hostPrefix = "runtime-";
            String resolvedHostExpression = "runtime-";

            return clientHandler.execute(new ClientExecutionParams<CopyImageSetRequest, CopyImageSetResponse>()
                    .withOperationName("CopyImageSet").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(copyImageSetRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CopyImageSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Create a data store.
     * </p>
     *
     * @param createDatastoreRequest
     * @return Result of the CreateDatastore operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ServiceQuotaExceededException
     *         The request caused a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.CreateDatastore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/CreateDatastore"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateDatastoreResponse createDatastore(CreateDatastoreRequest createDatastoreRequest) throws ThrottlingException,
            ConflictException, AccessDeniedException, ValidationException, InternalServerException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, MedicalImagingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDatastoreResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateDatastoreResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDatastoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Medical Imaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDatastore");

            return clientHandler.execute(new ClientExecutionParams<CreateDatastoreRequest, CreateDatastoreResponse>()
                    .withOperationName("CreateDatastore").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createDatastoreRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateDatastoreRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Delete a data store.
     * </p>
     * <note>
     * <p>
     * Before a data store can be deleted, you must first delete all image sets within it.
     * </p>
     * </note>
     *
     * @param deleteDatastoreRequest
     * @return Result of the DeleteDatastore operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.DeleteDatastore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/DeleteDatastore"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteDatastoreResponse deleteDatastore(DeleteDatastoreRequest deleteDatastoreRequest) throws ThrottlingException,
            ConflictException, AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, MedicalImagingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDatastoreResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteDatastoreResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDatastoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Medical Imaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDatastore");

            return clientHandler.execute(new ClientExecutionParams<DeleteDatastoreRequest, DeleteDatastoreResponse>()
                    .withOperationName("DeleteDatastore").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteDatastoreRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteDatastoreRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Delete an image set.
     * </p>
     *
     * @param deleteImageSetRequest
     * @return Result of the DeleteImageSet operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.DeleteImageSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/DeleteImageSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteImageSetResponse deleteImageSet(DeleteImageSetRequest deleteImageSetRequest) throws ThrottlingException,
            ConflictException, AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, MedicalImagingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteImageSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteImageSetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteImageSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Medical Imaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteImageSet");
            String hostPrefix = "runtime-";
            String resolvedHostExpression = "runtime-";

            return clientHandler.execute(new ClientExecutionParams<DeleteImageSetRequest, DeleteImageSetResponse>()
                    .withOperationName("DeleteImageSet").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(deleteImageSetRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteImageSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get the import job properties to learn more about the job or job progress.
     * </p>
     *
     * @param getDicomImportJobRequest
     * @return Result of the GetDICOMImportJob operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.GetDICOMImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetDICOMImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetDicomImportJobResponse getDICOMImportJob(GetDicomImportJobRequest getDicomImportJobRequest)
            throws ThrottlingException, ConflictException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, MedicalImagingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDicomImportJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetDicomImportJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDicomImportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Medical Imaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDICOMImportJob");

            return clientHandler.execute(new ClientExecutionParams<GetDicomImportJobRequest, GetDicomImportJobResponse>()
                    .withOperationName("GetDICOMImportJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getDicomImportJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetDicomImportJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get data store properties.
     * </p>
     *
     * @param getDatastoreRequest
     * @return Result of the GetDatastore operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.GetDatastore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetDatastore" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetDatastoreResponse getDatastore(GetDatastoreRequest getDatastoreRequest) throws ThrottlingException,
            AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, MedicalImagingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDatastoreResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetDatastoreResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDatastoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Medical Imaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDatastore");

            return clientHandler.execute(new ClientExecutionParams<GetDatastoreRequest, GetDatastoreResponse>()
                    .withOperationName("GetDatastore").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getDatastoreRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetDatastoreRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get an image frame (pixel data) for an image set.
     * </p>
     *
     * @param getImageFrameRequest
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled GetImageFrameResponse
     *        and an InputStream to the response content are provided as parameters to the callback. The callback may
     *        return a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows '
     *        <p>
     *        The blob containing the aggregated image frame information.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.GetImageFrame
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageFrame" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public <ReturnT> ReturnT getImageFrame(GetImageFrameRequest getImageFrameRequest,
            ResponseTransformer<GetImageFrameResponse, ReturnT> responseTransformer) throws ThrottlingException,
            ConflictException, AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, MedicalImagingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                .isPayloadJson(false).build();

        HttpResponseHandler<GetImageFrameResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetImageFrameResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getImageFrameRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Medical Imaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetImageFrame");
            String hostPrefix = "runtime-";
            String resolvedHostExpression = "runtime-";

            return clientHandler.execute(
                    new ClientExecutionParams<GetImageFrameRequest, GetImageFrameResponse>().withOperationName("GetImageFrame")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .hostPrefixExpression(resolvedHostExpression).withInput(getImageFrameRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetImageFrameRequestMarshaller(protocolFactory)), responseTransformer);
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get image set properties.
     * </p>
     *
     * @param getImageSetRequest
     * @return Result of the GetImageSet operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.GetImageSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageSet" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetImageSetResponse getImageSet(GetImageSetRequest getImageSetRequest) throws ThrottlingException, ConflictException,
            AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, MedicalImagingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetImageSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetImageSetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getImageSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Medical Imaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetImageSet");
            String hostPrefix = "runtime-";
            String resolvedHostExpression = "runtime-";

            return clientHandler.execute(new ClientExecutionParams<GetImageSetRequest, GetImageSetResponse>()
                    .withOperationName("GetImageSet").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(getImageSetRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetImageSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get metadata attributes for an image set.
     * </p>
     *
     * @param getImageSetMetadataRequest
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled
     *        GetImageSetMetadataResponse and an InputStream to the response content are provided as parameters to the
     *        callback. The callback may return a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows '
     *        <p>
     *        The blob containing the aggregated metadata information for the image set.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.GetImageSetMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageSetMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public <ReturnT> ReturnT getImageSetMetadata(GetImageSetMetadataRequest getImageSetMetadataRequest,
            ResponseTransformer<GetImageSetMetadataResponse, ReturnT> responseTransformer) throws ThrottlingException,
            ConflictException, AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, MedicalImagingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                .isPayloadJson(false).build();

        HttpResponseHandler<GetImageSetMetadataResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetImageSetMetadataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getImageSetMetadataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Medical Imaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetImageSetMetadata");
            String hostPrefix = "runtime-";
            String resolvedHostExpression = "runtime-";

            return clientHandler.execute(
                    new ClientExecutionParams<GetImageSetMetadataRequest, GetImageSetMetadataResponse>()
                            .withOperationName("GetImageSetMetadata").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withInput(getImageSetMetadataRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetImageSetMetadataRequestMarshaller(protocolFactory)), responseTransformer);
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List import jobs created by this AWS account for a specific data store.
     * </p>
     *
     * @param listDicomImportJobsRequest
     * @return Result of the ListDICOMImportJobs operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.ListDICOMImportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListDICOMImportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDicomImportJobsResponse listDICOMImportJobs(ListDicomImportJobsRequest listDicomImportJobsRequest)
            throws ThrottlingException, ConflictException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, MedicalImagingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDicomImportJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListDicomImportJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDicomImportJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Medical Imaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDICOMImportJobs");

            return clientHandler.execute(new ClientExecutionParams<ListDicomImportJobsRequest, ListDicomImportJobsResponse>()
                    .withOperationName("ListDICOMImportJobs").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listDicomImportJobsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDicomImportJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List data stores created by this AWS account.
     * </p>
     *
     * @param listDatastoresRequest
     * @return Result of the ListDatastores operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.ListDatastores
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListDatastores"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDatastoresResponse listDatastores(ListDatastoresRequest listDatastoresRequest) throws ThrottlingException,
            AccessDeniedException, ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            MedicalImagingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDatastoresResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListDatastoresResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDatastoresRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Medical Imaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDatastores");

            return clientHandler.execute(new ClientExecutionParams<ListDatastoresRequest, ListDatastoresResponse>()
                    .withOperationName("ListDatastores").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listDatastoresRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDatastoresRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List image set versions.
     * </p>
     *
     * @param listImageSetVersionsRequest
     * @return Result of the ListImageSetVersions operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.ListImageSetVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListImageSetVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListImageSetVersionsResponse listImageSetVersions(ListImageSetVersionsRequest listImageSetVersionsRequest)
            throws ThrottlingException, ConflictException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, MedicalImagingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListImageSetVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListImageSetVersionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listImageSetVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Medical Imaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListImageSetVersions");
            String hostPrefix = "runtime-";
            String resolvedHostExpression = "runtime-";

            return clientHandler.execute(new ClientExecutionParams<ListImageSetVersionsRequest, ListImageSetVersionsResponse>()
                    .withOperationName("ListImageSetVersions").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(listImageSetVersionsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListImageSetVersionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all tags associated with a medical imaging resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, MedicalImagingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Medical Imaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Search image sets based on defined input attributes.
     * </p>
     *
     * @param searchImageSetsRequest
     * @return Result of the SearchImageSets operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.SearchImageSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/SearchImageSets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SearchImageSetsResponse searchImageSets(SearchImageSetsRequest searchImageSetsRequest) throws ThrottlingException,
            ConflictException, AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, MedicalImagingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SearchImageSetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SearchImageSetsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchImageSetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Medical Imaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchImageSets");
            String hostPrefix = "runtime-";
            String resolvedHostExpression = "runtime-";

            return clientHandler.execute(new ClientExecutionParams<SearchImageSetsRequest, SearchImageSetsResponse>()
                    .withOperationName("SearchImageSets").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(searchImageSetsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SearchImageSetsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Start importing bulk data into an <code>ACTIVE</code> data store. The import job imports DICOM P10 files found in
     * the S3 prefix specified by the <code>inputS3Uri</code> parameter. The import job stores processing results in the
     * file specified by the <code>outputS3Uri</code> parameter.
     * </p>
     *
     * @param startDicomImportJobRequest
     * @return Result of the StartDICOMImportJob operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ServiceQuotaExceededException
     *         The request caused a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.StartDICOMImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/StartDICOMImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartDicomImportJobResponse startDICOMImportJob(StartDicomImportJobRequest startDicomImportJobRequest)
            throws ThrottlingException, ConflictException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            MedicalImagingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartDicomImportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartDicomImportJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startDicomImportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Medical Imaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartDICOMImportJob");

            return clientHandler.execute(new ClientExecutionParams<StartDicomImportJobRequest, StartDicomImportJobResponse>()
                    .withOperationName("StartDICOMImportJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startDicomImportJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartDicomImportJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds a user-specifed key and value tag to a medical imaging resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ThrottlingException,
            AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, MedicalImagingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Medical Imaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from a medical imaging resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ThrottlingException,
            AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, MedicalImagingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Medical Imaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Update image set metadata attributes.
     * </p>
     *
     * @param updateImageSetMetadataRequest
     * @return Result of the UpdateImageSetMetadata operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         The user does not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ServiceQuotaExceededException
     *         The request caused a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MedicalImagingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MedicalImagingClient.UpdateImageSetMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/UpdateImageSetMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateImageSetMetadataResponse updateImageSetMetadata(UpdateImageSetMetadataRequest updateImageSetMetadataRequest)
            throws ThrottlingException, ConflictException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            MedicalImagingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateImageSetMetadataResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateImageSetMetadataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateImageSetMetadataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Medical Imaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateImageSetMetadata");
            String hostPrefix = "runtime-";
            String resolvedHostExpression = "runtime-";

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateImageSetMetadataRequest, UpdateImageSetMetadataResponse>()
                            .withOperationName("UpdateImageSetMetadata").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withInput(updateImageSetMetadataRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateImageSetMetadataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MedicalImagingException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public final MedicalImagingServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
