/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediapackagev2;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.mediapackagev2.internal.MediaPackageV2ServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.mediapackagev2.model.AccessDeniedException;
import software.amazon.awssdk.services.mediapackagev2.model.ConflictException;
import software.amazon.awssdk.services.mediapackagev2.model.CreateChannelGroupRequest;
import software.amazon.awssdk.services.mediapackagev2.model.CreateChannelGroupResponse;
import software.amazon.awssdk.services.mediapackagev2.model.CreateChannelRequest;
import software.amazon.awssdk.services.mediapackagev2.model.CreateChannelResponse;
import software.amazon.awssdk.services.mediapackagev2.model.CreateOriginEndpointRequest;
import software.amazon.awssdk.services.mediapackagev2.model.CreateOriginEndpointResponse;
import software.amazon.awssdk.services.mediapackagev2.model.DeleteChannelGroupRequest;
import software.amazon.awssdk.services.mediapackagev2.model.DeleteChannelGroupResponse;
import software.amazon.awssdk.services.mediapackagev2.model.DeleteChannelPolicyRequest;
import software.amazon.awssdk.services.mediapackagev2.model.DeleteChannelPolicyResponse;
import software.amazon.awssdk.services.mediapackagev2.model.DeleteChannelRequest;
import software.amazon.awssdk.services.mediapackagev2.model.DeleteChannelResponse;
import software.amazon.awssdk.services.mediapackagev2.model.DeleteOriginEndpointPolicyRequest;
import software.amazon.awssdk.services.mediapackagev2.model.DeleteOriginEndpointPolicyResponse;
import software.amazon.awssdk.services.mediapackagev2.model.DeleteOriginEndpointRequest;
import software.amazon.awssdk.services.mediapackagev2.model.DeleteOriginEndpointResponse;
import software.amazon.awssdk.services.mediapackagev2.model.GetChannelGroupRequest;
import software.amazon.awssdk.services.mediapackagev2.model.GetChannelGroupResponse;
import software.amazon.awssdk.services.mediapackagev2.model.GetChannelPolicyRequest;
import software.amazon.awssdk.services.mediapackagev2.model.GetChannelPolicyResponse;
import software.amazon.awssdk.services.mediapackagev2.model.GetChannelRequest;
import software.amazon.awssdk.services.mediapackagev2.model.GetChannelResponse;
import software.amazon.awssdk.services.mediapackagev2.model.GetOriginEndpointPolicyRequest;
import software.amazon.awssdk.services.mediapackagev2.model.GetOriginEndpointPolicyResponse;
import software.amazon.awssdk.services.mediapackagev2.model.GetOriginEndpointRequest;
import software.amazon.awssdk.services.mediapackagev2.model.GetOriginEndpointResponse;
import software.amazon.awssdk.services.mediapackagev2.model.InternalServerException;
import software.amazon.awssdk.services.mediapackagev2.model.ListChannelGroupsRequest;
import software.amazon.awssdk.services.mediapackagev2.model.ListChannelGroupsResponse;
import software.amazon.awssdk.services.mediapackagev2.model.ListChannelsRequest;
import software.amazon.awssdk.services.mediapackagev2.model.ListChannelsResponse;
import software.amazon.awssdk.services.mediapackagev2.model.ListOriginEndpointsRequest;
import software.amazon.awssdk.services.mediapackagev2.model.ListOriginEndpointsResponse;
import software.amazon.awssdk.services.mediapackagev2.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.mediapackagev2.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.mediapackagev2.model.MediaPackageV2Exception;
import software.amazon.awssdk.services.mediapackagev2.model.PutChannelPolicyRequest;
import software.amazon.awssdk.services.mediapackagev2.model.PutChannelPolicyResponse;
import software.amazon.awssdk.services.mediapackagev2.model.PutOriginEndpointPolicyRequest;
import software.amazon.awssdk.services.mediapackagev2.model.PutOriginEndpointPolicyResponse;
import software.amazon.awssdk.services.mediapackagev2.model.ResourceNotFoundException;
import software.amazon.awssdk.services.mediapackagev2.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.mediapackagev2.model.TagResourceRequest;
import software.amazon.awssdk.services.mediapackagev2.model.TagResourceResponse;
import software.amazon.awssdk.services.mediapackagev2.model.ThrottlingException;
import software.amazon.awssdk.services.mediapackagev2.model.UntagResourceRequest;
import software.amazon.awssdk.services.mediapackagev2.model.UntagResourceResponse;
import software.amazon.awssdk.services.mediapackagev2.model.UpdateChannelGroupRequest;
import software.amazon.awssdk.services.mediapackagev2.model.UpdateChannelGroupResponse;
import software.amazon.awssdk.services.mediapackagev2.model.UpdateChannelRequest;
import software.amazon.awssdk.services.mediapackagev2.model.UpdateChannelResponse;
import software.amazon.awssdk.services.mediapackagev2.model.UpdateOriginEndpointRequest;
import software.amazon.awssdk.services.mediapackagev2.model.UpdateOriginEndpointResponse;
import software.amazon.awssdk.services.mediapackagev2.model.ValidationException;
import software.amazon.awssdk.services.mediapackagev2.transform.CreateChannelGroupRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.CreateChannelRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.CreateOriginEndpointRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.DeleteChannelGroupRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.DeleteChannelPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.DeleteChannelRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.DeleteOriginEndpointPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.DeleteOriginEndpointRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.GetChannelGroupRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.GetChannelPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.GetChannelRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.GetOriginEndpointPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.GetOriginEndpointRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.ListChannelGroupsRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.ListChannelsRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.ListOriginEndpointsRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.PutChannelPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.PutOriginEndpointPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.UpdateChannelGroupRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.UpdateChannelRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.UpdateOriginEndpointRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link MediaPackageV2AsyncClient}.
 *
 * @see MediaPackageV2AsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMediaPackageV2AsyncClient implements MediaPackageV2AsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultMediaPackageV2AsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final MediaPackageV2ServiceClientConfiguration serviceClientConfiguration;

    protected DefaultMediaPackageV2AsyncClient(MediaPackageV2ServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Create a channel to start receiving content streams. The channel represents the input to MediaPackage for
     * incoming live content from an encoder such as AWS Elemental MediaLive. The channel receives content, and after
     * packaging it, outputs it through an origin endpoint to downstream devices (such as video players or CDNs) that
     * request the content. You can create only one channel with each request. We recommend that you spread out channels
     * between channel groups, such as putting redundant channels in the same AWS Region in different channel groups.
     * </p>
     *
     * @param createChannelRequest
     * @return A Java Future containing the result of the CreateChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ConflictException Updating or deleting this resource can cause an inconsistent state.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see Access Management in the IAM User Guide.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageV2AsyncClient.CreateChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/CreateChannel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateChannelResponse> createChannel(CreateChannelRequest createChannelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createChannelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateChannelRequest, CreateChannelResponse>()
                            .withOperationName("CreateChannel")
                            .withMarshaller(new CreateChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createChannelRequest));
            CompletableFuture<CreateChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create a channel group to group your channels and origin endpoints. A channel group is the top-level resource
     * that consists of channels and origin endpoints that are associated with it and that provides predictable URLs for
     * stream delivery. All channels and origin endpoints within the channel group are guaranteed to share the DNS. You
     * can create only one channel group with each request.
     * </p>
     *
     * @param createChannelGroupRequest
     * @return A Java Future containing the result of the CreateChannelGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ConflictException Updating or deleting this resource can cause an inconsistent state.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see Access Management in the IAM User Guide.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageV2AsyncClient.CreateChannelGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/CreateChannelGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateChannelGroupResponse> createChannelGroup(CreateChannelGroupRequest createChannelGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createChannelGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createChannelGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateChannelGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateChannelGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateChannelGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateChannelGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateChannelGroupRequest, CreateChannelGroupResponse>()
                            .withOperationName("CreateChannelGroup")
                            .withMarshaller(new CreateChannelGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createChannelGroupRequest));
            CompletableFuture<CreateChannelGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The endpoint is attached to a channel, and represents the output of the live content. You can associate multiple
     * endpoints to a single channel. Each endpoint gives players and downstream CDNs (such as Amazon CloudFront) access
     * to the content for playback. Content can't be served from a channel until it has an endpoint. You can create only
     * one endpoint with each request.
     * </p>
     *
     * @param createOriginEndpointRequest
     * @return A Java Future containing the result of the CreateOriginEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ConflictException Updating or deleting this resource can cause an inconsistent state.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see Access Management in the IAM User Guide.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageV2AsyncClient.CreateOriginEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/CreateOriginEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateOriginEndpointResponse> createOriginEndpoint(
            CreateOriginEndpointRequest createOriginEndpointRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createOriginEndpointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createOriginEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateOriginEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateOriginEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateOriginEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateOriginEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateOriginEndpointRequest, CreateOriginEndpointResponse>()
                            .withOperationName("CreateOriginEndpoint")
                            .withMarshaller(new CreateOriginEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createOriginEndpointRequest));
            CompletableFuture<CreateOriginEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete a channel to stop AWS Elemental MediaPackage from receiving further content. You must delete the channel's
     * origin endpoints before you can delete the channel.
     * </p>
     *
     * @param deleteChannelRequest
     * @return A Java Future containing the result of the DeleteChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ConflictException Updating or deleting this resource can cause an inconsistent state.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see Access Management in the IAM User Guide.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageV2AsyncClient.DeleteChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/DeleteChannel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteChannelResponse> deleteChannel(DeleteChannelRequest deleteChannelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteChannelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteChannelRequest, DeleteChannelResponse>()
                            .withOperationName("DeleteChannel")
                            .withMarshaller(new DeleteChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteChannelRequest));
            CompletableFuture<DeleteChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete a channel group. You must delete the channel group's channels and origin endpoints before you can delete
     * the channel group. If you delete a channel group, you'll lose access to the egress domain and will have to create
     * a new channel group to replace it.
     * </p>
     *
     * @param deleteChannelGroupRequest
     * @return A Java Future containing the result of the DeleteChannelGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ConflictException Updating or deleting this resource can cause an inconsistent state.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see Access Management in the IAM User Guide.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageV2AsyncClient.DeleteChannelGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/DeleteChannelGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteChannelGroupResponse> deleteChannelGroup(DeleteChannelGroupRequest deleteChannelGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteChannelGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteChannelGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteChannelGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteChannelGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteChannelGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteChannelGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteChannelGroupRequest, DeleteChannelGroupResponse>()
                            .withOperationName("DeleteChannelGroup")
                            .withMarshaller(new DeleteChannelGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteChannelGroupRequest));
            CompletableFuture<DeleteChannelGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete a channel policy.
     * </p>
     *
     * @param deleteChannelPolicyRequest
     * @return A Java Future containing the result of the DeleteChannelPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ConflictException Updating or deleting this resource can cause an inconsistent state.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see Access Management in the IAM User Guide.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageV2AsyncClient.DeleteChannelPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/DeleteChannelPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteChannelPolicyResponse> deleteChannelPolicy(
            DeleteChannelPolicyRequest deleteChannelPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteChannelPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteChannelPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteChannelPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteChannelPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteChannelPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteChannelPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteChannelPolicyRequest, DeleteChannelPolicyResponse>()
                            .withOperationName("DeleteChannelPolicy")
                            .withMarshaller(new DeleteChannelPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteChannelPolicyRequest));
            CompletableFuture<DeleteChannelPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Origin endpoints can serve content until they're deleted. Delete the endpoint if it should no longer respond to
     * playback requests. You must delete all endpoints from a channel before you can delete the channel.
     * </p>
     *
     * @param deleteOriginEndpointRequest
     * @return A Java Future containing the result of the DeleteOriginEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see Access Management in the IAM User Guide.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageV2AsyncClient.DeleteOriginEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/DeleteOriginEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteOriginEndpointResponse> deleteOriginEndpoint(
            DeleteOriginEndpointRequest deleteOriginEndpointRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteOriginEndpointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteOriginEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteOriginEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteOriginEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteOriginEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteOriginEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteOriginEndpointRequest, DeleteOriginEndpointResponse>()
                            .withOperationName("DeleteOriginEndpoint")
                            .withMarshaller(new DeleteOriginEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteOriginEndpointRequest));
            CompletableFuture<DeleteOriginEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete an origin endpoint policy.
     * </p>
     *
     * @param deleteOriginEndpointPolicyRequest
     * @return A Java Future containing the result of the DeleteOriginEndpointPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ConflictException Updating or deleting this resource can cause an inconsistent state.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see Access Management in the IAM User Guide.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageV2AsyncClient.DeleteOriginEndpointPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/DeleteOriginEndpointPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteOriginEndpointPolicyResponse> deleteOriginEndpointPolicy(
            DeleteOriginEndpointPolicyRequest deleteOriginEndpointPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteOriginEndpointPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteOriginEndpointPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteOriginEndpointPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteOriginEndpointPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteOriginEndpointPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteOriginEndpointPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteOriginEndpointPolicyRequest, DeleteOriginEndpointPolicyResponse>()
                            .withOperationName("DeleteOriginEndpointPolicy")
                            .withMarshaller(new DeleteOriginEndpointPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteOriginEndpointPolicyRequest));
            CompletableFuture<DeleteOriginEndpointPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the specified channel that's configured in AWS Elemental MediaPackage, including the origin endpoints
     * that are associated with it.
     * </p>
     *
     * @param getChannelRequest
     * @return A Java Future containing the result of the GetChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see Access Management in the IAM User Guide.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageV2AsyncClient.GetChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/GetChannel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetChannelResponse> getChannel(GetChannelRequest getChannelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getChannelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetChannelRequest, GetChannelResponse>().withOperationName("GetChannel")
                            .withMarshaller(new GetChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getChannelRequest));
            CompletableFuture<GetChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the specified channel group that's configured in AWS Elemental MediaPackage, including the channels and
     * origin endpoints that are associated with it.
     * </p>
     *
     * @param getChannelGroupRequest
     * @return A Java Future containing the result of the GetChannelGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see Access Management in the IAM User Guide.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageV2AsyncClient.GetChannelGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/GetChannelGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetChannelGroupResponse> getChannelGroup(GetChannelGroupRequest getChannelGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getChannelGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getChannelGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetChannelGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetChannelGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetChannelGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetChannelGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetChannelGroupRequest, GetChannelGroupResponse>()
                            .withOperationName("GetChannelGroup")
                            .withMarshaller(new GetChannelGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getChannelGroupRequest));
            CompletableFuture<GetChannelGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the specified channel policy that's configured in AWS Elemental MediaPackage. With policies, you can
     * specify who has access to AWS resources and what actions they can perform on those resources.
     * </p>
     *
     * @param getChannelPolicyRequest
     * @return A Java Future containing the result of the GetChannelPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see Access Management in the IAM User Guide.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageV2AsyncClient.GetChannelPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/GetChannelPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetChannelPolicyResponse> getChannelPolicy(GetChannelPolicyRequest getChannelPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getChannelPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getChannelPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetChannelPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetChannelPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetChannelPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetChannelPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetChannelPolicyRequest, GetChannelPolicyResponse>()
                            .withOperationName("GetChannelPolicy")
                            .withMarshaller(new GetChannelPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getChannelPolicyRequest));
            CompletableFuture<GetChannelPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the specified origin endpoint that's configured in AWS Elemental MediaPackage to obtain its playback
     * URL and to view the packaging settings that it's currently using.
     * </p>
     *
     * @param getOriginEndpointRequest
     * @return A Java Future containing the result of the GetOriginEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see Access Management in the IAM User Guide.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageV2AsyncClient.GetOriginEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/GetOriginEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetOriginEndpointResponse> getOriginEndpoint(GetOriginEndpointRequest getOriginEndpointRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getOriginEndpointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getOriginEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetOriginEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetOriginEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetOriginEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetOriginEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetOriginEndpointRequest, GetOriginEndpointResponse>()
                            .withOperationName("GetOriginEndpoint")
                            .withMarshaller(new GetOriginEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getOriginEndpointRequest));
            CompletableFuture<GetOriginEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the specified origin endpoint policy that's configured in AWS Elemental MediaPackage.
     * </p>
     *
     * @param getOriginEndpointPolicyRequest
     * @return A Java Future containing the result of the GetOriginEndpointPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see Access Management in the IAM User Guide.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageV2AsyncClient.GetOriginEndpointPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/GetOriginEndpointPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetOriginEndpointPolicyResponse> getOriginEndpointPolicy(
            GetOriginEndpointPolicyRequest getOriginEndpointPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getOriginEndpointPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getOriginEndpointPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetOriginEndpointPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetOriginEndpointPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetOriginEndpointPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetOriginEndpointPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetOriginEndpointPolicyRequest, GetOriginEndpointPolicyResponse>()
                            .withOperationName("GetOriginEndpointPolicy")
                            .withMarshaller(new GetOriginEndpointPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getOriginEndpointPolicyRequest));
            CompletableFuture<GetOriginEndpointPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves all channel groups that are configured in AWS Elemental MediaPackage, including the channels and origin
     * endpoints that are associated with it.
     * </p>
     *
     * @param listChannelGroupsRequest
     * @return A Java Future containing the result of the ListChannelGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see Access Management in the IAM User Guide.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageV2AsyncClient.ListChannelGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/ListChannelGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListChannelGroupsResponse> listChannelGroups(ListChannelGroupsRequest listChannelGroupsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listChannelGroupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listChannelGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListChannelGroups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListChannelGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListChannelGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListChannelGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListChannelGroupsRequest, ListChannelGroupsResponse>()
                            .withOperationName("ListChannelGroups")
                            .withMarshaller(new ListChannelGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listChannelGroupsRequest));
            CompletableFuture<ListChannelGroupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves all channels in a specific channel group that are configured in AWS Elemental MediaPackage, including
     * the origin endpoints that are associated with it.
     * </p>
     *
     * @param listChannelsRequest
     * @return A Java Future containing the result of the ListChannels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see Access Management in the IAM User Guide.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageV2AsyncClient.ListChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/ListChannels" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListChannelsResponse> listChannels(ListChannelsRequest listChannelsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listChannelsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listChannelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListChannels");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListChannelsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListChannelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListChannelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListChannelsRequest, ListChannelsResponse>()
                            .withOperationName("ListChannels").withMarshaller(new ListChannelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listChannelsRequest));
            CompletableFuture<ListChannelsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves all origin endpoints in a specific channel that are configured in AWS Elemental MediaPackage.
     * </p>
     *
     * @param listOriginEndpointsRequest
     * @return A Java Future containing the result of the ListOriginEndpoints operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see Access Management in the IAM User Guide.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageV2AsyncClient.ListOriginEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/ListOriginEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListOriginEndpointsResponse> listOriginEndpoints(
            ListOriginEndpointsRequest listOriginEndpointsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listOriginEndpointsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listOriginEndpointsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOriginEndpoints");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListOriginEndpointsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListOriginEndpointsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListOriginEndpointsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListOriginEndpointsRequest, ListOriginEndpointsResponse>()
                            .withOperationName("ListOriginEndpoints")
                            .withMarshaller(new ListOriginEndpointsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listOriginEndpointsRequest));
            CompletableFuture<ListOriginEndpointsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags assigned to a resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageV2AsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Attaches an IAM policy to the specified channel. With policies, you can specify who has access to AWS resources
     * and what actions they can perform on those resources. You can attach only one policy with each request.
     * </p>
     *
     * @param putChannelPolicyRequest
     * @return A Java Future containing the result of the PutChannelPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ConflictException Updating or deleting this resource can cause an inconsistent state.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see Access Management in the IAM User Guide.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageV2AsyncClient.PutChannelPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/PutChannelPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutChannelPolicyResponse> putChannelPolicy(PutChannelPolicyRequest putChannelPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putChannelPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putChannelPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutChannelPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutChannelPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutChannelPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutChannelPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutChannelPolicyRequest, PutChannelPolicyResponse>()
                            .withOperationName("PutChannelPolicy")
                            .withMarshaller(new PutChannelPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putChannelPolicyRequest));
            CompletableFuture<PutChannelPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Attaches an IAM policy to the specified origin endpoint. You can attach only one policy with each request.
     * </p>
     *
     * @param putOriginEndpointPolicyRequest
     * @return A Java Future containing the result of the PutOriginEndpointPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ConflictException Updating or deleting this resource can cause an inconsistent state.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see Access Management in the IAM User Guide.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageV2AsyncClient.PutOriginEndpointPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/PutOriginEndpointPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutOriginEndpointPolicyResponse> putOriginEndpointPolicy(
            PutOriginEndpointPolicyRequest putOriginEndpointPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putOriginEndpointPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putOriginEndpointPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutOriginEndpointPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutOriginEndpointPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutOriginEndpointPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutOriginEndpointPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutOriginEndpointPolicyRequest, PutOriginEndpointPolicyResponse>()
                            .withOperationName("PutOriginEndpointPolicy")
                            .withMarshaller(new PutOriginEndpointPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putOriginEndpointPolicyRequest));
            CompletableFuture<PutOriginEndpointPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Assigns one of more tags (key-value pairs) to the specified MediaPackage resource.
     * </p>
     * <p>
     * Tags can help you organize and categorize your resources. You can also use them to scope user permissions, by
     * granting a user permission to access or change only resources with certain tag values. You can use the
     * TagResource operation with a resource that already has tags. If you specify a new tag key for the resource, this
     * tag is appended to the list of tags associated with the resource. If you specify a tag key that is already
     * associated with the resource, the new tag value that you specify replaces the previous value for that tag.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageV2AsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageV2AsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update the specified channel. You can edit if MediaPackage sends ingest or egress access logs to the CloudWatch
     * log group, if content will be encrypted, the description on a channel, and your channel's policy settings. You
     * can't edit the name of the channel or CloudFront distribution details.
     * </p>
     * <p>
     * Any edits you make that impact the video output may not be reflected for a few minutes.
     * </p>
     *
     * @param updateChannelRequest
     * @return A Java Future containing the result of the UpdateChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ConflictException Updating or deleting this resource can cause an inconsistent state.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see Access Management in the IAM User Guide.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageV2AsyncClient.UpdateChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/UpdateChannel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateChannelResponse> updateChannel(UpdateChannelRequest updateChannelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateChannelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateChannelRequest, UpdateChannelResponse>()
                            .withOperationName("UpdateChannel")
                            .withMarshaller(new UpdateChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateChannelRequest));
            CompletableFuture<UpdateChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update the specified channel group. You can edit the description on a channel group for easier identification
     * later from the AWS Elemental MediaPackage console. You can't edit the name of the channel group.
     * </p>
     * <p>
     * Any edits you make that impact the video output may not be reflected for a few minutes.
     * </p>
     *
     * @param updateChannelGroupRequest
     * @return A Java Future containing the result of the UpdateChannelGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ConflictException Updating or deleting this resource can cause an inconsistent state.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see Access Management in the IAM User Guide.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageV2AsyncClient.UpdateChannelGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/UpdateChannelGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateChannelGroupResponse> updateChannelGroup(UpdateChannelGroupRequest updateChannelGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateChannelGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateChannelGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateChannelGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateChannelGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateChannelGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateChannelGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateChannelGroupRequest, UpdateChannelGroupResponse>()
                            .withOperationName("UpdateChannelGroup")
                            .withMarshaller(new UpdateChannelGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateChannelGroupRequest));
            CompletableFuture<UpdateChannelGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update the specified origin endpoint. Edit the packaging preferences on an endpoint to optimize the viewing
     * experience. You can't edit the name of the endpoint.
     * </p>
     * <p>
     * Any edits you make that impact the video output may not be reflected for a few minutes.
     * </p>
     *
     * @param updateOriginEndpointRequest
     * @return A Java Future containing the result of the UpdateOriginEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ConflictException Updating or deleting this resource can cause an inconsistent state.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see Access Management in the IAM User Guide.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageV2AsyncClient.UpdateOriginEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/UpdateOriginEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateOriginEndpointResponse> updateOriginEndpoint(
            UpdateOriginEndpointRequest updateOriginEndpointRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateOriginEndpointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateOriginEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateOriginEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateOriginEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateOriginEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateOriginEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateOriginEndpointRequest, UpdateOriginEndpointResponse>()
                            .withOperationName("UpdateOriginEndpoint")
                            .withMarshaller(new UpdateOriginEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateOriginEndpointRequest));
            CompletableFuture<UpdateOriginEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final MediaPackageV2ServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MediaPackageV2Exception::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        MediaPackageV2ServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = MediaPackageV2ServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
