/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotsecuretunneling;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.iotsecuretunneling.internal.IoTSecureTunnelingServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.iotsecuretunneling.model.CloseTunnelRequest;
import software.amazon.awssdk.services.iotsecuretunneling.model.CloseTunnelResponse;
import software.amazon.awssdk.services.iotsecuretunneling.model.DescribeTunnelRequest;
import software.amazon.awssdk.services.iotsecuretunneling.model.DescribeTunnelResponse;
import software.amazon.awssdk.services.iotsecuretunneling.model.IoTSecureTunnelingException;
import software.amazon.awssdk.services.iotsecuretunneling.model.LimitExceededException;
import software.amazon.awssdk.services.iotsecuretunneling.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.iotsecuretunneling.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.iotsecuretunneling.model.ListTunnelsRequest;
import software.amazon.awssdk.services.iotsecuretunneling.model.ListTunnelsResponse;
import software.amazon.awssdk.services.iotsecuretunneling.model.OpenTunnelRequest;
import software.amazon.awssdk.services.iotsecuretunneling.model.OpenTunnelResponse;
import software.amazon.awssdk.services.iotsecuretunneling.model.ResourceNotFoundException;
import software.amazon.awssdk.services.iotsecuretunneling.model.RotateTunnelAccessTokenRequest;
import software.amazon.awssdk.services.iotsecuretunneling.model.RotateTunnelAccessTokenResponse;
import software.amazon.awssdk.services.iotsecuretunneling.model.TagResourceRequest;
import software.amazon.awssdk.services.iotsecuretunneling.model.TagResourceResponse;
import software.amazon.awssdk.services.iotsecuretunneling.model.UntagResourceRequest;
import software.amazon.awssdk.services.iotsecuretunneling.model.UntagResourceResponse;
import software.amazon.awssdk.services.iotsecuretunneling.transform.CloseTunnelRequestMarshaller;
import software.amazon.awssdk.services.iotsecuretunneling.transform.DescribeTunnelRequestMarshaller;
import software.amazon.awssdk.services.iotsecuretunneling.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.iotsecuretunneling.transform.ListTunnelsRequestMarshaller;
import software.amazon.awssdk.services.iotsecuretunneling.transform.OpenTunnelRequestMarshaller;
import software.amazon.awssdk.services.iotsecuretunneling.transform.RotateTunnelAccessTokenRequestMarshaller;
import software.amazon.awssdk.services.iotsecuretunneling.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.iotsecuretunneling.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link IoTSecureTunnelingClient}.
 *
 * @see IoTSecureTunnelingClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultIoTSecureTunnelingClient implements IoTSecureTunnelingClient {
    private static final Logger log = Logger.loggerFor(DefaultIoTSecureTunnelingClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final IoTSecureTunnelingServiceClientConfiguration serviceClientConfiguration;

    protected DefaultIoTSecureTunnelingClient(IoTSecureTunnelingServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Closes a tunnel identified by the unique tunnel id. When a <code>CloseTunnel</code> request is received, we close
     * the WebSocket connections between the client and proxy server so no data can be transmitted.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >CloseTunnel</a> action.
     * </p>
     *
     * @param closeTunnelRequest
     * @return Result of the CloseTunnel operation returned by the service.
     * @throws ResourceNotFoundException
     *         Thrown when an operation is attempted on a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IoTSecureTunnelingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IoTSecureTunnelingClient.CloseTunnel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotsecuretunneling-2018-10-05/CloseTunnel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CloseTunnelResponse closeTunnel(CloseTunnelRequest closeTunnelRequest) throws ResourceNotFoundException,
            AwsServiceException, SdkClientException, IoTSecureTunnelingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CloseTunnelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CloseTunnelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(closeTunnelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, closeTunnelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTSecureTunneling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CloseTunnel");

            return clientHandler.execute(new ClientExecutionParams<CloseTunnelRequest, CloseTunnelResponse>()
                    .withOperationName("CloseTunnel").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(closeTunnelRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CloseTunnelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about a tunnel identified by the unique tunnel id.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >DescribeTunnel</a> action.
     * </p>
     *
     * @param describeTunnelRequest
     * @return Result of the DescribeTunnel operation returned by the service.
     * @throws ResourceNotFoundException
     *         Thrown when an operation is attempted on a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IoTSecureTunnelingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IoTSecureTunnelingClient.DescribeTunnel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotsecuretunneling-2018-10-05/DescribeTunnel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeTunnelResponse describeTunnel(DescribeTunnelRequest describeTunnelRequest) throws ResourceNotFoundException,
            AwsServiceException, SdkClientException, IoTSecureTunnelingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeTunnelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeTunnelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeTunnelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTunnelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTSecureTunneling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTunnel");

            return clientHandler.execute(new ClientExecutionParams<DescribeTunnelRequest, DescribeTunnelResponse>()
                    .withOperationName("DescribeTunnel").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeTunnelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeTunnelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the tags for the specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         Thrown when an operation is attempted on a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IoTSecureTunnelingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IoTSecureTunnelingClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotsecuretunneling-2018-10-05/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, AwsServiceException, SdkClientException, IoTSecureTunnelingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTSecureTunneling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List all tunnels for an Amazon Web Services account. Tunnels are listed by creation time in descending order,
     * newer tunnels will be listed before older tunnels.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >ListTunnels</a> action.
     * </p>
     *
     * @param listTunnelsRequest
     * @return Result of the ListTunnels operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IoTSecureTunnelingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IoTSecureTunnelingClient.ListTunnels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotsecuretunneling-2018-10-05/ListTunnels"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTunnelsResponse listTunnels(ListTunnelsRequest listTunnelsRequest) throws AwsServiceException, SdkClientException,
            IoTSecureTunnelingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTunnelsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTunnelsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTunnelsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTunnelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTSecureTunneling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTunnels");

            return clientHandler.execute(new ClientExecutionParams<ListTunnelsRequest, ListTunnelsResponse>()
                    .withOperationName("ListTunnels").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listTunnelsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTunnelsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new tunnel, and returns two client access tokens for clients to use to connect to the IoT Secure
     * Tunneling proxy server.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >OpenTunnel</a> action.
     * </p>
     *
     * @param openTunnelRequest
     * @return Result of the OpenTunnel operation returned by the service.
     * @throws LimitExceededException
     *         Thrown when a tunnel limit is exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IoTSecureTunnelingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IoTSecureTunnelingClient.OpenTunnel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotsecuretunneling-2018-10-05/OpenTunnel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public OpenTunnelResponse openTunnel(OpenTunnelRequest openTunnelRequest) throws LimitExceededException, AwsServiceException,
            SdkClientException, IoTSecureTunnelingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<OpenTunnelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                OpenTunnelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(openTunnelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, openTunnelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTSecureTunneling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "OpenTunnel");

            return clientHandler.execute(new ClientExecutionParams<OpenTunnelRequest, OpenTunnelResponse>()
                    .withOperationName("OpenTunnel").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(openTunnelRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new OpenTunnelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Revokes the current client access token (CAT) and returns new CAT for clients to use when reconnecting to secure
     * tunneling to access the same tunnel.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >RotateTunnelAccessToken</a> action.
     * </p>
     * <note>
     * <p>
     * Rotating the CAT doesn't extend the tunnel duration. For example, say the tunnel duration is 12 hours and the
     * tunnel has already been open for 4 hours. When you rotate the access tokens, the new tokens that are generated
     * can only be used for the remaining 8 hours.
     * </p>
     * </note>
     *
     * @param rotateTunnelAccessTokenRequest
     * @return Result of the RotateTunnelAccessToken operation returned by the service.
     * @throws ResourceNotFoundException
     *         Thrown when an operation is attempted on a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IoTSecureTunnelingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IoTSecureTunnelingClient.RotateTunnelAccessToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotsecuretunneling-2018-10-05/RotateTunnelAccessToken"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RotateTunnelAccessTokenResponse rotateTunnelAccessToken(RotateTunnelAccessTokenRequest rotateTunnelAccessTokenRequest)
            throws ResourceNotFoundException, AwsServiceException, SdkClientException, IoTSecureTunnelingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RotateTunnelAccessTokenResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RotateTunnelAccessTokenResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(rotateTunnelAccessTokenRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, rotateTunnelAccessTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTSecureTunneling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RotateTunnelAccessToken");

            return clientHandler
                    .execute(new ClientExecutionParams<RotateTunnelAccessTokenRequest, RotateTunnelAccessTokenResponse>()
                            .withOperationName("RotateTunnelAccessToken").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(rotateTunnelAccessTokenRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RotateTunnelAccessTokenRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * A resource tag.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         Thrown when an operation is attempted on a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IoTSecureTunnelingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IoTSecureTunnelingClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotsecuretunneling-2018-10-05/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            AwsServiceException, SdkClientException, IoTSecureTunnelingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTSecureTunneling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes a tag from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         Thrown when an operation is attempted on a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IoTSecureTunnelingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IoTSecureTunnelingClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotsecuretunneling-2018-10-05/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            AwsServiceException, SdkClientException, IoTSecureTunnelingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTSecureTunneling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        IoTSecureTunnelingServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = IoTSecureTunnelingServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(IoTSecureTunnelingException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build());
    }

    @Override
    public final IoTSecureTunnelingServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
