/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotevents;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.iotevents.internal.IotEventsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.iotevents.model.CreateAlarmModelRequest;
import software.amazon.awssdk.services.iotevents.model.CreateAlarmModelResponse;
import software.amazon.awssdk.services.iotevents.model.CreateDetectorModelRequest;
import software.amazon.awssdk.services.iotevents.model.CreateDetectorModelResponse;
import software.amazon.awssdk.services.iotevents.model.CreateInputRequest;
import software.amazon.awssdk.services.iotevents.model.CreateInputResponse;
import software.amazon.awssdk.services.iotevents.model.DeleteAlarmModelRequest;
import software.amazon.awssdk.services.iotevents.model.DeleteAlarmModelResponse;
import software.amazon.awssdk.services.iotevents.model.DeleteDetectorModelRequest;
import software.amazon.awssdk.services.iotevents.model.DeleteDetectorModelResponse;
import software.amazon.awssdk.services.iotevents.model.DeleteInputRequest;
import software.amazon.awssdk.services.iotevents.model.DeleteInputResponse;
import software.amazon.awssdk.services.iotevents.model.DescribeAlarmModelRequest;
import software.amazon.awssdk.services.iotevents.model.DescribeAlarmModelResponse;
import software.amazon.awssdk.services.iotevents.model.DescribeDetectorModelAnalysisRequest;
import software.amazon.awssdk.services.iotevents.model.DescribeDetectorModelAnalysisResponse;
import software.amazon.awssdk.services.iotevents.model.DescribeDetectorModelRequest;
import software.amazon.awssdk.services.iotevents.model.DescribeDetectorModelResponse;
import software.amazon.awssdk.services.iotevents.model.DescribeInputRequest;
import software.amazon.awssdk.services.iotevents.model.DescribeInputResponse;
import software.amazon.awssdk.services.iotevents.model.DescribeLoggingOptionsRequest;
import software.amazon.awssdk.services.iotevents.model.DescribeLoggingOptionsResponse;
import software.amazon.awssdk.services.iotevents.model.GetDetectorModelAnalysisResultsRequest;
import software.amazon.awssdk.services.iotevents.model.GetDetectorModelAnalysisResultsResponse;
import software.amazon.awssdk.services.iotevents.model.InternalFailureException;
import software.amazon.awssdk.services.iotevents.model.InvalidRequestException;
import software.amazon.awssdk.services.iotevents.model.IotEventsException;
import software.amazon.awssdk.services.iotevents.model.LimitExceededException;
import software.amazon.awssdk.services.iotevents.model.ListAlarmModelVersionsRequest;
import software.amazon.awssdk.services.iotevents.model.ListAlarmModelVersionsResponse;
import software.amazon.awssdk.services.iotevents.model.ListAlarmModelsRequest;
import software.amazon.awssdk.services.iotevents.model.ListAlarmModelsResponse;
import software.amazon.awssdk.services.iotevents.model.ListDetectorModelVersionsRequest;
import software.amazon.awssdk.services.iotevents.model.ListDetectorModelVersionsResponse;
import software.amazon.awssdk.services.iotevents.model.ListDetectorModelsRequest;
import software.amazon.awssdk.services.iotevents.model.ListDetectorModelsResponse;
import software.amazon.awssdk.services.iotevents.model.ListInputRoutingsRequest;
import software.amazon.awssdk.services.iotevents.model.ListInputRoutingsResponse;
import software.amazon.awssdk.services.iotevents.model.ListInputsRequest;
import software.amazon.awssdk.services.iotevents.model.ListInputsResponse;
import software.amazon.awssdk.services.iotevents.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.iotevents.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.iotevents.model.PutLoggingOptionsRequest;
import software.amazon.awssdk.services.iotevents.model.PutLoggingOptionsResponse;
import software.amazon.awssdk.services.iotevents.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.iotevents.model.ResourceInUseException;
import software.amazon.awssdk.services.iotevents.model.ResourceNotFoundException;
import software.amazon.awssdk.services.iotevents.model.ServiceUnavailableException;
import software.amazon.awssdk.services.iotevents.model.StartDetectorModelAnalysisRequest;
import software.amazon.awssdk.services.iotevents.model.StartDetectorModelAnalysisResponse;
import software.amazon.awssdk.services.iotevents.model.TagResourceRequest;
import software.amazon.awssdk.services.iotevents.model.TagResourceResponse;
import software.amazon.awssdk.services.iotevents.model.ThrottlingException;
import software.amazon.awssdk.services.iotevents.model.UnsupportedOperationException;
import software.amazon.awssdk.services.iotevents.model.UntagResourceRequest;
import software.amazon.awssdk.services.iotevents.model.UntagResourceResponse;
import software.amazon.awssdk.services.iotevents.model.UpdateAlarmModelRequest;
import software.amazon.awssdk.services.iotevents.model.UpdateAlarmModelResponse;
import software.amazon.awssdk.services.iotevents.model.UpdateDetectorModelRequest;
import software.amazon.awssdk.services.iotevents.model.UpdateDetectorModelResponse;
import software.amazon.awssdk.services.iotevents.model.UpdateInputRequest;
import software.amazon.awssdk.services.iotevents.model.UpdateInputResponse;
import software.amazon.awssdk.services.iotevents.transform.CreateAlarmModelRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.CreateDetectorModelRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.CreateInputRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.DeleteAlarmModelRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.DeleteDetectorModelRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.DeleteInputRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.DescribeAlarmModelRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.DescribeDetectorModelAnalysisRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.DescribeDetectorModelRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.DescribeInputRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.DescribeLoggingOptionsRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.GetDetectorModelAnalysisResultsRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.ListAlarmModelVersionsRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.ListAlarmModelsRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.ListDetectorModelVersionsRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.ListDetectorModelsRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.ListInputRoutingsRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.ListInputsRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.PutLoggingOptionsRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.StartDetectorModelAnalysisRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.UpdateAlarmModelRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.UpdateDetectorModelRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.UpdateInputRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link IotEventsClient}.
 *
 * @see IotEventsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultIotEventsClient implements IotEventsClient {
    private static final Logger log = Logger.loggerFor(DefaultIotEventsClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final IotEventsServiceClientConfiguration serviceClientConfiguration;

    protected DefaultIotEventsClient(IotEventsServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an alarm model to monitor an AWS IoT Events input attribute. You can use the alarm to get notified when
     * the value is outside a specified range. For more information, see <a
     * href="https://docs.aws.amazon.com/iotevents/latest/developerguide/create-alarms.html">Create an alarm model</a>
     * in the <i>AWS IoT Events Developer Guide</i>.
     * </p>
     *
     * @param createAlarmModelRequest
     * @return Result of the CreateAlarmModel operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ResourceInUseException
     *         The resource is in use.
     * @throws ResourceAlreadyExistsException
     *         The resource already exists.
     * @throws LimitExceededException
     *         A limit was exceeded.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.CreateAlarmModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/CreateAlarmModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateAlarmModelResponse createAlarmModel(CreateAlarmModelRequest createAlarmModelRequest)
            throws InvalidRequestException, ResourceInUseException, ResourceAlreadyExistsException, LimitExceededException,
            ThrottlingException, InternalFailureException, ServiceUnavailableException, AwsServiceException, SdkClientException,
            IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateAlarmModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateAlarmModelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createAlarmModelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAlarmModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAlarmModel");

            return clientHandler.execute(new ClientExecutionParams<CreateAlarmModelRequest, CreateAlarmModelResponse>()
                    .withOperationName("CreateAlarmModel").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createAlarmModelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateAlarmModelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a detector model.
     * </p>
     *
     * @param createDetectorModelRequest
     * @return Result of the CreateDetectorModel operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ResourceInUseException
     *         The resource is in use.
     * @throws ResourceAlreadyExistsException
     *         The resource already exists.
     * @throws LimitExceededException
     *         A limit was exceeded.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.CreateDetectorModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/CreateDetectorModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateDetectorModelResponse createDetectorModel(CreateDetectorModelRequest createDetectorModelRequest)
            throws InvalidRequestException, ResourceInUseException, ResourceAlreadyExistsException, LimitExceededException,
            ThrottlingException, InternalFailureException, ServiceUnavailableException, AwsServiceException, SdkClientException,
            IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDetectorModelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateDetectorModelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDetectorModelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDetectorModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDetectorModel");

            return clientHandler.execute(new ClientExecutionParams<CreateDetectorModelRequest, CreateDetectorModelResponse>()
                    .withOperationName("CreateDetectorModel").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createDetectorModelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateDetectorModelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an input.
     * </p>
     *
     * @param createInputRequest
     * @return Result of the CreateInput operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ResourceAlreadyExistsException
     *         The resource already exists.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.CreateInput
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/CreateInput" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateInputResponse createInput(CreateInputRequest createInputRequest) throws InvalidRequestException,
            ThrottlingException, InternalFailureException, ServiceUnavailableException, ResourceAlreadyExistsException,
            AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateInputResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateInputResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createInputRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createInputRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateInput");

            return clientHandler.execute(new ClientExecutionParams<CreateInputRequest, CreateInputResponse>()
                    .withOperationName("CreateInput").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createInputRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateInputRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an alarm model. Any alarm instances that were created based on this alarm model are also deleted. This
     * action can't be undone.
     * </p>
     *
     * @param deleteAlarmModelRequest
     * @return Result of the DeleteAlarmModel operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ResourceInUseException
     *         The resource is in use.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.DeleteAlarmModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/DeleteAlarmModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteAlarmModelResponse deleteAlarmModel(DeleteAlarmModelRequest deleteAlarmModelRequest)
            throws InvalidRequestException, ResourceInUseException, ResourceNotFoundException, ThrottlingException,
            InternalFailureException, ServiceUnavailableException, AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAlarmModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteAlarmModelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAlarmModelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAlarmModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAlarmModel");

            return clientHandler.execute(new ClientExecutionParams<DeleteAlarmModelRequest, DeleteAlarmModelResponse>()
                    .withOperationName("DeleteAlarmModel").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteAlarmModelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteAlarmModelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a detector model. Any active instances of the detector model are also deleted.
     * </p>
     *
     * @param deleteDetectorModelRequest
     * @return Result of the DeleteDetectorModel operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ResourceInUseException
     *         The resource is in use.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.DeleteDetectorModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/DeleteDetectorModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteDetectorModelResponse deleteDetectorModel(DeleteDetectorModelRequest deleteDetectorModelRequest)
            throws InvalidRequestException, ResourceInUseException, ResourceNotFoundException, ThrottlingException,
            InternalFailureException, ServiceUnavailableException, AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDetectorModelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteDetectorModelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDetectorModelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDetectorModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDetectorModel");

            return clientHandler.execute(new ClientExecutionParams<DeleteDetectorModelRequest, DeleteDetectorModelResponse>()
                    .withOperationName("DeleteDetectorModel").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteDetectorModelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteDetectorModelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an input.
     * </p>
     *
     * @param deleteInputRequest
     * @return Result of the DeleteInput operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ResourceInUseException
     *         The resource is in use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.DeleteInput
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/DeleteInput" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteInputResponse deleteInput(DeleteInputRequest deleteInputRequest) throws InvalidRequestException,
            ResourceNotFoundException, ThrottlingException, InternalFailureException, ServiceUnavailableException,
            ResourceInUseException, AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteInputResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteInputResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteInputRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteInputRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteInput");

            return clientHandler.execute(new ClientExecutionParams<DeleteInputRequest, DeleteInputResponse>()
                    .withOperationName("DeleteInput").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteInputRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteInputRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about an alarm model. If you don't specify a value for the <code>alarmModelVersion</code>
     * parameter, the latest version is returned.
     * </p>
     *
     * @param describeAlarmModelRequest
     * @return Result of the DescribeAlarmModel operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.DescribeAlarmModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/DescribeAlarmModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeAlarmModelResponse describeAlarmModel(DescribeAlarmModelRequest describeAlarmModelRequest)
            throws InvalidRequestException, ResourceNotFoundException, ThrottlingException, InternalFailureException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAlarmModelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeAlarmModelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAlarmModelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAlarmModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAlarmModel");

            return clientHandler.execute(new ClientExecutionParams<DescribeAlarmModelRequest, DescribeAlarmModelResponse>()
                    .withOperationName("DescribeAlarmModel").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeAlarmModelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeAlarmModelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes a detector model. If the <code>version</code> parameter is not specified, information about the latest
     * version is returned.
     * </p>
     *
     * @param describeDetectorModelRequest
     * @return Result of the DescribeDetectorModel operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.DescribeDetectorModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/DescribeDetectorModel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeDetectorModelResponse describeDetectorModel(DescribeDetectorModelRequest describeDetectorModelRequest)
            throws InvalidRequestException, ResourceNotFoundException, ThrottlingException, InternalFailureException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeDetectorModelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeDetectorModelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeDetectorModelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDetectorModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDetectorModel");

            return clientHandler.execute(new ClientExecutionParams<DescribeDetectorModelRequest, DescribeDetectorModelResponse>()
                    .withOperationName("DescribeDetectorModel").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeDetectorModelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeDetectorModelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves runtime information about a detector model analysis.
     * </p>
     * <note>
     * <p>
     * After AWS IoT Events starts analyzing your detector model, you have up to 24 hours to retrieve the analysis
     * results.
     * </p>
     * </note>
     *
     * @param describeDetectorModelAnalysisRequest
     * @return Result of the DescribeDetectorModelAnalysis operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.DescribeDetectorModelAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/DescribeDetectorModelAnalysis"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeDetectorModelAnalysisResponse describeDetectorModelAnalysis(
            DescribeDetectorModelAnalysisRequest describeDetectorModelAnalysisRequest) throws InvalidRequestException,
            ResourceNotFoundException, ThrottlingException, InternalFailureException, ServiceUnavailableException,
            AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeDetectorModelAnalysisResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeDetectorModelAnalysisResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeDetectorModelAnalysisRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeDetectorModelAnalysisRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDetectorModelAnalysis");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeDetectorModelAnalysisRequest, DescribeDetectorModelAnalysisResponse>()
                            .withOperationName("DescribeDetectorModelAnalysis").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeDetectorModelAnalysisRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeDetectorModelAnalysisRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes an input.
     * </p>
     *
     * @param describeInputRequest
     * @return Result of the DescribeInput operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.DescribeInput
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/DescribeInput" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeInputResponse describeInput(DescribeInputRequest describeInputRequest) throws InvalidRequestException,
            ResourceNotFoundException, ThrottlingException, InternalFailureException, ServiceUnavailableException,
            AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeInputResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeInputResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeInputRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeInputRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeInput");

            return clientHandler.execute(new ClientExecutionParams<DescribeInputRequest, DescribeInputResponse>()
                    .withOperationName("DescribeInput").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeInputRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeInputRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the current settings of the AWS IoT Events logging options.
     * </p>
     *
     * @param describeLoggingOptionsRequest
     * @return Result of the DescribeLoggingOptions operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws UnsupportedOperationException
     *         The requested operation is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.DescribeLoggingOptions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/DescribeLoggingOptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeLoggingOptionsResponse describeLoggingOptions(DescribeLoggingOptionsRequest describeLoggingOptionsRequest)
            throws InvalidRequestException, ThrottlingException, InternalFailureException, ResourceNotFoundException,
            ServiceUnavailableException, UnsupportedOperationException, AwsServiceException, SdkClientException,
            IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeLoggingOptionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeLoggingOptionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeLoggingOptionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLoggingOptionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLoggingOptions");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeLoggingOptionsRequest, DescribeLoggingOptionsResponse>()
                            .withOperationName("DescribeLoggingOptions").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeLoggingOptionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeLoggingOptionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves one or more analysis results of the detector model.
     * </p>
     * <note>
     * <p>
     * After AWS IoT Events starts analyzing your detector model, you have up to 24 hours to retrieve the analysis
     * results.
     * </p>
     * </note>
     *
     * @param getDetectorModelAnalysisResultsRequest
     * @return Result of the GetDetectorModelAnalysisResults operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.GetDetectorModelAnalysisResults
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/GetDetectorModelAnalysisResults"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetDetectorModelAnalysisResultsResponse getDetectorModelAnalysisResults(
            GetDetectorModelAnalysisResultsRequest getDetectorModelAnalysisResultsRequest) throws InvalidRequestException,
            ResourceNotFoundException, ThrottlingException, InternalFailureException, ServiceUnavailableException,
            AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDetectorModelAnalysisResultsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetDetectorModelAnalysisResultsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDetectorModelAnalysisResultsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getDetectorModelAnalysisResultsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDetectorModelAnalysisResults");

            return clientHandler
                    .execute(new ClientExecutionParams<GetDetectorModelAnalysisResultsRequest, GetDetectorModelAnalysisResultsResponse>()
                            .withOperationName("GetDetectorModelAnalysisResults").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getDetectorModelAnalysisResultsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetDetectorModelAnalysisResultsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all the versions of an alarm model. The operation returns only the metadata associated with each alarm
     * model version.
     * </p>
     *
     * @param listAlarmModelVersionsRequest
     * @return Result of the ListAlarmModelVersions operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.ListAlarmModelVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/ListAlarmModelVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAlarmModelVersionsResponse listAlarmModelVersions(ListAlarmModelVersionsRequest listAlarmModelVersionsRequest)
            throws InvalidRequestException, ResourceNotFoundException, ThrottlingException, InternalFailureException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAlarmModelVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListAlarmModelVersionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAlarmModelVersionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAlarmModelVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAlarmModelVersions");

            return clientHandler
                    .execute(new ClientExecutionParams<ListAlarmModelVersionsRequest, ListAlarmModelVersionsResponse>()
                            .withOperationName("ListAlarmModelVersions").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listAlarmModelVersionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListAlarmModelVersionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the alarm models that you created. The operation returns only the metadata associated with each alarm
     * model.
     * </p>
     *
     * @param listAlarmModelsRequest
     * @return Result of the ListAlarmModels operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.ListAlarmModels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/ListAlarmModels" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListAlarmModelsResponse listAlarmModels(ListAlarmModelsRequest listAlarmModelsRequest) throws InvalidRequestException,
            ThrottlingException, InternalFailureException, ServiceUnavailableException, AwsServiceException, SdkClientException,
            IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAlarmModelsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListAlarmModelsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAlarmModelsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAlarmModelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAlarmModels");

            return clientHandler.execute(new ClientExecutionParams<ListAlarmModelsRequest, ListAlarmModelsResponse>()
                    .withOperationName("ListAlarmModels").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listAlarmModelsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListAlarmModelsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all the versions of a detector model. Only the metadata associated with each detector model version is
     * returned.
     * </p>
     *
     * @param listDetectorModelVersionsRequest
     * @return Result of the ListDetectorModelVersions operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.ListDetectorModelVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/ListDetectorModelVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDetectorModelVersionsResponse listDetectorModelVersions(
            ListDetectorModelVersionsRequest listDetectorModelVersionsRequest) throws InvalidRequestException,
            ResourceNotFoundException, ThrottlingException, InternalFailureException, ServiceUnavailableException,
            AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDetectorModelVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListDetectorModelVersionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDetectorModelVersionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDetectorModelVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDetectorModelVersions");

            return clientHandler
                    .execute(new ClientExecutionParams<ListDetectorModelVersionsRequest, ListDetectorModelVersionsResponse>()
                            .withOperationName("ListDetectorModelVersions").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listDetectorModelVersionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListDetectorModelVersionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the detector models you have created. Only the metadata associated with each detector model is returned.
     * </p>
     *
     * @param listDetectorModelsRequest
     * @return Result of the ListDetectorModels operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.ListDetectorModels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/ListDetectorModels" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListDetectorModelsResponse listDetectorModels(ListDetectorModelsRequest listDetectorModelsRequest)
            throws InvalidRequestException, ThrottlingException, InternalFailureException, ServiceUnavailableException,
            AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDetectorModelsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListDetectorModelsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDetectorModelsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDetectorModelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDetectorModels");

            return clientHandler.execute(new ClientExecutionParams<ListDetectorModelsRequest, ListDetectorModelsResponse>()
                    .withOperationName("ListDetectorModels").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listDetectorModelsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDetectorModelsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists one or more input routings.
     * </p>
     *
     * @param listInputRoutingsRequest
     * @return Result of the ListInputRoutings operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.ListInputRoutings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/ListInputRoutings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListInputRoutingsResponse listInputRoutings(ListInputRoutingsRequest listInputRoutingsRequest)
            throws InvalidRequestException, ThrottlingException, InternalFailureException, ServiceUnavailableException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListInputRoutingsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListInputRoutingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listInputRoutingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInputRoutingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInputRoutings");

            return clientHandler.execute(new ClientExecutionParams<ListInputRoutingsRequest, ListInputRoutingsResponse>()
                    .withOperationName("ListInputRoutings").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listInputRoutingsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListInputRoutingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the inputs you have created.
     * </p>
     *
     * @param listInputsRequest
     * @return Result of the ListInputs operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.ListInputs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/ListInputs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListInputsResponse listInputs(ListInputsRequest listInputsRequest) throws InvalidRequestException,
            ThrottlingException, InternalFailureException, ServiceUnavailableException, AwsServiceException, SdkClientException,
            IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListInputsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListInputsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listInputsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInputsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInputs");

            return clientHandler.execute(new ClientExecutionParams<ListInputsRequest, ListInputsResponse>()
                    .withOperationName("ListInputs").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listInputsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListInputsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the tags (metadata) you have assigned to the resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ResourceInUseException
     *         The resource is in use.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InvalidRequestException, ResourceNotFoundException, ResourceInUseException, ThrottlingException,
            InternalFailureException, AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Sets or updates the AWS IoT Events logging options.
     * </p>
     * <p>
     * If you update the value of any <code>loggingOptions</code> field, it takes up to one minute for the change to
     * take effect. If you change the policy attached to the role you specified in the <code>roleArn</code> field (for
     * example, to correct an invalid policy), it takes up to five minutes for that change to take effect.
     * </p>
     *
     * @param putLoggingOptionsRequest
     * @return Result of the PutLoggingOptions operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws UnsupportedOperationException
     *         The requested operation is not supported.
     * @throws ResourceInUseException
     *         The resource is in use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.PutLoggingOptions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/PutLoggingOptions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutLoggingOptionsResponse putLoggingOptions(PutLoggingOptionsRequest putLoggingOptionsRequest)
            throws InvalidRequestException, ThrottlingException, InternalFailureException, ServiceUnavailableException,
            UnsupportedOperationException, ResourceInUseException, AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutLoggingOptionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutLoggingOptionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putLoggingOptionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putLoggingOptionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutLoggingOptions");

            return clientHandler.execute(new ClientExecutionParams<PutLoggingOptionsRequest, PutLoggingOptionsResponse>()
                    .withOperationName("PutLoggingOptions").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putLoggingOptionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutLoggingOptionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Performs an analysis of your detector model. For more information, see <a
     * href="https://docs.aws.amazon.com/iotevents/latest/developerguide/iotevents-analyze-api.html">Troubleshooting a
     * detector model</a> in the <i>AWS IoT Events Developer Guide</i>.
     * </p>
     *
     * @param startDetectorModelAnalysisRequest
     * @return Result of the StartDetectorModelAnalysis operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws LimitExceededException
     *         A limit was exceeded.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.StartDetectorModelAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/StartDetectorModelAnalysis"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartDetectorModelAnalysisResponse startDetectorModelAnalysis(
            StartDetectorModelAnalysisRequest startDetectorModelAnalysisRequest) throws InvalidRequestException,
            LimitExceededException, ThrottlingException, InternalFailureException, ServiceUnavailableException,
            AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartDetectorModelAnalysisResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartDetectorModelAnalysisResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startDetectorModelAnalysisRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startDetectorModelAnalysisRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartDetectorModelAnalysis");

            return clientHandler
                    .execute(new ClientExecutionParams<StartDetectorModelAnalysisRequest, StartDetectorModelAnalysisResponse>()
                            .withOperationName("StartDetectorModelAnalysis").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(startDetectorModelAnalysisRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartDetectorModelAnalysisRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds to or modifies the tags of the given resource. Tags are metadata that can be used to manage a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ResourceInUseException
     *         The resource is in use.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws LimitExceededException
     *         A limit was exceeded.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InvalidRequestException,
            ResourceNotFoundException, ResourceInUseException, ThrottlingException, LimitExceededException,
            InternalFailureException, AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the given tags (metadata) from the resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ResourceInUseException
     *         The resource is in use.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InvalidRequestException,
            ResourceNotFoundException, ResourceInUseException, ThrottlingException, InternalFailureException,
            AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an alarm model. Any alarms that were created based on the previous version are deleted and then created
     * again as new data arrives.
     * </p>
     *
     * @param updateAlarmModelRequest
     * @return Result of the UpdateAlarmModel operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ResourceInUseException
     *         The resource is in use.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.UpdateAlarmModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/UpdateAlarmModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateAlarmModelResponse updateAlarmModel(UpdateAlarmModelRequest updateAlarmModelRequest)
            throws InvalidRequestException, ResourceInUseException, ResourceNotFoundException, ThrottlingException,
            InternalFailureException, ServiceUnavailableException, AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateAlarmModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateAlarmModelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateAlarmModelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAlarmModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAlarmModel");

            return clientHandler.execute(new ClientExecutionParams<UpdateAlarmModelRequest, UpdateAlarmModelResponse>()
                    .withOperationName("UpdateAlarmModel").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateAlarmModelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateAlarmModelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a detector model. Detectors (instances) spawned by the previous version are deleted and then re-created
     * as new inputs arrive.
     * </p>
     *
     * @param updateDetectorModelRequest
     * @return Result of the UpdateDetectorModel operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ResourceInUseException
     *         The resource is in use.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.UpdateDetectorModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/UpdateDetectorModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateDetectorModelResponse updateDetectorModel(UpdateDetectorModelRequest updateDetectorModelRequest)
            throws InvalidRequestException, ResourceInUseException, ResourceNotFoundException, ThrottlingException,
            InternalFailureException, ServiceUnavailableException, AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDetectorModelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateDetectorModelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDetectorModelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDetectorModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDetectorModel");

            return clientHandler.execute(new ClientExecutionParams<UpdateDetectorModelRequest, UpdateDetectorModelResponse>()
                    .withOperationName("UpdateDetectorModel").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateDetectorModelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateDetectorModelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an input.
     * </p>
     *
     * @param updateInputRequest
     * @return Result of the UpdateInput operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ResourceInUseException
     *         The resource is in use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.UpdateInput
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/UpdateInput" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateInputResponse updateInput(UpdateInputRequest updateInputRequest) throws InvalidRequestException,
            ThrottlingException, ResourceNotFoundException, InternalFailureException, ServiceUnavailableException,
            ResourceInUseException, AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateInputResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateInputResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateInputRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateInputRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateInput");

            return clientHandler.execute(new ClientExecutionParams<UpdateInputRequest, UpdateInputResponse>()
                    .withOperationName("UpdateInput").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(updateInputRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateInputRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        IotEventsServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = IotEventsServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(IotEventsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailureException")
                                .exceptionBuilderSupplier(InternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(410).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedOperationException")
                                .exceptionBuilderSupplier(UnsupportedOperationException::builder).httpStatusCode(501).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).httpStatusCode(409).build());
    }

    @Override
    public final IotEventsServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
