/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.grafana.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration settings for in-bound network access to your workspace.
 * </p>
 * <p>
 * When this is configured, only listed IP addresses and VPC endpoints will be able to access your workspace. Standard
 * Grafana authentication and authorization are still required.
 * </p>
 * <p>
 * Access is granted to a caller that is in either the IP address list or the VPC endpoint list - they do not need to be
 * in both.
 * </p>
 * <p>
 * If this is not configured, or is removed, then all IP addresses and VPC endpoints are allowed. Standard Grafana
 * authentication and authorization are still required.
 * </p>
 * <note>
 * <p>
 * While both <code>prefixListIds</code> and <code>vpceIds</code> are required, you can pass in an empty array of
 * strings for either parameter if you do not want to allow any of that type.
 * </p>
 * <p>
 * If both are passed as empty arrays, no traffic is allowed to the workspace, because only <i>explicitly</i> allowed
 * connections are accepted.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class NetworkAccessConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<NetworkAccessConfiguration.Builder, NetworkAccessConfiguration> {
    private static final SdkField<List<String>> PREFIX_LIST_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("prefixListIds")
            .getter(getter(NetworkAccessConfiguration::prefixListIds))
            .setter(setter(Builder::prefixListIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("prefixListIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> VPCE_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("vpceIds")
            .getter(getter(NetworkAccessConfiguration::vpceIds))
            .setter(setter(Builder::vpceIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("vpceIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PREFIX_LIST_IDS_FIELD,
            VPCE_IDS_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<String> prefixListIds;

    private final List<String> vpceIds;

    private NetworkAccessConfiguration(BuilderImpl builder) {
        this.prefixListIds = builder.prefixListIds;
        this.vpceIds = builder.vpceIds;
    }

    /**
     * For responses, this returns true if the service returned a value for the PrefixListIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPrefixListIds() {
        return prefixListIds != null && !(prefixListIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of prefix list IDs. A prefix list is a list of CIDR ranges of IP addresses. The IP addresses specified
     * are allowed to access your workspace. If the list is not included in the configuration (passed an empty array)
     * then no IP addresses are allowed to access the workspace. You create a prefix list using the Amazon VPC console.
     * </p>
     * <p>
     * Prefix list IDs have the format <code>pl-<i>1a2b3c4d</i> </code>.
     * </p>
     * <p>
     * For more information about prefix lists, see <a
     * href="https://docs.aws.amazon.com/vpc/latest/userguide/managed-prefix-lists.html">Group CIDR blocks using managed
     * prefix lists</a>in the <i>Amazon Virtual Private Cloud User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPrefixListIds} method.
     * </p>
     * 
     * @return An array of prefix list IDs. A prefix list is a list of CIDR ranges of IP addresses. The IP addresses
     *         specified are allowed to access your workspace. If the list is not included in the configuration (passed
     *         an empty array) then no IP addresses are allowed to access the workspace. You create a prefix list using
     *         the Amazon VPC console.</p>
     *         <p>
     *         Prefix list IDs have the format <code>pl-<i>1a2b3c4d</i> </code>.
     *         </p>
     *         <p>
     *         For more information about prefix lists, see <a
     *         href="https://docs.aws.amazon.com/vpc/latest/userguide/managed-prefix-lists.html">Group CIDR blocks using
     *         managed prefix lists</a>in the <i>Amazon Virtual Private Cloud User Guide</i>.
     */
    public final List<String> prefixListIds() {
        return prefixListIds;
    }

    /**
     * For responses, this returns true if the service returned a value for the VpceIds property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasVpceIds() {
        return vpceIds != null && !(vpceIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of Amazon VPC endpoint IDs for the workspace. You can create VPC endpoints to your Amazon Managed
     * Grafana workspace for access from within a VPC. If a <code>NetworkAccessConfiguration</code> is specified then
     * only VPC endpoints specified here are allowed to access the workspace. If you pass in an empty array of strings,
     * then no VPCs are allowed to access the workspace.
     * </p>
     * <p>
     * VPC endpoint IDs have the format <code>vpce-<i>1a2b3c4d</i> </code>.
     * </p>
     * <p>
     * For more information about creating an interface VPC endpoint, see <a
     * href="https://docs.aws.amazon.com/grafana/latest/userguide/VPC-endpoints">Interface VPC endpoints</a> in the
     * <i>Amazon Managed Grafana User Guide</i>.
     * </p>
     * <note>
     * <p>
     * The only VPC endpoints that can be specified here are interface VPC endpoints for Grafana workspaces (using the
     * <code>com.amazonaws.[region].grafana-workspace</code> service endpoint). Other VPC endpoints are ignored.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasVpceIds} method.
     * </p>
     * 
     * @return An array of Amazon VPC endpoint IDs for the workspace. You can create VPC endpoints to your Amazon
     *         Managed Grafana workspace for access from within a VPC. If a <code>NetworkAccessConfiguration</code> is
     *         specified then only VPC endpoints specified here are allowed to access the workspace. If you pass in an
     *         empty array of strings, then no VPCs are allowed to access the workspace.</p>
     *         <p>
     *         VPC endpoint IDs have the format <code>vpce-<i>1a2b3c4d</i> </code>.
     *         </p>
     *         <p>
     *         For more information about creating an interface VPC endpoint, see <a
     *         href="https://docs.aws.amazon.com/grafana/latest/userguide/VPC-endpoints">Interface VPC endpoints</a> in
     *         the <i>Amazon Managed Grafana User Guide</i>.
     *         </p>
     *         <note>
     *         <p>
     *         The only VPC endpoints that can be specified here are interface VPC endpoints for Grafana workspaces
     *         (using the <code>com.amazonaws.[region].grafana-workspace</code> service endpoint). Other VPC endpoints
     *         are ignored.
     *         </p>
     */
    public final List<String> vpceIds() {
        return vpceIds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasPrefixListIds() ? prefixListIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasVpceIds() ? vpceIds() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof NetworkAccessConfiguration)) {
            return false;
        }
        NetworkAccessConfiguration other = (NetworkAccessConfiguration) obj;
        return hasPrefixListIds() == other.hasPrefixListIds() && Objects.equals(prefixListIds(), other.prefixListIds())
                && hasVpceIds() == other.hasVpceIds() && Objects.equals(vpceIds(), other.vpceIds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("NetworkAccessConfiguration").add("PrefixListIds", hasPrefixListIds() ? prefixListIds() : null)
                .add("VpceIds", hasVpceIds() ? vpceIds() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "prefixListIds":
            return Optional.ofNullable(clazz.cast(prefixListIds()));
        case "vpceIds":
            return Optional.ofNullable(clazz.cast(vpceIds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<NetworkAccessConfiguration, T> g) {
        return obj -> g.apply((NetworkAccessConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, NetworkAccessConfiguration> {
        /**
         * <p>
         * An array of prefix list IDs. A prefix list is a list of CIDR ranges of IP addresses. The IP addresses
         * specified are allowed to access your workspace. If the list is not included in the configuration (passed an
         * empty array) then no IP addresses are allowed to access the workspace. You create a prefix list using the
         * Amazon VPC console.
         * </p>
         * <p>
         * Prefix list IDs have the format <code>pl-<i>1a2b3c4d</i> </code>.
         * </p>
         * <p>
         * For more information about prefix lists, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/managed-prefix-lists.html">Group CIDR blocks using
         * managed prefix lists</a>in the <i>Amazon Virtual Private Cloud User Guide</i>.
         * </p>
         * 
         * @param prefixListIds
         *        An array of prefix list IDs. A prefix list is a list of CIDR ranges of IP addresses. The IP addresses
         *        specified are allowed to access your workspace. If the list is not included in the configuration
         *        (passed an empty array) then no IP addresses are allowed to access the workspace. You create a prefix
         *        list using the Amazon VPC console.</p>
         *        <p>
         *        Prefix list IDs have the format <code>pl-<i>1a2b3c4d</i> </code>.
         *        </p>
         *        <p>
         *        For more information about prefix lists, see <a
         *        href="https://docs.aws.amazon.com/vpc/latest/userguide/managed-prefix-lists.html">Group CIDR blocks
         *        using managed prefix lists</a>in the <i>Amazon Virtual Private Cloud User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder prefixListIds(Collection<String> prefixListIds);

        /**
         * <p>
         * An array of prefix list IDs. A prefix list is a list of CIDR ranges of IP addresses. The IP addresses
         * specified are allowed to access your workspace. If the list is not included in the configuration (passed an
         * empty array) then no IP addresses are allowed to access the workspace. You create a prefix list using the
         * Amazon VPC console.
         * </p>
         * <p>
         * Prefix list IDs have the format <code>pl-<i>1a2b3c4d</i> </code>.
         * </p>
         * <p>
         * For more information about prefix lists, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/managed-prefix-lists.html">Group CIDR blocks using
         * managed prefix lists</a>in the <i>Amazon Virtual Private Cloud User Guide</i>.
         * </p>
         * 
         * @param prefixListIds
         *        An array of prefix list IDs. A prefix list is a list of CIDR ranges of IP addresses. The IP addresses
         *        specified are allowed to access your workspace. If the list is not included in the configuration
         *        (passed an empty array) then no IP addresses are allowed to access the workspace. You create a prefix
         *        list using the Amazon VPC console.</p>
         *        <p>
         *        Prefix list IDs have the format <code>pl-<i>1a2b3c4d</i> </code>.
         *        </p>
         *        <p>
         *        For more information about prefix lists, see <a
         *        href="https://docs.aws.amazon.com/vpc/latest/userguide/managed-prefix-lists.html">Group CIDR blocks
         *        using managed prefix lists</a>in the <i>Amazon Virtual Private Cloud User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder prefixListIds(String... prefixListIds);

        /**
         * <p>
         * An array of Amazon VPC endpoint IDs for the workspace. You can create VPC endpoints to your Amazon Managed
         * Grafana workspace for access from within a VPC. If a <code>NetworkAccessConfiguration</code> is specified
         * then only VPC endpoints specified here are allowed to access the workspace. If you pass in an empty array of
         * strings, then no VPCs are allowed to access the workspace.
         * </p>
         * <p>
         * VPC endpoint IDs have the format <code>vpce-<i>1a2b3c4d</i> </code>.
         * </p>
         * <p>
         * For more information about creating an interface VPC endpoint, see <a
         * href="https://docs.aws.amazon.com/grafana/latest/userguide/VPC-endpoints">Interface VPC endpoints</a> in the
         * <i>Amazon Managed Grafana User Guide</i>.
         * </p>
         * <note>
         * <p>
         * The only VPC endpoints that can be specified here are interface VPC endpoints for Grafana workspaces (using
         * the <code>com.amazonaws.[region].grafana-workspace</code> service endpoint). Other VPC endpoints are ignored.
         * </p>
         * </note>
         * 
         * @param vpceIds
         *        An array of Amazon VPC endpoint IDs for the workspace. You can create VPC endpoints to your Amazon
         *        Managed Grafana workspace for access from within a VPC. If a <code>NetworkAccessConfiguration</code>
         *        is specified then only VPC endpoints specified here are allowed to access the workspace. If you pass
         *        in an empty array of strings, then no VPCs are allowed to access the workspace.</p>
         *        <p>
         *        VPC endpoint IDs have the format <code>vpce-<i>1a2b3c4d</i> </code>.
         *        </p>
         *        <p>
         *        For more information about creating an interface VPC endpoint, see <a
         *        href="https://docs.aws.amazon.com/grafana/latest/userguide/VPC-endpoints">Interface VPC endpoints</a>
         *        in the <i>Amazon Managed Grafana User Guide</i>.
         *        </p>
         *        <note>
         *        <p>
         *        The only VPC endpoints that can be specified here are interface VPC endpoints for Grafana workspaces
         *        (using the <code>com.amazonaws.[region].grafana-workspace</code> service endpoint). Other VPC
         *        endpoints are ignored.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpceIds(Collection<String> vpceIds);

        /**
         * <p>
         * An array of Amazon VPC endpoint IDs for the workspace. You can create VPC endpoints to your Amazon Managed
         * Grafana workspace for access from within a VPC. If a <code>NetworkAccessConfiguration</code> is specified
         * then only VPC endpoints specified here are allowed to access the workspace. If you pass in an empty array of
         * strings, then no VPCs are allowed to access the workspace.
         * </p>
         * <p>
         * VPC endpoint IDs have the format <code>vpce-<i>1a2b3c4d</i> </code>.
         * </p>
         * <p>
         * For more information about creating an interface VPC endpoint, see <a
         * href="https://docs.aws.amazon.com/grafana/latest/userguide/VPC-endpoints">Interface VPC endpoints</a> in the
         * <i>Amazon Managed Grafana User Guide</i>.
         * </p>
         * <note>
         * <p>
         * The only VPC endpoints that can be specified here are interface VPC endpoints for Grafana workspaces (using
         * the <code>com.amazonaws.[region].grafana-workspace</code> service endpoint). Other VPC endpoints are ignored.
         * </p>
         * </note>
         * 
         * @param vpceIds
         *        An array of Amazon VPC endpoint IDs for the workspace. You can create VPC endpoints to your Amazon
         *        Managed Grafana workspace for access from within a VPC. If a <code>NetworkAccessConfiguration</code>
         *        is specified then only VPC endpoints specified here are allowed to access the workspace. If you pass
         *        in an empty array of strings, then no VPCs are allowed to access the workspace.</p>
         *        <p>
         *        VPC endpoint IDs have the format <code>vpce-<i>1a2b3c4d</i> </code>.
         *        </p>
         *        <p>
         *        For more information about creating an interface VPC endpoint, see <a
         *        href="https://docs.aws.amazon.com/grafana/latest/userguide/VPC-endpoints">Interface VPC endpoints</a>
         *        in the <i>Amazon Managed Grafana User Guide</i>.
         *        </p>
         *        <note>
         *        <p>
         *        The only VPC endpoints that can be specified here are interface VPC endpoints for Grafana workspaces
         *        (using the <code>com.amazonaws.[region].grafana-workspace</code> service endpoint). Other VPC
         *        endpoints are ignored.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpceIds(String... vpceIds);
    }

    static final class BuilderImpl implements Builder {
        private List<String> prefixListIds = DefaultSdkAutoConstructList.getInstance();

        private List<String> vpceIds = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(NetworkAccessConfiguration model) {
            prefixListIds(model.prefixListIds);
            vpceIds(model.vpceIds);
        }

        public final Collection<String> getPrefixListIds() {
            if (prefixListIds instanceof SdkAutoConstructList) {
                return null;
            }
            return prefixListIds;
        }

        public final void setPrefixListIds(Collection<String> prefixListIds) {
            this.prefixListIds = PrefixListIdsCopier.copy(prefixListIds);
        }

        @Override
        public final Builder prefixListIds(Collection<String> prefixListIds) {
            this.prefixListIds = PrefixListIdsCopier.copy(prefixListIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder prefixListIds(String... prefixListIds) {
            prefixListIds(Arrays.asList(prefixListIds));
            return this;
        }

        public final Collection<String> getVpceIds() {
            if (vpceIds instanceof SdkAutoConstructList) {
                return null;
            }
            return vpceIds;
        }

        public final void setVpceIds(Collection<String> vpceIds) {
            this.vpceIds = VpceIdsCopier.copy(vpceIds);
        }

        @Override
        public final Builder vpceIds(Collection<String> vpceIds) {
            this.vpceIds = VpceIdsCopier.copy(vpceIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder vpceIds(String... vpceIds) {
            vpceIds(Arrays.asList(vpceIds));
            return this;
        }

        @Override
        public NetworkAccessConfiguration build() {
            return new NetworkAccessConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
