/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamesparks;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.gamesparks.model.AccessDeniedException;
import software.amazon.awssdk.services.gamesparks.model.ConflictException;
import software.amazon.awssdk.services.gamesparks.model.CreateGameRequest;
import software.amazon.awssdk.services.gamesparks.model.CreateGameResponse;
import software.amazon.awssdk.services.gamesparks.model.CreateSnapshotRequest;
import software.amazon.awssdk.services.gamesparks.model.CreateSnapshotResponse;
import software.amazon.awssdk.services.gamesparks.model.CreateStageRequest;
import software.amazon.awssdk.services.gamesparks.model.CreateStageResponse;
import software.amazon.awssdk.services.gamesparks.model.DeleteGameRequest;
import software.amazon.awssdk.services.gamesparks.model.DeleteGameResponse;
import software.amazon.awssdk.services.gamesparks.model.DeleteStageRequest;
import software.amazon.awssdk.services.gamesparks.model.DeleteStageResponse;
import software.amazon.awssdk.services.gamesparks.model.DisconnectPlayerRequest;
import software.amazon.awssdk.services.gamesparks.model.DisconnectPlayerResponse;
import software.amazon.awssdk.services.gamesparks.model.ExportSnapshotRequest;
import software.amazon.awssdk.services.gamesparks.model.ExportSnapshotResponse;
import software.amazon.awssdk.services.gamesparks.model.GameSparksException;
import software.amazon.awssdk.services.gamesparks.model.GetExtensionRequest;
import software.amazon.awssdk.services.gamesparks.model.GetExtensionResponse;
import software.amazon.awssdk.services.gamesparks.model.GetExtensionVersionRequest;
import software.amazon.awssdk.services.gamesparks.model.GetExtensionVersionResponse;
import software.amazon.awssdk.services.gamesparks.model.GetGameConfigurationRequest;
import software.amazon.awssdk.services.gamesparks.model.GetGameConfigurationResponse;
import software.amazon.awssdk.services.gamesparks.model.GetGameRequest;
import software.amazon.awssdk.services.gamesparks.model.GetGameResponse;
import software.amazon.awssdk.services.gamesparks.model.GetGeneratedCodeJobRequest;
import software.amazon.awssdk.services.gamesparks.model.GetGeneratedCodeJobResponse;
import software.amazon.awssdk.services.gamesparks.model.GetPlayerConnectionStatusRequest;
import software.amazon.awssdk.services.gamesparks.model.GetPlayerConnectionStatusResponse;
import software.amazon.awssdk.services.gamesparks.model.GetSnapshotRequest;
import software.amazon.awssdk.services.gamesparks.model.GetSnapshotResponse;
import software.amazon.awssdk.services.gamesparks.model.GetStageDeploymentRequest;
import software.amazon.awssdk.services.gamesparks.model.GetStageDeploymentResponse;
import software.amazon.awssdk.services.gamesparks.model.GetStageRequest;
import software.amazon.awssdk.services.gamesparks.model.GetStageResponse;
import software.amazon.awssdk.services.gamesparks.model.ImportGameConfigurationRequest;
import software.amazon.awssdk.services.gamesparks.model.ImportGameConfigurationResponse;
import software.amazon.awssdk.services.gamesparks.model.InternalServerException;
import software.amazon.awssdk.services.gamesparks.model.ListExtensionVersionsRequest;
import software.amazon.awssdk.services.gamesparks.model.ListExtensionVersionsResponse;
import software.amazon.awssdk.services.gamesparks.model.ListExtensionsRequest;
import software.amazon.awssdk.services.gamesparks.model.ListExtensionsResponse;
import software.amazon.awssdk.services.gamesparks.model.ListGamesRequest;
import software.amazon.awssdk.services.gamesparks.model.ListGamesResponse;
import software.amazon.awssdk.services.gamesparks.model.ListGeneratedCodeJobsRequest;
import software.amazon.awssdk.services.gamesparks.model.ListGeneratedCodeJobsResponse;
import software.amazon.awssdk.services.gamesparks.model.ListSnapshotsRequest;
import software.amazon.awssdk.services.gamesparks.model.ListSnapshotsResponse;
import software.amazon.awssdk.services.gamesparks.model.ListStageDeploymentsRequest;
import software.amazon.awssdk.services.gamesparks.model.ListStageDeploymentsResponse;
import software.amazon.awssdk.services.gamesparks.model.ListStagesRequest;
import software.amazon.awssdk.services.gamesparks.model.ListStagesResponse;
import software.amazon.awssdk.services.gamesparks.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.gamesparks.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.gamesparks.model.ResourceNotFoundException;
import software.amazon.awssdk.services.gamesparks.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.gamesparks.model.StartGeneratedCodeJobRequest;
import software.amazon.awssdk.services.gamesparks.model.StartGeneratedCodeJobResponse;
import software.amazon.awssdk.services.gamesparks.model.StartStageDeploymentRequest;
import software.amazon.awssdk.services.gamesparks.model.StartStageDeploymentResponse;
import software.amazon.awssdk.services.gamesparks.model.TagResourceRequest;
import software.amazon.awssdk.services.gamesparks.model.TagResourceResponse;
import software.amazon.awssdk.services.gamesparks.model.ThrottlingException;
import software.amazon.awssdk.services.gamesparks.model.UntagResourceRequest;
import software.amazon.awssdk.services.gamesparks.model.UntagResourceResponse;
import software.amazon.awssdk.services.gamesparks.model.UpdateGameConfigurationRequest;
import software.amazon.awssdk.services.gamesparks.model.UpdateGameConfigurationResponse;
import software.amazon.awssdk.services.gamesparks.model.UpdateGameRequest;
import software.amazon.awssdk.services.gamesparks.model.UpdateGameResponse;
import software.amazon.awssdk.services.gamesparks.model.UpdateSnapshotRequest;
import software.amazon.awssdk.services.gamesparks.model.UpdateSnapshotResponse;
import software.amazon.awssdk.services.gamesparks.model.UpdateStageRequest;
import software.amazon.awssdk.services.gamesparks.model.UpdateStageResponse;
import software.amazon.awssdk.services.gamesparks.model.ValidationException;
import software.amazon.awssdk.services.gamesparks.transform.CreateGameRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.CreateSnapshotRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.CreateStageRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.DeleteGameRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.DeleteStageRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.DisconnectPlayerRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.ExportSnapshotRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.GetExtensionRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.GetExtensionVersionRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.GetGameConfigurationRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.GetGameRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.GetGeneratedCodeJobRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.GetPlayerConnectionStatusRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.GetSnapshotRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.GetStageDeploymentRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.GetStageRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.ImportGameConfigurationRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.ListExtensionVersionsRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.ListExtensionsRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.ListGamesRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.ListGeneratedCodeJobsRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.ListSnapshotsRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.ListStageDeploymentsRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.ListStagesRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.StartGeneratedCodeJobRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.StartStageDeploymentRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.UpdateGameConfigurationRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.UpdateGameRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.UpdateSnapshotRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.UpdateStageRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link GameSparksClient}.
 *
 * @see GameSparksClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultGameSparksClient implements GameSparksClient {
    private static final Logger log = Logger.loggerFor(DefaultGameSparksClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final GameSparksServiceClientConfiguration serviceClientConfiguration;

    protected DefaultGameSparksClient(GameSparksServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a new game with an empty configuration. After creating your game, you can update the configuration using
     * <code>UpdateGameConfiguration</code> or <code>ImportGameConfiguration</code>.
     * </p>
     *
     * @param createGameRequest
     * @return Result of the CreateGame operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ConflictException
     *         The resource already exists, or another operation is in progress.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws ServiceQuotaExceededException
     *         The request would result in exceeding service quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.CreateGame
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/CreateGame" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateGameResponse createGame(CreateGameRequest createGameRequest) throws ValidationException, AccessDeniedException,
            ThrottlingException, ConflictException, InternalServerException, ServiceQuotaExceededException, AwsServiceException,
            SdkClientException, GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateGameResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateGameResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGameRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGame");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateGameRequest, CreateGameResponse>().withOperationName("CreateGame")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createGameRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateGameRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a snapshot of the game configuration.
     * </p>
     *
     * @param createSnapshotRequest
     * @return Result of the CreateSnapshot operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ConflictException
     *         The resource already exists, or another operation is in progress.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.CreateSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/CreateSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateSnapshotResponse createSnapshot(CreateSnapshotRequest createSnapshotRequest) throws ValidationException,
            AccessDeniedException, ThrottlingException, ConflictException, ResourceNotFoundException, InternalServerException,
            AwsServiceException, SdkClientException, GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateSnapshotResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSnapshot");

            return clientHandler.execute(new ClientExecutionParams<CreateSnapshotRequest, CreateSnapshotResponse>()
                    .withOperationName("CreateSnapshot").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createSnapshotRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateSnapshotRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new stage for stage-by-stage game development and deployment.
     * </p>
     *
     * @param createStageRequest
     * @return Result of the CreateStage operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ConflictException
     *         The resource already exists, or another operation is in progress.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.CreateStage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/CreateStage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateStageResponse createStage(CreateStageRequest createStageRequest) throws ValidationException,
            AccessDeniedException, ThrottlingException, ConflictException, ResourceNotFoundException, InternalServerException,
            AwsServiceException, SdkClientException, GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateStageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateStageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createStageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateStage");

            return clientHandler.execute(new ClientExecutionParams<CreateStageRequest, CreateStageResponse>()
                    .withOperationName("CreateStage").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createStageRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateStageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a game.
     * </p>
     *
     * @param deleteGameRequest
     * @return Result of the DeleteGame operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ConflictException
     *         The resource already exists, or another operation is in progress.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.DeleteGame
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/DeleteGame" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteGameResponse deleteGame(DeleteGameRequest deleteGameRequest) throws ValidationException, AccessDeniedException,
            ThrottlingException, ConflictException, InternalServerException, AwsServiceException, SdkClientException,
            GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteGameResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteGameResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteGameRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteGame");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteGameRequest, DeleteGameResponse>().withOperationName("DeleteGame")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteGameRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteGameRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a stage from a game, along with the associated game runtime.
     * </p>
     *
     * @param deleteStageRequest
     * @return Result of the DeleteStage operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ConflictException
     *         The resource already exists, or another operation is in progress.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.DeleteStage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/DeleteStage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteStageResponse deleteStage(DeleteStageRequest deleteStageRequest) throws ValidationException,
            AccessDeniedException, ThrottlingException, ConflictException, ResourceNotFoundException, InternalServerException,
            AwsServiceException, SdkClientException, GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteStageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteStageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteStageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteStage");

            return clientHandler.execute(new ClientExecutionParams<DeleteStageRequest, DeleteStageResponse>()
                    .withOperationName("DeleteStage").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteStageRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteStageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disconnects a player from the game runtime.
     * </p>
     * <p>
     * If a player has multiple connections, this operation attempts to close all of them.
     * </p>
     *
     * @param disconnectPlayerRequest
     * @return Result of the DisconnectPlayer operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.DisconnectPlayer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/DisconnectPlayer" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DisconnectPlayerResponse disconnectPlayer(DisconnectPlayerRequest disconnectPlayerRequest) throws ValidationException,
            AccessDeniedException, ThrottlingException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisconnectPlayerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DisconnectPlayerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disconnectPlayerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisconnectPlayer");

            return clientHandler.execute(new ClientExecutionParams<DisconnectPlayerRequest, DisconnectPlayerResponse>()
                    .withOperationName("DisconnectPlayer").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(disconnectPlayerRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DisconnectPlayerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Exports a game configuration snapshot.
     * </p>
     *
     * @param exportSnapshotRequest
     * @return Result of the ExportSnapshot operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.ExportSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/ExportSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ExportSnapshotResponse exportSnapshot(ExportSnapshotRequest exportSnapshotRequest) throws ValidationException,
            AccessDeniedException, ThrottlingException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ExportSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ExportSnapshotResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exportSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportSnapshot");

            return clientHandler.execute(new ClientExecutionParams<ExportSnapshotRequest, ExportSnapshotResponse>()
                    .withOperationName("ExportSnapshot").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(exportSnapshotRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ExportSnapshotRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets details about a specified extension.
     * </p>
     *
     * @param getExtensionRequest
     * @return Result of the GetExtension operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.GetExtension
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/GetExtension" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetExtensionResponse getExtension(GetExtensionRequest getExtensionRequest) throws ValidationException,
            AccessDeniedException, ThrottlingException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetExtensionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetExtensionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getExtensionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetExtension");

            return clientHandler.execute(new ClientExecutionParams<GetExtensionRequest, GetExtensionResponse>()
                    .withOperationName("GetExtension").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getExtensionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetExtensionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets details about a specified extension version.
     * </p>
     *
     * @param getExtensionVersionRequest
     * @return Result of the GetExtensionVersion operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.GetExtensionVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/GetExtensionVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetExtensionVersionResponse getExtensionVersion(GetExtensionVersionRequest getExtensionVersionRequest)
            throws ValidationException, AccessDeniedException, ThrottlingException, ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetExtensionVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetExtensionVersionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getExtensionVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetExtensionVersion");

            return clientHandler.execute(new ClientExecutionParams<GetExtensionVersionRequest, GetExtensionVersionResponse>()
                    .withOperationName("GetExtensionVersion").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getExtensionVersionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetExtensionVersionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets details about a game.
     * </p>
     *
     * @param getGameRequest
     * @return Result of the GetGame operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.GetGame
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/GetGame" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetGameResponse getGame(GetGameRequest getGameRequest) throws ValidationException, AccessDeniedException,
            ThrottlingException, ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetGameResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetGameResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGameRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGame");

            return clientHandler.execute(new ClientExecutionParams<GetGameRequest, GetGameResponse>()
                    .withOperationName("GetGame").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getGameRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetGameRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the configuration of the game.
     * </p>
     *
     * @param getGameConfigurationRequest
     * @return Result of the GetGameConfiguration operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.GetGameConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/GetGameConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetGameConfigurationResponse getGameConfiguration(GetGameConfigurationRequest getGameConfigurationRequest)
            throws ValidationException, AccessDeniedException, ThrottlingException, ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetGameConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetGameConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGameConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGameConfiguration");

            return clientHandler.execute(new ClientExecutionParams<GetGameConfigurationRequest, GetGameConfigurationResponse>()
                    .withOperationName("GetGameConfiguration").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getGameConfigurationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetGameConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets details about a job that is generating code for a snapshot.
     * </p>
     *
     * @param getGeneratedCodeJobRequest
     * @return Result of the GetGeneratedCodeJob operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.GetGeneratedCodeJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/GetGeneratedCodeJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetGeneratedCodeJobResponse getGeneratedCodeJob(GetGeneratedCodeJobRequest getGeneratedCodeJobRequest)
            throws ValidationException, AccessDeniedException, ThrottlingException, ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetGeneratedCodeJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetGeneratedCodeJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGeneratedCodeJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGeneratedCodeJob");

            return clientHandler.execute(new ClientExecutionParams<GetGeneratedCodeJobRequest, GetGeneratedCodeJobResponse>()
                    .withOperationName("GetGeneratedCodeJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getGeneratedCodeJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetGeneratedCodeJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the status of a player's connection to the game runtime.
     * </p>
     * <p>
     * It's possible for a single player to have multiple connections to the game runtime. If a player is not connected,
     * this operation returns an empty list.
     * </p>
     *
     * @param getPlayerConnectionStatusRequest
     * @return Result of the GetPlayerConnectionStatus operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.GetPlayerConnectionStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/GetPlayerConnectionStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetPlayerConnectionStatusResponse getPlayerConnectionStatus(
            GetPlayerConnectionStatusRequest getPlayerConnectionStatusRequest) throws ValidationException, AccessDeniedException,
            ThrottlingException, ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetPlayerConnectionStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetPlayerConnectionStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPlayerConnectionStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPlayerConnectionStatus");

            return clientHandler
                    .execute(new ClientExecutionParams<GetPlayerConnectionStatusRequest, GetPlayerConnectionStatusResponse>()
                            .withOperationName("GetPlayerConnectionStatus").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getPlayerConnectionStatusRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetPlayerConnectionStatusRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a copy of the game configuration in a snapshot.
     * </p>
     *
     * @param getSnapshotRequest
     * @return Result of the GetSnapshot operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.GetSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/GetSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetSnapshotResponse getSnapshot(GetSnapshotRequest getSnapshotRequest) throws ValidationException,
            AccessDeniedException, ThrottlingException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetSnapshotResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSnapshot");

            return clientHandler.execute(new ClientExecutionParams<GetSnapshotRequest, GetSnapshotResponse>()
                    .withOperationName("GetSnapshot").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getSnapshotRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSnapshotRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about a stage.
     * </p>
     *
     * @param getStageRequest
     * @return Result of the GetStage operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.GetStage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/GetStage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetStageResponse getStage(GetStageRequest getStageRequest) throws ValidationException, AccessDeniedException,
            ThrottlingException, ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetStageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetStageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getStageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetStage");

            return clientHandler.execute(new ClientExecutionParams<GetStageRequest, GetStageResponse>()
                    .withOperationName("GetStage").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getStageRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetStageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about a stage deployment.
     * </p>
     *
     * @param getStageDeploymentRequest
     * @return Result of the GetStageDeployment operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.GetStageDeployment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/GetStageDeployment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetStageDeploymentResponse getStageDeployment(GetStageDeploymentRequest getStageDeploymentRequest)
            throws ValidationException, AccessDeniedException, ThrottlingException, ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetStageDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetStageDeploymentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getStageDeploymentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetStageDeployment");

            return clientHandler.execute(new ClientExecutionParams<GetStageDeploymentRequest, GetStageDeploymentResponse>()
                    .withOperationName("GetStageDeployment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getStageDeploymentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetStageDeploymentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Imports a game configuration.
     * </p>
     * <p>
     * This operation replaces the current configuration of the game with the provided input. This is not a reversible
     * operation. If you want to preserve the previous configuration, use <code>CreateSnapshot</code> to make a new
     * snapshot before importing.
     * </p>
     *
     * @param importGameConfigurationRequest
     * @return Result of the ImportGameConfiguration operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws ServiceQuotaExceededException
     *         The request would result in exceeding service quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.ImportGameConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/ImportGameConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ImportGameConfigurationResponse importGameConfiguration(ImportGameConfigurationRequest importGameConfigurationRequest)
            throws ValidationException, AccessDeniedException, ThrottlingException, ResourceNotFoundException,
            InternalServerException, ServiceQuotaExceededException, AwsServiceException, SdkClientException, GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ImportGameConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ImportGameConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, importGameConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ImportGameConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<ImportGameConfigurationRequest, ImportGameConfigurationResponse>()
                            .withOperationName("ImportGameConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(importGameConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ImportGameConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a paginated list of available versions for the extension.
     * </p>
     * <p>
     * Each time an API change is made to an extension, the version is incremented. The list retrieved by this operation
     * shows the versions that are currently available.
     * </p>
     *
     * @param listExtensionVersionsRequest
     * @return Result of the ListExtensionVersions operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.ListExtensionVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/ListExtensionVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListExtensionVersionsResponse listExtensionVersions(ListExtensionVersionsRequest listExtensionVersionsRequest)
            throws ValidationException, AccessDeniedException, ThrottlingException, ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListExtensionVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListExtensionVersionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listExtensionVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListExtensionVersions");

            return clientHandler.execute(new ClientExecutionParams<ListExtensionVersionsRequest, ListExtensionVersionsResponse>()
                    .withOperationName("ListExtensionVersions").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listExtensionVersionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListExtensionVersionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a paginated list of available extensions.
     * </p>
     * <p>
     * Extensions provide features that games can use from scripts.
     * </p>
     *
     * @param listExtensionsRequest
     * @return Result of the ListExtensions operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.ListExtensions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/ListExtensions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListExtensionsResponse listExtensions(ListExtensionsRequest listExtensionsRequest) throws ValidationException,
            AccessDeniedException, ThrottlingException, InternalServerException, AwsServiceException, SdkClientException,
            GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListExtensionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListExtensionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listExtensionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListExtensions");

            return clientHandler.execute(new ClientExecutionParams<ListExtensionsRequest, ListExtensionsResponse>()
                    .withOperationName("ListExtensions").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listExtensionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListExtensionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a paginated list of games.
     * </p>
     *
     * @param listGamesRequest
     * @return Result of the ListGames operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.ListGames
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/ListGames" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListGamesResponse listGames(ListGamesRequest listGamesRequest) throws ValidationException, AccessDeniedException,
            ThrottlingException, InternalServerException, AwsServiceException, SdkClientException, GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListGamesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListGamesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGamesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGames");

            return clientHandler.execute(new ClientExecutionParams<ListGamesRequest, ListGamesResponse>()
                    .withOperationName("ListGames").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listGamesRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new ListGamesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a paginated list of code generation jobs for a snapshot.
     * </p>
     *
     * @param listGeneratedCodeJobsRequest
     * @return Result of the ListGeneratedCodeJobs operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.ListGeneratedCodeJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/ListGeneratedCodeJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListGeneratedCodeJobsResponse listGeneratedCodeJobs(ListGeneratedCodeJobsRequest listGeneratedCodeJobsRequest)
            throws ValidationException, AccessDeniedException, ThrottlingException, ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListGeneratedCodeJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListGeneratedCodeJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGeneratedCodeJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGeneratedCodeJobs");

            return clientHandler.execute(new ClientExecutionParams<ListGeneratedCodeJobsRequest, ListGeneratedCodeJobsResponse>()
                    .withOperationName("ListGeneratedCodeJobs").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listGeneratedCodeJobsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListGeneratedCodeJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a paginated list of snapshot summaries from the game.
     * </p>
     *
     * @param listSnapshotsRequest
     * @return Result of the ListSnapshots operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.ListSnapshots
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/ListSnapshots" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListSnapshotsResponse listSnapshots(ListSnapshotsRequest listSnapshotsRequest) throws ValidationException,
            AccessDeniedException, ThrottlingException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSnapshotsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListSnapshotsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSnapshotsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSnapshots");

            return clientHandler.execute(new ClientExecutionParams<ListSnapshotsRequest, ListSnapshotsResponse>()
                    .withOperationName("ListSnapshots").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listSnapshotsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListSnapshotsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a paginated list of stage deployment summaries from the game.
     * </p>
     *
     * @param listStageDeploymentsRequest
     * @return Result of the ListStageDeployments operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.ListStageDeployments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/ListStageDeployments"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListStageDeploymentsResponse listStageDeployments(ListStageDeploymentsRequest listStageDeploymentsRequest)
            throws ValidationException, AccessDeniedException, ThrottlingException, ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListStageDeploymentsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListStageDeploymentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStageDeploymentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStageDeployments");

            return clientHandler.execute(new ClientExecutionParams<ListStageDeploymentsRequest, ListStageDeploymentsResponse>()
                    .withOperationName("ListStageDeployments").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listStageDeploymentsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListStageDeploymentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a paginated list of stage summaries from the game.
     * </p>
     *
     * @param listStagesRequest
     * @return Result of the ListStages operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.ListStages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/ListStages" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListStagesResponse listStages(ListStagesRequest listStagesRequest) throws ValidationException, AccessDeniedException,
            ThrottlingException, ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListStagesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListStagesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStagesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStages");

            return clientHandler
                    .execute(new ClientExecutionParams<ListStagesRequest, ListStagesResponse>().withOperationName("ListStages")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listStagesRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListStagesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the tags associated with a GameSparks resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ValidationException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts an asynchronous process that generates client code for system-defined and custom messages. The resulting
     * code is collected as a .zip file and uploaded to a pre-signed Amazon S3 URL.
     * </p>
     *
     * @param startGeneratedCodeJobRequest
     * @return Result of the StartGeneratedCodeJob operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.StartGeneratedCodeJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/StartGeneratedCodeJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartGeneratedCodeJobResponse startGeneratedCodeJob(StartGeneratedCodeJobRequest startGeneratedCodeJobRequest)
            throws ValidationException, AccessDeniedException, ThrottlingException, ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartGeneratedCodeJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartGeneratedCodeJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startGeneratedCodeJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartGeneratedCodeJob");

            return clientHandler.execute(new ClientExecutionParams<StartGeneratedCodeJobRequest, StartGeneratedCodeJobResponse>()
                    .withOperationName("StartGeneratedCodeJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startGeneratedCodeJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartGeneratedCodeJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deploys a snapshot to the stage and creates a new game runtime.
     * </p>
     * <p>
     * After you call this operation, you can check the deployment status by using <code>GetStageDeployment</code>.
     * </p>
     * <p>
     * If there are any players connected to the previous game runtime, then both runtimes persist. Existing connections
     * to the previous runtime are maintained. When players disconnect and reconnect, they connect to the new runtime.
     * After there are no connections to the previous game runtime, it is deleted.
     * </p>
     *
     * @param startStageDeploymentRequest
     * @return Result of the StartStageDeployment operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ConflictException
     *         The resource already exists, or another operation is in progress.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.StartStageDeployment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/StartStageDeployment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartStageDeploymentResponse startStageDeployment(StartStageDeploymentRequest startStageDeploymentRequest)
            throws ValidationException, AccessDeniedException, ThrottlingException, ConflictException, ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartStageDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartStageDeploymentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startStageDeploymentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartStageDeployment");

            return clientHandler.execute(new ClientExecutionParams<StartStageDeploymentRequest, StartStageDeploymentResponse>()
                    .withOperationName("StartStageDeployment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startStageDeploymentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartStageDeploymentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds tags to a GameSparks resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ValidationException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from a GameSparks resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ValidationException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates details of the game.
     * </p>
     *
     * @param updateGameRequest
     * @return Result of the UpdateGame operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.UpdateGame
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/UpdateGame" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateGameResponse updateGame(UpdateGameRequest updateGameRequest) throws ValidationException, AccessDeniedException,
            ThrottlingException, ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateGameResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateGameResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGameRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGame");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateGameRequest, UpdateGameResponse>().withOperationName("UpdateGame")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateGameRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateGameRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates one or more sections of the game configuration.
     * </p>
     *
     * @param updateGameConfigurationRequest
     * @return Result of the UpdateGameConfiguration operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws ServiceQuotaExceededException
     *         The request would result in exceeding service quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.UpdateGameConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/UpdateGameConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateGameConfigurationResponse updateGameConfiguration(UpdateGameConfigurationRequest updateGameConfigurationRequest)
            throws ValidationException, AccessDeniedException, ThrottlingException, ResourceNotFoundException,
            InternalServerException, ServiceQuotaExceededException, AwsServiceException, SdkClientException, GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateGameConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateGameConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGameConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGameConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateGameConfigurationRequest, UpdateGameConfigurationResponse>()
                            .withOperationName("UpdateGameConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateGameConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateGameConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the metadata of a GameSparks snapshot.
     * </p>
     *
     * @param updateSnapshotRequest
     * @return Result of the UpdateSnapshot operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.UpdateSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/UpdateSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateSnapshotResponse updateSnapshot(UpdateSnapshotRequest updateSnapshotRequest) throws ValidationException,
            AccessDeniedException, ThrottlingException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateSnapshotResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSnapshot");

            return clientHandler.execute(new ClientExecutionParams<UpdateSnapshotRequest, UpdateSnapshotResponse>()
                    .withOperationName("UpdateSnapshot").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateSnapshotRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateSnapshotRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the metadata of a stage.
     * </p>
     *
     * @param updateStageRequest
     * @return Result of the UpdateStage operation returned by the service.
     * @throws ValidationException
     *         One of the parameters in the request is invalid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws InternalServerException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GameSparksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GameSparksClient.UpdateStage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/UpdateStage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateStageResponse updateStage(UpdateStageRequest updateStageRequest) throws ValidationException,
            AccessDeniedException, ThrottlingException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, GameSparksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateStageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateStageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateStageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateStage");

            return clientHandler.execute(new ClientExecutionParams<UpdateStageRequest, UpdateStageResponse>()
                    .withOperationName("UpdateStage").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateStageRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateStageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(GameSparksException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    @Override
    public final GameSparksServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
