/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration for this Microsoft Windows file system.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class WindowsFileSystemConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<WindowsFileSystemConfiguration.Builder, WindowsFileSystemConfiguration> {
    private static final SdkField<String> ACTIVE_DIRECTORY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ActiveDirectoryId").getter(getter(WindowsFileSystemConfiguration::activeDirectoryId))
            .setter(setter(Builder::activeDirectoryId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ActiveDirectoryId").build()).build();

    private static final SdkField<SelfManagedActiveDirectoryAttributes> SELF_MANAGED_ACTIVE_DIRECTORY_CONFIGURATION_FIELD = SdkField
            .<SelfManagedActiveDirectoryAttributes> builder(MarshallingType.SDK_POJO)
            .memberName("SelfManagedActiveDirectoryConfiguration")
            .getter(getter(WindowsFileSystemConfiguration::selfManagedActiveDirectoryConfiguration))
            .setter(setter(Builder::selfManagedActiveDirectoryConfiguration))
            .constructor(SelfManagedActiveDirectoryAttributes::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("SelfManagedActiveDirectoryConfiguration").build()).build();

    private static final SdkField<String> DEPLOYMENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DeploymentType").getter(getter(WindowsFileSystemConfiguration::deploymentTypeAsString))
            .setter(setter(Builder::deploymentType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeploymentType").build()).build();

    private static final SdkField<String> REMOTE_ADMINISTRATION_ENDPOINT_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("RemoteAdministrationEndpoint")
            .getter(getter(WindowsFileSystemConfiguration::remoteAdministrationEndpoint))
            .setter(setter(Builder::remoteAdministrationEndpoint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RemoteAdministrationEndpoint")
                    .build()).build();

    private static final SdkField<String> PREFERRED_SUBNET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PreferredSubnetId").getter(getter(WindowsFileSystemConfiguration::preferredSubnetId))
            .setter(setter(Builder::preferredSubnetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreferredSubnetId").build()).build();

    private static final SdkField<String> PREFERRED_FILE_SERVER_IP_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PreferredFileServerIp").getter(getter(WindowsFileSystemConfiguration::preferredFileServerIp))
            .setter(setter(Builder::preferredFileServerIp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreferredFileServerIp").build())
            .build();

    private static final SdkField<Integer> THROUGHPUT_CAPACITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ThroughputCapacity").getter(getter(WindowsFileSystemConfiguration::throughputCapacity))
            .setter(setter(Builder::throughputCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ThroughputCapacity").build())
            .build();

    private static final SdkField<List<String>> MAINTENANCE_OPERATIONS_IN_PROGRESS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("MaintenanceOperationsInProgress")
            .getter(getter(WindowsFileSystemConfiguration::maintenanceOperationsInProgressAsStrings))
            .setter(setter(Builder::maintenanceOperationsInProgressWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaintenanceOperationsInProgress")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> WEEKLY_MAINTENANCE_START_TIME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("WeeklyMaintenanceStartTime")
            .getter(getter(WindowsFileSystemConfiguration::weeklyMaintenanceStartTime))
            .setter(setter(Builder::weeklyMaintenanceStartTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WeeklyMaintenanceStartTime").build())
            .build();

    private static final SdkField<String> DAILY_AUTOMATIC_BACKUP_START_TIME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DailyAutomaticBackupStartTime")
            .getter(getter(WindowsFileSystemConfiguration::dailyAutomaticBackupStartTime))
            .setter(setter(Builder::dailyAutomaticBackupStartTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DailyAutomaticBackupStartTime")
                    .build()).build();

    private static final SdkField<Integer> AUTOMATIC_BACKUP_RETENTION_DAYS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("AutomaticBackupRetentionDays")
            .getter(getter(WindowsFileSystemConfiguration::automaticBackupRetentionDays))
            .setter(setter(Builder::automaticBackupRetentionDays))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutomaticBackupRetentionDays")
                    .build()).build();

    private static final SdkField<Boolean> COPY_TAGS_TO_BACKUPS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("CopyTagsToBackups").getter(getter(WindowsFileSystemConfiguration::copyTagsToBackups))
            .setter(setter(Builder::copyTagsToBackups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CopyTagsToBackups").build()).build();

    private static final SdkField<List<Alias>> ALIASES_FIELD = SdkField
            .<List<Alias>> builder(MarshallingType.LIST)
            .memberName("Aliases")
            .getter(getter(WindowsFileSystemConfiguration::aliases))
            .setter(setter(Builder::aliases))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Aliases").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Alias> builder(MarshallingType.SDK_POJO)
                                            .constructor(Alias::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<WindowsAuditLogConfiguration> AUDIT_LOG_CONFIGURATION_FIELD = SdkField
            .<WindowsAuditLogConfiguration> builder(MarshallingType.SDK_POJO).memberName("AuditLogConfiguration")
            .getter(getter(WindowsFileSystemConfiguration::auditLogConfiguration)).setter(setter(Builder::auditLogConfiguration))
            .constructor(WindowsAuditLogConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AuditLogConfiguration").build())
            .build();

    private static final SdkField<DiskIopsConfiguration> DISK_IOPS_CONFIGURATION_FIELD = SdkField
            .<DiskIopsConfiguration> builder(MarshallingType.SDK_POJO).memberName("DiskIopsConfiguration")
            .getter(getter(WindowsFileSystemConfiguration::diskIopsConfiguration)).setter(setter(Builder::diskIopsConfiguration))
            .constructor(DiskIopsConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DiskIopsConfiguration").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACTIVE_DIRECTORY_ID_FIELD,
            SELF_MANAGED_ACTIVE_DIRECTORY_CONFIGURATION_FIELD, DEPLOYMENT_TYPE_FIELD, REMOTE_ADMINISTRATION_ENDPOINT_FIELD,
            PREFERRED_SUBNET_ID_FIELD, PREFERRED_FILE_SERVER_IP_FIELD, THROUGHPUT_CAPACITY_FIELD,
            MAINTENANCE_OPERATIONS_IN_PROGRESS_FIELD, WEEKLY_MAINTENANCE_START_TIME_FIELD,
            DAILY_AUTOMATIC_BACKUP_START_TIME_FIELD, AUTOMATIC_BACKUP_RETENTION_DAYS_FIELD, COPY_TAGS_TO_BACKUPS_FIELD,
            ALIASES_FIELD, AUDIT_LOG_CONFIGURATION_FIELD, DISK_IOPS_CONFIGURATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String activeDirectoryId;

    private final SelfManagedActiveDirectoryAttributes selfManagedActiveDirectoryConfiguration;

    private final String deploymentType;

    private final String remoteAdministrationEndpoint;

    private final String preferredSubnetId;

    private final String preferredFileServerIp;

    private final Integer throughputCapacity;

    private final List<String> maintenanceOperationsInProgress;

    private final String weeklyMaintenanceStartTime;

    private final String dailyAutomaticBackupStartTime;

    private final Integer automaticBackupRetentionDays;

    private final Boolean copyTagsToBackups;

    private final List<Alias> aliases;

    private final WindowsAuditLogConfiguration auditLogConfiguration;

    private final DiskIopsConfiguration diskIopsConfiguration;

    private WindowsFileSystemConfiguration(BuilderImpl builder) {
        this.activeDirectoryId = builder.activeDirectoryId;
        this.selfManagedActiveDirectoryConfiguration = builder.selfManagedActiveDirectoryConfiguration;
        this.deploymentType = builder.deploymentType;
        this.remoteAdministrationEndpoint = builder.remoteAdministrationEndpoint;
        this.preferredSubnetId = builder.preferredSubnetId;
        this.preferredFileServerIp = builder.preferredFileServerIp;
        this.throughputCapacity = builder.throughputCapacity;
        this.maintenanceOperationsInProgress = builder.maintenanceOperationsInProgress;
        this.weeklyMaintenanceStartTime = builder.weeklyMaintenanceStartTime;
        this.dailyAutomaticBackupStartTime = builder.dailyAutomaticBackupStartTime;
        this.automaticBackupRetentionDays = builder.automaticBackupRetentionDays;
        this.copyTagsToBackups = builder.copyTagsToBackups;
        this.aliases = builder.aliases;
        this.auditLogConfiguration = builder.auditLogConfiguration;
        this.diskIopsConfiguration = builder.diskIopsConfiguration;
    }

    /**
     * <p>
     * The ID for an existing Amazon Web Services Managed Microsoft Active Directory instance that the file system is
     * joined to.
     * </p>
     * 
     * @return The ID for an existing Amazon Web Services Managed Microsoft Active Directory instance that the file
     *         system is joined to.
     */
    public final String activeDirectoryId() {
        return activeDirectoryId;
    }

    /**
     * Returns the value of the SelfManagedActiveDirectoryConfiguration property for this object.
     * 
     * @return The value of the SelfManagedActiveDirectoryConfiguration property for this object.
     */
    public final SelfManagedActiveDirectoryAttributes selfManagedActiveDirectoryConfiguration() {
        return selfManagedActiveDirectoryConfiguration;
    }

    /**
     * <p>
     * Specifies the file system deployment type, valid values are the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>MULTI_AZ_1</code> - Specifies a high availability file system that is configured for Multi-AZ redundancy to
     * tolerate temporary Availability Zone (AZ) unavailability, and supports SSD and HDD storage.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SINGLE_AZ_1</code> - (Default) Specifies a file system that is configured for single AZ redundancy, only
     * supports SSD storage.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SINGLE_AZ_2</code> - Latest generation Single AZ file system. Specifies a file system that is configured
     * for single AZ redundancy and supports SSD and HDD storage.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/high-availability-multiAZ.html">Single-AZ and Multi-AZ
     * File Systems</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #deploymentType}
     * will return {@link WindowsDeploymentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #deploymentTypeAsString}.
     * </p>
     * 
     * @return Specifies the file system deployment type, valid values are the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>MULTI_AZ_1</code> - Specifies a high availability file system that is configured for Multi-AZ
     *         redundancy to tolerate temporary Availability Zone (AZ) unavailability, and supports SSD and HDD storage.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SINGLE_AZ_1</code> - (Default) Specifies a file system that is configured for single AZ redundancy,
     *         only supports SSD storage.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SINGLE_AZ_2</code> - Latest generation Single AZ file system. Specifies a file system that is
     *         configured for single AZ redundancy and supports SSD and HDD storage.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/high-availability-multiAZ.html">Single-AZ and
     *         Multi-AZ File Systems</a>.
     * @see WindowsDeploymentType
     */
    public final WindowsDeploymentType deploymentType() {
        return WindowsDeploymentType.fromValue(deploymentType);
    }

    /**
     * <p>
     * Specifies the file system deployment type, valid values are the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>MULTI_AZ_1</code> - Specifies a high availability file system that is configured for Multi-AZ redundancy to
     * tolerate temporary Availability Zone (AZ) unavailability, and supports SSD and HDD storage.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SINGLE_AZ_1</code> - (Default) Specifies a file system that is configured for single AZ redundancy, only
     * supports SSD storage.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SINGLE_AZ_2</code> - Latest generation Single AZ file system. Specifies a file system that is configured
     * for single AZ redundancy and supports SSD and HDD storage.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/high-availability-multiAZ.html">Single-AZ and Multi-AZ
     * File Systems</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #deploymentType}
     * will return {@link WindowsDeploymentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #deploymentTypeAsString}.
     * </p>
     * 
     * @return Specifies the file system deployment type, valid values are the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>MULTI_AZ_1</code> - Specifies a high availability file system that is configured for Multi-AZ
     *         redundancy to tolerate temporary Availability Zone (AZ) unavailability, and supports SSD and HDD storage.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SINGLE_AZ_1</code> - (Default) Specifies a file system that is configured for single AZ redundancy,
     *         only supports SSD storage.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SINGLE_AZ_2</code> - Latest generation Single AZ file system. Specifies a file system that is
     *         configured for single AZ redundancy and supports SSD and HDD storage.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/high-availability-multiAZ.html">Single-AZ and
     *         Multi-AZ File Systems</a>.
     * @see WindowsDeploymentType
     */
    public final String deploymentTypeAsString() {
        return deploymentType;
    }

    /**
     * <p>
     * For <code>MULTI_AZ_1</code> deployment types, use this endpoint when performing administrative tasks on the file
     * system using Amazon FSx Remote PowerShell.
     * </p>
     * <p>
     * For <code>SINGLE_AZ_1</code> and <code>SINGLE_AZ_2</code> deployment types, this is the DNS name of the file
     * system.
     * </p>
     * <p>
     * This endpoint is temporarily unavailable when the file system is undergoing maintenance.
     * </p>
     * 
     * @return For <code>MULTI_AZ_1</code> deployment types, use this endpoint when performing administrative tasks on
     *         the file system using Amazon FSx Remote PowerShell.</p>
     *         <p>
     *         For <code>SINGLE_AZ_1</code> and <code>SINGLE_AZ_2</code> deployment types, this is the DNS name of the
     *         file system.
     *         </p>
     *         <p>
     *         This endpoint is temporarily unavailable when the file system is undergoing maintenance.
     */
    public final String remoteAdministrationEndpoint() {
        return remoteAdministrationEndpoint;
    }

    /**
     * <p>
     * For <code>MULTI_AZ_1</code> deployment types, it specifies the ID of the subnet where the preferred file server
     * is located. Must be one of the two subnet IDs specified in <code>SubnetIds</code> property. Amazon FSx serves
     * traffic from this subnet except in the event of a failover to the secondary file server.
     * </p>
     * <p>
     * For <code>SINGLE_AZ_1</code> and <code>SINGLE_AZ_2</code> deployment types, this value is the same as that for
     * <code>SubnetIDs</code>. For more information, see <a href=
     * "https://docs.aws.amazon.com/fsx/latest/WindowsGuide/high-availability-multiAZ.html#single-multi-az-resources"
     * >Availability and durability: Single-AZ and Multi-AZ file systems</a>.
     * </p>
     * 
     * @return For <code>MULTI_AZ_1</code> deployment types, it specifies the ID of the subnet where the preferred file
     *         server is located. Must be one of the two subnet IDs specified in <code>SubnetIds</code> property. Amazon
     *         FSx serves traffic from this subnet except in the event of a failover to the secondary file server.</p>
     *         <p>
     *         For <code>SINGLE_AZ_1</code> and <code>SINGLE_AZ_2</code> deployment types, this value is the same as
     *         that for <code>SubnetIDs</code>. For more information, see <a href=
     *         "https://docs.aws.amazon.com/fsx/latest/WindowsGuide/high-availability-multiAZ.html#single-multi-az-resources"
     *         >Availability and durability: Single-AZ and Multi-AZ file systems</a>.
     */
    public final String preferredSubnetId() {
        return preferredSubnetId;
    }

    /**
     * <p>
     * For <code>MULTI_AZ_1</code> deployment types, the IP address of the primary, or preferred, file server.
     * </p>
     * <p>
     * Use this IP address when mounting the file system on Linux SMB clients or Windows SMB clients that are not joined
     * to a Microsoft Active Directory. Applicable for all Windows file system deployment types. This IP address is
     * temporarily unavailable when the file system is undergoing maintenance. For Linux and Windows SMB clients that
     * are joined to an Active Directory, use the file system's DNSName instead. For more information on mapping and
     * mounting file shares, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/accessing-file-shares.html">Accessing File Shares</a>.
     * </p>
     * 
     * @return For <code>MULTI_AZ_1</code> deployment types, the IP address of the primary, or preferred, file
     *         server.</p>
     *         <p>
     *         Use this IP address when mounting the file system on Linux SMB clients or Windows SMB clients that are
     *         not joined to a Microsoft Active Directory. Applicable for all Windows file system deployment types. This
     *         IP address is temporarily unavailable when the file system is undergoing maintenance. For Linux and
     *         Windows SMB clients that are joined to an Active Directory, use the file system's DNSName instead. For
     *         more information on mapping and mounting file shares, see <a
     *         href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/accessing-file-shares.html">Accessing File
     *         Shares</a>.
     */
    public final String preferredFileServerIp() {
        return preferredFileServerIp;
    }

    /**
     * <p>
     * The throughput of the Amazon FSx file system, measured in megabytes per second.
     * </p>
     * 
     * @return The throughput of the Amazon FSx file system, measured in megabytes per second.
     */
    public final Integer throughputCapacity() {
        return throughputCapacity;
    }

    /**
     * <p>
     * The list of maintenance operations in progress for this file system.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMaintenanceOperationsInProgress}
     * method.
     * </p>
     * 
     * @return The list of maintenance operations in progress for this file system.
     */
    public final List<FileSystemMaintenanceOperation> maintenanceOperationsInProgress() {
        return FileSystemMaintenanceOperationsCopier.copyStringToEnum(maintenanceOperationsInProgress);
    }

    /**
     * For responses, this returns true if the service returned a value for the MaintenanceOperationsInProgress
     * property. This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()}
     * method on the property). This is useful because the SDK will never return a null collection or map, but you may
     * need to differentiate between the service returning nothing (or null) and the service returning an empty
     * collection or map. For requests, this returns true if a value for the property was specified in the request
     * builder, and false if a value was not specified.
     */
    public final boolean hasMaintenanceOperationsInProgress() {
        return maintenanceOperationsInProgress != null && !(maintenanceOperationsInProgress instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of maintenance operations in progress for this file system.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMaintenanceOperationsInProgress}
     * method.
     * </p>
     * 
     * @return The list of maintenance operations in progress for this file system.
     */
    public final List<String> maintenanceOperationsInProgressAsStrings() {
        return maintenanceOperationsInProgress;
    }

    /**
     * <p>
     * The preferred start time to perform weekly maintenance, formatted d:HH:MM in the UTC time zone. d is the weekday
     * number, from 1 through 7, beginning with Monday and ending with Sunday.
     * </p>
     * 
     * @return The preferred start time to perform weekly maintenance, formatted d:HH:MM in the UTC time zone. d is the
     *         weekday number, from 1 through 7, beginning with Monday and ending with Sunday.
     */
    public final String weeklyMaintenanceStartTime() {
        return weeklyMaintenanceStartTime;
    }

    /**
     * <p>
     * The preferred time to take daily automatic backups, in the UTC time zone.
     * </p>
     * 
     * @return The preferred time to take daily automatic backups, in the UTC time zone.
     */
    public final String dailyAutomaticBackupStartTime() {
        return dailyAutomaticBackupStartTime;
    }

    /**
     * <p>
     * The number of days to retain automatic backups. Setting this to 0 disables automatic backups. You can retain
     * automatic backups for a maximum of 90 days.
     * </p>
     * 
     * @return The number of days to retain automatic backups. Setting this to 0 disables automatic backups. You can
     *         retain automatic backups for a maximum of 90 days.
     */
    public final Integer automaticBackupRetentionDays() {
        return automaticBackupRetentionDays;
    }

    /**
     * <p>
     * A boolean flag indicating whether tags on the file system should be copied to backups. This value defaults to
     * false. If it's set to true, all tags on the file system are copied to all automatic backups and any
     * user-initiated backups where the user doesn't specify any tags. If this value is true, and you specify one or
     * more tags, only the specified tags are copied to backups. If you specify one or more tags when creating a
     * user-initiated backup, no tags are copied from the file system, regardless of this value.
     * </p>
     * 
     * @return A boolean flag indicating whether tags on the file system should be copied to backups. This value
     *         defaults to false. If it's set to true, all tags on the file system are copied to all automatic backups
     *         and any user-initiated backups where the user doesn't specify any tags. If this value is true, and you
     *         specify one or more tags, only the specified tags are copied to backups. If you specify one or more tags
     *         when creating a user-initiated backup, no tags are copied from the file system, regardless of this value.
     */
    public final Boolean copyTagsToBackups() {
        return copyTagsToBackups;
    }

    /**
     * For responses, this returns true if the service returned a value for the Aliases property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAliases() {
        return aliases != null && !(aliases instanceof SdkAutoConstructList);
    }

    /**
     * Returns the value of the Aliases property for this object.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAliases} method.
     * </p>
     * 
     * @return The value of the Aliases property for this object.
     */
    public final List<Alias> aliases() {
        return aliases;
    }

    /**
     * <p>
     * The configuration that Amazon FSx for Windows File Server uses to audit and log user accesses of files, folders,
     * and file shares on the Amazon FSx for Windows File Server file system.
     * </p>
     * 
     * @return The configuration that Amazon FSx for Windows File Server uses to audit and log user accesses of files,
     *         folders, and file shares on the Amazon FSx for Windows File Server file system.
     */
    public final WindowsAuditLogConfiguration auditLogConfiguration() {
        return auditLogConfiguration;
    }

    /**
     * <p>
     * The SSD IOPS (input/output operations per second) configuration for an Amazon FSx for Windows file system. By
     * default, Amazon FSx automatically provisions 3 IOPS per GiB of storage capacity. You can provision additional
     * IOPS per GiB of storage, up to the maximum limit associated with your chosen throughput capacity.
     * </p>
     * 
     * @return The SSD IOPS (input/output operations per second) configuration for an Amazon FSx for Windows file
     *         system. By default, Amazon FSx automatically provisions 3 IOPS per GiB of storage capacity. You can
     *         provision additional IOPS per GiB of storage, up to the maximum limit associated with your chosen
     *         throughput capacity.
     */
    public final DiskIopsConfiguration diskIopsConfiguration() {
        return diskIopsConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(activeDirectoryId());
        hashCode = 31 * hashCode + Objects.hashCode(selfManagedActiveDirectoryConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(deploymentTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(remoteAdministrationEndpoint());
        hashCode = 31 * hashCode + Objects.hashCode(preferredSubnetId());
        hashCode = 31 * hashCode + Objects.hashCode(preferredFileServerIp());
        hashCode = 31 * hashCode + Objects.hashCode(throughputCapacity());
        hashCode = 31 * hashCode
                + Objects.hashCode(hasMaintenanceOperationsInProgress() ? maintenanceOperationsInProgressAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(weeklyMaintenanceStartTime());
        hashCode = 31 * hashCode + Objects.hashCode(dailyAutomaticBackupStartTime());
        hashCode = 31 * hashCode + Objects.hashCode(automaticBackupRetentionDays());
        hashCode = 31 * hashCode + Objects.hashCode(copyTagsToBackups());
        hashCode = 31 * hashCode + Objects.hashCode(hasAliases() ? aliases() : null);
        hashCode = 31 * hashCode + Objects.hashCode(auditLogConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(diskIopsConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof WindowsFileSystemConfiguration)) {
            return false;
        }
        WindowsFileSystemConfiguration other = (WindowsFileSystemConfiguration) obj;
        return Objects.equals(activeDirectoryId(), other.activeDirectoryId())
                && Objects.equals(selfManagedActiveDirectoryConfiguration(), other.selfManagedActiveDirectoryConfiguration())
                && Objects.equals(deploymentTypeAsString(), other.deploymentTypeAsString())
                && Objects.equals(remoteAdministrationEndpoint(), other.remoteAdministrationEndpoint())
                && Objects.equals(preferredSubnetId(), other.preferredSubnetId())
                && Objects.equals(preferredFileServerIp(), other.preferredFileServerIp())
                && Objects.equals(throughputCapacity(), other.throughputCapacity())
                && hasMaintenanceOperationsInProgress() == other.hasMaintenanceOperationsInProgress()
                && Objects.equals(maintenanceOperationsInProgressAsStrings(), other.maintenanceOperationsInProgressAsStrings())
                && Objects.equals(weeklyMaintenanceStartTime(), other.weeklyMaintenanceStartTime())
                && Objects.equals(dailyAutomaticBackupStartTime(), other.dailyAutomaticBackupStartTime())
                && Objects.equals(automaticBackupRetentionDays(), other.automaticBackupRetentionDays())
                && Objects.equals(copyTagsToBackups(), other.copyTagsToBackups()) && hasAliases() == other.hasAliases()
                && Objects.equals(aliases(), other.aliases())
                && Objects.equals(auditLogConfiguration(), other.auditLogConfiguration())
                && Objects.equals(diskIopsConfiguration(), other.diskIopsConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("WindowsFileSystemConfiguration")
                .add("ActiveDirectoryId", activeDirectoryId())
                .add("SelfManagedActiveDirectoryConfiguration", selfManagedActiveDirectoryConfiguration())
                .add("DeploymentType", deploymentTypeAsString())
                .add("RemoteAdministrationEndpoint", remoteAdministrationEndpoint())
                .add("PreferredSubnetId", preferredSubnetId())
                .add("PreferredFileServerIp", preferredFileServerIp())
                .add("ThroughputCapacity", throughputCapacity())
                .add("MaintenanceOperationsInProgress",
                        hasMaintenanceOperationsInProgress() ? maintenanceOperationsInProgressAsStrings() : null)
                .add("WeeklyMaintenanceStartTime", weeklyMaintenanceStartTime())
                .add("DailyAutomaticBackupStartTime", dailyAutomaticBackupStartTime())
                .add("AutomaticBackupRetentionDays", automaticBackupRetentionDays())
                .add("CopyTagsToBackups", copyTagsToBackups()).add("Aliases", hasAliases() ? aliases() : null)
                .add("AuditLogConfiguration", auditLogConfiguration()).add("DiskIopsConfiguration", diskIopsConfiguration())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ActiveDirectoryId":
            return Optional.ofNullable(clazz.cast(activeDirectoryId()));
        case "SelfManagedActiveDirectoryConfiguration":
            return Optional.ofNullable(clazz.cast(selfManagedActiveDirectoryConfiguration()));
        case "DeploymentType":
            return Optional.ofNullable(clazz.cast(deploymentTypeAsString()));
        case "RemoteAdministrationEndpoint":
            return Optional.ofNullable(clazz.cast(remoteAdministrationEndpoint()));
        case "PreferredSubnetId":
            return Optional.ofNullable(clazz.cast(preferredSubnetId()));
        case "PreferredFileServerIp":
            return Optional.ofNullable(clazz.cast(preferredFileServerIp()));
        case "ThroughputCapacity":
            return Optional.ofNullable(clazz.cast(throughputCapacity()));
        case "MaintenanceOperationsInProgress":
            return Optional.ofNullable(clazz.cast(maintenanceOperationsInProgressAsStrings()));
        case "WeeklyMaintenanceStartTime":
            return Optional.ofNullable(clazz.cast(weeklyMaintenanceStartTime()));
        case "DailyAutomaticBackupStartTime":
            return Optional.ofNullable(clazz.cast(dailyAutomaticBackupStartTime()));
        case "AutomaticBackupRetentionDays":
            return Optional.ofNullable(clazz.cast(automaticBackupRetentionDays()));
        case "CopyTagsToBackups":
            return Optional.ofNullable(clazz.cast(copyTagsToBackups()));
        case "Aliases":
            return Optional.ofNullable(clazz.cast(aliases()));
        case "AuditLogConfiguration":
            return Optional.ofNullable(clazz.cast(auditLogConfiguration()));
        case "DiskIopsConfiguration":
            return Optional.ofNullable(clazz.cast(diskIopsConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<WindowsFileSystemConfiguration, T> g) {
        return obj -> g.apply((WindowsFileSystemConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, WindowsFileSystemConfiguration> {
        /**
         * <p>
         * The ID for an existing Amazon Web Services Managed Microsoft Active Directory instance that the file system
         * is joined to.
         * </p>
         * 
         * @param activeDirectoryId
         *        The ID for an existing Amazon Web Services Managed Microsoft Active Directory instance that the file
         *        system is joined to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder activeDirectoryId(String activeDirectoryId);

        /**
         * Sets the value of the SelfManagedActiveDirectoryConfiguration property for this object.
         *
         * @param selfManagedActiveDirectoryConfiguration
         *        The new value for the SelfManagedActiveDirectoryConfiguration property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder selfManagedActiveDirectoryConfiguration(
                SelfManagedActiveDirectoryAttributes selfManagedActiveDirectoryConfiguration);

        /**
         * Sets the value of the SelfManagedActiveDirectoryConfiguration property for this object.
         *
         * This is a convenience method that creates an instance of the
         * {@link SelfManagedActiveDirectoryAttributes.Builder} avoiding the need to create one manually via
         * {@link SelfManagedActiveDirectoryAttributes#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SelfManagedActiveDirectoryAttributes.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #selfManagedActiveDirectoryConfiguration(SelfManagedActiveDirectoryAttributes)}.
         * 
         * @param selfManagedActiveDirectoryConfiguration
         *        a consumer that will call methods on {@link SelfManagedActiveDirectoryAttributes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #selfManagedActiveDirectoryConfiguration(SelfManagedActiveDirectoryAttributes)
         */
        default Builder selfManagedActiveDirectoryConfiguration(
                Consumer<SelfManagedActiveDirectoryAttributes.Builder> selfManagedActiveDirectoryConfiguration) {
            return selfManagedActiveDirectoryConfiguration(SelfManagedActiveDirectoryAttributes.builder()
                    .applyMutation(selfManagedActiveDirectoryConfiguration).build());
        }

        /**
         * <p>
         * Specifies the file system deployment type, valid values are the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>MULTI_AZ_1</code> - Specifies a high availability file system that is configured for Multi-AZ
         * redundancy to tolerate temporary Availability Zone (AZ) unavailability, and supports SSD and HDD storage.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SINGLE_AZ_1</code> - (Default) Specifies a file system that is configured for single AZ redundancy,
         * only supports SSD storage.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SINGLE_AZ_2</code> - Latest generation Single AZ file system. Specifies a file system that is
         * configured for single AZ redundancy and supports SSD and HDD storage.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/high-availability-multiAZ.html">Single-AZ and
         * Multi-AZ File Systems</a>.
         * </p>
         * 
         * @param deploymentType
         *        Specifies the file system deployment type, valid values are the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>MULTI_AZ_1</code> - Specifies a high availability file system that is configured for Multi-AZ
         *        redundancy to tolerate temporary Availability Zone (AZ) unavailability, and supports SSD and HDD
         *        storage.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SINGLE_AZ_1</code> - (Default) Specifies a file system that is configured for single AZ
         *        redundancy, only supports SSD storage.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SINGLE_AZ_2</code> - Latest generation Single AZ file system. Specifies a file system that is
         *        configured for single AZ redundancy and supports SSD and HDD storage.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/high-availability-multiAZ.html">Single-AZ
         *        and Multi-AZ File Systems</a>.
         * @see WindowsDeploymentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WindowsDeploymentType
         */
        Builder deploymentType(String deploymentType);

        /**
         * <p>
         * Specifies the file system deployment type, valid values are the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>MULTI_AZ_1</code> - Specifies a high availability file system that is configured for Multi-AZ
         * redundancy to tolerate temporary Availability Zone (AZ) unavailability, and supports SSD and HDD storage.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SINGLE_AZ_1</code> - (Default) Specifies a file system that is configured for single AZ redundancy,
         * only supports SSD storage.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SINGLE_AZ_2</code> - Latest generation Single AZ file system. Specifies a file system that is
         * configured for single AZ redundancy and supports SSD and HDD storage.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/high-availability-multiAZ.html">Single-AZ and
         * Multi-AZ File Systems</a>.
         * </p>
         * 
         * @param deploymentType
         *        Specifies the file system deployment type, valid values are the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>MULTI_AZ_1</code> - Specifies a high availability file system that is configured for Multi-AZ
         *        redundancy to tolerate temporary Availability Zone (AZ) unavailability, and supports SSD and HDD
         *        storage.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SINGLE_AZ_1</code> - (Default) Specifies a file system that is configured for single AZ
         *        redundancy, only supports SSD storage.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SINGLE_AZ_2</code> - Latest generation Single AZ file system. Specifies a file system that is
         *        configured for single AZ redundancy and supports SSD and HDD storage.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/high-availability-multiAZ.html">Single-AZ
         *        and Multi-AZ File Systems</a>.
         * @see WindowsDeploymentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WindowsDeploymentType
         */
        Builder deploymentType(WindowsDeploymentType deploymentType);

        /**
         * <p>
         * For <code>MULTI_AZ_1</code> deployment types, use this endpoint when performing administrative tasks on the
         * file system using Amazon FSx Remote PowerShell.
         * </p>
         * <p>
         * For <code>SINGLE_AZ_1</code> and <code>SINGLE_AZ_2</code> deployment types, this is the DNS name of the file
         * system.
         * </p>
         * <p>
         * This endpoint is temporarily unavailable when the file system is undergoing maintenance.
         * </p>
         * 
         * @param remoteAdministrationEndpoint
         *        For <code>MULTI_AZ_1</code> deployment types, use this endpoint when performing administrative tasks
         *        on the file system using Amazon FSx Remote PowerShell.</p>
         *        <p>
         *        For <code>SINGLE_AZ_1</code> and <code>SINGLE_AZ_2</code> deployment types, this is the DNS name of
         *        the file system.
         *        </p>
         *        <p>
         *        This endpoint is temporarily unavailable when the file system is undergoing maintenance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder remoteAdministrationEndpoint(String remoteAdministrationEndpoint);

        /**
         * <p>
         * For <code>MULTI_AZ_1</code> deployment types, it specifies the ID of the subnet where the preferred file
         * server is located. Must be one of the two subnet IDs specified in <code>SubnetIds</code> property. Amazon FSx
         * serves traffic from this subnet except in the event of a failover to the secondary file server.
         * </p>
         * <p>
         * For <code>SINGLE_AZ_1</code> and <code>SINGLE_AZ_2</code> deployment types, this value is the same as that
         * for <code>SubnetIDs</code>. For more information, see <a href=
         * "https://docs.aws.amazon.com/fsx/latest/WindowsGuide/high-availability-multiAZ.html#single-multi-az-resources"
         * >Availability and durability: Single-AZ and Multi-AZ file systems</a>.
         * </p>
         * 
         * @param preferredSubnetId
         *        For <code>MULTI_AZ_1</code> deployment types, it specifies the ID of the subnet where the preferred
         *        file server is located. Must be one of the two subnet IDs specified in <code>SubnetIds</code>
         *        property. Amazon FSx serves traffic from this subnet except in the event of a failover to the
         *        secondary file server.</p>
         *        <p>
         *        For <code>SINGLE_AZ_1</code> and <code>SINGLE_AZ_2</code> deployment types, this value is the same as
         *        that for <code>SubnetIDs</code>. For more information, see <a href=
         *        "https://docs.aws.amazon.com/fsx/latest/WindowsGuide/high-availability-multiAZ.html#single-multi-az-resources"
         *        >Availability and durability: Single-AZ and Multi-AZ file systems</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredSubnetId(String preferredSubnetId);

        /**
         * <p>
         * For <code>MULTI_AZ_1</code> deployment types, the IP address of the primary, or preferred, file server.
         * </p>
         * <p>
         * Use this IP address when mounting the file system on Linux SMB clients or Windows SMB clients that are not
         * joined to a Microsoft Active Directory. Applicable for all Windows file system deployment types. This IP
         * address is temporarily unavailable when the file system is undergoing maintenance. For Linux and Windows SMB
         * clients that are joined to an Active Directory, use the file system's DNSName instead. For more information
         * on mapping and mounting file shares, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/accessing-file-shares.html">Accessing File
         * Shares</a>.
         * </p>
         * 
         * @param preferredFileServerIp
         *        For <code>MULTI_AZ_1</code> deployment types, the IP address of the primary, or preferred, file
         *        server.</p>
         *        <p>
         *        Use this IP address when mounting the file system on Linux SMB clients or Windows SMB clients that are
         *        not joined to a Microsoft Active Directory. Applicable for all Windows file system deployment types.
         *        This IP address is temporarily unavailable when the file system is undergoing maintenance. For Linux
         *        and Windows SMB clients that are joined to an Active Directory, use the file system's DNSName instead.
         *        For more information on mapping and mounting file shares, see <a
         *        href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/accessing-file-shares.html">Accessing File
         *        Shares</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredFileServerIp(String preferredFileServerIp);

        /**
         * <p>
         * The throughput of the Amazon FSx file system, measured in megabytes per second.
         * </p>
         * 
         * @param throughputCapacity
         *        The throughput of the Amazon FSx file system, measured in megabytes per second.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder throughputCapacity(Integer throughputCapacity);

        /**
         * <p>
         * The list of maintenance operations in progress for this file system.
         * </p>
         * 
         * @param maintenanceOperationsInProgress
         *        The list of maintenance operations in progress for this file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maintenanceOperationsInProgressWithStrings(Collection<String> maintenanceOperationsInProgress);

        /**
         * <p>
         * The list of maintenance operations in progress for this file system.
         * </p>
         * 
         * @param maintenanceOperationsInProgress
         *        The list of maintenance operations in progress for this file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maintenanceOperationsInProgressWithStrings(String... maintenanceOperationsInProgress);

        /**
         * <p>
         * The list of maintenance operations in progress for this file system.
         * </p>
         * 
         * @param maintenanceOperationsInProgress
         *        The list of maintenance operations in progress for this file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maintenanceOperationsInProgress(Collection<FileSystemMaintenanceOperation> maintenanceOperationsInProgress);

        /**
         * <p>
         * The list of maintenance operations in progress for this file system.
         * </p>
         * 
         * @param maintenanceOperationsInProgress
         *        The list of maintenance operations in progress for this file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maintenanceOperationsInProgress(FileSystemMaintenanceOperation... maintenanceOperationsInProgress);

        /**
         * <p>
         * The preferred start time to perform weekly maintenance, formatted d:HH:MM in the UTC time zone. d is the
         * weekday number, from 1 through 7, beginning with Monday and ending with Sunday.
         * </p>
         * 
         * @param weeklyMaintenanceStartTime
         *        The preferred start time to perform weekly maintenance, formatted d:HH:MM in the UTC time zone. d is
         *        the weekday number, from 1 through 7, beginning with Monday and ending with Sunday.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder weeklyMaintenanceStartTime(String weeklyMaintenanceStartTime);

        /**
         * <p>
         * The preferred time to take daily automatic backups, in the UTC time zone.
         * </p>
         * 
         * @param dailyAutomaticBackupStartTime
         *        The preferred time to take daily automatic backups, in the UTC time zone.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dailyAutomaticBackupStartTime(String dailyAutomaticBackupStartTime);

        /**
         * <p>
         * The number of days to retain automatic backups. Setting this to 0 disables automatic backups. You can retain
         * automatic backups for a maximum of 90 days.
         * </p>
         * 
         * @param automaticBackupRetentionDays
         *        The number of days to retain automatic backups. Setting this to 0 disables automatic backups. You can
         *        retain automatic backups for a maximum of 90 days.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder automaticBackupRetentionDays(Integer automaticBackupRetentionDays);

        /**
         * <p>
         * A boolean flag indicating whether tags on the file system should be copied to backups. This value defaults to
         * false. If it's set to true, all tags on the file system are copied to all automatic backups and any
         * user-initiated backups where the user doesn't specify any tags. If this value is true, and you specify one or
         * more tags, only the specified tags are copied to backups. If you specify one or more tags when creating a
         * user-initiated backup, no tags are copied from the file system, regardless of this value.
         * </p>
         * 
         * @param copyTagsToBackups
         *        A boolean flag indicating whether tags on the file system should be copied to backups. This value
         *        defaults to false. If it's set to true, all tags on the file system are copied to all automatic
         *        backups and any user-initiated backups where the user doesn't specify any tags. If this value is true,
         *        and you specify one or more tags, only the specified tags are copied to backups. If you specify one or
         *        more tags when creating a user-initiated backup, no tags are copied from the file system, regardless
         *        of this value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder copyTagsToBackups(Boolean copyTagsToBackups);

        /**
         * Sets the value of the Aliases property for this object.
         *
         * @param aliases
         *        The new value for the Aliases property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder aliases(Collection<Alias> aliases);

        /**
         * Sets the value of the Aliases property for this object.
         *
         * @param aliases
         *        The new value for the Aliases property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder aliases(Alias... aliases);

        /**
         * Sets the value of the Aliases property for this object.
         *
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.fsx.model.Alias.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.fsx.model.Alias#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.fsx.model.Alias.Builder#build()}
         * is called immediately and its result is passed to {@link #aliases(List<Alias>)}.
         * 
         * @param aliases
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.fsx.model.Alias.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #aliases(java.util.Collection<Alias>)
         */
        Builder aliases(Consumer<Alias.Builder>... aliases);

        /**
         * <p>
         * The configuration that Amazon FSx for Windows File Server uses to audit and log user accesses of files,
         * folders, and file shares on the Amazon FSx for Windows File Server file system.
         * </p>
         * 
         * @param auditLogConfiguration
         *        The configuration that Amazon FSx for Windows File Server uses to audit and log user accesses of
         *        files, folders, and file shares on the Amazon FSx for Windows File Server file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder auditLogConfiguration(WindowsAuditLogConfiguration auditLogConfiguration);

        /**
         * <p>
         * The configuration that Amazon FSx for Windows File Server uses to audit and log user accesses of files,
         * folders, and file shares on the Amazon FSx for Windows File Server file system.
         * </p>
         * This is a convenience method that creates an instance of the {@link WindowsAuditLogConfiguration.Builder}
         * avoiding the need to create one manually via {@link WindowsAuditLogConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link WindowsAuditLogConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #auditLogConfiguration(WindowsAuditLogConfiguration)}.
         * 
         * @param auditLogConfiguration
         *        a consumer that will call methods on {@link WindowsAuditLogConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #auditLogConfiguration(WindowsAuditLogConfiguration)
         */
        default Builder auditLogConfiguration(Consumer<WindowsAuditLogConfiguration.Builder> auditLogConfiguration) {
            return auditLogConfiguration(WindowsAuditLogConfiguration.builder().applyMutation(auditLogConfiguration).build());
        }

        /**
         * <p>
         * The SSD IOPS (input/output operations per second) configuration for an Amazon FSx for Windows file system. By
         * default, Amazon FSx automatically provisions 3 IOPS per GiB of storage capacity. You can provision additional
         * IOPS per GiB of storage, up to the maximum limit associated with your chosen throughput capacity.
         * </p>
         * 
         * @param diskIopsConfiguration
         *        The SSD IOPS (input/output operations per second) configuration for an Amazon FSx for Windows file
         *        system. By default, Amazon FSx automatically provisions 3 IOPS per GiB of storage capacity. You can
         *        provision additional IOPS per GiB of storage, up to the maximum limit associated with your chosen
         *        throughput capacity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder diskIopsConfiguration(DiskIopsConfiguration diskIopsConfiguration);

        /**
         * <p>
         * The SSD IOPS (input/output operations per second) configuration for an Amazon FSx for Windows file system. By
         * default, Amazon FSx automatically provisions 3 IOPS per GiB of storage capacity. You can provision additional
         * IOPS per GiB of storage, up to the maximum limit associated with your chosen throughput capacity.
         * </p>
         * This is a convenience method that creates an instance of the {@link DiskIopsConfiguration.Builder} avoiding
         * the need to create one manually via {@link DiskIopsConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DiskIopsConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #diskIopsConfiguration(DiskIopsConfiguration)}.
         * 
         * @param diskIopsConfiguration
         *        a consumer that will call methods on {@link DiskIopsConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #diskIopsConfiguration(DiskIopsConfiguration)
         */
        default Builder diskIopsConfiguration(Consumer<DiskIopsConfiguration.Builder> diskIopsConfiguration) {
            return diskIopsConfiguration(DiskIopsConfiguration.builder().applyMutation(diskIopsConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String activeDirectoryId;

        private SelfManagedActiveDirectoryAttributes selfManagedActiveDirectoryConfiguration;

        private String deploymentType;

        private String remoteAdministrationEndpoint;

        private String preferredSubnetId;

        private String preferredFileServerIp;

        private Integer throughputCapacity;

        private List<String> maintenanceOperationsInProgress = DefaultSdkAutoConstructList.getInstance();

        private String weeklyMaintenanceStartTime;

        private String dailyAutomaticBackupStartTime;

        private Integer automaticBackupRetentionDays;

        private Boolean copyTagsToBackups;

        private List<Alias> aliases = DefaultSdkAutoConstructList.getInstance();

        private WindowsAuditLogConfiguration auditLogConfiguration;

        private DiskIopsConfiguration diskIopsConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(WindowsFileSystemConfiguration model) {
            activeDirectoryId(model.activeDirectoryId);
            selfManagedActiveDirectoryConfiguration(model.selfManagedActiveDirectoryConfiguration);
            deploymentType(model.deploymentType);
            remoteAdministrationEndpoint(model.remoteAdministrationEndpoint);
            preferredSubnetId(model.preferredSubnetId);
            preferredFileServerIp(model.preferredFileServerIp);
            throughputCapacity(model.throughputCapacity);
            maintenanceOperationsInProgressWithStrings(model.maintenanceOperationsInProgress);
            weeklyMaintenanceStartTime(model.weeklyMaintenanceStartTime);
            dailyAutomaticBackupStartTime(model.dailyAutomaticBackupStartTime);
            automaticBackupRetentionDays(model.automaticBackupRetentionDays);
            copyTagsToBackups(model.copyTagsToBackups);
            aliases(model.aliases);
            auditLogConfiguration(model.auditLogConfiguration);
            diskIopsConfiguration(model.diskIopsConfiguration);
        }

        public final String getActiveDirectoryId() {
            return activeDirectoryId;
        }

        public final void setActiveDirectoryId(String activeDirectoryId) {
            this.activeDirectoryId = activeDirectoryId;
        }

        @Override
        public final Builder activeDirectoryId(String activeDirectoryId) {
            this.activeDirectoryId = activeDirectoryId;
            return this;
        }

        public final SelfManagedActiveDirectoryAttributes.Builder getSelfManagedActiveDirectoryConfiguration() {
            return selfManagedActiveDirectoryConfiguration != null ? selfManagedActiveDirectoryConfiguration.toBuilder() : null;
        }

        public final void setSelfManagedActiveDirectoryConfiguration(
                SelfManagedActiveDirectoryAttributes.BuilderImpl selfManagedActiveDirectoryConfiguration) {
            this.selfManagedActiveDirectoryConfiguration = selfManagedActiveDirectoryConfiguration != null ? selfManagedActiveDirectoryConfiguration
                    .build() : null;
        }

        @Override
        public final Builder selfManagedActiveDirectoryConfiguration(
                SelfManagedActiveDirectoryAttributes selfManagedActiveDirectoryConfiguration) {
            this.selfManagedActiveDirectoryConfiguration = selfManagedActiveDirectoryConfiguration;
            return this;
        }

        public final String getDeploymentType() {
            return deploymentType;
        }

        public final void setDeploymentType(String deploymentType) {
            this.deploymentType = deploymentType;
        }

        @Override
        public final Builder deploymentType(String deploymentType) {
            this.deploymentType = deploymentType;
            return this;
        }

        @Override
        public final Builder deploymentType(WindowsDeploymentType deploymentType) {
            this.deploymentType(deploymentType == null ? null : deploymentType.toString());
            return this;
        }

        public final String getRemoteAdministrationEndpoint() {
            return remoteAdministrationEndpoint;
        }

        public final void setRemoteAdministrationEndpoint(String remoteAdministrationEndpoint) {
            this.remoteAdministrationEndpoint = remoteAdministrationEndpoint;
        }

        @Override
        public final Builder remoteAdministrationEndpoint(String remoteAdministrationEndpoint) {
            this.remoteAdministrationEndpoint = remoteAdministrationEndpoint;
            return this;
        }

        public final String getPreferredSubnetId() {
            return preferredSubnetId;
        }

        public final void setPreferredSubnetId(String preferredSubnetId) {
            this.preferredSubnetId = preferredSubnetId;
        }

        @Override
        public final Builder preferredSubnetId(String preferredSubnetId) {
            this.preferredSubnetId = preferredSubnetId;
            return this;
        }

        public final String getPreferredFileServerIp() {
            return preferredFileServerIp;
        }

        public final void setPreferredFileServerIp(String preferredFileServerIp) {
            this.preferredFileServerIp = preferredFileServerIp;
        }

        @Override
        public final Builder preferredFileServerIp(String preferredFileServerIp) {
            this.preferredFileServerIp = preferredFileServerIp;
            return this;
        }

        public final Integer getThroughputCapacity() {
            return throughputCapacity;
        }

        public final void setThroughputCapacity(Integer throughputCapacity) {
            this.throughputCapacity = throughputCapacity;
        }

        @Override
        public final Builder throughputCapacity(Integer throughputCapacity) {
            this.throughputCapacity = throughputCapacity;
            return this;
        }

        public final Collection<String> getMaintenanceOperationsInProgress() {
            if (maintenanceOperationsInProgress instanceof SdkAutoConstructList) {
                return null;
            }
            return maintenanceOperationsInProgress;
        }

        public final void setMaintenanceOperationsInProgress(Collection<String> maintenanceOperationsInProgress) {
            this.maintenanceOperationsInProgress = FileSystemMaintenanceOperationsCopier.copy(maintenanceOperationsInProgress);
        }

        @Override
        public final Builder maintenanceOperationsInProgressWithStrings(Collection<String> maintenanceOperationsInProgress) {
            this.maintenanceOperationsInProgress = FileSystemMaintenanceOperationsCopier.copy(maintenanceOperationsInProgress);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder maintenanceOperationsInProgressWithStrings(String... maintenanceOperationsInProgress) {
            maintenanceOperationsInProgressWithStrings(Arrays.asList(maintenanceOperationsInProgress));
            return this;
        }

        @Override
        public final Builder maintenanceOperationsInProgress(
                Collection<FileSystemMaintenanceOperation> maintenanceOperationsInProgress) {
            this.maintenanceOperationsInProgress = FileSystemMaintenanceOperationsCopier
                    .copyEnumToString(maintenanceOperationsInProgress);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder maintenanceOperationsInProgress(FileSystemMaintenanceOperation... maintenanceOperationsInProgress) {
            maintenanceOperationsInProgress(Arrays.asList(maintenanceOperationsInProgress));
            return this;
        }

        public final String getWeeklyMaintenanceStartTime() {
            return weeklyMaintenanceStartTime;
        }

        public final void setWeeklyMaintenanceStartTime(String weeklyMaintenanceStartTime) {
            this.weeklyMaintenanceStartTime = weeklyMaintenanceStartTime;
        }

        @Override
        public final Builder weeklyMaintenanceStartTime(String weeklyMaintenanceStartTime) {
            this.weeklyMaintenanceStartTime = weeklyMaintenanceStartTime;
            return this;
        }

        public final String getDailyAutomaticBackupStartTime() {
            return dailyAutomaticBackupStartTime;
        }

        public final void setDailyAutomaticBackupStartTime(String dailyAutomaticBackupStartTime) {
            this.dailyAutomaticBackupStartTime = dailyAutomaticBackupStartTime;
        }

        @Override
        public final Builder dailyAutomaticBackupStartTime(String dailyAutomaticBackupStartTime) {
            this.dailyAutomaticBackupStartTime = dailyAutomaticBackupStartTime;
            return this;
        }

        public final Integer getAutomaticBackupRetentionDays() {
            return automaticBackupRetentionDays;
        }

        public final void setAutomaticBackupRetentionDays(Integer automaticBackupRetentionDays) {
            this.automaticBackupRetentionDays = automaticBackupRetentionDays;
        }

        @Override
        public final Builder automaticBackupRetentionDays(Integer automaticBackupRetentionDays) {
            this.automaticBackupRetentionDays = automaticBackupRetentionDays;
            return this;
        }

        public final Boolean getCopyTagsToBackups() {
            return copyTagsToBackups;
        }

        public final void setCopyTagsToBackups(Boolean copyTagsToBackups) {
            this.copyTagsToBackups = copyTagsToBackups;
        }

        @Override
        public final Builder copyTagsToBackups(Boolean copyTagsToBackups) {
            this.copyTagsToBackups = copyTagsToBackups;
            return this;
        }

        public final List<Alias.Builder> getAliases() {
            List<Alias.Builder> result = AliasesCopier.copyToBuilder(this.aliases);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAliases(Collection<Alias.BuilderImpl> aliases) {
            this.aliases = AliasesCopier.copyFromBuilder(aliases);
        }

        @Override
        public final Builder aliases(Collection<Alias> aliases) {
            this.aliases = AliasesCopier.copy(aliases);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder aliases(Alias... aliases) {
            aliases(Arrays.asList(aliases));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder aliases(Consumer<Alias.Builder>... aliases) {
            aliases(Stream.of(aliases).map(c -> Alias.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final WindowsAuditLogConfiguration.Builder getAuditLogConfiguration() {
            return auditLogConfiguration != null ? auditLogConfiguration.toBuilder() : null;
        }

        public final void setAuditLogConfiguration(WindowsAuditLogConfiguration.BuilderImpl auditLogConfiguration) {
            this.auditLogConfiguration = auditLogConfiguration != null ? auditLogConfiguration.build() : null;
        }

        @Override
        public final Builder auditLogConfiguration(WindowsAuditLogConfiguration auditLogConfiguration) {
            this.auditLogConfiguration = auditLogConfiguration;
            return this;
        }

        public final DiskIopsConfiguration.Builder getDiskIopsConfiguration() {
            return diskIopsConfiguration != null ? diskIopsConfiguration.toBuilder() : null;
        }

        public final void setDiskIopsConfiguration(DiskIopsConfiguration.BuilderImpl diskIopsConfiguration) {
            this.diskIopsConfiguration = diskIopsConfiguration != null ? diskIopsConfiguration.build() : null;
        }

        @Override
        public final Builder diskIopsConfiguration(DiskIopsConfiguration diskIopsConfiguration) {
            this.diskIopsConfiguration = diskIopsConfiguration;
            return this;
        }

        @Override
        public WindowsFileSystemConfiguration build() {
            return new WindowsFileSystemConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
