/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.firehose.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the configuration of a destination in Splunk.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SplunkDestinationConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<SplunkDestinationConfiguration.Builder, SplunkDestinationConfiguration> {
    private static final SdkField<String> HEC_ENDPOINT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SplunkDestinationConfiguration::hecEndpoint)).setter(setter(Builder::hecEndpoint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HECEndpoint").build()).build();

    private static final SdkField<String> HEC_ENDPOINT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SplunkDestinationConfiguration::hecEndpointTypeAsString)).setter(setter(Builder::hecEndpointType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HECEndpointType").build()).build();

    private static final SdkField<String> HEC_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SplunkDestinationConfiguration::hecToken)).setter(setter(Builder::hecToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HECToken").build()).build();

    private static final SdkField<Integer> HEC_ACKNOWLEDGMENT_TIMEOUT_IN_SECONDS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(SplunkDestinationConfiguration::hecAcknowledgmentTimeoutInSeconds))
            .setter(setter(Builder::hecAcknowledgmentTimeoutInSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HECAcknowledgmentTimeoutInSeconds")
                    .build()).build();

    private static final SdkField<SplunkRetryOptions> RETRY_OPTIONS_FIELD = SdkField
            .<SplunkRetryOptions> builder(MarshallingType.SDK_POJO).getter(getter(SplunkDestinationConfiguration::retryOptions))
            .setter(setter(Builder::retryOptions)).constructor(SplunkRetryOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RetryOptions").build()).build();

    private static final SdkField<String> S3_BACKUP_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SplunkDestinationConfiguration::s3BackupModeAsString)).setter(setter(Builder::s3BackupMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3BackupMode").build()).build();

    private static final SdkField<S3DestinationConfiguration> S3_CONFIGURATION_FIELD = SdkField
            .<S3DestinationConfiguration> builder(MarshallingType.SDK_POJO)
            .getter(getter(SplunkDestinationConfiguration::s3Configuration)).setter(setter(Builder::s3Configuration))
            .constructor(S3DestinationConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3Configuration").build()).build();

    private static final SdkField<ProcessingConfiguration> PROCESSING_CONFIGURATION_FIELD = SdkField
            .<ProcessingConfiguration> builder(MarshallingType.SDK_POJO)
            .getter(getter(SplunkDestinationConfiguration::processingConfiguration))
            .setter(setter(Builder::processingConfiguration)).constructor(ProcessingConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProcessingConfiguration").build())
            .build();

    private static final SdkField<CloudWatchLoggingOptions> CLOUD_WATCH_LOGGING_OPTIONS_FIELD = SdkField
            .<CloudWatchLoggingOptions> builder(MarshallingType.SDK_POJO)
            .getter(getter(SplunkDestinationConfiguration::cloudWatchLoggingOptions))
            .setter(setter(Builder::cloudWatchLoggingOptions)).constructor(CloudWatchLoggingOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CloudWatchLoggingOptions").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(HEC_ENDPOINT_FIELD,
            HEC_ENDPOINT_TYPE_FIELD, HEC_TOKEN_FIELD, HEC_ACKNOWLEDGMENT_TIMEOUT_IN_SECONDS_FIELD, RETRY_OPTIONS_FIELD,
            S3_BACKUP_MODE_FIELD, S3_CONFIGURATION_FIELD, PROCESSING_CONFIGURATION_FIELD, CLOUD_WATCH_LOGGING_OPTIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String hecEndpoint;

    private final String hecEndpointType;

    private final String hecToken;

    private final Integer hecAcknowledgmentTimeoutInSeconds;

    private final SplunkRetryOptions retryOptions;

    private final String s3BackupMode;

    private final S3DestinationConfiguration s3Configuration;

    private final ProcessingConfiguration processingConfiguration;

    private final CloudWatchLoggingOptions cloudWatchLoggingOptions;

    private SplunkDestinationConfiguration(BuilderImpl builder) {
        this.hecEndpoint = builder.hecEndpoint;
        this.hecEndpointType = builder.hecEndpointType;
        this.hecToken = builder.hecToken;
        this.hecAcknowledgmentTimeoutInSeconds = builder.hecAcknowledgmentTimeoutInSeconds;
        this.retryOptions = builder.retryOptions;
        this.s3BackupMode = builder.s3BackupMode;
        this.s3Configuration = builder.s3Configuration;
        this.processingConfiguration = builder.processingConfiguration;
        this.cloudWatchLoggingOptions = builder.cloudWatchLoggingOptions;
    }

    /**
     * <p>
     * The HTTP Event Collector (HEC) endpoint to which Kinesis Data Firehose sends your data.
     * </p>
     * 
     * @return The HTTP Event Collector (HEC) endpoint to which Kinesis Data Firehose sends your data.
     */
    public String hecEndpoint() {
        return hecEndpoint;
    }

    /**
     * <p>
     * This type can be either "Raw" or "Event."
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #hecEndpointType}
     * will return {@link HECEndpointType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #hecEndpointTypeAsString}.
     * </p>
     * 
     * @return This type can be either "Raw" or "Event."
     * @see HECEndpointType
     */
    public HECEndpointType hecEndpointType() {
        return HECEndpointType.fromValue(hecEndpointType);
    }

    /**
     * <p>
     * This type can be either "Raw" or "Event."
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #hecEndpointType}
     * will return {@link HECEndpointType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #hecEndpointTypeAsString}.
     * </p>
     * 
     * @return This type can be either "Raw" or "Event."
     * @see HECEndpointType
     */
    public String hecEndpointTypeAsString() {
        return hecEndpointType;
    }

    /**
     * <p>
     * This is a GUID that you obtain from your Splunk cluster when you create a new HEC endpoint.
     * </p>
     * 
     * @return This is a GUID that you obtain from your Splunk cluster when you create a new HEC endpoint.
     */
    public String hecToken() {
        return hecToken;
    }

    /**
     * <p>
     * The amount of time that Kinesis Data Firehose waits to receive an acknowledgment from Splunk after it sends it
     * data. At the end of the timeout period, Kinesis Data Firehose either tries to send the data again or considers it
     * an error, based on your retry settings.
     * </p>
     * 
     * @return The amount of time that Kinesis Data Firehose waits to receive an acknowledgment from Splunk after it
     *         sends it data. At the end of the timeout period, Kinesis Data Firehose either tries to send the data
     *         again or considers it an error, based on your retry settings.
     */
    public Integer hecAcknowledgmentTimeoutInSeconds() {
        return hecAcknowledgmentTimeoutInSeconds;
    }

    /**
     * <p>
     * The retry behavior in case Kinesis Data Firehose is unable to deliver data to Splunk, or if it doesn't receive an
     * acknowledgment of receipt from Splunk.
     * </p>
     * 
     * @return The retry behavior in case Kinesis Data Firehose is unable to deliver data to Splunk, or if it doesn't
     *         receive an acknowledgment of receipt from Splunk.
     */
    public SplunkRetryOptions retryOptions() {
        return retryOptions;
    }

    /**
     * <p>
     * Defines how documents should be delivered to Amazon S3. When set to <code>FailedDocumentsOnly</code>, Kinesis
     * Data Firehose writes any data that could not be indexed to the configured Amazon S3 destination. When set to
     * <code>AllDocuments</code>, Kinesis Data Firehose delivers all incoming records to Amazon S3, and also writes
     * failed documents to Amazon S3. Default value is <code>FailedDocumentsOnly</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #s3BackupMode} will
     * return {@link SplunkS3BackupMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #s3BackupModeAsString}.
     * </p>
     * 
     * @return Defines how documents should be delivered to Amazon S3. When set to <code>FailedDocumentsOnly</code>,
     *         Kinesis Data Firehose writes any data that could not be indexed to the configured Amazon S3 destination.
     *         When set to <code>AllDocuments</code>, Kinesis Data Firehose delivers all incoming records to Amazon S3,
     *         and also writes failed documents to Amazon S3. Default value is <code>FailedDocumentsOnly</code>.
     * @see SplunkS3BackupMode
     */
    public SplunkS3BackupMode s3BackupMode() {
        return SplunkS3BackupMode.fromValue(s3BackupMode);
    }

    /**
     * <p>
     * Defines how documents should be delivered to Amazon S3. When set to <code>FailedDocumentsOnly</code>, Kinesis
     * Data Firehose writes any data that could not be indexed to the configured Amazon S3 destination. When set to
     * <code>AllDocuments</code>, Kinesis Data Firehose delivers all incoming records to Amazon S3, and also writes
     * failed documents to Amazon S3. Default value is <code>FailedDocumentsOnly</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #s3BackupMode} will
     * return {@link SplunkS3BackupMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #s3BackupModeAsString}.
     * </p>
     * 
     * @return Defines how documents should be delivered to Amazon S3. When set to <code>FailedDocumentsOnly</code>,
     *         Kinesis Data Firehose writes any data that could not be indexed to the configured Amazon S3 destination.
     *         When set to <code>AllDocuments</code>, Kinesis Data Firehose delivers all incoming records to Amazon S3,
     *         and also writes failed documents to Amazon S3. Default value is <code>FailedDocumentsOnly</code>.
     * @see SplunkS3BackupMode
     */
    public String s3BackupModeAsString() {
        return s3BackupMode;
    }

    /**
     * <p>
     * The configuration for the backup Amazon S3 location.
     * </p>
     * 
     * @return The configuration for the backup Amazon S3 location.
     */
    public S3DestinationConfiguration s3Configuration() {
        return s3Configuration;
    }

    /**
     * <p>
     * The data processing configuration.
     * </p>
     * 
     * @return The data processing configuration.
     */
    public ProcessingConfiguration processingConfiguration() {
        return processingConfiguration;
    }

    /**
     * <p>
     * The Amazon CloudWatch logging options for your delivery stream.
     * </p>
     * 
     * @return The Amazon CloudWatch logging options for your delivery stream.
     */
    public CloudWatchLoggingOptions cloudWatchLoggingOptions() {
        return cloudWatchLoggingOptions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hecEndpoint());
        hashCode = 31 * hashCode + Objects.hashCode(hecEndpointTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hecToken());
        hashCode = 31 * hashCode + Objects.hashCode(hecAcknowledgmentTimeoutInSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(retryOptions());
        hashCode = 31 * hashCode + Objects.hashCode(s3BackupModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(s3Configuration());
        hashCode = 31 * hashCode + Objects.hashCode(processingConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(cloudWatchLoggingOptions());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SplunkDestinationConfiguration)) {
            return false;
        }
        SplunkDestinationConfiguration other = (SplunkDestinationConfiguration) obj;
        return Objects.equals(hecEndpoint(), other.hecEndpoint())
                && Objects.equals(hecEndpointTypeAsString(), other.hecEndpointTypeAsString())
                && Objects.equals(hecToken(), other.hecToken())
                && Objects.equals(hecAcknowledgmentTimeoutInSeconds(), other.hecAcknowledgmentTimeoutInSeconds())
                && Objects.equals(retryOptions(), other.retryOptions())
                && Objects.equals(s3BackupModeAsString(), other.s3BackupModeAsString())
                && Objects.equals(s3Configuration(), other.s3Configuration())
                && Objects.equals(processingConfiguration(), other.processingConfiguration())
                && Objects.equals(cloudWatchLoggingOptions(), other.cloudWatchLoggingOptions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("SplunkDestinationConfiguration").add("HECEndpoint", hecEndpoint())
                .add("HECEndpointType", hecEndpointTypeAsString()).add("HECToken", hecToken())
                .add("HECAcknowledgmentTimeoutInSeconds", hecAcknowledgmentTimeoutInSeconds())
                .add("RetryOptions", retryOptions()).add("S3BackupMode", s3BackupModeAsString())
                .add("S3Configuration", s3Configuration()).add("ProcessingConfiguration", processingConfiguration())
                .add("CloudWatchLoggingOptions", cloudWatchLoggingOptions()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "HECEndpoint":
            return Optional.ofNullable(clazz.cast(hecEndpoint()));
        case "HECEndpointType":
            return Optional.ofNullable(clazz.cast(hecEndpointTypeAsString()));
        case "HECToken":
            return Optional.ofNullable(clazz.cast(hecToken()));
        case "HECAcknowledgmentTimeoutInSeconds":
            return Optional.ofNullable(clazz.cast(hecAcknowledgmentTimeoutInSeconds()));
        case "RetryOptions":
            return Optional.ofNullable(clazz.cast(retryOptions()));
        case "S3BackupMode":
            return Optional.ofNullable(clazz.cast(s3BackupModeAsString()));
        case "S3Configuration":
            return Optional.ofNullable(clazz.cast(s3Configuration()));
        case "ProcessingConfiguration":
            return Optional.ofNullable(clazz.cast(processingConfiguration()));
        case "CloudWatchLoggingOptions":
            return Optional.ofNullable(clazz.cast(cloudWatchLoggingOptions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SplunkDestinationConfiguration, T> g) {
        return obj -> g.apply((SplunkDestinationConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SplunkDestinationConfiguration> {
        /**
         * <p>
         * The HTTP Event Collector (HEC) endpoint to which Kinesis Data Firehose sends your data.
         * </p>
         * 
         * @param hecEndpoint
         *        The HTTP Event Collector (HEC) endpoint to which Kinesis Data Firehose sends your data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hecEndpoint(String hecEndpoint);

        /**
         * <p>
         * This type can be either "Raw" or "Event."
         * </p>
         * 
         * @param hecEndpointType
         *        This type can be either "Raw" or "Event."
         * @see HECEndpointType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HECEndpointType
         */
        Builder hecEndpointType(String hecEndpointType);

        /**
         * <p>
         * This type can be either "Raw" or "Event."
         * </p>
         * 
         * @param hecEndpointType
         *        This type can be either "Raw" or "Event."
         * @see HECEndpointType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HECEndpointType
         */
        Builder hecEndpointType(HECEndpointType hecEndpointType);

        /**
         * <p>
         * This is a GUID that you obtain from your Splunk cluster when you create a new HEC endpoint.
         * </p>
         * 
         * @param hecToken
         *        This is a GUID that you obtain from your Splunk cluster when you create a new HEC endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hecToken(String hecToken);

        /**
         * <p>
         * The amount of time that Kinesis Data Firehose waits to receive an acknowledgment from Splunk after it sends
         * it data. At the end of the timeout period, Kinesis Data Firehose either tries to send the data again or
         * considers it an error, based on your retry settings.
         * </p>
         * 
         * @param hecAcknowledgmentTimeoutInSeconds
         *        The amount of time that Kinesis Data Firehose waits to receive an acknowledgment from Splunk after it
         *        sends it data. At the end of the timeout period, Kinesis Data Firehose either tries to send the data
         *        again or considers it an error, based on your retry settings.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hecAcknowledgmentTimeoutInSeconds(Integer hecAcknowledgmentTimeoutInSeconds);

        /**
         * <p>
         * The retry behavior in case Kinesis Data Firehose is unable to deliver data to Splunk, or if it doesn't
         * receive an acknowledgment of receipt from Splunk.
         * </p>
         * 
         * @param retryOptions
         *        The retry behavior in case Kinesis Data Firehose is unable to deliver data to Splunk, or if it doesn't
         *        receive an acknowledgment of receipt from Splunk.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder retryOptions(SplunkRetryOptions retryOptions);

        /**
         * <p>
         * The retry behavior in case Kinesis Data Firehose is unable to deliver data to Splunk, or if it doesn't
         * receive an acknowledgment of receipt from Splunk.
         * </p>
         * This is a convenience that creates an instance of the {@link SplunkRetryOptions.Builder} avoiding the need to
         * create one manually via {@link SplunkRetryOptions#builder()}.
         *
         * When the {@link Consumer} completes, {@link SplunkRetryOptions.Builder#build()} is called immediately and its
         * result is passed to {@link #retryOptions(SplunkRetryOptions)}.
         * 
         * @param retryOptions
         *        a consumer that will call methods on {@link SplunkRetryOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #retryOptions(SplunkRetryOptions)
         */
        default Builder retryOptions(Consumer<SplunkRetryOptions.Builder> retryOptions) {
            return retryOptions(SplunkRetryOptions.builder().applyMutation(retryOptions).build());
        }

        /**
         * <p>
         * Defines how documents should be delivered to Amazon S3. When set to <code>FailedDocumentsOnly</code>, Kinesis
         * Data Firehose writes any data that could not be indexed to the configured Amazon S3 destination. When set to
         * <code>AllDocuments</code>, Kinesis Data Firehose delivers all incoming records to Amazon S3, and also writes
         * failed documents to Amazon S3. Default value is <code>FailedDocumentsOnly</code>.
         * </p>
         * 
         * @param s3BackupMode
         *        Defines how documents should be delivered to Amazon S3. When set to <code>FailedDocumentsOnly</code>,
         *        Kinesis Data Firehose writes any data that could not be indexed to the configured Amazon S3
         *        destination. When set to <code>AllDocuments</code>, Kinesis Data Firehose delivers all incoming
         *        records to Amazon S3, and also writes failed documents to Amazon S3. Default value is
         *        <code>FailedDocumentsOnly</code>.
         * @see SplunkS3BackupMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SplunkS3BackupMode
         */
        Builder s3BackupMode(String s3BackupMode);

        /**
         * <p>
         * Defines how documents should be delivered to Amazon S3. When set to <code>FailedDocumentsOnly</code>, Kinesis
         * Data Firehose writes any data that could not be indexed to the configured Amazon S3 destination. When set to
         * <code>AllDocuments</code>, Kinesis Data Firehose delivers all incoming records to Amazon S3, and also writes
         * failed documents to Amazon S3. Default value is <code>FailedDocumentsOnly</code>.
         * </p>
         * 
         * @param s3BackupMode
         *        Defines how documents should be delivered to Amazon S3. When set to <code>FailedDocumentsOnly</code>,
         *        Kinesis Data Firehose writes any data that could not be indexed to the configured Amazon S3
         *        destination. When set to <code>AllDocuments</code>, Kinesis Data Firehose delivers all incoming
         *        records to Amazon S3, and also writes failed documents to Amazon S3. Default value is
         *        <code>FailedDocumentsOnly</code>.
         * @see SplunkS3BackupMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SplunkS3BackupMode
         */
        Builder s3BackupMode(SplunkS3BackupMode s3BackupMode);

        /**
         * <p>
         * The configuration for the backup Amazon S3 location.
         * </p>
         * 
         * @param s3Configuration
         *        The configuration for the backup Amazon S3 location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Configuration(S3DestinationConfiguration s3Configuration);

        /**
         * <p>
         * The configuration for the backup Amazon S3 location.
         * </p>
         * This is a convenience that creates an instance of the {@link S3DestinationConfiguration.Builder} avoiding the
         * need to create one manually via {@link S3DestinationConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link S3DestinationConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #s3Configuration(S3DestinationConfiguration)}.
         * 
         * @param s3Configuration
         *        a consumer that will call methods on {@link S3DestinationConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3Configuration(S3DestinationConfiguration)
         */
        default Builder s3Configuration(Consumer<S3DestinationConfiguration.Builder> s3Configuration) {
            return s3Configuration(S3DestinationConfiguration.builder().applyMutation(s3Configuration).build());
        }

        /**
         * <p>
         * The data processing configuration.
         * </p>
         * 
         * @param processingConfiguration
         *        The data processing configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder processingConfiguration(ProcessingConfiguration processingConfiguration);

        /**
         * <p>
         * The data processing configuration.
         * </p>
         * This is a convenience that creates an instance of the {@link ProcessingConfiguration.Builder} avoiding the
         * need to create one manually via {@link ProcessingConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link ProcessingConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #processingConfiguration(ProcessingConfiguration)}.
         * 
         * @param processingConfiguration
         *        a consumer that will call methods on {@link ProcessingConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #processingConfiguration(ProcessingConfiguration)
         */
        default Builder processingConfiguration(Consumer<ProcessingConfiguration.Builder> processingConfiguration) {
            return processingConfiguration(ProcessingConfiguration.builder().applyMutation(processingConfiguration).build());
        }

        /**
         * <p>
         * The Amazon CloudWatch logging options for your delivery stream.
         * </p>
         * 
         * @param cloudWatchLoggingOptions
         *        The Amazon CloudWatch logging options for your delivery stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cloudWatchLoggingOptions(CloudWatchLoggingOptions cloudWatchLoggingOptions);

        /**
         * <p>
         * The Amazon CloudWatch logging options for your delivery stream.
         * </p>
         * This is a convenience that creates an instance of the {@link CloudWatchLoggingOptions.Builder} avoiding the
         * need to create one manually via {@link CloudWatchLoggingOptions#builder()}.
         *
         * When the {@link Consumer} completes, {@link CloudWatchLoggingOptions.Builder#build()} is called immediately
         * and its result is passed to {@link #cloudWatchLoggingOptions(CloudWatchLoggingOptions)}.
         * 
         * @param cloudWatchLoggingOptions
         *        a consumer that will call methods on {@link CloudWatchLoggingOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cloudWatchLoggingOptions(CloudWatchLoggingOptions)
         */
        default Builder cloudWatchLoggingOptions(Consumer<CloudWatchLoggingOptions.Builder> cloudWatchLoggingOptions) {
            return cloudWatchLoggingOptions(CloudWatchLoggingOptions.builder().applyMutation(cloudWatchLoggingOptions).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String hecEndpoint;

        private String hecEndpointType;

        private String hecToken;

        private Integer hecAcknowledgmentTimeoutInSeconds;

        private SplunkRetryOptions retryOptions;

        private String s3BackupMode;

        private S3DestinationConfiguration s3Configuration;

        private ProcessingConfiguration processingConfiguration;

        private CloudWatchLoggingOptions cloudWatchLoggingOptions;

        private BuilderImpl() {
        }

        private BuilderImpl(SplunkDestinationConfiguration model) {
            hecEndpoint(model.hecEndpoint);
            hecEndpointType(model.hecEndpointType);
            hecToken(model.hecToken);
            hecAcknowledgmentTimeoutInSeconds(model.hecAcknowledgmentTimeoutInSeconds);
            retryOptions(model.retryOptions);
            s3BackupMode(model.s3BackupMode);
            s3Configuration(model.s3Configuration);
            processingConfiguration(model.processingConfiguration);
            cloudWatchLoggingOptions(model.cloudWatchLoggingOptions);
        }

        public final String getHecEndpoint() {
            return hecEndpoint;
        }

        @Override
        public final Builder hecEndpoint(String hecEndpoint) {
            this.hecEndpoint = hecEndpoint;
            return this;
        }

        public final void setHecEndpoint(String hecEndpoint) {
            this.hecEndpoint = hecEndpoint;
        }

        public final String getHecEndpointTypeAsString() {
            return hecEndpointType;
        }

        @Override
        public final Builder hecEndpointType(String hecEndpointType) {
            this.hecEndpointType = hecEndpointType;
            return this;
        }

        @Override
        public final Builder hecEndpointType(HECEndpointType hecEndpointType) {
            this.hecEndpointType(hecEndpointType.toString());
            return this;
        }

        public final void setHecEndpointType(String hecEndpointType) {
            this.hecEndpointType = hecEndpointType;
        }

        public final String getHecToken() {
            return hecToken;
        }

        @Override
        public final Builder hecToken(String hecToken) {
            this.hecToken = hecToken;
            return this;
        }

        public final void setHecToken(String hecToken) {
            this.hecToken = hecToken;
        }

        public final Integer getHecAcknowledgmentTimeoutInSeconds() {
            return hecAcknowledgmentTimeoutInSeconds;
        }

        @Override
        public final Builder hecAcknowledgmentTimeoutInSeconds(Integer hecAcknowledgmentTimeoutInSeconds) {
            this.hecAcknowledgmentTimeoutInSeconds = hecAcknowledgmentTimeoutInSeconds;
            return this;
        }

        public final void setHecAcknowledgmentTimeoutInSeconds(Integer hecAcknowledgmentTimeoutInSeconds) {
            this.hecAcknowledgmentTimeoutInSeconds = hecAcknowledgmentTimeoutInSeconds;
        }

        public final SplunkRetryOptions.Builder getRetryOptions() {
            return retryOptions != null ? retryOptions.toBuilder() : null;
        }

        @Override
        public final Builder retryOptions(SplunkRetryOptions retryOptions) {
            this.retryOptions = retryOptions;
            return this;
        }

        public final void setRetryOptions(SplunkRetryOptions.BuilderImpl retryOptions) {
            this.retryOptions = retryOptions != null ? retryOptions.build() : null;
        }

        public final String getS3BackupModeAsString() {
            return s3BackupMode;
        }

        @Override
        public final Builder s3BackupMode(String s3BackupMode) {
            this.s3BackupMode = s3BackupMode;
            return this;
        }

        @Override
        public final Builder s3BackupMode(SplunkS3BackupMode s3BackupMode) {
            this.s3BackupMode(s3BackupMode.toString());
            return this;
        }

        public final void setS3BackupMode(String s3BackupMode) {
            this.s3BackupMode = s3BackupMode;
        }

        public final S3DestinationConfiguration.Builder getS3Configuration() {
            return s3Configuration != null ? s3Configuration.toBuilder() : null;
        }

        @Override
        public final Builder s3Configuration(S3DestinationConfiguration s3Configuration) {
            this.s3Configuration = s3Configuration;
            return this;
        }

        public final void setS3Configuration(S3DestinationConfiguration.BuilderImpl s3Configuration) {
            this.s3Configuration = s3Configuration != null ? s3Configuration.build() : null;
        }

        public final ProcessingConfiguration.Builder getProcessingConfiguration() {
            return processingConfiguration != null ? processingConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder processingConfiguration(ProcessingConfiguration processingConfiguration) {
            this.processingConfiguration = processingConfiguration;
            return this;
        }

        public final void setProcessingConfiguration(ProcessingConfiguration.BuilderImpl processingConfiguration) {
            this.processingConfiguration = processingConfiguration != null ? processingConfiguration.build() : null;
        }

        public final CloudWatchLoggingOptions.Builder getCloudWatchLoggingOptions() {
            return cloudWatchLoggingOptions != null ? cloudWatchLoggingOptions.toBuilder() : null;
        }

        @Override
        public final Builder cloudWatchLoggingOptions(CloudWatchLoggingOptions cloudWatchLoggingOptions) {
            this.cloudWatchLoggingOptions = cloudWatchLoggingOptions;
            return this;
        }

        public final void setCloudWatchLoggingOptions(CloudWatchLoggingOptions.BuilderImpl cloudWatchLoggingOptions) {
            this.cloudWatchLoggingOptions = cloudWatchLoggingOptions != null ? cloudWatchLoggingOptions.build() : null;
        }

        @Override
        public SplunkDestinationConfiguration build() {
            return new SplunkDestinationConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
