/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emr.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Configuration defining a new instance group.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstanceGroupConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<InstanceGroupConfig.Builder, InstanceGroupConfig> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(InstanceGroupConfig::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> MARKET_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Market")
            .getter(getter(InstanceGroupConfig::marketAsString)).setter(setter(Builder::market))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Market").build()).build();

    private static final SdkField<String> INSTANCE_ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceRole").getter(getter(InstanceGroupConfig::instanceRoleAsString))
            .setter(setter(Builder::instanceRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceRole").build()).build();

    private static final SdkField<String> BID_PRICE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BidPrice").getter(getter(InstanceGroupConfig::bidPrice)).setter(setter(Builder::bidPrice))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BidPrice").build()).build();

    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceType").getter(getter(InstanceGroupConfig::instanceType)).setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType").build()).build();

    private static final SdkField<Integer> INSTANCE_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("InstanceCount").getter(getter(InstanceGroupConfig::instanceCount))
            .setter(setter(Builder::instanceCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceCount").build()).build();

    private static final SdkField<List<Configuration>> CONFIGURATIONS_FIELD = SdkField
            .<List<Configuration>> builder(MarshallingType.LIST)
            .memberName("Configurations")
            .getter(getter(InstanceGroupConfig::configurations))
            .setter(setter(Builder::configurations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Configurations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Configuration> builder(MarshallingType.SDK_POJO)
                                            .constructor(Configuration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<EbsConfiguration> EBS_CONFIGURATION_FIELD = SdkField
            .<EbsConfiguration> builder(MarshallingType.SDK_POJO).memberName("EbsConfiguration")
            .getter(getter(InstanceGroupConfig::ebsConfiguration)).setter(setter(Builder::ebsConfiguration))
            .constructor(EbsConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EbsConfiguration").build()).build();

    private static final SdkField<AutoScalingPolicy> AUTO_SCALING_POLICY_FIELD = SdkField
            .<AutoScalingPolicy> builder(MarshallingType.SDK_POJO).memberName("AutoScalingPolicy")
            .getter(getter(InstanceGroupConfig::autoScalingPolicy)).setter(setter(Builder::autoScalingPolicy))
            .constructor(AutoScalingPolicy::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoScalingPolicy").build()).build();

    private static final SdkField<String> CUSTOM_AMI_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CustomAmiId").getter(getter(InstanceGroupConfig::customAmiId)).setter(setter(Builder::customAmiId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomAmiId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, MARKET_FIELD,
            INSTANCE_ROLE_FIELD, BID_PRICE_FIELD, INSTANCE_TYPE_FIELD, INSTANCE_COUNT_FIELD, CONFIGURATIONS_FIELD,
            EBS_CONFIGURATION_FIELD, AUTO_SCALING_POLICY_FIELD, CUSTOM_AMI_ID_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String market;

    private final String instanceRole;

    private final String bidPrice;

    private final String instanceType;

    private final Integer instanceCount;

    private final List<Configuration> configurations;

    private final EbsConfiguration ebsConfiguration;

    private final AutoScalingPolicy autoScalingPolicy;

    private final String customAmiId;

    private InstanceGroupConfig(BuilderImpl builder) {
        this.name = builder.name;
        this.market = builder.market;
        this.instanceRole = builder.instanceRole;
        this.bidPrice = builder.bidPrice;
        this.instanceType = builder.instanceType;
        this.instanceCount = builder.instanceCount;
        this.configurations = builder.configurations;
        this.ebsConfiguration = builder.ebsConfiguration;
        this.autoScalingPolicy = builder.autoScalingPolicy;
        this.customAmiId = builder.customAmiId;
    }

    /**
     * <p>
     * Friendly name given to the instance group.
     * </p>
     * 
     * @return Friendly name given to the instance group.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * Market type of the Amazon EC2 instances used to create a cluster node.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #market} will
     * return {@link MarketType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #marketAsString}.
     * </p>
     * 
     * @return Market type of the Amazon EC2 instances used to create a cluster node.
     * @see MarketType
     */
    public final MarketType market() {
        return MarketType.fromValue(market);
    }

    /**
     * <p>
     * Market type of the Amazon EC2 instances used to create a cluster node.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #market} will
     * return {@link MarketType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #marketAsString}.
     * </p>
     * 
     * @return Market type of the Amazon EC2 instances used to create a cluster node.
     * @see MarketType
     */
    public final String marketAsString() {
        return market;
    }

    /**
     * <p>
     * The role of the instance group in the cluster.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceRole} will
     * return {@link InstanceRoleType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #instanceRoleAsString}.
     * </p>
     * 
     * @return The role of the instance group in the cluster.
     * @see InstanceRoleType
     */
    public final InstanceRoleType instanceRole() {
        return InstanceRoleType.fromValue(instanceRole);
    }

    /**
     * <p>
     * The role of the instance group in the cluster.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceRole} will
     * return {@link InstanceRoleType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #instanceRoleAsString}.
     * </p>
     * 
     * @return The role of the instance group in the cluster.
     * @see InstanceRoleType
     */
    public final String instanceRoleAsString() {
        return instanceRole;
    }

    /**
     * <p>
     * If specified, indicates that the instance group uses Spot Instances. This is the maximum price you are willing to
     * pay for Spot Instances. Specify <code>OnDemandPrice</code> to set the amount equal to the On-Demand price, or
     * specify an amount in USD.
     * </p>
     * 
     * @return If specified, indicates that the instance group uses Spot Instances. This is the maximum price you are
     *         willing to pay for Spot Instances. Specify <code>OnDemandPrice</code> to set the amount equal to the
     *         On-Demand price, or specify an amount in USD.
     */
    public final String bidPrice() {
        return bidPrice;
    }

    /**
     * <p>
     * The Amazon EC2 instance type for all instances in the instance group.
     * </p>
     * 
     * @return The Amazon EC2 instance type for all instances in the instance group.
     */
    public final String instanceType() {
        return instanceType;
    }

    /**
     * <p>
     * Target number of instances for the instance group.
     * </p>
     * 
     * @return Target number of instances for the instance group.
     */
    public final Integer instanceCount() {
        return instanceCount;
    }

    /**
     * For responses, this returns true if the service returned a value for the Configurations property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasConfigurations() {
        return configurations != null && !(configurations instanceof SdkAutoConstructList);
    }

    /**
     * <note>
     * <p>
     * Amazon EMR releases 4.x or higher.
     * </p>
     * </note>
     * <p>
     * The list of configurations supplied for an Amazon EMR cluster instance group. You can specify a separate
     * configuration for each instance group (master, core, and task).
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasConfigurations} method.
     * </p>
     * 
     * @return <p>
     *         Amazon EMR releases 4.x or higher.
     *         </p>
     *         </note>
     *         <p>
     *         The list of configurations supplied for an Amazon EMR cluster instance group. You can specify a separate
     *         configuration for each instance group (master, core, and task).
     */
    public final List<Configuration> configurations() {
        return configurations;
    }

    /**
     * <p>
     * EBS configurations that will be attached to each Amazon EC2 instance in the instance group.
     * </p>
     * 
     * @return EBS configurations that will be attached to each Amazon EC2 instance in the instance group.
     */
    public final EbsConfiguration ebsConfiguration() {
        return ebsConfiguration;
    }

    /**
     * <p>
     * An automatic scaling policy for a core instance group or task instance group in an Amazon EMR cluster. The
     * automatic scaling policy defines how an instance group dynamically adds and terminates Amazon EC2 instances in
     * response to the value of a CloudWatch metric. See <a>PutAutoScalingPolicy</a>.
     * </p>
     * 
     * @return An automatic scaling policy for a core instance group or task instance group in an Amazon EMR cluster.
     *         The automatic scaling policy defines how an instance group dynamically adds and terminates Amazon EC2
     *         instances in response to the value of a CloudWatch metric. See <a>PutAutoScalingPolicy</a>.
     */
    public final AutoScalingPolicy autoScalingPolicy() {
        return autoScalingPolicy;
    }

    /**
     * <p>
     * The custom AMI ID to use for the provisioned instance group.
     * </p>
     * 
     * @return The custom AMI ID to use for the provisioned instance group.
     */
    public final String customAmiId() {
        return customAmiId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(marketAsString());
        hashCode = 31 * hashCode + Objects.hashCode(instanceRoleAsString());
        hashCode = 31 * hashCode + Objects.hashCode(bidPrice());
        hashCode = 31 * hashCode + Objects.hashCode(instanceType());
        hashCode = 31 * hashCode + Objects.hashCode(instanceCount());
        hashCode = 31 * hashCode + Objects.hashCode(hasConfigurations() ? configurations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(ebsConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(autoScalingPolicy());
        hashCode = 31 * hashCode + Objects.hashCode(customAmiId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstanceGroupConfig)) {
            return false;
        }
        InstanceGroupConfig other = (InstanceGroupConfig) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(marketAsString(), other.marketAsString())
                && Objects.equals(instanceRoleAsString(), other.instanceRoleAsString())
                && Objects.equals(bidPrice(), other.bidPrice()) && Objects.equals(instanceType(), other.instanceType())
                && Objects.equals(instanceCount(), other.instanceCount()) && hasConfigurations() == other.hasConfigurations()
                && Objects.equals(configurations(), other.configurations())
                && Objects.equals(ebsConfiguration(), other.ebsConfiguration())
                && Objects.equals(autoScalingPolicy(), other.autoScalingPolicy())
                && Objects.equals(customAmiId(), other.customAmiId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InstanceGroupConfig").add("Name", name()).add("Market", marketAsString())
                .add("InstanceRole", instanceRoleAsString()).add("BidPrice", bidPrice()).add("InstanceType", instanceType())
                .add("InstanceCount", instanceCount()).add("Configurations", hasConfigurations() ? configurations() : null)
                .add("EbsConfiguration", ebsConfiguration()).add("AutoScalingPolicy", autoScalingPolicy())
                .add("CustomAmiId", customAmiId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Market":
            return Optional.ofNullable(clazz.cast(marketAsString()));
        case "InstanceRole":
            return Optional.ofNullable(clazz.cast(instanceRoleAsString()));
        case "BidPrice":
            return Optional.ofNullable(clazz.cast(bidPrice()));
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceType()));
        case "InstanceCount":
            return Optional.ofNullable(clazz.cast(instanceCount()));
        case "Configurations":
            return Optional.ofNullable(clazz.cast(configurations()));
        case "EbsConfiguration":
            return Optional.ofNullable(clazz.cast(ebsConfiguration()));
        case "AutoScalingPolicy":
            return Optional.ofNullable(clazz.cast(autoScalingPolicy()));
        case "CustomAmiId":
            return Optional.ofNullable(clazz.cast(customAmiId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InstanceGroupConfig, T> g) {
        return obj -> g.apply((InstanceGroupConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstanceGroupConfig> {
        /**
         * <p>
         * Friendly name given to the instance group.
         * </p>
         * 
         * @param name
         *        Friendly name given to the instance group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Market type of the Amazon EC2 instances used to create a cluster node.
         * </p>
         * 
         * @param market
         *        Market type of the Amazon EC2 instances used to create a cluster node.
         * @see MarketType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MarketType
         */
        Builder market(String market);

        /**
         * <p>
         * Market type of the Amazon EC2 instances used to create a cluster node.
         * </p>
         * 
         * @param market
         *        Market type of the Amazon EC2 instances used to create a cluster node.
         * @see MarketType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MarketType
         */
        Builder market(MarketType market);

        /**
         * <p>
         * The role of the instance group in the cluster.
         * </p>
         * 
         * @param instanceRole
         *        The role of the instance group in the cluster.
         * @see InstanceRoleType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceRoleType
         */
        Builder instanceRole(String instanceRole);

        /**
         * <p>
         * The role of the instance group in the cluster.
         * </p>
         * 
         * @param instanceRole
         *        The role of the instance group in the cluster.
         * @see InstanceRoleType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceRoleType
         */
        Builder instanceRole(InstanceRoleType instanceRole);

        /**
         * <p>
         * If specified, indicates that the instance group uses Spot Instances. This is the maximum price you are
         * willing to pay for Spot Instances. Specify <code>OnDemandPrice</code> to set the amount equal to the
         * On-Demand price, or specify an amount in USD.
         * </p>
         * 
         * @param bidPrice
         *        If specified, indicates that the instance group uses Spot Instances. This is the maximum price you are
         *        willing to pay for Spot Instances. Specify <code>OnDemandPrice</code> to set the amount equal to the
         *        On-Demand price, or specify an amount in USD.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bidPrice(String bidPrice);

        /**
         * <p>
         * The Amazon EC2 instance type for all instances in the instance group.
         * </p>
         * 
         * @param instanceType
         *        The Amazon EC2 instance type for all instances in the instance group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * Target number of instances for the instance group.
         * </p>
         * 
         * @param instanceCount
         *        Target number of instances for the instance group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceCount(Integer instanceCount);

        /**
         * <note>
         * <p>
         * Amazon EMR releases 4.x or higher.
         * </p>
         * </note>
         * <p>
         * The list of configurations supplied for an Amazon EMR cluster instance group. You can specify a separate
         * configuration for each instance group (master, core, and task).
         * </p>
         * 
         * @param configurations
         *        <p>
         *        Amazon EMR releases 4.x or higher.
         *        </p>
         *        </note>
         *        <p>
         *        The list of configurations supplied for an Amazon EMR cluster instance group. You can specify a
         *        separate configuration for each instance group (master, core, and task).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurations(Collection<Configuration> configurations);

        /**
         * <note>
         * <p>
         * Amazon EMR releases 4.x or higher.
         * </p>
         * </note>
         * <p>
         * The list of configurations supplied for an Amazon EMR cluster instance group. You can specify a separate
         * configuration for each instance group (master, core, and task).
         * </p>
         * 
         * @param configurations
         *        <p>
         *        Amazon EMR releases 4.x or higher.
         *        </p>
         *        </note>
         *        <p>
         *        The list of configurations supplied for an Amazon EMR cluster instance group. You can specify a
         *        separate configuration for each instance group (master, core, and task).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurations(Configuration... configurations);

        /**
         * <note>
         * <p>
         * Amazon EMR releases 4.x or higher.
         * </p>
         * </note>
         * <p>
         * The list of configurations supplied for an Amazon EMR cluster instance group. You can specify a separate
         * configuration for each instance group (master, core, and task).
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.emr.model.Configuration.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.emr.model.Configuration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.emr.model.Configuration.Builder#build()} is called immediately and its
         * result is passed to {@link #configurations(List<Configuration>)}.
         * 
         * @param configurations
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.emr.model.Configuration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #configurations(java.util.Collection<Configuration>)
         */
        Builder configurations(Consumer<Configuration.Builder>... configurations);

        /**
         * <p>
         * EBS configurations that will be attached to each Amazon EC2 instance in the instance group.
         * </p>
         * 
         * @param ebsConfiguration
         *        EBS configurations that will be attached to each Amazon EC2 instance in the instance group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ebsConfiguration(EbsConfiguration ebsConfiguration);

        /**
         * <p>
         * EBS configurations that will be attached to each Amazon EC2 instance in the instance group.
         * </p>
         * This is a convenience method that creates an instance of the {@link EbsConfiguration.Builder} avoiding the
         * need to create one manually via {@link EbsConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EbsConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #ebsConfiguration(EbsConfiguration)}.
         * 
         * @param ebsConfiguration
         *        a consumer that will call methods on {@link EbsConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ebsConfiguration(EbsConfiguration)
         */
        default Builder ebsConfiguration(Consumer<EbsConfiguration.Builder> ebsConfiguration) {
            return ebsConfiguration(EbsConfiguration.builder().applyMutation(ebsConfiguration).build());
        }

        /**
         * <p>
         * An automatic scaling policy for a core instance group or task instance group in an Amazon EMR cluster. The
         * automatic scaling policy defines how an instance group dynamically adds and terminates Amazon EC2 instances
         * in response to the value of a CloudWatch metric. See <a>PutAutoScalingPolicy</a>.
         * </p>
         * 
         * @param autoScalingPolicy
         *        An automatic scaling policy for a core instance group or task instance group in an Amazon EMR cluster.
         *        The automatic scaling policy defines how an instance group dynamically adds and terminates Amazon EC2
         *        instances in response to the value of a CloudWatch metric. See <a>PutAutoScalingPolicy</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoScalingPolicy(AutoScalingPolicy autoScalingPolicy);

        /**
         * <p>
         * An automatic scaling policy for a core instance group or task instance group in an Amazon EMR cluster. The
         * automatic scaling policy defines how an instance group dynamically adds and terminates Amazon EC2 instances
         * in response to the value of a CloudWatch metric. See <a>PutAutoScalingPolicy</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link AutoScalingPolicy.Builder} avoiding the
         * need to create one manually via {@link AutoScalingPolicy#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AutoScalingPolicy.Builder#build()} is called immediately and its
         * result is passed to {@link #autoScalingPolicy(AutoScalingPolicy)}.
         * 
         * @param autoScalingPolicy
         *        a consumer that will call methods on {@link AutoScalingPolicy.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #autoScalingPolicy(AutoScalingPolicy)
         */
        default Builder autoScalingPolicy(Consumer<AutoScalingPolicy.Builder> autoScalingPolicy) {
            return autoScalingPolicy(AutoScalingPolicy.builder().applyMutation(autoScalingPolicy).build());
        }

        /**
         * <p>
         * The custom AMI ID to use for the provisioned instance group.
         * </p>
         * 
         * @param customAmiId
         *        The custom AMI ID to use for the provisioned instance group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customAmiId(String customAmiId);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String market;

        private String instanceRole;

        private String bidPrice;

        private String instanceType;

        private Integer instanceCount;

        private List<Configuration> configurations = DefaultSdkAutoConstructList.getInstance();

        private EbsConfiguration ebsConfiguration;

        private AutoScalingPolicy autoScalingPolicy;

        private String customAmiId;

        private BuilderImpl() {
        }

        private BuilderImpl(InstanceGroupConfig model) {
            name(model.name);
            market(model.market);
            instanceRole(model.instanceRole);
            bidPrice(model.bidPrice);
            instanceType(model.instanceType);
            instanceCount(model.instanceCount);
            configurations(model.configurations);
            ebsConfiguration(model.ebsConfiguration);
            autoScalingPolicy(model.autoScalingPolicy);
            customAmiId(model.customAmiId);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getMarket() {
            return market;
        }

        public final void setMarket(String market) {
            this.market = market;
        }

        @Override
        public final Builder market(String market) {
            this.market = market;
            return this;
        }

        @Override
        public final Builder market(MarketType market) {
            this.market(market == null ? null : market.toString());
            return this;
        }

        public final String getInstanceRole() {
            return instanceRole;
        }

        public final void setInstanceRole(String instanceRole) {
            this.instanceRole = instanceRole;
        }

        @Override
        public final Builder instanceRole(String instanceRole) {
            this.instanceRole = instanceRole;
            return this;
        }

        @Override
        public final Builder instanceRole(InstanceRoleType instanceRole) {
            this.instanceRole(instanceRole == null ? null : instanceRole.toString());
            return this;
        }

        public final String getBidPrice() {
            return bidPrice;
        }

        public final void setBidPrice(String bidPrice) {
            this.bidPrice = bidPrice;
        }

        @Override
        public final Builder bidPrice(String bidPrice) {
            this.bidPrice = bidPrice;
            return this;
        }

        public final String getInstanceType() {
            return instanceType;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        public final Integer getInstanceCount() {
            return instanceCount;
        }

        public final void setInstanceCount(Integer instanceCount) {
            this.instanceCount = instanceCount;
        }

        @Override
        public final Builder instanceCount(Integer instanceCount) {
            this.instanceCount = instanceCount;
            return this;
        }

        public final List<Configuration.Builder> getConfigurations() {
            List<Configuration.Builder> result = ConfigurationListCopier.copyToBuilder(this.configurations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setConfigurations(Collection<Configuration.BuilderImpl> configurations) {
            this.configurations = ConfigurationListCopier.copyFromBuilder(configurations);
        }

        @Override
        public final Builder configurations(Collection<Configuration> configurations) {
            this.configurations = ConfigurationListCopier.copy(configurations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder configurations(Configuration... configurations) {
            configurations(Arrays.asList(configurations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder configurations(Consumer<Configuration.Builder>... configurations) {
            configurations(Stream.of(configurations).map(c -> Configuration.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final EbsConfiguration.Builder getEbsConfiguration() {
            return ebsConfiguration != null ? ebsConfiguration.toBuilder() : null;
        }

        public final void setEbsConfiguration(EbsConfiguration.BuilderImpl ebsConfiguration) {
            this.ebsConfiguration = ebsConfiguration != null ? ebsConfiguration.build() : null;
        }

        @Override
        public final Builder ebsConfiguration(EbsConfiguration ebsConfiguration) {
            this.ebsConfiguration = ebsConfiguration;
            return this;
        }

        public final AutoScalingPolicy.Builder getAutoScalingPolicy() {
            return autoScalingPolicy != null ? autoScalingPolicy.toBuilder() : null;
        }

        public final void setAutoScalingPolicy(AutoScalingPolicy.BuilderImpl autoScalingPolicy) {
            this.autoScalingPolicy = autoScalingPolicy != null ? autoScalingPolicy.build() : null;
        }

        @Override
        public final Builder autoScalingPolicy(AutoScalingPolicy autoScalingPolicy) {
            this.autoScalingPolicy = autoScalingPolicy;
            return this;
        }

        public final String getCustomAmiId() {
            return customAmiId;
        }

        public final void setCustomAmiId(String customAmiId) {
            this.customAmiId = customAmiId;
        }

        @Override
        public final Builder customAmiId(String customAmiId) {
            this.customAmiId = customAmiId;
            return this;
        }

        @Override
        public InstanceGroupConfig build() {
            return new InstanceGroupConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
