/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emr.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration that defines an instance fleet.
 * </p>
 * <note>
 * <p>
 * The instance fleet configuration is available only in Amazon EMR releases 4.8.0 and higher, excluding 5.0.x versions.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstanceFleetConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<InstanceFleetConfig.Builder, InstanceFleetConfig> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(InstanceFleetConfig::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> INSTANCE_FLEET_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceFleetType").getter(getter(InstanceFleetConfig::instanceFleetTypeAsString))
            .setter(setter(Builder::instanceFleetType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceFleetType").build()).build();

    private static final SdkField<Integer> TARGET_ON_DEMAND_CAPACITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("TargetOnDemandCapacity").getter(getter(InstanceFleetConfig::targetOnDemandCapacity))
            .setter(setter(Builder::targetOnDemandCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetOnDemandCapacity").build())
            .build();

    private static final SdkField<Integer> TARGET_SPOT_CAPACITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("TargetSpotCapacity").getter(getter(InstanceFleetConfig::targetSpotCapacity))
            .setter(setter(Builder::targetSpotCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetSpotCapacity").build())
            .build();

    private static final SdkField<List<InstanceTypeConfig>> INSTANCE_TYPE_CONFIGS_FIELD = SdkField
            .<List<InstanceTypeConfig>> builder(MarshallingType.LIST)
            .memberName("InstanceTypeConfigs")
            .getter(getter(InstanceFleetConfig::instanceTypeConfigs))
            .setter(setter(Builder::instanceTypeConfigs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceTypeConfigs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<InstanceTypeConfig> builder(MarshallingType.SDK_POJO)
                                            .constructor(InstanceTypeConfig::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<InstanceFleetProvisioningSpecifications> LAUNCH_SPECIFICATIONS_FIELD = SdkField
            .<InstanceFleetProvisioningSpecifications> builder(MarshallingType.SDK_POJO).memberName("LaunchSpecifications")
            .getter(getter(InstanceFleetConfig::launchSpecifications)).setter(setter(Builder::launchSpecifications))
            .constructor(InstanceFleetProvisioningSpecifications::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LaunchSpecifications").build())
            .build();

    private static final SdkField<InstanceFleetResizingSpecifications> RESIZE_SPECIFICATIONS_FIELD = SdkField
            .<InstanceFleetResizingSpecifications> builder(MarshallingType.SDK_POJO).memberName("ResizeSpecifications")
            .getter(getter(InstanceFleetConfig::resizeSpecifications)).setter(setter(Builder::resizeSpecifications))
            .constructor(InstanceFleetResizingSpecifications::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResizeSpecifications").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            INSTANCE_FLEET_TYPE_FIELD, TARGET_ON_DEMAND_CAPACITY_FIELD, TARGET_SPOT_CAPACITY_FIELD, INSTANCE_TYPE_CONFIGS_FIELD,
            LAUNCH_SPECIFICATIONS_FIELD, RESIZE_SPECIFICATIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String instanceFleetType;

    private final Integer targetOnDemandCapacity;

    private final Integer targetSpotCapacity;

    private final List<InstanceTypeConfig> instanceTypeConfigs;

    private final InstanceFleetProvisioningSpecifications launchSpecifications;

    private final InstanceFleetResizingSpecifications resizeSpecifications;

    private InstanceFleetConfig(BuilderImpl builder) {
        this.name = builder.name;
        this.instanceFleetType = builder.instanceFleetType;
        this.targetOnDemandCapacity = builder.targetOnDemandCapacity;
        this.targetSpotCapacity = builder.targetSpotCapacity;
        this.instanceTypeConfigs = builder.instanceTypeConfigs;
        this.launchSpecifications = builder.launchSpecifications;
        this.resizeSpecifications = builder.resizeSpecifications;
    }

    /**
     * <p>
     * The friendly name of the instance fleet.
     * </p>
     * 
     * @return The friendly name of the instance fleet.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The node type that the instance fleet hosts. Valid values are MASTER, CORE, and TASK.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceFleetType}
     * will return {@link InstanceFleetType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #instanceFleetTypeAsString}.
     * </p>
     * 
     * @return The node type that the instance fleet hosts. Valid values are MASTER, CORE, and TASK.
     * @see InstanceFleetType
     */
    public final InstanceFleetType instanceFleetType() {
        return InstanceFleetType.fromValue(instanceFleetType);
    }

    /**
     * <p>
     * The node type that the instance fleet hosts. Valid values are MASTER, CORE, and TASK.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceFleetType}
     * will return {@link InstanceFleetType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #instanceFleetTypeAsString}.
     * </p>
     * 
     * @return The node type that the instance fleet hosts. Valid values are MASTER, CORE, and TASK.
     * @see InstanceFleetType
     */
    public final String instanceFleetTypeAsString() {
        return instanceFleetType;
    }

    /**
     * <p>
     * The target capacity of On-Demand units for the instance fleet, which determines how many On-Demand Instances to
     * provision. When the instance fleet launches, Amazon EMR tries to provision On-Demand Instances as specified by
     * <a>InstanceTypeConfig</a>. Each instance configuration has a specified <code>WeightedCapacity</code>. When an
     * On-Demand Instance is provisioned, the <code>WeightedCapacity</code> units count toward the target capacity.
     * Amazon EMR provisions instances until the target capacity is totally fulfilled, even if this results in an
     * overage. For example, if there are 2 units remaining to fulfill capacity, and Amazon EMR can only provision an
     * instance with a <code>WeightedCapacity</code> of 5 units, the instance is provisioned, and the target capacity is
     * exceeded by 3 units.
     * </p>
     * <note>
     * <p>
     * If not specified or set to 0, only Spot Instances are provisioned for the instance fleet using
     * <code>TargetSpotCapacity</code>. At least one of <code>TargetSpotCapacity</code> and
     * <code>TargetOnDemandCapacity</code> should be greater than 0. For a master instance fleet, only one of
     * <code>TargetSpotCapacity</code> and <code>TargetOnDemandCapacity</code> can be specified, and its value must be
     * 1.
     * </p>
     * </note>
     * 
     * @return The target capacity of On-Demand units for the instance fleet, which determines how many On-Demand
     *         Instances to provision. When the instance fleet launches, Amazon EMR tries to provision On-Demand
     *         Instances as specified by <a>InstanceTypeConfig</a>. Each instance configuration has a specified
     *         <code>WeightedCapacity</code>. When an On-Demand Instance is provisioned, the
     *         <code>WeightedCapacity</code> units count toward the target capacity. Amazon EMR provisions instances
     *         until the target capacity is totally fulfilled, even if this results in an overage. For example, if there
     *         are 2 units remaining to fulfill capacity, and Amazon EMR can only provision an instance with a
     *         <code>WeightedCapacity</code> of 5 units, the instance is provisioned, and the target capacity is
     *         exceeded by 3 units.</p> <note>
     *         <p>
     *         If not specified or set to 0, only Spot Instances are provisioned for the instance fleet using
     *         <code>TargetSpotCapacity</code>. At least one of <code>TargetSpotCapacity</code> and
     *         <code>TargetOnDemandCapacity</code> should be greater than 0. For a master instance fleet, only one of
     *         <code>TargetSpotCapacity</code> and <code>TargetOnDemandCapacity</code> can be specified, and its value
     *         must be 1.
     *         </p>
     */
    public final Integer targetOnDemandCapacity() {
        return targetOnDemandCapacity;
    }

    /**
     * <p>
     * The target capacity of Spot units for the instance fleet, which determines how many Spot Instances to provision.
     * When the instance fleet launches, Amazon EMR tries to provision Spot Instances as specified by
     * <a>InstanceTypeConfig</a>. Each instance configuration has a specified <code>WeightedCapacity</code>. When a Spot
     * Instance is provisioned, the <code>WeightedCapacity</code> units count toward the target capacity. Amazon EMR
     * provisions instances until the target capacity is totally fulfilled, even if this results in an overage. For
     * example, if there are 2 units remaining to fulfill capacity, and Amazon EMR can only provision an instance with a
     * <code>WeightedCapacity</code> of 5 units, the instance is provisioned, and the target capacity is exceeded by 3
     * units.
     * </p>
     * <note>
     * <p>
     * If not specified or set to 0, only On-Demand Instances are provisioned for the instance fleet. At least one of
     * <code>TargetSpotCapacity</code> and <code>TargetOnDemandCapacity</code> should be greater than 0. For a master
     * instance fleet, only one of <code>TargetSpotCapacity</code> and <code>TargetOnDemandCapacity</code> can be
     * specified, and its value must be 1.
     * </p>
     * </note>
     * 
     * @return The target capacity of Spot units for the instance fleet, which determines how many Spot Instances to
     *         provision. When the instance fleet launches, Amazon EMR tries to provision Spot Instances as specified by
     *         <a>InstanceTypeConfig</a>. Each instance configuration has a specified <code>WeightedCapacity</code>.
     *         When a Spot Instance is provisioned, the <code>WeightedCapacity</code> units count toward the target
     *         capacity. Amazon EMR provisions instances until the target capacity is totally fulfilled, even if this
     *         results in an overage. For example, if there are 2 units remaining to fulfill capacity, and Amazon EMR
     *         can only provision an instance with a <code>WeightedCapacity</code> of 5 units, the instance is
     *         provisioned, and the target capacity is exceeded by 3 units.</p> <note>
     *         <p>
     *         If not specified or set to 0, only On-Demand Instances are provisioned for the instance fleet. At least
     *         one of <code>TargetSpotCapacity</code> and <code>TargetOnDemandCapacity</code> should be greater than 0.
     *         For a master instance fleet, only one of <code>TargetSpotCapacity</code> and
     *         <code>TargetOnDemandCapacity</code> can be specified, and its value must be 1.
     *         </p>
     */
    public final Integer targetSpotCapacity() {
        return targetSpotCapacity;
    }

    /**
     * For responses, this returns true if the service returned a value for the InstanceTypeConfigs property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasInstanceTypeConfigs() {
        return instanceTypeConfigs != null && !(instanceTypeConfigs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The instance type configurations that define the Amazon EC2 instances in the instance fleet.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInstanceTypeConfigs} method.
     * </p>
     * 
     * @return The instance type configurations that define the Amazon EC2 instances in the instance fleet.
     */
    public final List<InstanceTypeConfig> instanceTypeConfigs() {
        return instanceTypeConfigs;
    }

    /**
     * <p>
     * The launch specification for the instance fleet.
     * </p>
     * 
     * @return The launch specification for the instance fleet.
     */
    public final InstanceFleetProvisioningSpecifications launchSpecifications() {
        return launchSpecifications;
    }

    /**
     * <p>
     * The resize specification for the instance fleet.
     * </p>
     * 
     * @return The resize specification for the instance fleet.
     */
    public final InstanceFleetResizingSpecifications resizeSpecifications() {
        return resizeSpecifications;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(instanceFleetTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(targetOnDemandCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(targetSpotCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(hasInstanceTypeConfigs() ? instanceTypeConfigs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(launchSpecifications());
        hashCode = 31 * hashCode + Objects.hashCode(resizeSpecifications());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstanceFleetConfig)) {
            return false;
        }
        InstanceFleetConfig other = (InstanceFleetConfig) obj;
        return Objects.equals(name(), other.name())
                && Objects.equals(instanceFleetTypeAsString(), other.instanceFleetTypeAsString())
                && Objects.equals(targetOnDemandCapacity(), other.targetOnDemandCapacity())
                && Objects.equals(targetSpotCapacity(), other.targetSpotCapacity())
                && hasInstanceTypeConfigs() == other.hasInstanceTypeConfigs()
                && Objects.equals(instanceTypeConfigs(), other.instanceTypeConfigs())
                && Objects.equals(launchSpecifications(), other.launchSpecifications())
                && Objects.equals(resizeSpecifications(), other.resizeSpecifications());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InstanceFleetConfig").add("Name", name()).add("InstanceFleetType", instanceFleetTypeAsString())
                .add("TargetOnDemandCapacity", targetOnDemandCapacity()).add("TargetSpotCapacity", targetSpotCapacity())
                .add("InstanceTypeConfigs", hasInstanceTypeConfigs() ? instanceTypeConfigs() : null)
                .add("LaunchSpecifications", launchSpecifications()).add("ResizeSpecifications", resizeSpecifications()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "InstanceFleetType":
            return Optional.ofNullable(clazz.cast(instanceFleetTypeAsString()));
        case "TargetOnDemandCapacity":
            return Optional.ofNullable(clazz.cast(targetOnDemandCapacity()));
        case "TargetSpotCapacity":
            return Optional.ofNullable(clazz.cast(targetSpotCapacity()));
        case "InstanceTypeConfigs":
            return Optional.ofNullable(clazz.cast(instanceTypeConfigs()));
        case "LaunchSpecifications":
            return Optional.ofNullable(clazz.cast(launchSpecifications()));
        case "ResizeSpecifications":
            return Optional.ofNullable(clazz.cast(resizeSpecifications()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InstanceFleetConfig, T> g) {
        return obj -> g.apply((InstanceFleetConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstanceFleetConfig> {
        /**
         * <p>
         * The friendly name of the instance fleet.
         * </p>
         * 
         * @param name
         *        The friendly name of the instance fleet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The node type that the instance fleet hosts. Valid values are MASTER, CORE, and TASK.
         * </p>
         * 
         * @param instanceFleetType
         *        The node type that the instance fleet hosts. Valid values are MASTER, CORE, and TASK.
         * @see InstanceFleetType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceFleetType
         */
        Builder instanceFleetType(String instanceFleetType);

        /**
         * <p>
         * The node type that the instance fleet hosts. Valid values are MASTER, CORE, and TASK.
         * </p>
         * 
         * @param instanceFleetType
         *        The node type that the instance fleet hosts. Valid values are MASTER, CORE, and TASK.
         * @see InstanceFleetType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceFleetType
         */
        Builder instanceFleetType(InstanceFleetType instanceFleetType);

        /**
         * <p>
         * The target capacity of On-Demand units for the instance fleet, which determines how many On-Demand Instances
         * to provision. When the instance fleet launches, Amazon EMR tries to provision On-Demand Instances as
         * specified by <a>InstanceTypeConfig</a>. Each instance configuration has a specified
         * <code>WeightedCapacity</code>. When an On-Demand Instance is provisioned, the <code>WeightedCapacity</code>
         * units count toward the target capacity. Amazon EMR provisions instances until the target capacity is totally
         * fulfilled, even if this results in an overage. For example, if there are 2 units remaining to fulfill
         * capacity, and Amazon EMR can only provision an instance with a <code>WeightedCapacity</code> of 5 units, the
         * instance is provisioned, and the target capacity is exceeded by 3 units.
         * </p>
         * <note>
         * <p>
         * If not specified or set to 0, only Spot Instances are provisioned for the instance fleet using
         * <code>TargetSpotCapacity</code>. At least one of <code>TargetSpotCapacity</code> and
         * <code>TargetOnDemandCapacity</code> should be greater than 0. For a master instance fleet, only one of
         * <code>TargetSpotCapacity</code> and <code>TargetOnDemandCapacity</code> can be specified, and its value must
         * be 1.
         * </p>
         * </note>
         * 
         * @param targetOnDemandCapacity
         *        The target capacity of On-Demand units for the instance fleet, which determines how many On-Demand
         *        Instances to provision. When the instance fleet launches, Amazon EMR tries to provision On-Demand
         *        Instances as specified by <a>InstanceTypeConfig</a>. Each instance configuration has a specified
         *        <code>WeightedCapacity</code>. When an On-Demand Instance is provisioned, the
         *        <code>WeightedCapacity</code> units count toward the target capacity. Amazon EMR provisions instances
         *        until the target capacity is totally fulfilled, even if this results in an overage. For example, if
         *        there are 2 units remaining to fulfill capacity, and Amazon EMR can only provision an instance with a
         *        <code>WeightedCapacity</code> of 5 units, the instance is provisioned, and the target capacity is
         *        exceeded by 3 units.</p> <note>
         *        <p>
         *        If not specified or set to 0, only Spot Instances are provisioned for the instance fleet using
         *        <code>TargetSpotCapacity</code>. At least one of <code>TargetSpotCapacity</code> and
         *        <code>TargetOnDemandCapacity</code> should be greater than 0. For a master instance fleet, only one of
         *        <code>TargetSpotCapacity</code> and <code>TargetOnDemandCapacity</code> can be specified, and its
         *        value must be 1.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetOnDemandCapacity(Integer targetOnDemandCapacity);

        /**
         * <p>
         * The target capacity of Spot units for the instance fleet, which determines how many Spot Instances to
         * provision. When the instance fleet launches, Amazon EMR tries to provision Spot Instances as specified by
         * <a>InstanceTypeConfig</a>. Each instance configuration has a specified <code>WeightedCapacity</code>. When a
         * Spot Instance is provisioned, the <code>WeightedCapacity</code> units count toward the target capacity.
         * Amazon EMR provisions instances until the target capacity is totally fulfilled, even if this results in an
         * overage. For example, if there are 2 units remaining to fulfill capacity, and Amazon EMR can only provision
         * an instance with a <code>WeightedCapacity</code> of 5 units, the instance is provisioned, and the target
         * capacity is exceeded by 3 units.
         * </p>
         * <note>
         * <p>
         * If not specified or set to 0, only On-Demand Instances are provisioned for the instance fleet. At least one
         * of <code>TargetSpotCapacity</code> and <code>TargetOnDemandCapacity</code> should be greater than 0. For a
         * master instance fleet, only one of <code>TargetSpotCapacity</code> and <code>TargetOnDemandCapacity</code>
         * can be specified, and its value must be 1.
         * </p>
         * </note>
         * 
         * @param targetSpotCapacity
         *        The target capacity of Spot units for the instance fleet, which determines how many Spot Instances to
         *        provision. When the instance fleet launches, Amazon EMR tries to provision Spot Instances as specified
         *        by <a>InstanceTypeConfig</a>. Each instance configuration has a specified
         *        <code>WeightedCapacity</code>. When a Spot Instance is provisioned, the <code>WeightedCapacity</code>
         *        units count toward the target capacity. Amazon EMR provisions instances until the target capacity is
         *        totally fulfilled, even if this results in an overage. For example, if there are 2 units remaining to
         *        fulfill capacity, and Amazon EMR can only provision an instance with a <code>WeightedCapacity</code>
         *        of 5 units, the instance is provisioned, and the target capacity is exceeded by 3 units.</p> <note>
         *        <p>
         *        If not specified or set to 0, only On-Demand Instances are provisioned for the instance fleet. At
         *        least one of <code>TargetSpotCapacity</code> and <code>TargetOnDemandCapacity</code> should be greater
         *        than 0. For a master instance fleet, only one of <code>TargetSpotCapacity</code> and
         *        <code>TargetOnDemandCapacity</code> can be specified, and its value must be 1.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetSpotCapacity(Integer targetSpotCapacity);

        /**
         * <p>
         * The instance type configurations that define the Amazon EC2 instances in the instance fleet.
         * </p>
         * 
         * @param instanceTypeConfigs
         *        The instance type configurations that define the Amazon EC2 instances in the instance fleet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceTypeConfigs(Collection<InstanceTypeConfig> instanceTypeConfigs);

        /**
         * <p>
         * The instance type configurations that define the Amazon EC2 instances in the instance fleet.
         * </p>
         * 
         * @param instanceTypeConfigs
         *        The instance type configurations that define the Amazon EC2 instances in the instance fleet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceTypeConfigs(InstanceTypeConfig... instanceTypeConfigs);

        /**
         * <p>
         * The instance type configurations that define the Amazon EC2 instances in the instance fleet.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.emr.model.InstanceTypeConfig.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.emr.model.InstanceTypeConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.emr.model.InstanceTypeConfig.Builder#build()} is called immediately
         * and its result is passed to {@link #instanceTypeConfigs(List<InstanceTypeConfig>)}.
         * 
         * @param instanceTypeConfigs
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.emr.model.InstanceTypeConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #instanceTypeConfigs(java.util.Collection<InstanceTypeConfig>)
         */
        Builder instanceTypeConfigs(Consumer<InstanceTypeConfig.Builder>... instanceTypeConfigs);

        /**
         * <p>
         * The launch specification for the instance fleet.
         * </p>
         * 
         * @param launchSpecifications
         *        The launch specification for the instance fleet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder launchSpecifications(InstanceFleetProvisioningSpecifications launchSpecifications);

        /**
         * <p>
         * The launch specification for the instance fleet.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link InstanceFleetProvisioningSpecifications.Builder} avoiding the need to create one manually via
         * {@link InstanceFleetProvisioningSpecifications#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link InstanceFleetProvisioningSpecifications.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #launchSpecifications(InstanceFleetProvisioningSpecifications)}.
         * 
         * @param launchSpecifications
         *        a consumer that will call methods on {@link InstanceFleetProvisioningSpecifications.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #launchSpecifications(InstanceFleetProvisioningSpecifications)
         */
        default Builder launchSpecifications(Consumer<InstanceFleetProvisioningSpecifications.Builder> launchSpecifications) {
            return launchSpecifications(InstanceFleetProvisioningSpecifications.builder().applyMutation(launchSpecifications)
                    .build());
        }

        /**
         * <p>
         * The resize specification for the instance fleet.
         * </p>
         * 
         * @param resizeSpecifications
         *        The resize specification for the instance fleet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resizeSpecifications(InstanceFleetResizingSpecifications resizeSpecifications);

        /**
         * <p>
         * The resize specification for the instance fleet.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link InstanceFleetResizingSpecifications.Builder} avoiding the need to create one manually via
         * {@link InstanceFleetResizingSpecifications#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link InstanceFleetResizingSpecifications.Builder#build()} is called
         * immediately and its result is passed to {@link #resizeSpecifications(InstanceFleetResizingSpecifications)}.
         * 
         * @param resizeSpecifications
         *        a consumer that will call methods on {@link InstanceFleetResizingSpecifications.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resizeSpecifications(InstanceFleetResizingSpecifications)
         */
        default Builder resizeSpecifications(Consumer<InstanceFleetResizingSpecifications.Builder> resizeSpecifications) {
            return resizeSpecifications(InstanceFleetResizingSpecifications.builder().applyMutation(resizeSpecifications).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String instanceFleetType;

        private Integer targetOnDemandCapacity;

        private Integer targetSpotCapacity;

        private List<InstanceTypeConfig> instanceTypeConfigs = DefaultSdkAutoConstructList.getInstance();

        private InstanceFleetProvisioningSpecifications launchSpecifications;

        private InstanceFleetResizingSpecifications resizeSpecifications;

        private BuilderImpl() {
        }

        private BuilderImpl(InstanceFleetConfig model) {
            name(model.name);
            instanceFleetType(model.instanceFleetType);
            targetOnDemandCapacity(model.targetOnDemandCapacity);
            targetSpotCapacity(model.targetSpotCapacity);
            instanceTypeConfigs(model.instanceTypeConfigs);
            launchSpecifications(model.launchSpecifications);
            resizeSpecifications(model.resizeSpecifications);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getInstanceFleetType() {
            return instanceFleetType;
        }

        public final void setInstanceFleetType(String instanceFleetType) {
            this.instanceFleetType = instanceFleetType;
        }

        @Override
        public final Builder instanceFleetType(String instanceFleetType) {
            this.instanceFleetType = instanceFleetType;
            return this;
        }

        @Override
        public final Builder instanceFleetType(InstanceFleetType instanceFleetType) {
            this.instanceFleetType(instanceFleetType == null ? null : instanceFleetType.toString());
            return this;
        }

        public final Integer getTargetOnDemandCapacity() {
            return targetOnDemandCapacity;
        }

        public final void setTargetOnDemandCapacity(Integer targetOnDemandCapacity) {
            this.targetOnDemandCapacity = targetOnDemandCapacity;
        }

        @Override
        public final Builder targetOnDemandCapacity(Integer targetOnDemandCapacity) {
            this.targetOnDemandCapacity = targetOnDemandCapacity;
            return this;
        }

        public final Integer getTargetSpotCapacity() {
            return targetSpotCapacity;
        }

        public final void setTargetSpotCapacity(Integer targetSpotCapacity) {
            this.targetSpotCapacity = targetSpotCapacity;
        }

        @Override
        public final Builder targetSpotCapacity(Integer targetSpotCapacity) {
            this.targetSpotCapacity = targetSpotCapacity;
            return this;
        }

        public final List<InstanceTypeConfig.Builder> getInstanceTypeConfigs() {
            List<InstanceTypeConfig.Builder> result = InstanceTypeConfigListCopier.copyToBuilder(this.instanceTypeConfigs);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setInstanceTypeConfigs(Collection<InstanceTypeConfig.BuilderImpl> instanceTypeConfigs) {
            this.instanceTypeConfigs = InstanceTypeConfigListCopier.copyFromBuilder(instanceTypeConfigs);
        }

        @Override
        public final Builder instanceTypeConfigs(Collection<InstanceTypeConfig> instanceTypeConfigs) {
            this.instanceTypeConfigs = InstanceTypeConfigListCopier.copy(instanceTypeConfigs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceTypeConfigs(InstanceTypeConfig... instanceTypeConfigs) {
            instanceTypeConfigs(Arrays.asList(instanceTypeConfigs));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceTypeConfigs(Consumer<InstanceTypeConfig.Builder>... instanceTypeConfigs) {
            instanceTypeConfigs(Stream.of(instanceTypeConfigs).map(c -> InstanceTypeConfig.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final InstanceFleetProvisioningSpecifications.Builder getLaunchSpecifications() {
            return launchSpecifications != null ? launchSpecifications.toBuilder() : null;
        }

        public final void setLaunchSpecifications(InstanceFleetProvisioningSpecifications.BuilderImpl launchSpecifications) {
            this.launchSpecifications = launchSpecifications != null ? launchSpecifications.build() : null;
        }

        @Override
        public final Builder launchSpecifications(InstanceFleetProvisioningSpecifications launchSpecifications) {
            this.launchSpecifications = launchSpecifications;
            return this;
        }

        public final InstanceFleetResizingSpecifications.Builder getResizeSpecifications() {
            return resizeSpecifications != null ? resizeSpecifications.toBuilder() : null;
        }

        public final void setResizeSpecifications(InstanceFleetResizingSpecifications.BuilderImpl resizeSpecifications) {
            this.resizeSpecifications = resizeSpecifications != null ? resizeSpecifications.build() : null;
        }

        @Override
        public final Builder resizeSpecifications(InstanceFleetResizingSpecifications resizeSpecifications) {
            this.resizeSpecifications = resizeSpecifications;
            return this;
        }

        @Override
        public InstanceFleetConfig build() {
            return new InstanceFleetConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
