/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emr.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * With Amazon EMR release version 4.0 and higher, the only accepted parameter is the application name. To pass
 * arguments to applications, you use configuration classifications specified using configuration JSON objects. For more
 * information, see <a href="https://docs.aws.amazon.com/emr/latest/ReleaseGuide/emr-configure-apps.html">Configuring
 * Applications</a>.
 * </p>
 * <p>
 * With earlier Amazon EMR releases, the application is any Amazon or third-party software that you can add to the
 * cluster. This structure contains a list of strings that indicates the software to use with the cluster and accepts a
 * user argument list. Amazon EMR accepts and forwards the argument list to the corresponding installation script as
 * bootstrap action argument.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Application implements SdkPojo, Serializable, ToCopyableBuilder<Application.Builder, Application> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Application::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Version")
            .getter(getter(Application::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Version").build()).build();

    private static final SdkField<List<String>> ARGS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Args")
            .getter(getter(Application::args))
            .setter(setter(Builder::args))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Args").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> ADDITIONAL_INFO_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("AdditionalInfo")
            .getter(getter(Application::additionalInfo))
            .setter(setter(Builder::additionalInfo))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdditionalInfo").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, VERSION_FIELD,
            ARGS_FIELD, ADDITIONAL_INFO_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String version;

    private final List<String> args;

    private final Map<String, String> additionalInfo;

    private Application(BuilderImpl builder) {
        this.name = builder.name;
        this.version = builder.version;
        this.args = builder.args;
        this.additionalInfo = builder.additionalInfo;
    }

    /**
     * <p>
     * The name of the application.
     * </p>
     * 
     * @return The name of the application.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The version of the application.
     * </p>
     * 
     * @return The version of the application.
     */
    public final String version() {
        return version;
    }

    /**
     * For responses, this returns true if the service returned a value for the Args property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasArgs() {
        return args != null && !(args instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Arguments for Amazon EMR to pass to the application.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasArgs} method.
     * </p>
     * 
     * @return Arguments for Amazon EMR to pass to the application.
     */
    public final List<String> args() {
        return args;
    }

    /**
     * For responses, this returns true if the service returned a value for the AdditionalInfo property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAdditionalInfo() {
        return additionalInfo != null && !(additionalInfo instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * This option is for advanced users only. This is meta information about third-party applications that third-party
     * vendors use for testing purposes.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAdditionalInfo} method.
     * </p>
     * 
     * @return This option is for advanced users only. This is meta information about third-party applications that
     *         third-party vendors use for testing purposes.
     */
    public final Map<String, String> additionalInfo() {
        return additionalInfo;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(hasArgs() ? args() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasAdditionalInfo() ? additionalInfo() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Application)) {
            return false;
        }
        Application other = (Application) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(version(), other.version()) && hasArgs() == other.hasArgs()
                && Objects.equals(args(), other.args()) && hasAdditionalInfo() == other.hasAdditionalInfo()
                && Objects.equals(additionalInfo(), other.additionalInfo());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Application").add("Name", name()).add("Version", version())
                .add("Args", hasArgs() ? args() : null).add("AdditionalInfo", hasAdditionalInfo() ? additionalInfo() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Version":
            return Optional.ofNullable(clazz.cast(version()));
        case "Args":
            return Optional.ofNullable(clazz.cast(args()));
        case "AdditionalInfo":
            return Optional.ofNullable(clazz.cast(additionalInfo()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Application, T> g) {
        return obj -> g.apply((Application) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Application> {
        /**
         * <p>
         * The name of the application.
         * </p>
         * 
         * @param name
         *        The name of the application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The version of the application.
         * </p>
         * 
         * @param version
         *        The version of the application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * Arguments for Amazon EMR to pass to the application.
         * </p>
         * 
         * @param args
         *        Arguments for Amazon EMR to pass to the application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder args(Collection<String> args);

        /**
         * <p>
         * Arguments for Amazon EMR to pass to the application.
         * </p>
         * 
         * @param args
         *        Arguments for Amazon EMR to pass to the application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder args(String... args);

        /**
         * <p>
         * This option is for advanced users only. This is meta information about third-party applications that
         * third-party vendors use for testing purposes.
         * </p>
         * 
         * @param additionalInfo
         *        This option is for advanced users only. This is meta information about third-party applications that
         *        third-party vendors use for testing purposes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder additionalInfo(Map<String, String> additionalInfo);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String version;

        private List<String> args = DefaultSdkAutoConstructList.getInstance();

        private Map<String, String> additionalInfo = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Application model) {
            name(model.name);
            version(model.version);
            args(model.args);
            additionalInfo(model.additionalInfo);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getVersion() {
            return version;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final Collection<String> getArgs() {
            if (args instanceof SdkAutoConstructList) {
                return null;
            }
            return args;
        }

        public final void setArgs(Collection<String> args) {
            this.args = StringListCopier.copy(args);
        }

        @Override
        public final Builder args(Collection<String> args) {
            this.args = StringListCopier.copy(args);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder args(String... args) {
            args(Arrays.asList(args));
            return this;
        }

        public final Map<String, String> getAdditionalInfo() {
            if (additionalInfo instanceof SdkAutoConstructMap) {
                return null;
            }
            return additionalInfo;
        }

        public final void setAdditionalInfo(Map<String, String> additionalInfo) {
            this.additionalInfo = StringMapCopier.copy(additionalInfo);
        }

        @Override
        public final Builder additionalInfo(Map<String, String> additionalInfo) {
            this.additionalInfo = StringMapCopier.copy(additionalInfo);
            return this;
        }

        @Override
        public Application build() {
            return new Application(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
