/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The EC2 Instance Connect Endpoint.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Ec2InstanceConnectEndpoint implements SdkPojo, Serializable,
        ToCopyableBuilder<Ec2InstanceConnectEndpoint.Builder, Ec2InstanceConnectEndpoint> {
    private static final SdkField<String> OWNER_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("OwnerId")
            .getter(getter(Ec2InstanceConnectEndpoint::ownerId))
            .setter(setter(Builder::ownerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OwnerId")
                    .unmarshallLocationName("ownerId").build()).build();

    private static final SdkField<String> INSTANCE_CONNECT_ENDPOINT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InstanceConnectEndpointId")
            .getter(getter(Ec2InstanceConnectEndpoint::instanceConnectEndpointId))
            .setter(setter(Builder::instanceConnectEndpointId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceConnectEndpointId")
                    .unmarshallLocationName("instanceConnectEndpointId").build()).build();

    private static final SdkField<String> INSTANCE_CONNECT_ENDPOINT_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InstanceConnectEndpointArn")
            .getter(getter(Ec2InstanceConnectEndpoint::instanceConnectEndpointArn))
            .setter(setter(Builder::instanceConnectEndpointArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceConnectEndpointArn")
                    .unmarshallLocationName("instanceConnectEndpointArn").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("State")
            .getter(getter(Ec2InstanceConnectEndpoint::stateAsString))
            .setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State")
                    .unmarshallLocationName("state").build()).build();

    private static final SdkField<String> STATE_MESSAGE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("StateMessage")
            .getter(getter(Ec2InstanceConnectEndpoint::stateMessage))
            .setter(setter(Builder::stateMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StateMessage")
                    .unmarshallLocationName("stateMessage").build()).build();

    private static final SdkField<String> DNS_NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DnsName")
            .getter(getter(Ec2InstanceConnectEndpoint::dnsName))
            .setter(setter(Builder::dnsName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsName")
                    .unmarshallLocationName("dnsName").build()).build();

    private static final SdkField<String> FIPS_DNS_NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("FipsDnsName")
            .getter(getter(Ec2InstanceConnectEndpoint::fipsDnsName))
            .setter(setter(Builder::fipsDnsName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FipsDnsName")
                    .unmarshallLocationName("fipsDnsName").build()).build();

    private static final SdkField<List<String>> NETWORK_INTERFACE_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("NetworkInterfaceIds")
            .getter(getter(Ec2InstanceConnectEndpoint::networkInterfaceIds))
            .setter(setter(Builder::networkInterfaceIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkInterfaceIdSet")
                    .unmarshallLocationName("networkInterfaceIdSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("VpcId")
            .getter(getter(Ec2InstanceConnectEndpoint::vpcId))
            .setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcId")
                    .unmarshallLocationName("vpcId").build()).build();

    private static final SdkField<String> AVAILABILITY_ZONE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AvailabilityZone")
            .getter(getter(Ec2InstanceConnectEndpoint::availabilityZone))
            .setter(setter(Builder::availabilityZone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZone")
                    .unmarshallLocationName("availabilityZone").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedAt")
            .getter(getter(Ec2InstanceConnectEndpoint::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedAt")
                    .unmarshallLocationName("createdAt").build()).build();

    private static final SdkField<String> SUBNET_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SubnetId")
            .getter(getter(Ec2InstanceConnectEndpoint::subnetId))
            .setter(setter(Builder::subnetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetId")
                    .unmarshallLocationName("subnetId").build()).build();

    private static final SdkField<Boolean> PRESERVE_CLIENT_IP_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("PreserveClientIp")
            .getter(getter(Ec2InstanceConnectEndpoint::preserveClientIp))
            .setter(setter(Builder::preserveClientIp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreserveClientIp")
                    .unmarshallLocationName("preserveClientIp").build()).build();

    private static final SdkField<List<String>> SECURITY_GROUP_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SecurityGroupIds")
            .getter(getter(Ec2InstanceConnectEndpoint::securityGroupIds))
            .setter(setter(Builder::securityGroupIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroupIdSet")
                    .unmarshallLocationName("securityGroupIdSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(Ec2InstanceConnectEndpoint::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(OWNER_ID_FIELD,
            INSTANCE_CONNECT_ENDPOINT_ID_FIELD, INSTANCE_CONNECT_ENDPOINT_ARN_FIELD, STATE_FIELD, STATE_MESSAGE_FIELD,
            DNS_NAME_FIELD, FIPS_DNS_NAME_FIELD, NETWORK_INTERFACE_IDS_FIELD, VPC_ID_FIELD, AVAILABILITY_ZONE_FIELD,
            CREATED_AT_FIELD, SUBNET_ID_FIELD, PRESERVE_CLIENT_IP_FIELD, SECURITY_GROUP_IDS_FIELD, TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String ownerId;

    private final String instanceConnectEndpointId;

    private final String instanceConnectEndpointArn;

    private final String state;

    private final String stateMessage;

    private final String dnsName;

    private final String fipsDnsName;

    private final List<String> networkInterfaceIds;

    private final String vpcId;

    private final String availabilityZone;

    private final Instant createdAt;

    private final String subnetId;

    private final Boolean preserveClientIp;

    private final List<String> securityGroupIds;

    private final List<Tag> tags;

    private Ec2InstanceConnectEndpoint(BuilderImpl builder) {
        this.ownerId = builder.ownerId;
        this.instanceConnectEndpointId = builder.instanceConnectEndpointId;
        this.instanceConnectEndpointArn = builder.instanceConnectEndpointArn;
        this.state = builder.state;
        this.stateMessage = builder.stateMessage;
        this.dnsName = builder.dnsName;
        this.fipsDnsName = builder.fipsDnsName;
        this.networkInterfaceIds = builder.networkInterfaceIds;
        this.vpcId = builder.vpcId;
        this.availabilityZone = builder.availabilityZone;
        this.createdAt = builder.createdAt;
        this.subnetId = builder.subnetId;
        this.preserveClientIp = builder.preserveClientIp;
        this.securityGroupIds = builder.securityGroupIds;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The ID of the Amazon Web Services account that created the EC2 Instance Connect Endpoint.
     * </p>
     * 
     * @return The ID of the Amazon Web Services account that created the EC2 Instance Connect Endpoint.
     */
    public final String ownerId() {
        return ownerId;
    }

    /**
     * <p>
     * The ID of the EC2 Instance Connect Endpoint.
     * </p>
     * 
     * @return The ID of the EC2 Instance Connect Endpoint.
     */
    public final String instanceConnectEndpointId() {
        return instanceConnectEndpointId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the EC2 Instance Connect Endpoint.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the EC2 Instance Connect Endpoint.
     */
    public final String instanceConnectEndpointArn() {
        return instanceConnectEndpointArn;
    }

    /**
     * <p>
     * The current state of the EC2 Instance Connect Endpoint.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link Ec2InstanceConnectEndpointState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the EC2 Instance Connect Endpoint.
     * @see Ec2InstanceConnectEndpointState
     */
    public final Ec2InstanceConnectEndpointState state() {
        return Ec2InstanceConnectEndpointState.fromValue(state);
    }

    /**
     * <p>
     * The current state of the EC2 Instance Connect Endpoint.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link Ec2InstanceConnectEndpointState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the EC2 Instance Connect Endpoint.
     * @see Ec2InstanceConnectEndpointState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The message for the current state of the EC2 Instance Connect Endpoint. Can include a failure message.
     * </p>
     * 
     * @return The message for the current state of the EC2 Instance Connect Endpoint. Can include a failure message.
     */
    public final String stateMessage() {
        return stateMessage;
    }

    /**
     * <p>
     * The DNS name of the EC2 Instance Connect Endpoint.
     * </p>
     * 
     * @return The DNS name of the EC2 Instance Connect Endpoint.
     */
    public final String dnsName() {
        return dnsName;
    }

    /**
     * <p/>
     * 
     * @return
     */
    public final String fipsDnsName() {
        return fipsDnsName;
    }

    /**
     * For responses, this returns true if the service returned a value for the NetworkInterfaceIds property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasNetworkInterfaceIds() {
        return networkInterfaceIds != null && !(networkInterfaceIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The ID of the elastic network interface that Amazon EC2 automatically created when creating the EC2 Instance
     * Connect Endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNetworkInterfaceIds} method.
     * </p>
     * 
     * @return The ID of the elastic network interface that Amazon EC2 automatically created when creating the EC2
     *         Instance Connect Endpoint.
     */
    public final List<String> networkInterfaceIds() {
        return networkInterfaceIds;
    }

    /**
     * <p>
     * The ID of the VPC in which the EC2 Instance Connect Endpoint was created.
     * </p>
     * 
     * @return The ID of the VPC in which the EC2 Instance Connect Endpoint was created.
     */
    public final String vpcId() {
        return vpcId;
    }

    /**
     * <p>
     * The Availability Zone of the EC2 Instance Connect Endpoint.
     * </p>
     * 
     * @return The Availability Zone of the EC2 Instance Connect Endpoint.
     */
    public final String availabilityZone() {
        return availabilityZone;
    }

    /**
     * <p>
     * The date and time that the EC2 Instance Connect Endpoint was created.
     * </p>
     * 
     * @return The date and time that the EC2 Instance Connect Endpoint was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The ID of the subnet in which the EC2 Instance Connect Endpoint was created.
     * </p>
     * 
     * @return The ID of the subnet in which the EC2 Instance Connect Endpoint was created.
     */
    public final String subnetId() {
        return subnetId;
    }

    /**
     * <p>
     * Indicates whether your client's IP address is preserved as the source. The value is <code>true</code> or
     * <code>false</code>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If <code>true</code>, your client's IP address is used when you connect to a resource.
     * </p>
     * </li>
     * <li>
     * <p>
     * If <code>false</code>, the elastic network interface IP address is used when you connect to a resource.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default: <code>true</code>
     * </p>
     * 
     * @return Indicates whether your client's IP address is preserved as the source. The value is <code>true</code> or
     *         <code>false</code>.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If <code>true</code>, your client's IP address is used when you connect to a resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If <code>false</code>, the elastic network interface IP address is used when you connect to a resource.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default: <code>true</code>
     */
    public final Boolean preserveClientIp() {
        return preserveClientIp;
    }

    /**
     * For responses, this returns true if the service returned a value for the SecurityGroupIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSecurityGroupIds() {
        return securityGroupIds != null && !(securityGroupIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The security groups associated with the endpoint. If you didn't specify a security group, the default security
     * group for your VPC is associated with the endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecurityGroupIds} method.
     * </p>
     * 
     * @return The security groups associated with the endpoint. If you didn't specify a security group, the default
     *         security group for your VPC is associated with the endpoint.
     */
    public final List<String> securityGroupIds() {
        return securityGroupIds;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags assigned to the EC2 Instance Connect Endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags assigned to the EC2 Instance Connect Endpoint.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ownerId());
        hashCode = 31 * hashCode + Objects.hashCode(instanceConnectEndpointId());
        hashCode = 31 * hashCode + Objects.hashCode(instanceConnectEndpointArn());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stateMessage());
        hashCode = 31 * hashCode + Objects.hashCode(dnsName());
        hashCode = 31 * hashCode + Objects.hashCode(fipsDnsName());
        hashCode = 31 * hashCode + Objects.hashCode(hasNetworkInterfaceIds() ? networkInterfaceIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZone());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(subnetId());
        hashCode = 31 * hashCode + Objects.hashCode(preserveClientIp());
        hashCode = 31 * hashCode + Objects.hashCode(hasSecurityGroupIds() ? securityGroupIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Ec2InstanceConnectEndpoint)) {
            return false;
        }
        Ec2InstanceConnectEndpoint other = (Ec2InstanceConnectEndpoint) obj;
        return Objects.equals(ownerId(), other.ownerId())
                && Objects.equals(instanceConnectEndpointId(), other.instanceConnectEndpointId())
                && Objects.equals(instanceConnectEndpointArn(), other.instanceConnectEndpointArn())
                && Objects.equals(stateAsString(), other.stateAsString()) && Objects.equals(stateMessage(), other.stateMessage())
                && Objects.equals(dnsName(), other.dnsName()) && Objects.equals(fipsDnsName(), other.fipsDnsName())
                && hasNetworkInterfaceIds() == other.hasNetworkInterfaceIds()
                && Objects.equals(networkInterfaceIds(), other.networkInterfaceIds()) && Objects.equals(vpcId(), other.vpcId())
                && Objects.equals(availabilityZone(), other.availabilityZone()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(subnetId(), other.subnetId()) && Objects.equals(preserveClientIp(), other.preserveClientIp())
                && hasSecurityGroupIds() == other.hasSecurityGroupIds()
                && Objects.equals(securityGroupIds(), other.securityGroupIds()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Ec2InstanceConnectEndpoint").add("OwnerId", ownerId())
                .add("InstanceConnectEndpointId", instanceConnectEndpointId())
                .add("InstanceConnectEndpointArn", instanceConnectEndpointArn()).add("State", stateAsString())
                .add("StateMessage", stateMessage()).add("DnsName", dnsName()).add("FipsDnsName", fipsDnsName())
                .add("NetworkInterfaceIds", hasNetworkInterfaceIds() ? networkInterfaceIds() : null).add("VpcId", vpcId())
                .add("AvailabilityZone", availabilityZone()).add("CreatedAt", createdAt()).add("SubnetId", subnetId())
                .add("PreserveClientIp", preserveClientIp())
                .add("SecurityGroupIds", hasSecurityGroupIds() ? securityGroupIds() : null)
                .add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "OwnerId":
            return Optional.ofNullable(clazz.cast(ownerId()));
        case "InstanceConnectEndpointId":
            return Optional.ofNullable(clazz.cast(instanceConnectEndpointId()));
        case "InstanceConnectEndpointArn":
            return Optional.ofNullable(clazz.cast(instanceConnectEndpointArn()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "StateMessage":
            return Optional.ofNullable(clazz.cast(stateMessage()));
        case "DnsName":
            return Optional.ofNullable(clazz.cast(dnsName()));
        case "FipsDnsName":
            return Optional.ofNullable(clazz.cast(fipsDnsName()));
        case "NetworkInterfaceIds":
            return Optional.ofNullable(clazz.cast(networkInterfaceIds()));
        case "VpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "AvailabilityZone":
            return Optional.ofNullable(clazz.cast(availabilityZone()));
        case "CreatedAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "SubnetId":
            return Optional.ofNullable(clazz.cast(subnetId()));
        case "PreserveClientIp":
            return Optional.ofNullable(clazz.cast(preserveClientIp()));
        case "SecurityGroupIds":
            return Optional.ofNullable(clazz.cast(securityGroupIds()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Ec2InstanceConnectEndpoint, T> g) {
        return obj -> g.apply((Ec2InstanceConnectEndpoint) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Ec2InstanceConnectEndpoint> {
        /**
         * <p>
         * The ID of the Amazon Web Services account that created the EC2 Instance Connect Endpoint.
         * </p>
         * 
         * @param ownerId
         *        The ID of the Amazon Web Services account that created the EC2 Instance Connect Endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerId(String ownerId);

        /**
         * <p>
         * The ID of the EC2 Instance Connect Endpoint.
         * </p>
         * 
         * @param instanceConnectEndpointId
         *        The ID of the EC2 Instance Connect Endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceConnectEndpointId(String instanceConnectEndpointId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the EC2 Instance Connect Endpoint.
         * </p>
         * 
         * @param instanceConnectEndpointArn
         *        The Amazon Resource Name (ARN) of the EC2 Instance Connect Endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceConnectEndpointArn(String instanceConnectEndpointArn);

        /**
         * <p>
         * The current state of the EC2 Instance Connect Endpoint.
         * </p>
         * 
         * @param state
         *        The current state of the EC2 Instance Connect Endpoint.
         * @see Ec2InstanceConnectEndpointState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Ec2InstanceConnectEndpointState
         */
        Builder state(String state);

        /**
         * <p>
         * The current state of the EC2 Instance Connect Endpoint.
         * </p>
         * 
         * @param state
         *        The current state of the EC2 Instance Connect Endpoint.
         * @see Ec2InstanceConnectEndpointState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Ec2InstanceConnectEndpointState
         */
        Builder state(Ec2InstanceConnectEndpointState state);

        /**
         * <p>
         * The message for the current state of the EC2 Instance Connect Endpoint. Can include a failure message.
         * </p>
         * 
         * @param stateMessage
         *        The message for the current state of the EC2 Instance Connect Endpoint. Can include a failure message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateMessage(String stateMessage);

        /**
         * <p>
         * The DNS name of the EC2 Instance Connect Endpoint.
         * </p>
         * 
         * @param dnsName
         *        The DNS name of the EC2 Instance Connect Endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsName(String dnsName);

        /**
         * <p/>
         * 
         * @param fipsDnsName
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fipsDnsName(String fipsDnsName);

        /**
         * <p>
         * The ID of the elastic network interface that Amazon EC2 automatically created when creating the EC2 Instance
         * Connect Endpoint.
         * </p>
         * 
         * @param networkInterfaceIds
         *        The ID of the elastic network interface that Amazon EC2 automatically created when creating the EC2
         *        Instance Connect Endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaceIds(Collection<String> networkInterfaceIds);

        /**
         * <p>
         * The ID of the elastic network interface that Amazon EC2 automatically created when creating the EC2 Instance
         * Connect Endpoint.
         * </p>
         * 
         * @param networkInterfaceIds
         *        The ID of the elastic network interface that Amazon EC2 automatically created when creating the EC2
         *        Instance Connect Endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaceIds(String... networkInterfaceIds);

        /**
         * <p>
         * The ID of the VPC in which the EC2 Instance Connect Endpoint was created.
         * </p>
         * 
         * @param vpcId
         *        The ID of the VPC in which the EC2 Instance Connect Endpoint was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * The Availability Zone of the EC2 Instance Connect Endpoint.
         * </p>
         * 
         * @param availabilityZone
         *        The Availability Zone of the EC2 Instance Connect Endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZone(String availabilityZone);

        /**
         * <p>
         * The date and time that the EC2 Instance Connect Endpoint was created.
         * </p>
         * 
         * @param createdAt
         *        The date and time that the EC2 Instance Connect Endpoint was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The ID of the subnet in which the EC2 Instance Connect Endpoint was created.
         * </p>
         * 
         * @param subnetId
         *        The ID of the subnet in which the EC2 Instance Connect Endpoint was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetId(String subnetId);

        /**
         * <p>
         * Indicates whether your client's IP address is preserved as the source. The value is <code>true</code> or
         * <code>false</code>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If <code>true</code>, your client's IP address is used when you connect to a resource.
         * </p>
         * </li>
         * <li>
         * <p>
         * If <code>false</code>, the elastic network interface IP address is used when you connect to a resource.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: <code>true</code>
         * </p>
         * 
         * @param preserveClientIp
         *        Indicates whether your client's IP address is preserved as the source. The value is <code>true</code>
         *        or <code>false</code>.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If <code>true</code>, your client's IP address is used when you connect to a resource.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If <code>false</code>, the elastic network interface IP address is used when you connect to a
         *        resource.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: <code>true</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preserveClientIp(Boolean preserveClientIp);

        /**
         * <p>
         * The security groups associated with the endpoint. If you didn't specify a security group, the default
         * security group for your VPC is associated with the endpoint.
         * </p>
         * 
         * @param securityGroupIds
         *        The security groups associated with the endpoint. If you didn't specify a security group, the default
         *        security group for your VPC is associated with the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(Collection<String> securityGroupIds);

        /**
         * <p>
         * The security groups associated with the endpoint. If you didn't specify a security group, the default
         * security group for your VPC is associated with the endpoint.
         * </p>
         * 
         * @param securityGroupIds
         *        The security groups associated with the endpoint. If you didn't specify a security group, the default
         *        security group for your VPC is associated with the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(String... securityGroupIds);

        /**
         * <p>
         * The tags assigned to the EC2 Instance Connect Endpoint.
         * </p>
         * 
         * @param tags
         *        The tags assigned to the EC2 Instance Connect Endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tags assigned to the EC2 Instance Connect Endpoint.
         * </p>
         * 
         * @param tags
         *        The tags assigned to the EC2 Instance Connect Endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tags assigned to the EC2 Instance Connect Endpoint.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ec2.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ec2.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String ownerId;

        private String instanceConnectEndpointId;

        private String instanceConnectEndpointArn;

        private String state;

        private String stateMessage;

        private String dnsName;

        private String fipsDnsName;

        private List<String> networkInterfaceIds = DefaultSdkAutoConstructList.getInstance();

        private String vpcId;

        private String availabilityZone;

        private Instant createdAt;

        private String subnetId;

        private Boolean preserveClientIp;

        private List<String> securityGroupIds = DefaultSdkAutoConstructList.getInstance();

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Ec2InstanceConnectEndpoint model) {
            ownerId(model.ownerId);
            instanceConnectEndpointId(model.instanceConnectEndpointId);
            instanceConnectEndpointArn(model.instanceConnectEndpointArn);
            state(model.state);
            stateMessage(model.stateMessage);
            dnsName(model.dnsName);
            fipsDnsName(model.fipsDnsName);
            networkInterfaceIds(model.networkInterfaceIds);
            vpcId(model.vpcId);
            availabilityZone(model.availabilityZone);
            createdAt(model.createdAt);
            subnetId(model.subnetId);
            preserveClientIp(model.preserveClientIp);
            securityGroupIds(model.securityGroupIds);
            tags(model.tags);
        }

        public final String getOwnerId() {
            return ownerId;
        }

        public final void setOwnerId(String ownerId) {
            this.ownerId = ownerId;
        }

        @Override
        public final Builder ownerId(String ownerId) {
            this.ownerId = ownerId;
            return this;
        }

        public final String getInstanceConnectEndpointId() {
            return instanceConnectEndpointId;
        }

        public final void setInstanceConnectEndpointId(String instanceConnectEndpointId) {
            this.instanceConnectEndpointId = instanceConnectEndpointId;
        }

        @Override
        public final Builder instanceConnectEndpointId(String instanceConnectEndpointId) {
            this.instanceConnectEndpointId = instanceConnectEndpointId;
            return this;
        }

        public final String getInstanceConnectEndpointArn() {
            return instanceConnectEndpointArn;
        }

        public final void setInstanceConnectEndpointArn(String instanceConnectEndpointArn) {
            this.instanceConnectEndpointArn = instanceConnectEndpointArn;
        }

        @Override
        public final Builder instanceConnectEndpointArn(String instanceConnectEndpointArn) {
            this.instanceConnectEndpointArn = instanceConnectEndpointArn;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(Ec2InstanceConnectEndpointState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getStateMessage() {
            return stateMessage;
        }

        public final void setStateMessage(String stateMessage) {
            this.stateMessage = stateMessage;
        }

        @Override
        public final Builder stateMessage(String stateMessage) {
            this.stateMessage = stateMessage;
            return this;
        }

        public final String getDnsName() {
            return dnsName;
        }

        public final void setDnsName(String dnsName) {
            this.dnsName = dnsName;
        }

        @Override
        public final Builder dnsName(String dnsName) {
            this.dnsName = dnsName;
            return this;
        }

        public final String getFipsDnsName() {
            return fipsDnsName;
        }

        public final void setFipsDnsName(String fipsDnsName) {
            this.fipsDnsName = fipsDnsName;
        }

        @Override
        public final Builder fipsDnsName(String fipsDnsName) {
            this.fipsDnsName = fipsDnsName;
            return this;
        }

        public final Collection<String> getNetworkInterfaceIds() {
            if (networkInterfaceIds instanceof SdkAutoConstructList) {
                return null;
            }
            return networkInterfaceIds;
        }

        public final void setNetworkInterfaceIds(Collection<String> networkInterfaceIds) {
            this.networkInterfaceIds = NetworkInterfaceIdSetCopier.copy(networkInterfaceIds);
        }

        @Override
        public final Builder networkInterfaceIds(Collection<String> networkInterfaceIds) {
            this.networkInterfaceIds = NetworkInterfaceIdSetCopier.copy(networkInterfaceIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder networkInterfaceIds(String... networkInterfaceIds) {
            networkInterfaceIds(Arrays.asList(networkInterfaceIds));
            return this;
        }

        public final String getVpcId() {
            return vpcId;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final String getAvailabilityZone() {
            return availabilityZone;
        }

        public final void setAvailabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
        }

        @Override
        public final Builder availabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final String getSubnetId() {
            return subnetId;
        }

        public final void setSubnetId(String subnetId) {
            this.subnetId = subnetId;
        }

        @Override
        public final Builder subnetId(String subnetId) {
            this.subnetId = subnetId;
            return this;
        }

        public final Boolean getPreserveClientIp() {
            return preserveClientIp;
        }

        public final void setPreserveClientIp(Boolean preserveClientIp) {
            this.preserveClientIp = preserveClientIp;
        }

        @Override
        public final Builder preserveClientIp(Boolean preserveClientIp) {
            this.preserveClientIp = preserveClientIp;
            return this;
        }

        public final Collection<String> getSecurityGroupIds() {
            if (securityGroupIds instanceof SdkAutoConstructList) {
                return null;
            }
            return securityGroupIds;
        }

        public final void setSecurityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = SecurityGroupIdSetCopier.copy(securityGroupIds);
        }

        @Override
        public final Builder securityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = SecurityGroupIdSetCopier.copy(securityGroupIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroupIds(String... securityGroupIds) {
            securityGroupIds(Arrays.asList(securityGroupIds));
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public Ec2InstanceConnectEndpoint build() {
            return new Ec2InstanceConnectEndpoint(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
