/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ebs;

import java.nio.file.Path;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ResponseBytes;
import software.amazon.awssdk.core.ResponseInputStream;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.core.sync.ResponseTransformer;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.ebs.model.AccessDeniedException;
import software.amazon.awssdk.services.ebs.model.CompleteSnapshotRequest;
import software.amazon.awssdk.services.ebs.model.CompleteSnapshotResponse;
import software.amazon.awssdk.services.ebs.model.ConcurrentLimitExceededException;
import software.amazon.awssdk.services.ebs.model.ConflictException;
import software.amazon.awssdk.services.ebs.model.EbsException;
import software.amazon.awssdk.services.ebs.model.GetSnapshotBlockRequest;
import software.amazon.awssdk.services.ebs.model.GetSnapshotBlockResponse;
import software.amazon.awssdk.services.ebs.model.InternalServerException;
import software.amazon.awssdk.services.ebs.model.ListChangedBlocksRequest;
import software.amazon.awssdk.services.ebs.model.ListChangedBlocksResponse;
import software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksRequest;
import software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksResponse;
import software.amazon.awssdk.services.ebs.model.PutSnapshotBlockRequest;
import software.amazon.awssdk.services.ebs.model.PutSnapshotBlockResponse;
import software.amazon.awssdk.services.ebs.model.RequestThrottledException;
import software.amazon.awssdk.services.ebs.model.ResourceNotFoundException;
import software.amazon.awssdk.services.ebs.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.ebs.model.StartSnapshotRequest;
import software.amazon.awssdk.services.ebs.model.StartSnapshotResponse;
import software.amazon.awssdk.services.ebs.model.ValidationException;
import software.amazon.awssdk.services.ebs.paginators.ListChangedBlocksIterable;
import software.amazon.awssdk.services.ebs.paginators.ListSnapshotBlocksIterable;

/**
 * Service client for accessing Amazon EBS. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * You can use the Amazon Elastic Block Store (Amazon EBS) direct APIs to create Amazon EBS snapshots, write data
 * directly to your snapshots, read data on your snapshots, and identify the differences or changes between two
 * snapshots. If you’re an independent software vendor (ISV) who offers backup services for Amazon EBS, the EBS direct
 * APIs make it more efficient and cost-effective to track incremental changes on your Amazon EBS volumes through
 * snapshots. This can be done without having to create new volumes from snapshots, and then use Amazon Elastic Compute
 * Cloud (Amazon EC2) instances to compare the differences.
 * </p>
 * <p>
 * You can create incremental snapshots directly from data on-premises into volumes and the cloud to use for quick
 * disaster recovery. With the ability to write and read snapshots, you can write your on-premises data to an snapshot
 * during a disaster. Then after recovery, you can restore it back to Amazon Web Services or on-premises from the
 * snapshot. You no longer need to build and maintain complex mechanisms to copy data to and from Amazon EBS.
 * </p>
 * <p>
 * This API reference provides detailed information about the actions, data types, parameters, and errors of the EBS
 * direct APIs. For more information about the elements that make up the EBS direct APIs, and examples of how to use
 * them effectively, see <a
 * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-accessing-snapshot.html">Accessing the Contents of an
 * Amazon EBS Snapshot</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>. For more information about the
 * supported Amazon Web Services Regions, endpoints, and service quotas for the EBS direct APIs, see <a
 * href="https://docs.aws.amazon.com/general/latest/gr/ebs-service.html">Amazon Elastic Block Store Endpoints and
 * Quotas</a> in the <i>Amazon Web Services General Reference</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface EbsClient extends AwsClient {
    String SERVICE_NAME = "ebs";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "ebs";

    /**
     * <p>
     * Seals and completes the snapshot after all of the required blocks of data have been written to it. Completing the
     * snapshot changes the status to <code>completed</code>. You cannot write new blocks to a snapshot after it has
     * been completed.
     * </p>
     * <note>
     * <p>
     * You should always retry requests that receive server (<code>5xx</code>) error responses, and
     * <code>ThrottlingException</code> and <code>RequestThrottledException</code> client error responses. For more
     * information see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error
     * retries</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note>
     *
     * @param completeSnapshotRequest
     * @return Result of the CompleteSnapshot operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws RequestThrottledException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the
     *         snapshot. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws ServiceQuotaExceededException
     *         Your current service quotas do not allow you to perform this action.
     * @throws InternalServerException
     *         An internal error has occurred. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.CompleteSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/CompleteSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    default CompleteSnapshotResponse completeSnapshot(CompleteSnapshotRequest completeSnapshotRequest)
            throws AccessDeniedException, ValidationException, ResourceNotFoundException, RequestThrottledException,
            ServiceQuotaExceededException, InternalServerException, AwsServiceException, SdkClientException, EbsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Seals and completes the snapshot after all of the required blocks of data have been written to it. Completing the
     * snapshot changes the status to <code>completed</code>. You cannot write new blocks to a snapshot after it has
     * been completed.
     * </p>
     * <note>
     * <p>
     * You should always retry requests that receive server (<code>5xx</code>) error responses, and
     * <code>ThrottlingException</code> and <code>RequestThrottledException</code> client error responses. For more
     * information see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error
     * retries</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CompleteSnapshotRequest.Builder} avoiding the need
     * to create one manually via {@link CompleteSnapshotRequest#builder()}
     * </p>
     *
     * @param completeSnapshotRequest
     *        A {@link Consumer} that will call methods on {@link CompleteSnapshotRequest.Builder} to create a request.
     * @return Result of the CompleteSnapshot operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws RequestThrottledException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the
     *         snapshot. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws ServiceQuotaExceededException
     *         Your current service quotas do not allow you to perform this action.
     * @throws InternalServerException
     *         An internal error has occurred. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.CompleteSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/CompleteSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    default CompleteSnapshotResponse completeSnapshot(Consumer<CompleteSnapshotRequest.Builder> completeSnapshotRequest)
            throws AccessDeniedException, ValidationException, ResourceNotFoundException, RequestThrottledException,
            ServiceQuotaExceededException, InternalServerException, AwsServiceException, SdkClientException, EbsException {
        return completeSnapshot(CompleteSnapshotRequest.builder().applyMutation(completeSnapshotRequest).build());
    }

    /**
     * <p>
     * Returns the data in a block in an Amazon Elastic Block Store snapshot.
     * </p>
     * <note>
     * <p>
     * You should always retry requests that receive server (<code>5xx</code>) error responses, and
     * <code>ThrottlingException</code> and <code>RequestThrottledException</code> client error responses. For more
     * information see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error
     * retries</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note>
     *
     * @param getSnapshotBlockRequest
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled
     *        GetSnapshotBlockResponse and an InputStream to the response content are provided as parameters to the
     *        callback. The callback may return a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows '
     *        <p>
     *        The data content of the block.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws RequestThrottledException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the
     *         snapshot. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws ServiceQuotaExceededException
     *         Your current service quotas do not allow you to perform this action.
     * @throws InternalServerException
     *         An internal error has occurred. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.GetSnapshotBlock
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/GetSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    default <ReturnT> ReturnT getSnapshotBlock(GetSnapshotBlockRequest getSnapshotBlockRequest,
            ResponseTransformer<GetSnapshotBlockResponse, ReturnT> responseTransformer) throws AccessDeniedException,
            ValidationException, ResourceNotFoundException, RequestThrottledException, ServiceQuotaExceededException,
            InternalServerException, AwsServiceException, SdkClientException, EbsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the data in a block in an Amazon Elastic Block Store snapshot.
     * </p>
     * <note>
     * <p>
     * You should always retry requests that receive server (<code>5xx</code>) error responses, and
     * <code>ThrottlingException</code> and <code>RequestThrottledException</code> client error responses. For more
     * information see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error
     * retries</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetSnapshotBlockRequest.Builder} avoiding the need
     * to create one manually via {@link GetSnapshotBlockRequest#builder()}
     * </p>
     *
     * @param getSnapshotBlockRequest
     *        A {@link Consumer} that will call methods on {@link GetSnapshotBlockRequest.Builder} to create a request.
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled
     *        GetSnapshotBlockResponse and an InputStream to the response content are provided as parameters to the
     *        callback. The callback may return a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows '
     *        <p>
     *        The data content of the block.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws RequestThrottledException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the
     *         snapshot. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws ServiceQuotaExceededException
     *         Your current service quotas do not allow you to perform this action.
     * @throws InternalServerException
     *         An internal error has occurred. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.GetSnapshotBlock
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/GetSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    default <ReturnT> ReturnT getSnapshotBlock(Consumer<GetSnapshotBlockRequest.Builder> getSnapshotBlockRequest,
            ResponseTransformer<GetSnapshotBlockResponse, ReturnT> responseTransformer) throws AccessDeniedException,
            ValidationException, ResourceNotFoundException, RequestThrottledException, ServiceQuotaExceededException,
            InternalServerException, AwsServiceException, SdkClientException, EbsException {
        return getSnapshotBlock(GetSnapshotBlockRequest.builder().applyMutation(getSnapshotBlockRequest).build(),
                responseTransformer);
    }

    /**
     * <p>
     * Returns the data in a block in an Amazon Elastic Block Store snapshot.
     * </p>
     * <note>
     * <p>
     * You should always retry requests that receive server (<code>5xx</code>) error responses, and
     * <code>ThrottlingException</code> and <code>RequestThrottledException</code> client error responses. For more
     * information see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error
     * retries</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note>
     *
     * @param getSnapshotBlockRequest
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows '
     *        <p>
     *        The data content of the block.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws RequestThrottledException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the
     *         snapshot. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws ServiceQuotaExceededException
     *         Your current service quotas do not allow you to perform this action.
     * @throws InternalServerException
     *         An internal error has occurred. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.GetSnapshotBlock
     * @see #getSnapshotBlock(GetSnapshotBlockRequest, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/GetSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    default GetSnapshotBlockResponse getSnapshotBlock(GetSnapshotBlockRequest getSnapshotBlockRequest, Path destinationPath)
            throws AccessDeniedException, ValidationException, ResourceNotFoundException, RequestThrottledException,
            ServiceQuotaExceededException, InternalServerException, AwsServiceException, SdkClientException, EbsException {
        return getSnapshotBlock(getSnapshotBlockRequest, ResponseTransformer.toFile(destinationPath));
    }

    /**
     * <p>
     * Returns the data in a block in an Amazon Elastic Block Store snapshot.
     * </p>
     * <note>
     * <p>
     * You should always retry requests that receive server (<code>5xx</code>) error responses, and
     * <code>ThrottlingException</code> and <code>RequestThrottledException</code> client error responses. For more
     * information see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error
     * retries</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetSnapshotBlockRequest.Builder} avoiding the need
     * to create one manually via {@link GetSnapshotBlockRequest#builder()}
     * </p>
     *
     * @param getSnapshotBlockRequest
     *        A {@link Consumer} that will call methods on {@link GetSnapshotBlockRequest.Builder} to create a request.
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows '
     *        <p>
     *        The data content of the block.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws RequestThrottledException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the
     *         snapshot. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws ServiceQuotaExceededException
     *         Your current service quotas do not allow you to perform this action.
     * @throws InternalServerException
     *         An internal error has occurred. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.GetSnapshotBlock
     * @see #getSnapshotBlock(GetSnapshotBlockRequest, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/GetSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    default GetSnapshotBlockResponse getSnapshotBlock(Consumer<GetSnapshotBlockRequest.Builder> getSnapshotBlockRequest,
            Path destinationPath) throws AccessDeniedException, ValidationException, ResourceNotFoundException,
            RequestThrottledException, ServiceQuotaExceededException, InternalServerException, AwsServiceException,
            SdkClientException, EbsException {
        return getSnapshotBlock(GetSnapshotBlockRequest.builder().applyMutation(getSnapshotBlockRequest).build(), destinationPath);
    }

    /**
     * <p>
     * Returns the data in a block in an Amazon Elastic Block Store snapshot.
     * </p>
     * <note>
     * <p>
     * You should always retry requests that receive server (<code>5xx</code>) error responses, and
     * <code>ThrottlingException</code> and <code>RequestThrottledException</code> client error responses. For more
     * information see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error
     * retries</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note>
     *
     * @param getSnapshotBlockRequest
     * @return A {@link ResponseInputStream} containing data streamed from service. Note that this is an unmanaged
     *         reference to the underlying HTTP connection so great care must be taken to ensure all data if fully read
     *         from the input stream and that it is properly closed. Failure to do so may result in sub-optimal behavior
     *         and exhausting connections in the connection pool. The unmarshalled response object can be obtained via
     *         {@link ResponseInputStream#response()}. The service documentation for the response content is as follows
     *         '
     *         <p>
     *         The data content of the block.
     *         </p>
     *         '.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws RequestThrottledException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the
     *         snapshot. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws ServiceQuotaExceededException
     *         Your current service quotas do not allow you to perform this action.
     * @throws InternalServerException
     *         An internal error has occurred. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.GetSnapshotBlock
     * @see #getObject(getSnapshotBlock, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/GetSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    default ResponseInputStream<GetSnapshotBlockResponse> getSnapshotBlock(GetSnapshotBlockRequest getSnapshotBlockRequest)
            throws AccessDeniedException, ValidationException, ResourceNotFoundException, RequestThrottledException,
            ServiceQuotaExceededException, InternalServerException, AwsServiceException, SdkClientException, EbsException {
        return getSnapshotBlock(getSnapshotBlockRequest, ResponseTransformer.toInputStream());
    }

    /**
     * <p>
     * Returns the data in a block in an Amazon Elastic Block Store snapshot.
     * </p>
     * <note>
     * <p>
     * You should always retry requests that receive server (<code>5xx</code>) error responses, and
     * <code>ThrottlingException</code> and <code>RequestThrottledException</code> client error responses. For more
     * information see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error
     * retries</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetSnapshotBlockRequest.Builder} avoiding the need
     * to create one manually via {@link GetSnapshotBlockRequest#builder()}
     * </p>
     *
     * @param getSnapshotBlockRequest
     *        A {@link Consumer} that will call methods on {@link GetSnapshotBlockRequest.Builder} to create a request.
     * @return A {@link ResponseInputStream} containing data streamed from service. Note that this is an unmanaged
     *         reference to the underlying HTTP connection so great care must be taken to ensure all data if fully read
     *         from the input stream and that it is properly closed. Failure to do so may result in sub-optimal behavior
     *         and exhausting connections in the connection pool. The unmarshalled response object can be obtained via
     *         {@link ResponseInputStream#response()}. The service documentation for the response content is as follows
     *         '
     *         <p>
     *         The data content of the block.
     *         </p>
     *         '.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws RequestThrottledException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the
     *         snapshot. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws ServiceQuotaExceededException
     *         Your current service quotas do not allow you to perform this action.
     * @throws InternalServerException
     *         An internal error has occurred. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.GetSnapshotBlock
     * @see #getObject(getSnapshotBlock, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/GetSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    default ResponseInputStream<GetSnapshotBlockResponse> getSnapshotBlock(
            Consumer<GetSnapshotBlockRequest.Builder> getSnapshotBlockRequest) throws AccessDeniedException, ValidationException,
            ResourceNotFoundException, RequestThrottledException, ServiceQuotaExceededException, InternalServerException,
            AwsServiceException, SdkClientException, EbsException {
        return getSnapshotBlock(GetSnapshotBlockRequest.builder().applyMutation(getSnapshotBlockRequest).build());
    }

    /**
     * <p>
     * Returns the data in a block in an Amazon Elastic Block Store snapshot.
     * </p>
     * <note>
     * <p>
     * You should always retry requests that receive server (<code>5xx</code>) error responses, and
     * <code>ThrottlingException</code> and <code>RequestThrottledException</code> client error responses. For more
     * information see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error
     * retries</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note>
     *
     * @param getSnapshotBlockRequest
     * @return A {@link ResponseBytes} that loads the data streamed from the service into memory and exposes it in
     *         convenient in-memory representations like a byte buffer or string. The unmarshalled response object can
     *         be obtained via {@link ResponseBytes#response()}. The service documentation for the response content is
     *         as follows '
     *         <p>
     *         The data content of the block.
     *         </p>
     *         '.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws RequestThrottledException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the
     *         snapshot. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws ServiceQuotaExceededException
     *         Your current service quotas do not allow you to perform this action.
     * @throws InternalServerException
     *         An internal error has occurred. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.GetSnapshotBlock
     * @see #getObject(getSnapshotBlock, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/GetSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    default ResponseBytes<GetSnapshotBlockResponse> getSnapshotBlockAsBytes(GetSnapshotBlockRequest getSnapshotBlockRequest)
            throws AccessDeniedException, ValidationException, ResourceNotFoundException, RequestThrottledException,
            ServiceQuotaExceededException, InternalServerException, AwsServiceException, SdkClientException, EbsException {
        return getSnapshotBlock(getSnapshotBlockRequest, ResponseTransformer.toBytes());
    }

    /**
     * <p>
     * Returns the data in a block in an Amazon Elastic Block Store snapshot.
     * </p>
     * <note>
     * <p>
     * You should always retry requests that receive server (<code>5xx</code>) error responses, and
     * <code>ThrottlingException</code> and <code>RequestThrottledException</code> client error responses. For more
     * information see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error
     * retries</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetSnapshotBlockRequest.Builder} avoiding the need
     * to create one manually via {@link GetSnapshotBlockRequest#builder()}
     * </p>
     *
     * @param getSnapshotBlockRequest
     *        A {@link Consumer} that will call methods on {@link GetSnapshotBlockRequest.Builder} to create a request.
     * @return A {@link ResponseBytes} that loads the data streamed from the service into memory and exposes it in
     *         convenient in-memory representations like a byte buffer or string. The unmarshalled response object can
     *         be obtained via {@link ResponseBytes#response()}. The service documentation for the response content is
     *         as follows '
     *         <p>
     *         The data content of the block.
     *         </p>
     *         '.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws RequestThrottledException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the
     *         snapshot. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws ServiceQuotaExceededException
     *         Your current service quotas do not allow you to perform this action.
     * @throws InternalServerException
     *         An internal error has occurred. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.GetSnapshotBlock
     * @see #getObject(getSnapshotBlock, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/GetSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    default ResponseBytes<GetSnapshotBlockResponse> getSnapshotBlockAsBytes(
            Consumer<GetSnapshotBlockRequest.Builder> getSnapshotBlockRequest) throws AccessDeniedException, ValidationException,
            ResourceNotFoundException, RequestThrottledException, ServiceQuotaExceededException, InternalServerException,
            AwsServiceException, SdkClientException, EbsException {
        return getSnapshotBlockAsBytes(GetSnapshotBlockRequest.builder().applyMutation(getSnapshotBlockRequest).build());
    }

    /**
     * <p>
     * Returns information about the blocks that are different between two Amazon Elastic Block Store snapshots of the
     * same volume/snapshot lineage.
     * </p>
     * <note>
     * <p>
     * You should always retry requests that receive server (<code>5xx</code>) error responses, and
     * <code>ThrottlingException</code> and <code>RequestThrottledException</code> client error responses. For more
     * information see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error
     * retries</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note>
     *
     * @param listChangedBlocksRequest
     * @return Result of the ListChangedBlocks operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws RequestThrottledException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the
     *         snapshot. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws ServiceQuotaExceededException
     *         Your current service quotas do not allow you to perform this action.
     * @throws InternalServerException
     *         An internal error has occurred. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.ListChangedBlocks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/ListChangedBlocks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListChangedBlocksResponse listChangedBlocks(ListChangedBlocksRequest listChangedBlocksRequest)
            throws AccessDeniedException, ValidationException, ResourceNotFoundException, RequestThrottledException,
            ServiceQuotaExceededException, InternalServerException, AwsServiceException, SdkClientException, EbsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns information about the blocks that are different between two Amazon Elastic Block Store snapshots of the
     * same volume/snapshot lineage.
     * </p>
     * <note>
     * <p>
     * You should always retry requests that receive server (<code>5xx</code>) error responses, and
     * <code>ThrottlingException</code> and <code>RequestThrottledException</code> client error responses. For more
     * information see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error
     * retries</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListChangedBlocksRequest.Builder} avoiding the need
     * to create one manually via {@link ListChangedBlocksRequest#builder()}
     * </p>
     *
     * @param listChangedBlocksRequest
     *        A {@link Consumer} that will call methods on {@link ListChangedBlocksRequest.Builder} to create a request.
     * @return Result of the ListChangedBlocks operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws RequestThrottledException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the
     *         snapshot. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws ServiceQuotaExceededException
     *         Your current service quotas do not allow you to perform this action.
     * @throws InternalServerException
     *         An internal error has occurred. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.ListChangedBlocks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/ListChangedBlocks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListChangedBlocksResponse listChangedBlocks(Consumer<ListChangedBlocksRequest.Builder> listChangedBlocksRequest)
            throws AccessDeniedException, ValidationException, ResourceNotFoundException, RequestThrottledException,
            ServiceQuotaExceededException, InternalServerException, AwsServiceException, SdkClientException, EbsException {
        return listChangedBlocks(ListChangedBlocksRequest.builder().applyMutation(listChangedBlocksRequest).build());
    }

    /**
     * <p>
     * Returns information about the blocks that are different between two Amazon Elastic Block Store snapshots of the
     * same volume/snapshot lineage.
     * </p>
     * <note>
     * <p>
     * You should always retry requests that receive server (<code>5xx</code>) error responses, and
     * <code>ThrottlingException</code> and <code>RequestThrottledException</code> client error responses. For more
     * information see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error
     * retries</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #listChangedBlocks(software.amazon.awssdk.services.ebs.model.ListChangedBlocksRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ebs.paginators.ListChangedBlocksIterable responses = client.listChangedBlocksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ebs.paginators.ListChangedBlocksIterable responses = client
     *             .listChangedBlocksPaginator(request);
     *     for (software.amazon.awssdk.services.ebs.model.ListChangedBlocksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ebs.paginators.ListChangedBlocksIterable responses = client.listChangedBlocksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listChangedBlocks(software.amazon.awssdk.services.ebs.model.ListChangedBlocksRequest)} operation.</b>
     * </p>
     *
     * @param listChangedBlocksRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws RequestThrottledException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the
     *         snapshot. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws ServiceQuotaExceededException
     *         Your current service quotas do not allow you to perform this action.
     * @throws InternalServerException
     *         An internal error has occurred. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.ListChangedBlocks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/ListChangedBlocks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListChangedBlocksIterable listChangedBlocksPaginator(ListChangedBlocksRequest listChangedBlocksRequest)
            throws AccessDeniedException, ValidationException, ResourceNotFoundException, RequestThrottledException,
            ServiceQuotaExceededException, InternalServerException, AwsServiceException, SdkClientException, EbsException {
        return new ListChangedBlocksIterable(this, listChangedBlocksRequest);
    }

    /**
     * <p>
     * Returns information about the blocks that are different between two Amazon Elastic Block Store snapshots of the
     * same volume/snapshot lineage.
     * </p>
     * <note>
     * <p>
     * You should always retry requests that receive server (<code>5xx</code>) error responses, and
     * <code>ThrottlingException</code> and <code>RequestThrottledException</code> client error responses. For more
     * information see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error
     * retries</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #listChangedBlocks(software.amazon.awssdk.services.ebs.model.ListChangedBlocksRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ebs.paginators.ListChangedBlocksIterable responses = client.listChangedBlocksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ebs.paginators.ListChangedBlocksIterable responses = client
     *             .listChangedBlocksPaginator(request);
     *     for (software.amazon.awssdk.services.ebs.model.ListChangedBlocksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ebs.paginators.ListChangedBlocksIterable responses = client.listChangedBlocksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listChangedBlocks(software.amazon.awssdk.services.ebs.model.ListChangedBlocksRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListChangedBlocksRequest.Builder} avoiding the need
     * to create one manually via {@link ListChangedBlocksRequest#builder()}
     * </p>
     *
     * @param listChangedBlocksRequest
     *        A {@link Consumer} that will call methods on {@link ListChangedBlocksRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws RequestThrottledException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the
     *         snapshot. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws ServiceQuotaExceededException
     *         Your current service quotas do not allow you to perform this action.
     * @throws InternalServerException
     *         An internal error has occurred. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.ListChangedBlocks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/ListChangedBlocks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListChangedBlocksIterable listChangedBlocksPaginator(
            Consumer<ListChangedBlocksRequest.Builder> listChangedBlocksRequest) throws AccessDeniedException,
            ValidationException, ResourceNotFoundException, RequestThrottledException, ServiceQuotaExceededException,
            InternalServerException, AwsServiceException, SdkClientException, EbsException {
        return listChangedBlocksPaginator(ListChangedBlocksRequest.builder().applyMutation(listChangedBlocksRequest).build());
    }

    /**
     * <p>
     * Returns information about the blocks in an Amazon Elastic Block Store snapshot.
     * </p>
     * <note>
     * <p>
     * You should always retry requests that receive server (<code>5xx</code>) error responses, and
     * <code>ThrottlingException</code> and <code>RequestThrottledException</code> client error responses. For more
     * information see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error
     * retries</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note>
     *
     * @param listSnapshotBlocksRequest
     * @return Result of the ListSnapshotBlocks operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws RequestThrottledException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the
     *         snapshot. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws ServiceQuotaExceededException
     *         Your current service quotas do not allow you to perform this action.
     * @throws InternalServerException
     *         An internal error has occurred. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.ListSnapshotBlocks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/ListSnapshotBlocks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListSnapshotBlocksResponse listSnapshotBlocks(ListSnapshotBlocksRequest listSnapshotBlocksRequest)
            throws AccessDeniedException, ValidationException, ResourceNotFoundException, RequestThrottledException,
            ServiceQuotaExceededException, InternalServerException, AwsServiceException, SdkClientException, EbsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns information about the blocks in an Amazon Elastic Block Store snapshot.
     * </p>
     * <note>
     * <p>
     * You should always retry requests that receive server (<code>5xx</code>) error responses, and
     * <code>ThrottlingException</code> and <code>RequestThrottledException</code> client error responses. For more
     * information see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error
     * retries</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListSnapshotBlocksRequest.Builder} avoiding the
     * need to create one manually via {@link ListSnapshotBlocksRequest#builder()}
     * </p>
     *
     * @param listSnapshotBlocksRequest
     *        A {@link Consumer} that will call methods on {@link ListSnapshotBlocksRequest.Builder} to create a
     *        request.
     * @return Result of the ListSnapshotBlocks operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws RequestThrottledException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the
     *         snapshot. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws ServiceQuotaExceededException
     *         Your current service quotas do not allow you to perform this action.
     * @throws InternalServerException
     *         An internal error has occurred. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.ListSnapshotBlocks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/ListSnapshotBlocks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListSnapshotBlocksResponse listSnapshotBlocks(Consumer<ListSnapshotBlocksRequest.Builder> listSnapshotBlocksRequest)
            throws AccessDeniedException, ValidationException, ResourceNotFoundException, RequestThrottledException,
            ServiceQuotaExceededException, InternalServerException, AwsServiceException, SdkClientException, EbsException {
        return listSnapshotBlocks(ListSnapshotBlocksRequest.builder().applyMutation(listSnapshotBlocksRequest).build());
    }

    /**
     * <p>
     * Returns information about the blocks in an Amazon Elastic Block Store snapshot.
     * </p>
     * <note>
     * <p>
     * You should always retry requests that receive server (<code>5xx</code>) error responses, and
     * <code>ThrottlingException</code> and <code>RequestThrottledException</code> client error responses. For more
     * information see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error
     * retries</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #listSnapshotBlocks(software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ebs.paginators.ListSnapshotBlocksIterable responses = client.listSnapshotBlocksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ebs.paginators.ListSnapshotBlocksIterable responses = client
     *             .listSnapshotBlocksPaginator(request);
     *     for (software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ebs.paginators.ListSnapshotBlocksIterable responses = client.listSnapshotBlocksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSnapshotBlocks(software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksRequest)} operation.</b>
     * </p>
     *
     * @param listSnapshotBlocksRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws RequestThrottledException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the
     *         snapshot. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws ServiceQuotaExceededException
     *         Your current service quotas do not allow you to perform this action.
     * @throws InternalServerException
     *         An internal error has occurred. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.ListSnapshotBlocks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/ListSnapshotBlocks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListSnapshotBlocksIterable listSnapshotBlocksPaginator(ListSnapshotBlocksRequest listSnapshotBlocksRequest)
            throws AccessDeniedException, ValidationException, ResourceNotFoundException, RequestThrottledException,
            ServiceQuotaExceededException, InternalServerException, AwsServiceException, SdkClientException, EbsException {
        return new ListSnapshotBlocksIterable(this, listSnapshotBlocksRequest);
    }

    /**
     * <p>
     * Returns information about the blocks in an Amazon Elastic Block Store snapshot.
     * </p>
     * <note>
     * <p>
     * You should always retry requests that receive server (<code>5xx</code>) error responses, and
     * <code>ThrottlingException</code> and <code>RequestThrottledException</code> client error responses. For more
     * information see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error
     * retries</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #listSnapshotBlocks(software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ebs.paginators.ListSnapshotBlocksIterable responses = client.listSnapshotBlocksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ebs.paginators.ListSnapshotBlocksIterable responses = client
     *             .listSnapshotBlocksPaginator(request);
     *     for (software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ebs.paginators.ListSnapshotBlocksIterable responses = client.listSnapshotBlocksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSnapshotBlocks(software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListSnapshotBlocksRequest.Builder} avoiding the
     * need to create one manually via {@link ListSnapshotBlocksRequest#builder()}
     * </p>
     *
     * @param listSnapshotBlocksRequest
     *        A {@link Consumer} that will call methods on {@link ListSnapshotBlocksRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws RequestThrottledException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the
     *         snapshot. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws ServiceQuotaExceededException
     *         Your current service quotas do not allow you to perform this action.
     * @throws InternalServerException
     *         An internal error has occurred. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.ListSnapshotBlocks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/ListSnapshotBlocks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListSnapshotBlocksIterable listSnapshotBlocksPaginator(
            Consumer<ListSnapshotBlocksRequest.Builder> listSnapshotBlocksRequest) throws AccessDeniedException,
            ValidationException, ResourceNotFoundException, RequestThrottledException, ServiceQuotaExceededException,
            InternalServerException, AwsServiceException, SdkClientException, EbsException {
        return listSnapshotBlocksPaginator(ListSnapshotBlocksRequest.builder().applyMutation(listSnapshotBlocksRequest).build());
    }

    /**
     * <p>
     * Writes a block of data to a snapshot. If the specified block contains data, the existing data is overwritten. The
     * target snapshot must be in the <code>pending</code> state.
     * </p>
     * <p>
     * Data written to a snapshot must be aligned with 512-KiB sectors.
     * </p>
     * <note>
     * <p>
     * You should always retry requests that receive server (<code>5xx</code>) error responses, and
     * <code>ThrottlingException</code> and <code>RequestThrottledException</code> client error responses. For more
     * information see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error
     * retries</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note>
     *
     * @param putSnapshotBlockRequest
     * @param requestBody
     *        The content to send to the service. A {@link RequestBody} can be created using one of several factory
     *        methods for various sources of data. For example, to create a request body from a file you can do the
     *        following.
     * 
     *        <pre>
     * {@code RequestBody.fromFile(new File("myfile.txt"))}
     * </pre>
     * 
     *        See documentation in {@link RequestBody} for additional details and which sources of data are supported.
     *        The service documentation for the request content is as follows '
     *        <p>
     *        The data to write to the block.
     *        </p>
     *        <p>
     *        The block data is not signed as part of the Signature Version 4 signing process. As a result, you must
     *        generate and provide a Base64-encoded SHA256 checksum for the block data using the <b>x-amz-Checksum</b>
     *        header. Also, you must specify the checksum algorithm using the <b>x-amz-Checksum-Algorithm</b> header.
     *        The checksum that you provide is part of the Signature Version 4 signing process. It is validated against
     *        a checksum generated by Amazon EBS to ensure the validity and authenticity of the data. If the checksums
     *        do not correspond, the request fails. For more information, see <a href=
     *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-accessing-snapshot.html#ebsapis-using-checksums">
     *        Using checksums with the EBS direct APIs</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     *        </p>
     *        '
     * @return Result of the PutSnapshotBlock operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws RequestThrottledException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the
     *         snapshot. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws ServiceQuotaExceededException
     *         Your current service quotas do not allow you to perform this action.
     * @throws InternalServerException
     *         An internal error has occurred. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.PutSnapshotBlock
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/PutSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    default PutSnapshotBlockResponse putSnapshotBlock(PutSnapshotBlockRequest putSnapshotBlockRequest, RequestBody requestBody)
            throws AccessDeniedException, ValidationException, ResourceNotFoundException, RequestThrottledException,
            ServiceQuotaExceededException, InternalServerException, AwsServiceException, SdkClientException, EbsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Writes a block of data to a snapshot. If the specified block contains data, the existing data is overwritten. The
     * target snapshot must be in the <code>pending</code> state.
     * </p>
     * <p>
     * Data written to a snapshot must be aligned with 512-KiB sectors.
     * </p>
     * <note>
     * <p>
     * You should always retry requests that receive server (<code>5xx</code>) error responses, and
     * <code>ThrottlingException</code> and <code>RequestThrottledException</code> client error responses. For more
     * information see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error
     * retries</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutSnapshotBlockRequest.Builder} avoiding the need
     * to create one manually via {@link PutSnapshotBlockRequest#builder()}
     * </p>
     *
     * @param putSnapshotBlockRequest
     *        A {@link Consumer} that will call methods on {@link PutSnapshotBlockRequest.Builder} to create a request.
     * @param requestBody
     *        The content to send to the service. A {@link RequestBody} can be created using one of several factory
     *        methods for various sources of data. For example, to create a request body from a file you can do the
     *        following.
     * 
     *        <pre>
     * {@code RequestBody.fromFile(new File("myfile.txt"))}
     * </pre>
     * 
     *        See documentation in {@link RequestBody} for additional details and which sources of data are supported.
     *        The service documentation for the request content is as follows '
     *        <p>
     *        The data to write to the block.
     *        </p>
     *        <p>
     *        The block data is not signed as part of the Signature Version 4 signing process. As a result, you must
     *        generate and provide a Base64-encoded SHA256 checksum for the block data using the <b>x-amz-Checksum</b>
     *        header. Also, you must specify the checksum algorithm using the <b>x-amz-Checksum-Algorithm</b> header.
     *        The checksum that you provide is part of the Signature Version 4 signing process. It is validated against
     *        a checksum generated by Amazon EBS to ensure the validity and authenticity of the data. If the checksums
     *        do not correspond, the request fails. For more information, see <a href=
     *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-accessing-snapshot.html#ebsapis-using-checksums">
     *        Using checksums with the EBS direct APIs</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     *        </p>
     *        '
     * @return Result of the PutSnapshotBlock operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws RequestThrottledException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the
     *         snapshot. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws ServiceQuotaExceededException
     *         Your current service quotas do not allow you to perform this action.
     * @throws InternalServerException
     *         An internal error has occurred. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.PutSnapshotBlock
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/PutSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    default PutSnapshotBlockResponse putSnapshotBlock(Consumer<PutSnapshotBlockRequest.Builder> putSnapshotBlockRequest,
            RequestBody requestBody) throws AccessDeniedException, ValidationException, ResourceNotFoundException,
            RequestThrottledException, ServiceQuotaExceededException, InternalServerException, AwsServiceException,
            SdkClientException, EbsException {
        return putSnapshotBlock(PutSnapshotBlockRequest.builder().applyMutation(putSnapshotBlockRequest).build(), requestBody);
    }

    /**
     * <p>
     * Writes a block of data to a snapshot. If the specified block contains data, the existing data is overwritten. The
     * target snapshot must be in the <code>pending</code> state.
     * </p>
     * <p>
     * Data written to a snapshot must be aligned with 512-KiB sectors.
     * </p>
     * <note>
     * <p>
     * You should always retry requests that receive server (<code>5xx</code>) error responses, and
     * <code>ThrottlingException</code> and <code>RequestThrottledException</code> client error responses. For more
     * information see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error
     * retries</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note>
     *
     * @param putSnapshotBlockRequest
     * @param sourcePath
     *        {@link Path} to file containing data to send to the service. File will be read entirely and may be read
     *        multiple times in the event of a retry. If the file does not exist or the current user does not have
     *        access to read it then an exception will be thrown. The service documentation for the request content is
     *        as follows '
     *        <p>
     *        The data to write to the block.
     *        </p>
     *        <p>
     *        The block data is not signed as part of the Signature Version 4 signing process. As a result, you must
     *        generate and provide a Base64-encoded SHA256 checksum for the block data using the <b>x-amz-Checksum</b>
     *        header. Also, you must specify the checksum algorithm using the <b>x-amz-Checksum-Algorithm</b> header.
     *        The checksum that you provide is part of the Signature Version 4 signing process. It is validated against
     *        a checksum generated by Amazon EBS to ensure the validity and authenticity of the data. If the checksums
     *        do not correspond, the request fails. For more information, see <a href=
     *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-accessing-snapshot.html#ebsapis-using-checksums">
     *        Using checksums with the EBS direct APIs</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     *        </p>
     *        '
     * @return Result of the PutSnapshotBlock operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws RequestThrottledException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the
     *         snapshot. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws ServiceQuotaExceededException
     *         Your current service quotas do not allow you to perform this action.
     * @throws InternalServerException
     *         An internal error has occurred. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.PutSnapshotBlock
     * @see #putSnapshotBlock(PutSnapshotBlockRequest, RequestBody)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/PutSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    default PutSnapshotBlockResponse putSnapshotBlock(PutSnapshotBlockRequest putSnapshotBlockRequest, Path sourcePath)
            throws AccessDeniedException, ValidationException, ResourceNotFoundException, RequestThrottledException,
            ServiceQuotaExceededException, InternalServerException, AwsServiceException, SdkClientException, EbsException {
        return putSnapshotBlock(putSnapshotBlockRequest, RequestBody.fromFile(sourcePath));
    }

    /**
     * <p>
     * Writes a block of data to a snapshot. If the specified block contains data, the existing data is overwritten. The
     * target snapshot must be in the <code>pending</code> state.
     * </p>
     * <p>
     * Data written to a snapshot must be aligned with 512-KiB sectors.
     * </p>
     * <note>
     * <p>
     * You should always retry requests that receive server (<code>5xx</code>) error responses, and
     * <code>ThrottlingException</code> and <code>RequestThrottledException</code> client error responses. For more
     * information see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error
     * retries</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutSnapshotBlockRequest.Builder} avoiding the need
     * to create one manually via {@link PutSnapshotBlockRequest#builder()}
     * </p>
     *
     * @param putSnapshotBlockRequest
     *        A {@link Consumer} that will call methods on {@link PutSnapshotBlockRequest.Builder} to create a request.
     * @param sourcePath
     *        {@link Path} to file containing data to send to the service. File will be read entirely and may be read
     *        multiple times in the event of a retry. If the file does not exist or the current user does not have
     *        access to read it then an exception will be thrown. The service documentation for the request content is
     *        as follows '
     *        <p>
     *        The data to write to the block.
     *        </p>
     *        <p>
     *        The block data is not signed as part of the Signature Version 4 signing process. As a result, you must
     *        generate and provide a Base64-encoded SHA256 checksum for the block data using the <b>x-amz-Checksum</b>
     *        header. Also, you must specify the checksum algorithm using the <b>x-amz-Checksum-Algorithm</b> header.
     *        The checksum that you provide is part of the Signature Version 4 signing process. It is validated against
     *        a checksum generated by Amazon EBS to ensure the validity and authenticity of the data. If the checksums
     *        do not correspond, the request fails. For more information, see <a href=
     *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-accessing-snapshot.html#ebsapis-using-checksums">
     *        Using checksums with the EBS direct APIs</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     *        </p>
     *        '
     * @return Result of the PutSnapshotBlock operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws RequestThrottledException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the
     *         snapshot. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws ServiceQuotaExceededException
     *         Your current service quotas do not allow you to perform this action.
     * @throws InternalServerException
     *         An internal error has occurred. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.PutSnapshotBlock
     * @see #putSnapshotBlock(PutSnapshotBlockRequest, RequestBody)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/PutSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    default PutSnapshotBlockResponse putSnapshotBlock(Consumer<PutSnapshotBlockRequest.Builder> putSnapshotBlockRequest,
            Path sourcePath) throws AccessDeniedException, ValidationException, ResourceNotFoundException,
            RequestThrottledException, ServiceQuotaExceededException, InternalServerException, AwsServiceException,
            SdkClientException, EbsException {
        return putSnapshotBlock(PutSnapshotBlockRequest.builder().applyMutation(putSnapshotBlockRequest).build(), sourcePath);
    }

    /**
     * <p>
     * Creates a new Amazon EBS snapshot. The new snapshot enters the <code>pending</code> state after the request
     * completes.
     * </p>
     * <p>
     * After creating the snapshot, use <a
     * href="https://docs.aws.amazon.com/ebs/latest/APIReference/API_PutSnapshotBlock.html"> PutSnapshotBlock</a> to
     * write blocks of data to the snapshot.
     * </p>
     * <note>
     * <p>
     * You should always retry requests that receive server (<code>5xx</code>) error responses, and
     * <code>ThrottlingException</code> and <code>RequestThrottledException</code> client error responses. For more
     * information see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error
     * retries</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note>
     *
     * @param startSnapshotRequest
     * @return Result of the StartSnapshot operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws RequestThrottledException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the
     *         snapshot. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws ServiceQuotaExceededException
     *         Your current service quotas do not allow you to perform this action.
     * @throws InternalServerException
     *         An internal error has occurred. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws ConcurrentLimitExceededException
     *         You have reached the limit for concurrent API requests. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-accessing-snapshot.html#ebsapi-performance"
     *         >Optimizing performance of the EBS direct APIs</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * @throws ConflictException
     *         The request uses the same client token as a previous, but non-identical request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.StartSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/StartSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    default StartSnapshotResponse startSnapshot(StartSnapshotRequest startSnapshotRequest) throws AccessDeniedException,
            ValidationException, RequestThrottledException, ResourceNotFoundException, ServiceQuotaExceededException,
            InternalServerException, ConcurrentLimitExceededException, ConflictException, AwsServiceException,
            SdkClientException, EbsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a new Amazon EBS snapshot. The new snapshot enters the <code>pending</code> state after the request
     * completes.
     * </p>
     * <p>
     * After creating the snapshot, use <a
     * href="https://docs.aws.amazon.com/ebs/latest/APIReference/API_PutSnapshotBlock.html"> PutSnapshotBlock</a> to
     * write blocks of data to the snapshot.
     * </p>
     * <note>
     * <p>
     * You should always retry requests that receive server (<code>5xx</code>) error responses, and
     * <code>ThrottlingException</code> and <code>RequestThrottledException</code> client error responses. For more
     * information see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error
     * retries</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartSnapshotRequest.Builder} avoiding the need to
     * create one manually via {@link StartSnapshotRequest#builder()}
     * </p>
     *
     * @param startSnapshotRequest
     *        A {@link Consumer} that will call methods on {@link StartSnapshotRequest.Builder} to create a request.
     * @return Result of the StartSnapshot operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws RequestThrottledException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the
     *         snapshot. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws ServiceQuotaExceededException
     *         Your current service quotas do not allow you to perform this action.
     * @throws InternalServerException
     *         An internal error has occurred. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.
     * @throws ConcurrentLimitExceededException
     *         You have reached the limit for concurrent API requests. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-accessing-snapshot.html#ebsapi-performance"
     *         >Optimizing performance of the EBS direct APIs</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * @throws ConflictException
     *         The request uses the same client token as a previous, but non-identical request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.StartSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/StartSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    default StartSnapshotResponse startSnapshot(Consumer<StartSnapshotRequest.Builder> startSnapshotRequest)
            throws AccessDeniedException, ValidationException, RequestThrottledException, ResourceNotFoundException,
            ServiceQuotaExceededException, InternalServerException, ConcurrentLimitExceededException, ConflictException,
            AwsServiceException, SdkClientException, EbsException {
        return startSnapshot(StartSnapshotRequest.builder().applyMutation(startSnapshotRequest).build());
    }

    /**
     * Create a {@link EbsClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static EbsClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link EbsClient}.
     */
    static EbsClientBuilder builder() {
        return new DefaultEbsClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    default EbsServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }
}
