/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dataexchange;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.dataexchange.internal.DataExchangeServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.dataexchange.model.AccessDeniedException;
import software.amazon.awssdk.services.dataexchange.model.CancelJobRequest;
import software.amazon.awssdk.services.dataexchange.model.CancelJobResponse;
import software.amazon.awssdk.services.dataexchange.model.ConflictException;
import software.amazon.awssdk.services.dataexchange.model.CreateDataSetRequest;
import software.amazon.awssdk.services.dataexchange.model.CreateDataSetResponse;
import software.amazon.awssdk.services.dataexchange.model.CreateEventActionRequest;
import software.amazon.awssdk.services.dataexchange.model.CreateEventActionResponse;
import software.amazon.awssdk.services.dataexchange.model.CreateJobRequest;
import software.amazon.awssdk.services.dataexchange.model.CreateJobResponse;
import software.amazon.awssdk.services.dataexchange.model.CreateRevisionRequest;
import software.amazon.awssdk.services.dataexchange.model.CreateRevisionResponse;
import software.amazon.awssdk.services.dataexchange.model.DataExchangeException;
import software.amazon.awssdk.services.dataexchange.model.DeleteAssetRequest;
import software.amazon.awssdk.services.dataexchange.model.DeleteAssetResponse;
import software.amazon.awssdk.services.dataexchange.model.DeleteDataSetRequest;
import software.amazon.awssdk.services.dataexchange.model.DeleteDataSetResponse;
import software.amazon.awssdk.services.dataexchange.model.DeleteEventActionRequest;
import software.amazon.awssdk.services.dataexchange.model.DeleteEventActionResponse;
import software.amazon.awssdk.services.dataexchange.model.DeleteRevisionRequest;
import software.amazon.awssdk.services.dataexchange.model.DeleteRevisionResponse;
import software.amazon.awssdk.services.dataexchange.model.GetAssetRequest;
import software.amazon.awssdk.services.dataexchange.model.GetAssetResponse;
import software.amazon.awssdk.services.dataexchange.model.GetDataSetRequest;
import software.amazon.awssdk.services.dataexchange.model.GetDataSetResponse;
import software.amazon.awssdk.services.dataexchange.model.GetEventActionRequest;
import software.amazon.awssdk.services.dataexchange.model.GetEventActionResponse;
import software.amazon.awssdk.services.dataexchange.model.GetJobRequest;
import software.amazon.awssdk.services.dataexchange.model.GetJobResponse;
import software.amazon.awssdk.services.dataexchange.model.GetRevisionRequest;
import software.amazon.awssdk.services.dataexchange.model.GetRevisionResponse;
import software.amazon.awssdk.services.dataexchange.model.InternalServerException;
import software.amazon.awssdk.services.dataexchange.model.ListDataSetRevisionsRequest;
import software.amazon.awssdk.services.dataexchange.model.ListDataSetRevisionsResponse;
import software.amazon.awssdk.services.dataexchange.model.ListDataSetsRequest;
import software.amazon.awssdk.services.dataexchange.model.ListDataSetsResponse;
import software.amazon.awssdk.services.dataexchange.model.ListEventActionsRequest;
import software.amazon.awssdk.services.dataexchange.model.ListEventActionsResponse;
import software.amazon.awssdk.services.dataexchange.model.ListJobsRequest;
import software.amazon.awssdk.services.dataexchange.model.ListJobsResponse;
import software.amazon.awssdk.services.dataexchange.model.ListRevisionAssetsRequest;
import software.amazon.awssdk.services.dataexchange.model.ListRevisionAssetsResponse;
import software.amazon.awssdk.services.dataexchange.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.dataexchange.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.dataexchange.model.ResourceNotFoundException;
import software.amazon.awssdk.services.dataexchange.model.RevokeRevisionRequest;
import software.amazon.awssdk.services.dataexchange.model.RevokeRevisionResponse;
import software.amazon.awssdk.services.dataexchange.model.SendDataSetNotificationRequest;
import software.amazon.awssdk.services.dataexchange.model.SendDataSetNotificationResponse;
import software.amazon.awssdk.services.dataexchange.model.ServiceLimitExceededException;
import software.amazon.awssdk.services.dataexchange.model.StartJobRequest;
import software.amazon.awssdk.services.dataexchange.model.StartJobResponse;
import software.amazon.awssdk.services.dataexchange.model.TagResourceRequest;
import software.amazon.awssdk.services.dataexchange.model.TagResourceResponse;
import software.amazon.awssdk.services.dataexchange.model.ThrottlingException;
import software.amazon.awssdk.services.dataexchange.model.UntagResourceRequest;
import software.amazon.awssdk.services.dataexchange.model.UntagResourceResponse;
import software.amazon.awssdk.services.dataexchange.model.UpdateAssetRequest;
import software.amazon.awssdk.services.dataexchange.model.UpdateAssetResponse;
import software.amazon.awssdk.services.dataexchange.model.UpdateDataSetRequest;
import software.amazon.awssdk.services.dataexchange.model.UpdateDataSetResponse;
import software.amazon.awssdk.services.dataexchange.model.UpdateEventActionRequest;
import software.amazon.awssdk.services.dataexchange.model.UpdateEventActionResponse;
import software.amazon.awssdk.services.dataexchange.model.UpdateRevisionRequest;
import software.amazon.awssdk.services.dataexchange.model.UpdateRevisionResponse;
import software.amazon.awssdk.services.dataexchange.model.ValidationException;
import software.amazon.awssdk.services.dataexchange.transform.CancelJobRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.CreateDataSetRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.CreateEventActionRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.CreateJobRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.CreateRevisionRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.DeleteAssetRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.DeleteDataSetRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.DeleteEventActionRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.DeleteRevisionRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.GetAssetRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.GetDataSetRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.GetEventActionRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.GetJobRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.GetRevisionRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.ListDataSetRevisionsRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.ListDataSetsRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.ListEventActionsRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.ListJobsRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.ListRevisionAssetsRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.RevokeRevisionRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.SendDataSetNotificationRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.StartJobRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.UpdateAssetRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.UpdateDataSetRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.UpdateEventActionRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.UpdateRevisionRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link DataExchangeAsyncClient}.
 *
 * @see DataExchangeAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultDataExchangeAsyncClient implements DataExchangeAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultDataExchangeAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultDataExchangeAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * This operation cancels a job. Jobs can be cancelled only when they are in the WAITING state.
     * </p>
     *
     * @param cancelJobRequest
     * @return A Java Future containing the result of the CancelJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>ConflictException The request couldn't be completed because it conflicted with the current state of
     *         the resource.</li>
     *         <li>ValidationException The request was invalid.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataExchangeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataExchangeAsyncClient.CancelJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/CancelJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CancelJobResponse> cancelJob(CancelJobRequest cancelJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelJobRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CancelJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelJobRequest, CancelJobResponse>().withOperationName("CancelJob")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CancelJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(cancelJobRequest));
            CompletableFuture<CancelJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation creates a data set.
     * </p>
     *
     * @param createDataSetRequest
     * @return A Java Future containing the result of the CreateDataSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceLimitExceededException The request has exceeded the quotas imposed by the service.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>AccessDeniedException Access to the resource is denied.</li>
     *         <li>ValidationException The request was invalid.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataExchangeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataExchangeAsyncClient.CreateDataSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/CreateDataSet" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDataSetResponse> createDataSet(CreateDataSetRequest createDataSetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDataSetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDataSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDataSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateDataSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDataSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDataSetRequest, CreateDataSetResponse>()
                            .withOperationName("CreateDataSet").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateDataSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createDataSetRequest));
            CompletableFuture<CreateDataSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation creates an event action.
     * </p>
     *
     * @param createEventActionRequest
     * @return A Java Future containing the result of the CreateEventAction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceLimitExceededException The request has exceeded the quotas imposed by the service.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>AccessDeniedException Access to the resource is denied.</li>
     *         <li>ValidationException The request was invalid.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataExchangeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataExchangeAsyncClient.CreateEventAction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/CreateEventAction"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateEventActionResponse> createEventAction(CreateEventActionRequest createEventActionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createEventActionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createEventActionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateEventAction");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateEventActionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateEventActionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateEventActionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateEventActionRequest, CreateEventActionResponse>()
                            .withOperationName("CreateEventAction").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateEventActionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createEventActionRequest));
            CompletableFuture<CreateEventActionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation creates a job.
     * </p>
     *
     * @param createJobRequest
     * @return A Java Future containing the result of the CreateJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>AccessDeniedException Access to the resource is denied.</li>
     *         <li>ConflictException The request couldn't be completed because it conflicted with the current state of
     *         the resource.</li>
     *         <li>ValidationException The request was invalid.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataExchangeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataExchangeAsyncClient.CreateJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/CreateJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateJobResponse> createJob(CreateJobRequest createJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createJobRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateJobRequest, CreateJobResponse>().withOperationName("CreateJob")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(createJobRequest));
            CompletableFuture<CreateJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation creates a revision for a data set.
     * </p>
     *
     * @param createRevisionRequest
     * @return A Java Future containing the result of the CreateRevision operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>AccessDeniedException Access to the resource is denied.</li>
     *         <li>ValidationException The request was invalid.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataExchangeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataExchangeAsyncClient.CreateRevision
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/CreateRevision" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateRevisionResponse> createRevision(CreateRevisionRequest createRevisionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createRevisionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRevisionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRevision");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateRevisionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateRevisionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateRevisionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateRevisionRequest, CreateRevisionResponse>()
                            .withOperationName("CreateRevision").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateRevisionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createRevisionRequest));
            CompletableFuture<CreateRevisionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation deletes an asset.
     * </p>
     *
     * @param deleteAssetRequest
     * @return A Java Future containing the result of the DeleteAsset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>AccessDeniedException Access to the resource is denied.</li>
     *         <li>ConflictException The request couldn't be completed because it conflicted with the current state of
     *         the resource.</li>
     *         <li>ValidationException The request was invalid.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataExchangeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataExchangeAsyncClient.DeleteAsset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/DeleteAsset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAssetResponse> deleteAsset(DeleteAssetRequest deleteAssetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAssetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAssetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAsset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAssetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteAssetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAssetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAssetRequest, DeleteAssetResponse>()
                            .withOperationName("DeleteAsset").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteAssetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteAssetRequest));
            CompletableFuture<DeleteAssetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation deletes a data set.
     * </p>
     *
     * @param deleteDataSetRequest
     * @return A Java Future containing the result of the DeleteDataSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>AccessDeniedException Access to the resource is denied.</li>
     *         <li>ConflictException The request couldn't be completed because it conflicted with the current state of
     *         the resource.</li>
     *         <li>ValidationException The request was invalid.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataExchangeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataExchangeAsyncClient.DeleteDataSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/DeleteDataSet" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDataSetResponse> deleteDataSet(DeleteDataSetRequest deleteDataSetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDataSetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDataSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDataSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteDataSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDataSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDataSetRequest, DeleteDataSetResponse>()
                            .withOperationName("DeleteDataSet").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteDataSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteDataSetRequest));
            CompletableFuture<DeleteDataSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation deletes the event action.
     * </p>
     *
     * @param deleteEventActionRequest
     * @return A Java Future containing the result of the DeleteEventAction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>ValidationException The request was invalid.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataExchangeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataExchangeAsyncClient.DeleteEventAction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/DeleteEventAction"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteEventActionResponse> deleteEventAction(DeleteEventActionRequest deleteEventActionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteEventActionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEventActionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEventAction");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteEventActionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteEventActionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteEventActionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteEventActionRequest, DeleteEventActionResponse>()
                            .withOperationName("DeleteEventAction").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteEventActionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteEventActionRequest));
            CompletableFuture<DeleteEventActionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation deletes a revision.
     * </p>
     *
     * @param deleteRevisionRequest
     * @return A Java Future containing the result of the DeleteRevision operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>AccessDeniedException Access to the resource is denied.</li>
     *         <li>ConflictException The request couldn't be completed because it conflicted with the current state of
     *         the resource.</li>
     *         <li>ValidationException The request was invalid.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataExchangeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataExchangeAsyncClient.DeleteRevision
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/DeleteRevision" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRevisionResponse> deleteRevision(DeleteRevisionRequest deleteRevisionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRevisionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRevisionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRevision");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRevisionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteRevisionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRevisionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRevisionRequest, DeleteRevisionResponse>()
                            .withOperationName("DeleteRevision").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteRevisionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteRevisionRequest));
            CompletableFuture<DeleteRevisionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation returns information about an asset.
     * </p>
     *
     * @param getAssetRequest
     * @return A Java Future containing the result of the GetAsset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>ValidationException The request was invalid.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataExchangeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataExchangeAsyncClient.GetAsset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/GetAsset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetAssetResponse> getAsset(GetAssetRequest getAssetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAssetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAssetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAsset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAssetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetAssetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAssetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAssetRequest, GetAssetResponse>().withOperationName("GetAsset")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetAssetRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getAssetRequest));
            CompletableFuture<GetAssetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation returns information about a data set.
     * </p>
     *
     * @param getDataSetRequest
     * @return A Java Future containing the result of the GetDataSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>ValidationException The request was invalid.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataExchangeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataExchangeAsyncClient.GetDataSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/GetDataSet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetDataSetResponse> getDataSet(GetDataSetRequest getDataSetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDataSetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDataSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDataSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDataSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetDataSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDataSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDataSetRequest, GetDataSetResponse>().withOperationName("GetDataSet")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetDataSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDataSetRequest));
            CompletableFuture<GetDataSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation retrieves information about an event action.
     * </p>
     *
     * @param getEventActionRequest
     * @return A Java Future containing the result of the GetEventAction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>ValidationException The request was invalid.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataExchangeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataExchangeAsyncClient.GetEventAction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/GetEventAction" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEventActionResponse> getEventAction(GetEventActionRequest getEventActionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getEventActionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEventActionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEventAction");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEventActionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetEventActionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEventActionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEventActionRequest, GetEventActionResponse>()
                            .withOperationName("GetEventAction").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetEventActionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getEventActionRequest));
            CompletableFuture<GetEventActionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation returns information about a job.
     * </p>
     *
     * @param getJobRequest
     * @return A Java Future containing the result of the GetJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>ValidationException The request was invalid.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataExchangeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataExchangeAsyncClient.GetJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/GetJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetJobResponse> getJob(GetJobRequest getJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getJobRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetJobRequest, GetJobResponse>().withOperationName("GetJob")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new GetJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getJobRequest));
            CompletableFuture<GetJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation returns information about a revision.
     * </p>
     *
     * @param getRevisionRequest
     * @return A Java Future containing the result of the GetRevision operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>ValidationException The request was invalid.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataExchangeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataExchangeAsyncClient.GetRevision
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/GetRevision" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetRevisionResponse> getRevision(GetRevisionRequest getRevisionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRevisionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRevisionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRevision");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRevisionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetRevisionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRevisionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRevisionRequest, GetRevisionResponse>()
                            .withOperationName("GetRevision").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetRevisionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getRevisionRequest));
            CompletableFuture<GetRevisionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation lists a data set's revisions sorted by CreatedAt in descending order.
     * </p>
     *
     * @param listDataSetRevisionsRequest
     * @return A Java Future containing the result of the ListDataSetRevisions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>ValidationException The request was invalid.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataExchangeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataExchangeAsyncClient.ListDataSetRevisions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/ListDataSetRevisions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDataSetRevisionsResponse> listDataSetRevisions(
            ListDataSetRevisionsRequest listDataSetRevisionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDataSetRevisionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDataSetRevisionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDataSetRevisions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDataSetRevisionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDataSetRevisionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDataSetRevisionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDataSetRevisionsRequest, ListDataSetRevisionsResponse>()
                            .withOperationName("ListDataSetRevisions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDataSetRevisionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDataSetRevisionsRequest));
            CompletableFuture<ListDataSetRevisionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation lists your data sets. When listing by origin OWNED, results are sorted by CreatedAt in descending
     * order. When listing by origin ENTITLED, there is no order and the maxResults parameter is ignored.
     * </p>
     *
     * @param listDataSetsRequest
     * @return A Java Future containing the result of the ListDataSets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>ValidationException The request was invalid.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataExchangeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataExchangeAsyncClient.ListDataSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/ListDataSets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListDataSetsResponse> listDataSets(ListDataSetsRequest listDataSetsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDataSetsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDataSetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDataSets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDataSetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListDataSetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDataSetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDataSetsRequest, ListDataSetsResponse>()
                            .withOperationName("ListDataSets").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDataSetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDataSetsRequest));
            CompletableFuture<ListDataSetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation lists your event actions.
     * </p>
     *
     * @param listEventActionsRequest
     * @return A Java Future containing the result of the ListEventActions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>ValidationException The request was invalid.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataExchangeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataExchangeAsyncClient.ListEventActions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/ListEventActions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListEventActionsResponse> listEventActions(ListEventActionsRequest listEventActionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listEventActionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEventActionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEventActions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEventActionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEventActionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEventActionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEventActionsRequest, ListEventActionsResponse>()
                            .withOperationName("ListEventActions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListEventActionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listEventActionsRequest));
            CompletableFuture<ListEventActionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation lists your jobs sorted by CreatedAt in descending order.
     * </p>
     *
     * @param listJobsRequest
     * @return A Java Future containing the result of the ListJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>ValidationException The request was invalid.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataExchangeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataExchangeAsyncClient.ListJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListJobsResponse> listJobs(ListJobsRequest listJobsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listJobsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListJobsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListJobsRequest, ListJobsResponse>().withOperationName("ListJobs")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListJobsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(listJobsRequest));
            CompletableFuture<ListJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation lists a revision's assets sorted alphabetically in descending order.
     * </p>
     *
     * @param listRevisionAssetsRequest
     * @return A Java Future containing the result of the ListRevisionAssets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>ValidationException The request was invalid.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataExchangeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataExchangeAsyncClient.ListRevisionAssets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/ListRevisionAssets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRevisionAssetsResponse> listRevisionAssets(ListRevisionAssetsRequest listRevisionAssetsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRevisionAssetsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRevisionAssetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRevisionAssets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRevisionAssetsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRevisionAssetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRevisionAssetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRevisionAssetsRequest, ListRevisionAssetsResponse>()
                            .withOperationName("ListRevisionAssets").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListRevisionAssetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listRevisionAssetsRequest));
            CompletableFuture<ListRevisionAssetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation lists the tags on the resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataExchangeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataExchangeAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation revokes subscribers' access to a revision.
     * </p>
     *
     * @param revokeRevisionRequest
     * @return A Java Future containing the result of the RevokeRevision operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>AccessDeniedException Access to the resource is denied.</li>
     *         <li>ConflictException The request couldn't be completed because it conflicted with the current state of
     *         the resource.</li>
     *         <li>ValidationException The request was invalid.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataExchangeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataExchangeAsyncClient.RevokeRevision
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/RevokeRevision" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<RevokeRevisionResponse> revokeRevision(RevokeRevisionRequest revokeRevisionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(revokeRevisionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, revokeRevisionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RevokeRevision");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RevokeRevisionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RevokeRevisionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RevokeRevisionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RevokeRevisionRequest, RevokeRevisionResponse>()
                            .withOperationName("RevokeRevision").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RevokeRevisionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(revokeRevisionRequest));
            CompletableFuture<RevokeRevisionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The type of event associated with the data set.
     * </p>
     *
     * @param sendDataSetNotificationRequest
     * @return A Java Future containing the result of the SendDataSetNotification operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>AccessDeniedException Access to the resource is denied.</li>
     *         <li>ConflictException The request couldn't be completed because it conflicted with the current state of
     *         the resource.</li>
     *         <li>ValidationException The request was invalid.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataExchangeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataExchangeAsyncClient.SendDataSetNotification
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/SendDataSetNotification"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SendDataSetNotificationResponse> sendDataSetNotification(
            SendDataSetNotificationRequest sendDataSetNotificationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(sendDataSetNotificationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendDataSetNotificationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendDataSetNotification");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SendDataSetNotificationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SendDataSetNotificationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SendDataSetNotificationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SendDataSetNotificationRequest, SendDataSetNotificationResponse>()
                            .withOperationName("SendDataSetNotification").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SendDataSetNotificationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(sendDataSetNotificationRequest));
            CompletableFuture<SendDataSetNotificationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation starts a job.
     * </p>
     *
     * @param startJobRequest
     * @return A Java Future containing the result of the StartJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>AccessDeniedException Access to the resource is denied.</li>
     *         <li>ConflictException The request couldn't be completed because it conflicted with the current state of
     *         the resource.</li>
     *         <li>ValidationException The request was invalid.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataExchangeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataExchangeAsyncClient.StartJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/StartJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartJobResponse> startJob(StartJobRequest startJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startJobRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartJobRequest, StartJobResponse>().withOperationName("StartJob")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(startJobRequest));
            CompletableFuture<StartJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation tags a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataExchangeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataExchangeAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation removes one or more tags from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataExchangeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataExchangeAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation updates an asset.
     * </p>
     *
     * @param updateAssetRequest
     * @return A Java Future containing the result of the UpdateAsset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>AccessDeniedException Access to the resource is denied.</li>
     *         <li>ConflictException The request couldn't be completed because it conflicted with the current state of
     *         the resource.</li>
     *         <li>ValidationException The request was invalid.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataExchangeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataExchangeAsyncClient.UpdateAsset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/UpdateAsset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAssetResponse> updateAsset(UpdateAssetRequest updateAssetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateAssetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAssetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAsset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAssetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateAssetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAssetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAssetRequest, UpdateAssetResponse>()
                            .withOperationName("UpdateAsset").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateAssetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateAssetRequest));
            CompletableFuture<UpdateAssetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation updates a data set.
     * </p>
     *
     * @param updateDataSetRequest
     * @return A Java Future containing the result of the UpdateDataSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>AccessDeniedException Access to the resource is denied.</li>
     *         <li>ValidationException The request was invalid.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataExchangeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataExchangeAsyncClient.UpdateDataSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/UpdateDataSet" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDataSetResponse> updateDataSet(UpdateDataSetRequest updateDataSetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDataSetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDataSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDataSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDataSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateDataSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDataSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDataSetRequest, UpdateDataSetResponse>()
                            .withOperationName("UpdateDataSet").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateDataSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateDataSetRequest));
            CompletableFuture<UpdateDataSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation updates the event action.
     * </p>
     *
     * @param updateEventActionRequest
     * @return A Java Future containing the result of the UpdateEventAction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>AccessDeniedException Access to the resource is denied.</li>
     *         <li>ValidationException The request was invalid.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataExchangeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataExchangeAsyncClient.UpdateEventAction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/UpdateEventAction"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateEventActionResponse> updateEventAction(UpdateEventActionRequest updateEventActionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateEventActionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateEventActionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEventAction");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateEventActionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateEventActionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateEventActionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateEventActionRequest, UpdateEventActionResponse>()
                            .withOperationName("UpdateEventAction").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateEventActionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateEventActionRequest));
            CompletableFuture<UpdateEventActionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation updates a revision.
     * </p>
     *
     * @param updateRevisionRequest
     * @return A Java Future containing the result of the UpdateRevision operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>AccessDeniedException Access to the resource is denied.</li>
     *         <li>ConflictException The request couldn't be completed because it conflicted with the current state of
     *         the resource.</li>
     *         <li>ValidationException The request was invalid.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataExchangeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataExchangeAsyncClient.UpdateRevision
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/UpdateRevision" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateRevisionResponse> updateRevision(UpdateRevisionRequest updateRevisionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateRevisionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRevisionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRevision");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateRevisionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateRevisionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateRevisionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateRevisionRequest, UpdateRevisionResponse>()
                            .withOperationName("UpdateRevision").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateRevisionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateRevisionRequest));
            CompletableFuture<UpdateRevisionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final DataExchangeServiceClientConfiguration serviceClientConfiguration() {
        return new DataExchangeServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(DataExchangeException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceLimitExceededException")
                                .exceptionBuilderSupplier(ServiceLimitExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        DataExchangeServiceClientConfigurationBuilder serviceConfigBuilder = new DataExchangeServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
