/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codepipeline;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.codepipeline.internal.CodePipelineServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.codepipeline.model.AcknowledgeJobRequest;
import software.amazon.awssdk.services.codepipeline.model.AcknowledgeJobResponse;
import software.amazon.awssdk.services.codepipeline.model.AcknowledgeThirdPartyJobRequest;
import software.amazon.awssdk.services.codepipeline.model.AcknowledgeThirdPartyJobResponse;
import software.amazon.awssdk.services.codepipeline.model.ActionNotFoundException;
import software.amazon.awssdk.services.codepipeline.model.ActionTypeNotFoundException;
import software.amazon.awssdk.services.codepipeline.model.ApprovalAlreadyCompletedException;
import software.amazon.awssdk.services.codepipeline.model.CodePipelineException;
import software.amazon.awssdk.services.codepipeline.model.ConcurrentModificationException;
import software.amazon.awssdk.services.codepipeline.model.ConflictException;
import software.amazon.awssdk.services.codepipeline.model.CreateCustomActionTypeRequest;
import software.amazon.awssdk.services.codepipeline.model.CreateCustomActionTypeResponse;
import software.amazon.awssdk.services.codepipeline.model.CreatePipelineRequest;
import software.amazon.awssdk.services.codepipeline.model.CreatePipelineResponse;
import software.amazon.awssdk.services.codepipeline.model.DeleteCustomActionTypeRequest;
import software.amazon.awssdk.services.codepipeline.model.DeleteCustomActionTypeResponse;
import software.amazon.awssdk.services.codepipeline.model.DeletePipelineRequest;
import software.amazon.awssdk.services.codepipeline.model.DeletePipelineResponse;
import software.amazon.awssdk.services.codepipeline.model.DeleteWebhookRequest;
import software.amazon.awssdk.services.codepipeline.model.DeleteWebhookResponse;
import software.amazon.awssdk.services.codepipeline.model.DeregisterWebhookWithThirdPartyRequest;
import software.amazon.awssdk.services.codepipeline.model.DeregisterWebhookWithThirdPartyResponse;
import software.amazon.awssdk.services.codepipeline.model.DisableStageTransitionRequest;
import software.amazon.awssdk.services.codepipeline.model.DisableStageTransitionResponse;
import software.amazon.awssdk.services.codepipeline.model.DuplicatedStopRequestException;
import software.amazon.awssdk.services.codepipeline.model.EnableStageTransitionRequest;
import software.amazon.awssdk.services.codepipeline.model.EnableStageTransitionResponse;
import software.amazon.awssdk.services.codepipeline.model.GetActionTypeRequest;
import software.amazon.awssdk.services.codepipeline.model.GetActionTypeResponse;
import software.amazon.awssdk.services.codepipeline.model.GetJobDetailsRequest;
import software.amazon.awssdk.services.codepipeline.model.GetJobDetailsResponse;
import software.amazon.awssdk.services.codepipeline.model.GetPipelineExecutionRequest;
import software.amazon.awssdk.services.codepipeline.model.GetPipelineExecutionResponse;
import software.amazon.awssdk.services.codepipeline.model.GetPipelineRequest;
import software.amazon.awssdk.services.codepipeline.model.GetPipelineResponse;
import software.amazon.awssdk.services.codepipeline.model.GetPipelineStateRequest;
import software.amazon.awssdk.services.codepipeline.model.GetPipelineStateResponse;
import software.amazon.awssdk.services.codepipeline.model.GetThirdPartyJobDetailsRequest;
import software.amazon.awssdk.services.codepipeline.model.GetThirdPartyJobDetailsResponse;
import software.amazon.awssdk.services.codepipeline.model.InvalidActionDeclarationException;
import software.amazon.awssdk.services.codepipeline.model.InvalidApprovalTokenException;
import software.amazon.awssdk.services.codepipeline.model.InvalidArnException;
import software.amazon.awssdk.services.codepipeline.model.InvalidBlockerDeclarationException;
import software.amazon.awssdk.services.codepipeline.model.InvalidClientTokenException;
import software.amazon.awssdk.services.codepipeline.model.InvalidJobException;
import software.amazon.awssdk.services.codepipeline.model.InvalidJobStateException;
import software.amazon.awssdk.services.codepipeline.model.InvalidNextTokenException;
import software.amazon.awssdk.services.codepipeline.model.InvalidNonceException;
import software.amazon.awssdk.services.codepipeline.model.InvalidStageDeclarationException;
import software.amazon.awssdk.services.codepipeline.model.InvalidStructureException;
import software.amazon.awssdk.services.codepipeline.model.InvalidTagsException;
import software.amazon.awssdk.services.codepipeline.model.InvalidWebhookAuthenticationParametersException;
import software.amazon.awssdk.services.codepipeline.model.InvalidWebhookFilterPatternException;
import software.amazon.awssdk.services.codepipeline.model.JobNotFoundException;
import software.amazon.awssdk.services.codepipeline.model.LimitExceededException;
import software.amazon.awssdk.services.codepipeline.model.ListActionExecutionsRequest;
import software.amazon.awssdk.services.codepipeline.model.ListActionExecutionsResponse;
import software.amazon.awssdk.services.codepipeline.model.ListActionTypesRequest;
import software.amazon.awssdk.services.codepipeline.model.ListActionTypesResponse;
import software.amazon.awssdk.services.codepipeline.model.ListPipelineExecutionsRequest;
import software.amazon.awssdk.services.codepipeline.model.ListPipelineExecutionsResponse;
import software.amazon.awssdk.services.codepipeline.model.ListPipelinesRequest;
import software.amazon.awssdk.services.codepipeline.model.ListPipelinesResponse;
import software.amazon.awssdk.services.codepipeline.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.codepipeline.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.codepipeline.model.ListWebhooksRequest;
import software.amazon.awssdk.services.codepipeline.model.ListWebhooksResponse;
import software.amazon.awssdk.services.codepipeline.model.NotLatestPipelineExecutionException;
import software.amazon.awssdk.services.codepipeline.model.OutputVariablesSizeExceededException;
import software.amazon.awssdk.services.codepipeline.model.PipelineExecutionNotFoundException;
import software.amazon.awssdk.services.codepipeline.model.PipelineExecutionNotStoppableException;
import software.amazon.awssdk.services.codepipeline.model.PipelineNameInUseException;
import software.amazon.awssdk.services.codepipeline.model.PipelineNotFoundException;
import software.amazon.awssdk.services.codepipeline.model.PipelineVersionNotFoundException;
import software.amazon.awssdk.services.codepipeline.model.PollForJobsRequest;
import software.amazon.awssdk.services.codepipeline.model.PollForJobsResponse;
import software.amazon.awssdk.services.codepipeline.model.PollForThirdPartyJobsRequest;
import software.amazon.awssdk.services.codepipeline.model.PollForThirdPartyJobsResponse;
import software.amazon.awssdk.services.codepipeline.model.PutActionRevisionRequest;
import software.amazon.awssdk.services.codepipeline.model.PutActionRevisionResponse;
import software.amazon.awssdk.services.codepipeline.model.PutApprovalResultRequest;
import software.amazon.awssdk.services.codepipeline.model.PutApprovalResultResponse;
import software.amazon.awssdk.services.codepipeline.model.PutJobFailureResultRequest;
import software.amazon.awssdk.services.codepipeline.model.PutJobFailureResultResponse;
import software.amazon.awssdk.services.codepipeline.model.PutJobSuccessResultRequest;
import software.amazon.awssdk.services.codepipeline.model.PutJobSuccessResultResponse;
import software.amazon.awssdk.services.codepipeline.model.PutThirdPartyJobFailureResultRequest;
import software.amazon.awssdk.services.codepipeline.model.PutThirdPartyJobFailureResultResponse;
import software.amazon.awssdk.services.codepipeline.model.PutThirdPartyJobSuccessResultRequest;
import software.amazon.awssdk.services.codepipeline.model.PutThirdPartyJobSuccessResultResponse;
import software.amazon.awssdk.services.codepipeline.model.PutWebhookRequest;
import software.amazon.awssdk.services.codepipeline.model.PutWebhookResponse;
import software.amazon.awssdk.services.codepipeline.model.RegisterWebhookWithThirdPartyRequest;
import software.amazon.awssdk.services.codepipeline.model.RegisterWebhookWithThirdPartyResponse;
import software.amazon.awssdk.services.codepipeline.model.RequestFailedException;
import software.amazon.awssdk.services.codepipeline.model.ResourceNotFoundException;
import software.amazon.awssdk.services.codepipeline.model.RetryStageExecutionRequest;
import software.amazon.awssdk.services.codepipeline.model.RetryStageExecutionResponse;
import software.amazon.awssdk.services.codepipeline.model.StageNotFoundException;
import software.amazon.awssdk.services.codepipeline.model.StageNotRetryableException;
import software.amazon.awssdk.services.codepipeline.model.StartPipelineExecutionRequest;
import software.amazon.awssdk.services.codepipeline.model.StartPipelineExecutionResponse;
import software.amazon.awssdk.services.codepipeline.model.StopPipelineExecutionRequest;
import software.amazon.awssdk.services.codepipeline.model.StopPipelineExecutionResponse;
import software.amazon.awssdk.services.codepipeline.model.TagResourceRequest;
import software.amazon.awssdk.services.codepipeline.model.TagResourceResponse;
import software.amazon.awssdk.services.codepipeline.model.TooManyTagsException;
import software.amazon.awssdk.services.codepipeline.model.UntagResourceRequest;
import software.amazon.awssdk.services.codepipeline.model.UntagResourceResponse;
import software.amazon.awssdk.services.codepipeline.model.UpdateActionTypeRequest;
import software.amazon.awssdk.services.codepipeline.model.UpdateActionTypeResponse;
import software.amazon.awssdk.services.codepipeline.model.UpdatePipelineRequest;
import software.amazon.awssdk.services.codepipeline.model.UpdatePipelineResponse;
import software.amazon.awssdk.services.codepipeline.model.ValidationException;
import software.amazon.awssdk.services.codepipeline.model.WebhookNotFoundException;
import software.amazon.awssdk.services.codepipeline.transform.AcknowledgeJobRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.AcknowledgeThirdPartyJobRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.CreateCustomActionTypeRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.CreatePipelineRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.DeleteCustomActionTypeRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.DeletePipelineRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.DeleteWebhookRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.DeregisterWebhookWithThirdPartyRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.DisableStageTransitionRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.EnableStageTransitionRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.GetActionTypeRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.GetJobDetailsRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.GetPipelineExecutionRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.GetPipelineRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.GetPipelineStateRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.GetThirdPartyJobDetailsRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.ListActionExecutionsRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.ListActionTypesRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.ListPipelineExecutionsRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.ListPipelinesRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.ListWebhooksRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.PollForJobsRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.PollForThirdPartyJobsRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.PutActionRevisionRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.PutApprovalResultRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.PutJobFailureResultRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.PutJobSuccessResultRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.PutThirdPartyJobFailureResultRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.PutThirdPartyJobSuccessResultRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.PutWebhookRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.RegisterWebhookWithThirdPartyRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.RetryStageExecutionRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.StartPipelineExecutionRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.StopPipelineExecutionRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.UpdateActionTypeRequestMarshaller;
import software.amazon.awssdk.services.codepipeline.transform.UpdatePipelineRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link CodePipelineAsyncClient}.
 *
 * @see CodePipelineAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCodePipelineAsyncClient implements CodePipelineAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultCodePipelineAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final CodePipelineServiceClientConfiguration serviceClientConfiguration;

    protected DefaultCodePipelineAsyncClient(CodePipelineServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Returns information about a specified job and whether that job has been received by the job worker. Used for
     * custom actions only.
     * </p>
     *
     * @param acknowledgeJobRequest
     *        Represents the input of an AcknowledgeJob action.
     * @return A Java Future containing the result of the AcknowledgeJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>InvalidNonceException The nonce was specified in an invalid format.</li>
     *         <li>JobNotFoundException The job was specified in an invalid format or cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.AcknowledgeJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/AcknowledgeJob" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<AcknowledgeJobResponse> acknowledgeJob(AcknowledgeJobRequest acknowledgeJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(acknowledgeJobRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, acknowledgeJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcknowledgeJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AcknowledgeJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AcknowledgeJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AcknowledgeJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AcknowledgeJobRequest, AcknowledgeJobResponse>()
                            .withOperationName("AcknowledgeJob")
                            .withMarshaller(new AcknowledgeJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(acknowledgeJobRequest));
            CompletableFuture<AcknowledgeJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Confirms a job worker has received the specified job. Used for partner actions only.
     * </p>
     *
     * @param acknowledgeThirdPartyJobRequest
     *        Represents the input of an AcknowledgeThirdPartyJob action.
     * @return A Java Future containing the result of the AcknowledgeThirdPartyJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>InvalidNonceException The nonce was specified in an invalid format.</li>
     *         <li>JobNotFoundException The job was specified in an invalid format or cannot be found.</li>
     *         <li>InvalidClientTokenException The client token was specified in an invalid format</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.AcknowledgeThirdPartyJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/AcknowledgeThirdPartyJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AcknowledgeThirdPartyJobResponse> acknowledgeThirdPartyJob(
            AcknowledgeThirdPartyJobRequest acknowledgeThirdPartyJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(acknowledgeThirdPartyJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, acknowledgeThirdPartyJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcknowledgeThirdPartyJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AcknowledgeThirdPartyJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AcknowledgeThirdPartyJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AcknowledgeThirdPartyJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AcknowledgeThirdPartyJobRequest, AcknowledgeThirdPartyJobResponse>()
                            .withOperationName("AcknowledgeThirdPartyJob")
                            .withMarshaller(new AcknowledgeThirdPartyJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(acknowledgeThirdPartyJobRequest));
            CompletableFuture<AcknowledgeThirdPartyJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new custom action that can be used in all pipelines associated with the Amazon Web Services account.
     * Only used for custom actions.
     * </p>
     *
     * @param createCustomActionTypeRequest
     *        Represents the input of a CreateCustomActionType operation.
     * @return A Java Future containing the result of the CreateCustomActionType operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>LimitExceededException The number of pipelines associated with the Amazon Web Services account has
     *         exceeded the limit allowed for the account.</li>
     *         <li>TooManyTagsException The tags limit for a resource has been exceeded.</li>
     *         <li>InvalidTagsException The specified resource tags are invalid.</li>
     *         <li>ConcurrentModificationException Unable to modify the tag due to a simultaneous update request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.CreateCustomActionType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/CreateCustomActionType"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateCustomActionTypeResponse> createCustomActionType(
            CreateCustomActionTypeRequest createCustomActionTypeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createCustomActionTypeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCustomActionTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCustomActionType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateCustomActionTypeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateCustomActionTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateCustomActionTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateCustomActionTypeRequest, CreateCustomActionTypeResponse>()
                            .withOperationName("CreateCustomActionType")
                            .withMarshaller(new CreateCustomActionTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createCustomActionTypeRequest));
            CompletableFuture<CreateCustomActionTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a pipeline.
     * </p>
     * <note>
     * <p>
     * In the pipeline structure, you must include either <code>artifactStore</code> or <code>artifactStores</code> in
     * your pipeline, but you cannot use both. If you create a cross-region action in your pipeline, you must use
     * <code>artifactStores</code>.
     * </p>
     * </note>
     *
     * @param createPipelineRequest
     *        Represents the input of a <code>CreatePipeline</code> action.
     * @return A Java Future containing the result of the CreatePipeline operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>PipelineNameInUseException The specified pipeline name is already in use.</li>
     *         <li>InvalidStageDeclarationException The stage declaration was specified in an invalid format.</li>
     *         <li>InvalidActionDeclarationException The action declaration was specified in an invalid format.</li>
     *         <li>InvalidBlockerDeclarationException Reserved for future use.</li>
     *         <li>InvalidStructureException The structure was specified in an invalid format.</li>
     *         <li>LimitExceededException The number of pipelines associated with the Amazon Web Services account has
     *         exceeded the limit allowed for the account.</li>
     *         <li>TooManyTagsException The tags limit for a resource has been exceeded.</li>
     *         <li>InvalidTagsException The specified resource tags are invalid.</li>
     *         <li>ConcurrentModificationException Unable to modify the tag due to a simultaneous update request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.CreatePipeline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/CreatePipeline" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePipelineResponse> createPipeline(CreatePipelineRequest createPipelineRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPipelineRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePipeline");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePipelineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePipelineRequest, CreatePipelineResponse>()
                            .withOperationName("CreatePipeline")
                            .withMarshaller(new CreatePipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createPipelineRequest));
            CompletableFuture<CreatePipelineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Marks a custom action as deleted. <code>PollForJobs</code> for the custom action fails after the action is marked
     * for deletion. Used for custom actions only.
     * </p>
     * <important>
     * <p>
     * To re-create a custom action after it has been deleted you must use a string in the version field that has never
     * been used before. This string can be an incremented version number, for example. To restore a deleted custom
     * action, use a JSON file that is identical to the deleted action, including the original string in the version
     * field.
     * </p>
     * </important>
     *
     * @param deleteCustomActionTypeRequest
     *        Represents the input of a <code>DeleteCustomActionType</code> operation. The custom action will be marked
     *        as deleted.
     * @return A Java Future containing the result of the DeleteCustomActionType operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>ConcurrentModificationException Unable to modify the tag due to a simultaneous update request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.DeleteCustomActionType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/DeleteCustomActionType"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteCustomActionTypeResponse> deleteCustomActionType(
            DeleteCustomActionTypeRequest deleteCustomActionTypeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteCustomActionTypeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCustomActionTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCustomActionType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteCustomActionTypeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteCustomActionTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteCustomActionTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteCustomActionTypeRequest, DeleteCustomActionTypeResponse>()
                            .withOperationName("DeleteCustomActionType")
                            .withMarshaller(new DeleteCustomActionTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteCustomActionTypeRequest));
            CompletableFuture<DeleteCustomActionTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified pipeline.
     * </p>
     *
     * @param deletePipelineRequest
     *        Represents the input of a <code>DeletePipeline</code> action.
     * @return A Java Future containing the result of the DeletePipeline operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>ConcurrentModificationException Unable to modify the tag due to a simultaneous update request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.DeletePipeline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/DeletePipeline" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePipelineResponse> deletePipeline(DeletePipelineRequest deletePipelineRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePipelineRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePipeline");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePipelineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePipelineRequest, DeletePipelineResponse>()
                            .withOperationName("DeletePipeline")
                            .withMarshaller(new DeletePipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deletePipelineRequest));
            CompletableFuture<DeletePipelineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a previously created webhook by name. Deleting the webhook stops CodePipeline from starting a pipeline
     * every time an external event occurs. The API returns successfully when trying to delete a webhook that is already
     * deleted. If a deleted webhook is re-created by calling PutWebhook with the same name, it will have a different
     * URL.
     * </p>
     *
     * @param deleteWebhookRequest
     * @return A Java Future containing the result of the DeleteWebhook operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>ConcurrentModificationException Unable to modify the tag due to a simultaneous update request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.DeleteWebhook
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/DeleteWebhook" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteWebhookResponse> deleteWebhook(DeleteWebhookRequest deleteWebhookRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteWebhookRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWebhookRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWebhook");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteWebhookResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteWebhookResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteWebhookResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteWebhookRequest, DeleteWebhookResponse>()
                            .withOperationName("DeleteWebhook")
                            .withMarshaller(new DeleteWebhookRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteWebhookRequest));
            CompletableFuture<DeleteWebhookResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the connection between the webhook that was created by CodePipeline and the external tool with events to
     * be detected. Currently supported only for webhooks that target an action type of GitHub.
     * </p>
     *
     * @param deregisterWebhookWithThirdPartyRequest
     * @return A Java Future containing the result of the DeregisterWebhookWithThirdParty operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>WebhookNotFoundException The specified webhook was entered in an invalid format or cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.DeregisterWebhookWithThirdParty
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/DeregisterWebhookWithThirdParty"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeregisterWebhookWithThirdPartyResponse> deregisterWebhookWithThirdParty(
            DeregisterWebhookWithThirdPartyRequest deregisterWebhookWithThirdPartyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deregisterWebhookWithThirdPartyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deregisterWebhookWithThirdPartyRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeregisterWebhookWithThirdParty");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeregisterWebhookWithThirdPartyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeregisterWebhookWithThirdPartyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeregisterWebhookWithThirdPartyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeregisterWebhookWithThirdPartyRequest, DeregisterWebhookWithThirdPartyResponse>()
                            .withOperationName("DeregisterWebhookWithThirdParty")
                            .withMarshaller(new DeregisterWebhookWithThirdPartyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deregisterWebhookWithThirdPartyRequest));
            CompletableFuture<DeregisterWebhookWithThirdPartyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Prevents artifacts in a pipeline from transitioning to the next stage in the pipeline.
     * </p>
     *
     * @param disableStageTransitionRequest
     *        Represents the input of a <code>DisableStageTransition</code> action.
     * @return A Java Future containing the result of the DisableStageTransition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>PipelineNotFoundException The pipeline was specified in an invalid format or cannot be found.</li>
     *         <li>StageNotFoundException The stage was specified in an invalid format or cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.DisableStageTransition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/DisableStageTransition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisableStageTransitionResponse> disableStageTransition(
            DisableStageTransitionRequest disableStageTransitionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disableStageTransitionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disableStageTransitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableStageTransition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisableStageTransitionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisableStageTransitionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisableStageTransitionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisableStageTransitionRequest, DisableStageTransitionResponse>()
                            .withOperationName("DisableStageTransition")
                            .withMarshaller(new DisableStageTransitionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disableStageTransitionRequest));
            CompletableFuture<DisableStageTransitionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables artifacts in a pipeline to transition to a stage in a pipeline.
     * </p>
     *
     * @param enableStageTransitionRequest
     *        Represents the input of an <code>EnableStageTransition</code> action.
     * @return A Java Future containing the result of the EnableStageTransition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>PipelineNotFoundException The pipeline was specified in an invalid format or cannot be found.</li>
     *         <li>StageNotFoundException The stage was specified in an invalid format or cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.EnableStageTransition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/EnableStageTransition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<EnableStageTransitionResponse> enableStageTransition(
            EnableStageTransitionRequest enableStageTransitionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(enableStageTransitionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enableStageTransitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableStageTransition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EnableStageTransitionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, EnableStageTransitionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<EnableStageTransitionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EnableStageTransitionRequest, EnableStageTransitionResponse>()
                            .withOperationName("EnableStageTransition")
                            .withMarshaller(new EnableStageTransitionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(enableStageTransitionRequest));
            CompletableFuture<EnableStageTransitionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about an action type created for an external provider, where the action is to be used by
     * customers of the external provider. The action can be created with any supported integration model.
     * </p>
     *
     * @param getActionTypeRequest
     * @return A Java Future containing the result of the GetActionType operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ActionTypeNotFoundException The specified action type cannot be found.</li>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.GetActionType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/GetActionType" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetActionTypeResponse> getActionType(GetActionTypeRequest getActionTypeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getActionTypeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getActionTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetActionType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetActionTypeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetActionTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetActionTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetActionTypeRequest, GetActionTypeResponse>()
                            .withOperationName("GetActionType")
                            .withMarshaller(new GetActionTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getActionTypeRequest));
            CompletableFuture<GetActionTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a job. Used for custom actions only.
     * </p>
     * <important>
     * <p>
     * When this API is called, CodePipeline returns temporary credentials for the S3 bucket used to store artifacts for
     * the pipeline, if the action requires access to that S3 bucket for input or output artifacts. This API also
     * returns any secret values defined for the action.
     * </p>
     * </important>
     *
     * @param getJobDetailsRequest
     *        Represents the input of a <code>GetJobDetails</code> action.
     * @return A Java Future containing the result of the GetJobDetails operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>JobNotFoundException The job was specified in an invalid format or cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.GetJobDetails
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/GetJobDetails" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetJobDetailsResponse> getJobDetails(GetJobDetailsRequest getJobDetailsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getJobDetailsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getJobDetailsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetJobDetails");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetJobDetailsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetJobDetailsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetJobDetailsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetJobDetailsRequest, GetJobDetailsResponse>()
                            .withOperationName("GetJobDetails")
                            .withMarshaller(new GetJobDetailsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getJobDetailsRequest));
            CompletableFuture<GetJobDetailsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the metadata, structure, stages, and actions of a pipeline. Can be used to return the entire structure of
     * a pipeline in JSON format, which can then be modified and used to update the pipeline structure with
     * <a>UpdatePipeline</a>.
     * </p>
     *
     * @param getPipelineRequest
     *        Represents the input of a <code>GetPipeline</code> action.
     * @return A Java Future containing the result of the GetPipeline operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>PipelineNotFoundException The pipeline was specified in an invalid format or cannot be found.</li>
     *         <li>PipelineVersionNotFoundException The pipeline version was specified in an invalid format or cannot be
     *         found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.GetPipeline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/GetPipeline" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetPipelineResponse> getPipeline(GetPipelineRequest getPipelineRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPipelineRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPipeline");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPipelineResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetPipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPipelineRequest, GetPipelineResponse>()
                            .withOperationName("GetPipeline").withMarshaller(new GetPipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPipelineRequest));
            CompletableFuture<GetPipelineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about an execution of a pipeline, including details about artifacts, the pipeline execution
     * ID, and the name, version, and status of the pipeline.
     * </p>
     *
     * @param getPipelineExecutionRequest
     *        Represents the input of a <code>GetPipelineExecution</code> action.
     * @return A Java Future containing the result of the GetPipelineExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>PipelineNotFoundException The pipeline was specified in an invalid format or cannot be found.</li>
     *         <li>PipelineExecutionNotFoundException The pipeline execution was specified in an invalid format or
     *         cannot be found, or an execution ID does not belong to the specified pipeline.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.GetPipelineExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/GetPipelineExecution"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPipelineExecutionResponse> getPipelineExecution(
            GetPipelineExecutionRequest getPipelineExecutionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPipelineExecutionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPipelineExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPipelineExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPipelineExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPipelineExecutionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPipelineExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPipelineExecutionRequest, GetPipelineExecutionResponse>()
                            .withOperationName("GetPipelineExecution")
                            .withMarshaller(new GetPipelineExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPipelineExecutionRequest));
            CompletableFuture<GetPipelineExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the state of a pipeline, including the stages and actions.
     * </p>
     * <note>
     * <p>
     * Values returned in the <code>revisionId</code> and <code>revisionUrl</code> fields indicate the source revision
     * information, such as the commit ID, for the current state.
     * </p>
     * </note>
     *
     * @param getPipelineStateRequest
     *        Represents the input of a <code>GetPipelineState</code> action.
     * @return A Java Future containing the result of the GetPipelineState operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>PipelineNotFoundException The pipeline was specified in an invalid format or cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.GetPipelineState
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/GetPipelineState" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPipelineStateResponse> getPipelineState(GetPipelineStateRequest getPipelineStateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPipelineStateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPipelineStateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPipelineState");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPipelineStateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPipelineStateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPipelineStateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPipelineStateRequest, GetPipelineStateResponse>()
                            .withOperationName("GetPipelineState")
                            .withMarshaller(new GetPipelineStateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPipelineStateRequest));
            CompletableFuture<GetPipelineStateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Requests the details of a job for a third party action. Used for partner actions only.
     * </p>
     * <important>
     * <p>
     * When this API is called, CodePipeline returns temporary credentials for the S3 bucket used to store artifacts for
     * the pipeline, if the action requires access to that S3 bucket for input or output artifacts. This API also
     * returns any secret values defined for the action.
     * </p>
     * </important>
     *
     * @param getThirdPartyJobDetailsRequest
     *        Represents the input of a <code>GetThirdPartyJobDetails</code> action.
     * @return A Java Future containing the result of the GetThirdPartyJobDetails operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>JobNotFoundException The job was specified in an invalid format or cannot be found.</li>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>InvalidClientTokenException The client token was specified in an invalid format</li>
     *         <li>InvalidJobException The job was specified in an invalid format or cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.GetThirdPartyJobDetails
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/GetThirdPartyJobDetails"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetThirdPartyJobDetailsResponse> getThirdPartyJobDetails(
            GetThirdPartyJobDetailsRequest getThirdPartyJobDetailsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getThirdPartyJobDetailsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getThirdPartyJobDetailsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetThirdPartyJobDetails");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetThirdPartyJobDetailsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetThirdPartyJobDetailsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetThirdPartyJobDetailsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetThirdPartyJobDetailsRequest, GetThirdPartyJobDetailsResponse>()
                            .withOperationName("GetThirdPartyJobDetails")
                            .withMarshaller(new GetThirdPartyJobDetailsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getThirdPartyJobDetailsRequest));
            CompletableFuture<GetThirdPartyJobDetailsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the action executions that have occurred in a pipeline.
     * </p>
     *
     * @param listActionExecutionsRequest
     * @return A Java Future containing the result of the ListActionExecutions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>PipelineNotFoundException The pipeline was specified in an invalid format or cannot be found.</li>
     *         <li>InvalidNextTokenException The next token was specified in an invalid format. Make sure that the next
     *         token you provide is the token returned by a previous call.</li>
     *         <li>PipelineExecutionNotFoundException The pipeline execution was specified in an invalid format or
     *         cannot be found, or an execution ID does not belong to the specified pipeline.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.ListActionExecutions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/ListActionExecutions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListActionExecutionsResponse> listActionExecutions(
            ListActionExecutionsRequest listActionExecutionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listActionExecutionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listActionExecutionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListActionExecutions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListActionExecutionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListActionExecutionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListActionExecutionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListActionExecutionsRequest, ListActionExecutionsResponse>()
                            .withOperationName("ListActionExecutions")
                            .withMarshaller(new ListActionExecutionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listActionExecutionsRequest));
            CompletableFuture<ListActionExecutionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a summary of all CodePipeline action types associated with your account.
     * </p>
     *
     * @param listActionTypesRequest
     *        Represents the input of a <code>ListActionTypes</code> action.
     * @return A Java Future containing the result of the ListActionTypes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>InvalidNextTokenException The next token was specified in an invalid format. Make sure that the next
     *         token you provide is the token returned by a previous call.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.ListActionTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/ListActionTypes" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListActionTypesResponse> listActionTypes(ListActionTypesRequest listActionTypesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listActionTypesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listActionTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListActionTypes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListActionTypesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListActionTypesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListActionTypesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListActionTypesRequest, ListActionTypesResponse>()
                            .withOperationName("ListActionTypes")
                            .withMarshaller(new ListActionTypesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listActionTypesRequest));
            CompletableFuture<ListActionTypesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a summary of the most recent executions for a pipeline.
     * </p>
     *
     * @param listPipelineExecutionsRequest
     *        Represents the input of a <code>ListPipelineExecutions</code> action.
     * @return A Java Future containing the result of the ListPipelineExecutions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>PipelineNotFoundException The pipeline was specified in an invalid format or cannot be found.</li>
     *         <li>InvalidNextTokenException The next token was specified in an invalid format. Make sure that the next
     *         token you provide is the token returned by a previous call.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.ListPipelineExecutions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/ListPipelineExecutions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPipelineExecutionsResponse> listPipelineExecutions(
            ListPipelineExecutionsRequest listPipelineExecutionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPipelineExecutionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPipelineExecutionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPipelineExecutions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPipelineExecutionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPipelineExecutionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPipelineExecutionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPipelineExecutionsRequest, ListPipelineExecutionsResponse>()
                            .withOperationName("ListPipelineExecutions")
                            .withMarshaller(new ListPipelineExecutionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPipelineExecutionsRequest));
            CompletableFuture<ListPipelineExecutionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a summary of all of the pipelines associated with your account.
     * </p>
     *
     * @param listPipelinesRequest
     *        Represents the input of a <code>ListPipelines</code> action.
     * @return A Java Future containing the result of the ListPipelines operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>InvalidNextTokenException The next token was specified in an invalid format. Make sure that the next
     *         token you provide is the token returned by a previous call.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.ListPipelines
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/ListPipelines" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPipelinesResponse> listPipelines(ListPipelinesRequest listPipelinesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPipelinesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPipelinesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPipelines");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPipelinesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListPipelinesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPipelinesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPipelinesRequest, ListPipelinesResponse>()
                            .withOperationName("ListPipelines")
                            .withMarshaller(new ListPipelinesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPipelinesRequest));
            CompletableFuture<ListPipelinesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the set of key-value pairs (metadata) that are used to manage the resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>ResourceNotFoundException The resource was specified in an invalid format.</li>
     *         <li>InvalidNextTokenException The next token was specified in an invalid format. Make sure that the next
     *         token you provide is the token returned by a previous call.</li>
     *         <li>InvalidArnException The specified resource ARN is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a listing of all the webhooks in this Amazon Web Services Region for this account. The output lists all
     * webhooks and includes the webhook URL and ARN and the configuration for each webhook.
     * </p>
     *
     * @param listWebhooksRequest
     * @return A Java Future containing the result of the ListWebhooks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>InvalidNextTokenException The next token was specified in an invalid format. Make sure that the next
     *         token you provide is the token returned by a previous call.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.ListWebhooks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/ListWebhooks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListWebhooksResponse> listWebhooks(ListWebhooksRequest listWebhooksRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listWebhooksRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWebhooksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWebhooks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWebhooksResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListWebhooksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListWebhooksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWebhooksRequest, ListWebhooksResponse>()
                            .withOperationName("ListWebhooks").withMarshaller(new ListWebhooksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listWebhooksRequest));
            CompletableFuture<ListWebhooksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about any jobs for CodePipeline to act on. <code>PollForJobs</code> is valid only for action
     * types with "Custom" in the owner field. If the action type contains <code>AWS</code> or <code>ThirdParty</code>
     * in the owner field, the <code>PollForJobs</code> action returns an error.
     * </p>
     * <important>
     * <p>
     * When this API is called, CodePipeline returns temporary credentials for the S3 bucket used to store artifacts for
     * the pipeline, if the action requires access to that S3 bucket for input or output artifacts. This API also
     * returns any secret values defined for the action.
     * </p>
     * </important>
     *
     * @param pollForJobsRequest
     *        Represents the input of a <code>PollForJobs</code> action.
     * @return A Java Future containing the result of the PollForJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>ActionTypeNotFoundException The specified action type cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.PollForJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/PollForJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PollForJobsResponse> pollForJobs(PollForJobsRequest pollForJobsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(pollForJobsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, pollForJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PollForJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PollForJobsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PollForJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PollForJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PollForJobsRequest, PollForJobsResponse>()
                            .withOperationName("PollForJobs").withMarshaller(new PollForJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(pollForJobsRequest));
            CompletableFuture<PollForJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Determines whether there are any third party jobs for a job worker to act on. Used for partner actions only.
     * </p>
     * <important>
     * <p>
     * When this API is called, CodePipeline returns temporary credentials for the S3 bucket used to store artifacts for
     * the pipeline, if the action requires access to that S3 bucket for input or output artifacts.
     * </p>
     * </important>
     *
     * @param pollForThirdPartyJobsRequest
     *        Represents the input of a <code>PollForThirdPartyJobs</code> action.
     * @return A Java Future containing the result of the PollForThirdPartyJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ActionTypeNotFoundException The specified action type cannot be found.</li>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.PollForThirdPartyJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/PollForThirdPartyJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PollForThirdPartyJobsResponse> pollForThirdPartyJobs(
            PollForThirdPartyJobsRequest pollForThirdPartyJobsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(pollForThirdPartyJobsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, pollForThirdPartyJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PollForThirdPartyJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PollForThirdPartyJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PollForThirdPartyJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PollForThirdPartyJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PollForThirdPartyJobsRequest, PollForThirdPartyJobsResponse>()
                            .withOperationName("PollForThirdPartyJobs")
                            .withMarshaller(new PollForThirdPartyJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(pollForThirdPartyJobsRequest));
            CompletableFuture<PollForThirdPartyJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides information to CodePipeline about new revisions to a source.
     * </p>
     *
     * @param putActionRevisionRequest
     *        Represents the input of a <code>PutActionRevision</code> action.
     * @return A Java Future containing the result of the PutActionRevision operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>PipelineNotFoundException The pipeline was specified in an invalid format or cannot be found.</li>
     *         <li>StageNotFoundException The stage was specified in an invalid format or cannot be found.</li>
     *         <li>ActionNotFoundException The specified action cannot be found.</li>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.PutActionRevision
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/PutActionRevision"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutActionRevisionResponse> putActionRevision(PutActionRevisionRequest putActionRevisionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putActionRevisionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putActionRevisionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutActionRevision");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutActionRevisionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutActionRevisionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutActionRevisionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutActionRevisionRequest, PutActionRevisionResponse>()
                            .withOperationName("PutActionRevision")
                            .withMarshaller(new PutActionRevisionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putActionRevisionRequest));
            CompletableFuture<PutActionRevisionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides the response to a manual approval request to CodePipeline. Valid responses include Approved and
     * Rejected.
     * </p>
     *
     * @param putApprovalResultRequest
     *        Represents the input of a <code>PutApprovalResult</code> action.
     * @return A Java Future containing the result of the PutApprovalResult operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidApprovalTokenException The approval request already received a response or has expired.</li>
     *         <li>ApprovalAlreadyCompletedException The approval action has already been approved or rejected.</li>
     *         <li>PipelineNotFoundException The pipeline was specified in an invalid format or cannot be found.</li>
     *         <li>StageNotFoundException The stage was specified in an invalid format or cannot be found.</li>
     *         <li>ActionNotFoundException The specified action cannot be found.</li>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.PutApprovalResult
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/PutApprovalResult"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutApprovalResultResponse> putApprovalResult(PutApprovalResultRequest putApprovalResultRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putApprovalResultRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putApprovalResultRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutApprovalResult");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutApprovalResultResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutApprovalResultResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutApprovalResultResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutApprovalResultRequest, PutApprovalResultResponse>()
                            .withOperationName("PutApprovalResult")
                            .withMarshaller(new PutApprovalResultRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putApprovalResultRequest));
            CompletableFuture<PutApprovalResultResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Represents the failure of a job as returned to the pipeline by a job worker. Used for custom actions only.
     * </p>
     *
     * @param putJobFailureResultRequest
     *        Represents the input of a <code>PutJobFailureResult</code> action.
     * @return A Java Future containing the result of the PutJobFailureResult operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>JobNotFoundException The job was specified in an invalid format or cannot be found.</li>
     *         <li>InvalidJobStateException The job state was specified in an invalid format.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.PutJobFailureResult
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/PutJobFailureResult"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutJobFailureResultResponse> putJobFailureResult(
            PutJobFailureResultRequest putJobFailureResultRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putJobFailureResultRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putJobFailureResultRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutJobFailureResult");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutJobFailureResultResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutJobFailureResultResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutJobFailureResultResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutJobFailureResultRequest, PutJobFailureResultResponse>()
                            .withOperationName("PutJobFailureResult")
                            .withMarshaller(new PutJobFailureResultRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putJobFailureResultRequest));
            CompletableFuture<PutJobFailureResultResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Represents the success of a job as returned to the pipeline by a job worker. Used for custom actions only.
     * </p>
     *
     * @param putJobSuccessResultRequest
     *        Represents the input of a <code>PutJobSuccessResult</code> action.
     * @return A Java Future containing the result of the PutJobSuccessResult operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>JobNotFoundException The job was specified in an invalid format or cannot be found.</li>
     *         <li>InvalidJobStateException The job state was specified in an invalid format.</li>
     *         <li>OutputVariablesSizeExceededException Exceeded the total size limit for all variables in the pipeline.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.PutJobSuccessResult
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/PutJobSuccessResult"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutJobSuccessResultResponse> putJobSuccessResult(
            PutJobSuccessResultRequest putJobSuccessResultRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putJobSuccessResultRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putJobSuccessResultRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutJobSuccessResult");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutJobSuccessResultResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutJobSuccessResultResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutJobSuccessResultResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutJobSuccessResultRequest, PutJobSuccessResultResponse>()
                            .withOperationName("PutJobSuccessResult")
                            .withMarshaller(new PutJobSuccessResultRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putJobSuccessResultRequest));
            CompletableFuture<PutJobSuccessResultResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Represents the failure of a third party job as returned to the pipeline by a job worker. Used for partner actions
     * only.
     * </p>
     *
     * @param putThirdPartyJobFailureResultRequest
     *        Represents the input of a <code>PutThirdPartyJobFailureResult</code> action.
     * @return A Java Future containing the result of the PutThirdPartyJobFailureResult operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>JobNotFoundException The job was specified in an invalid format or cannot be found.</li>
     *         <li>InvalidJobStateException The job state was specified in an invalid format.</li>
     *         <li>InvalidClientTokenException The client token was specified in an invalid format</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.PutThirdPartyJobFailureResult
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/PutThirdPartyJobFailureResult"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutThirdPartyJobFailureResultResponse> putThirdPartyJobFailureResult(
            PutThirdPartyJobFailureResultRequest putThirdPartyJobFailureResultRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putThirdPartyJobFailureResultRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putThirdPartyJobFailureResultRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutThirdPartyJobFailureResult");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutThirdPartyJobFailureResultResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutThirdPartyJobFailureResultResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutThirdPartyJobFailureResultResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutThirdPartyJobFailureResultRequest, PutThirdPartyJobFailureResultResponse>()
                            .withOperationName("PutThirdPartyJobFailureResult")
                            .withMarshaller(new PutThirdPartyJobFailureResultRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putThirdPartyJobFailureResultRequest));
            CompletableFuture<PutThirdPartyJobFailureResultResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Represents the success of a third party job as returned to the pipeline by a job worker. Used for partner actions
     * only.
     * </p>
     *
     * @param putThirdPartyJobSuccessResultRequest
     *        Represents the input of a <code>PutThirdPartyJobSuccessResult</code> action.
     * @return A Java Future containing the result of the PutThirdPartyJobSuccessResult operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>JobNotFoundException The job was specified in an invalid format or cannot be found.</li>
     *         <li>InvalidJobStateException The job state was specified in an invalid format.</li>
     *         <li>InvalidClientTokenException The client token was specified in an invalid format</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.PutThirdPartyJobSuccessResult
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/PutThirdPartyJobSuccessResult"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutThirdPartyJobSuccessResultResponse> putThirdPartyJobSuccessResult(
            PutThirdPartyJobSuccessResultRequest putThirdPartyJobSuccessResultRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putThirdPartyJobSuccessResultRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putThirdPartyJobSuccessResultRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutThirdPartyJobSuccessResult");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutThirdPartyJobSuccessResultResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutThirdPartyJobSuccessResultResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutThirdPartyJobSuccessResultResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutThirdPartyJobSuccessResultRequest, PutThirdPartyJobSuccessResultResponse>()
                            .withOperationName("PutThirdPartyJobSuccessResult")
                            .withMarshaller(new PutThirdPartyJobSuccessResultRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putThirdPartyJobSuccessResultRequest));
            CompletableFuture<PutThirdPartyJobSuccessResultResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Defines a webhook and returns a unique webhook URL generated by CodePipeline. This URL can be supplied to third
     * party source hosting providers to call every time there's a code change. When CodePipeline receives a POST
     * request on this URL, the pipeline defined in the webhook is started as long as the POST request satisfied the
     * authentication and filtering requirements supplied when defining the webhook. RegisterWebhookWithThirdParty and
     * DeregisterWebhookWithThirdParty APIs can be used to automatically configure supported third parties to call the
     * generated webhook URL.
     * </p>
     *
     * @param putWebhookRequest
     * @return A Java Future containing the result of the PutWebhook operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>LimitExceededException The number of pipelines associated with the Amazon Web Services account has
     *         exceeded the limit allowed for the account.</li>
     *         <li>InvalidWebhookFilterPatternException The specified event filter rule is in an invalid format.</li>
     *         <li>InvalidWebhookAuthenticationParametersException The specified authentication type is in an invalid
     *         format.</li>
     *         <li>PipelineNotFoundException The pipeline was specified in an invalid format or cannot be found.</li>
     *         <li>TooManyTagsException The tags limit for a resource has been exceeded.</li>
     *         <li>InvalidTagsException The specified resource tags are invalid.</li>
     *         <li>ConcurrentModificationException Unable to modify the tag due to a simultaneous update request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.PutWebhook
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/PutWebhook" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutWebhookResponse> putWebhook(PutWebhookRequest putWebhookRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putWebhookRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putWebhookRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutWebhook");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutWebhookResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutWebhookResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutWebhookResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutWebhookRequest, PutWebhookResponse>().withOperationName("PutWebhook")
                            .withMarshaller(new PutWebhookRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putWebhookRequest));
            CompletableFuture<PutWebhookResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Configures a connection between the webhook that was created and the external tool with events to be detected.
     * </p>
     *
     * @param registerWebhookWithThirdPartyRequest
     * @return A Java Future containing the result of the RegisterWebhookWithThirdParty operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>WebhookNotFoundException The specified webhook was entered in an invalid format or cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.RegisterWebhookWithThirdParty
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/RegisterWebhookWithThirdParty"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RegisterWebhookWithThirdPartyResponse> registerWebhookWithThirdParty(
            RegisterWebhookWithThirdPartyRequest registerWebhookWithThirdPartyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(registerWebhookWithThirdPartyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                registerWebhookWithThirdPartyRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterWebhookWithThirdParty");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterWebhookWithThirdPartyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RegisterWebhookWithThirdPartyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RegisterWebhookWithThirdPartyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RegisterWebhookWithThirdPartyRequest, RegisterWebhookWithThirdPartyResponse>()
                            .withOperationName("RegisterWebhookWithThirdParty")
                            .withMarshaller(new RegisterWebhookWithThirdPartyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(registerWebhookWithThirdPartyRequest));
            CompletableFuture<RegisterWebhookWithThirdPartyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * You can retry a stage that has failed without having to run a pipeline again from the beginning. You do this by
     * either retrying the failed actions in a stage or by retrying all actions in the stage starting from the first
     * action in the stage. When you retry the failed actions in a stage, all actions that are still in progress
     * continue working, and failed actions are triggered again. When you retry a failed stage from the first action in
     * the stage, the stage cannot have any actions in progress. Before a stage can be retried, it must either have all
     * actions failed or some actions failed and some succeeded.
     * </p>
     *
     * @param retryStageExecutionRequest
     *        Represents the input of a <code>RetryStageExecution</code> action.
     * @return A Java Future containing the result of the RetryStageExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>ConflictException Your request cannot be handled because the pipeline is busy handling ongoing
     *         activities. Try again later.</li>
     *         <li>PipelineNotFoundException The pipeline was specified in an invalid format or cannot be found.</li>
     *         <li>StageNotFoundException The stage was specified in an invalid format or cannot be found.</li>
     *         <li>StageNotRetryableException Unable to retry. The pipeline structure or stage state might have changed
     *         while actions awaited retry, or the stage contains no failed actions.</li>
     *         <li>NotLatestPipelineExecutionException The stage has failed in a later run of the pipeline and the
     *         <code>pipelineExecutionId</code> associated with the request is out of date.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.RetryStageExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/RetryStageExecution"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RetryStageExecutionResponse> retryStageExecution(
            RetryStageExecutionRequest retryStageExecutionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(retryStageExecutionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, retryStageExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RetryStageExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RetryStageExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RetryStageExecutionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RetryStageExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RetryStageExecutionRequest, RetryStageExecutionResponse>()
                            .withOperationName("RetryStageExecution")
                            .withMarshaller(new RetryStageExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(retryStageExecutionRequest));
            CompletableFuture<RetryStageExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts the specified pipeline. Specifically, it begins processing the latest commit to the source location
     * specified as part of the pipeline.
     * </p>
     *
     * @param startPipelineExecutionRequest
     *        Represents the input of a <code>StartPipelineExecution</code> action.
     * @return A Java Future containing the result of the StartPipelineExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>ConflictException Your request cannot be handled because the pipeline is busy handling ongoing
     *         activities. Try again later.</li>
     *         <li>PipelineNotFoundException The pipeline was specified in an invalid format or cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.StartPipelineExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/StartPipelineExecution"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartPipelineExecutionResponse> startPipelineExecution(
            StartPipelineExecutionRequest startPipelineExecutionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startPipelineExecutionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startPipelineExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartPipelineExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartPipelineExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartPipelineExecutionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartPipelineExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartPipelineExecutionRequest, StartPipelineExecutionResponse>()
                            .withOperationName("StartPipelineExecution")
                            .withMarshaller(new StartPipelineExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startPipelineExecutionRequest));
            CompletableFuture<StartPipelineExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops the specified pipeline execution. You choose to either stop the pipeline execution by completing
     * in-progress actions without starting subsequent actions, or by abandoning in-progress actions. While completing
     * or abandoning in-progress actions, the pipeline execution is in a <code>Stopping</code> state. After all
     * in-progress actions are completed or abandoned, the pipeline execution is in a <code>Stopped</code> state.
     * </p>
     *
     * @param stopPipelineExecutionRequest
     * @return A Java Future containing the result of the StopPipelineExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>ConflictException Your request cannot be handled because the pipeline is busy handling ongoing
     *         activities. Try again later.</li>
     *         <li>PipelineNotFoundException The pipeline was specified in an invalid format or cannot be found.</li>
     *         <li>PipelineExecutionNotStoppableException Unable to stop the pipeline execution. The execution might
     *         already be in a <code>Stopped</code> state, or it might no longer be in progress.</li>
     *         <li>DuplicatedStopRequestException The pipeline execution is already in a <code>Stopping</code> state. If
     *         you already chose to stop and wait, you cannot make that request again. You can choose to stop and
     *         abandon now, but be aware that this option can lead to failed tasks or out of sequence tasks. If you
     *         already chose to stop and abandon, you cannot make that request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.StopPipelineExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/StopPipelineExecution"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StopPipelineExecutionResponse> stopPipelineExecution(
            StopPipelineExecutionRequest stopPipelineExecutionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopPipelineExecutionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopPipelineExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopPipelineExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopPipelineExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopPipelineExecutionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopPipelineExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopPipelineExecutionRequest, StopPipelineExecutionResponse>()
                            .withOperationName("StopPipelineExecution")
                            .withMarshaller(new StopPipelineExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopPipelineExecutionRequest));
            CompletableFuture<StopPipelineExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds to or modifies the tags of the given resource. Tags are metadata that can be used to manage a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>ResourceNotFoundException The resource was specified in an invalid format.</li>
     *         <li>InvalidArnException The specified resource ARN is invalid.</li>
     *         <li>TooManyTagsException The tags limit for a resource has been exceeded.</li>
     *         <li>InvalidTagsException The specified resource tags are invalid.</li>
     *         <li>ConcurrentModificationException Unable to modify the tag due to a simultaneous update request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from an Amazon Web Services resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>ResourceNotFoundException The resource was specified in an invalid format.</li>
     *         <li>InvalidArnException The specified resource ARN is invalid.</li>
     *         <li>InvalidTagsException The specified resource tags are invalid.</li>
     *         <li>ConcurrentModificationException Unable to modify the tag due to a simultaneous update request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an action type that was created with any supported integration model, where the action type is to be used
     * by customers of the action type provider. Use a JSON file with the action definition and
     * <code>UpdateActionType</code> to provide the full structure.
     * </p>
     *
     * @param updateActionTypeRequest
     * @return A Java Future containing the result of the UpdateActionType operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RequestFailedException The request failed because of an unknown error, exception, or failure.</li>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>ActionTypeNotFoundException The specified action type cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.UpdateActionType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/UpdateActionType" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateActionTypeResponse> updateActionType(UpdateActionTypeRequest updateActionTypeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateActionTypeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateActionTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateActionType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateActionTypeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateActionTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateActionTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateActionTypeRequest, UpdateActionTypeResponse>()
                            .withOperationName("UpdateActionType")
                            .withMarshaller(new UpdateActionTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateActionTypeRequest));
            CompletableFuture<UpdateActionTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a specified pipeline with edits or changes to its structure. Use a JSON file with the pipeline structure
     * and <code>UpdatePipeline</code> to provide the full structure of the pipeline. Updating the pipeline increases
     * the version number of the pipeline by 1.
     * </p>
     *
     * @param updatePipelineRequest
     *        Represents the input of an <code>UpdatePipeline</code> action.
     * @return A Java Future containing the result of the UpdatePipeline operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The validation was specified in an invalid format.</li>
     *         <li>InvalidStageDeclarationException The stage declaration was specified in an invalid format.</li>
     *         <li>InvalidActionDeclarationException The action declaration was specified in an invalid format.</li>
     *         <li>InvalidBlockerDeclarationException Reserved for future use.</li>
     *         <li>InvalidStructureException The structure was specified in an invalid format.</li>
     *         <li>LimitExceededException The number of pipelines associated with the Amazon Web Services account has
     *         exceeded the limit allowed for the account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodePipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodePipelineAsyncClient.UpdatePipeline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/UpdatePipeline" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePipelineResponse> updatePipeline(UpdatePipelineRequest updatePipelineRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePipelineRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodePipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePipeline");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePipelineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdatePipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdatePipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePipelineRequest, UpdatePipelineResponse>()
                            .withOperationName("UpdatePipeline")
                            .withMarshaller(new UpdatePipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updatePipelineRequest));
            CompletableFuture<UpdatePipelineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final CodePipelineServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CodePipelineException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidWebhookAuthenticationParametersException")
                                .exceptionBuilderSupplier(InvalidWebhookAuthenticationParametersException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidActionDeclarationException")
                                .exceptionBuilderSupplier(InvalidActionDeclarationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidArnException")
                                .exceptionBuilderSupplier(InvalidArnException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PipelineExecutionNotFoundException")
                                .exceptionBuilderSupplier(PipelineExecutionNotFoundException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PipelineNameInUseException")
                                .exceptionBuilderSupplier(PipelineNameInUseException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidJobException")
                                .exceptionBuilderSupplier(InvalidJobException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ActionTypeNotFoundException")
                                .exceptionBuilderSupplier(ActionTypeNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("StageNotFoundException")
                                .exceptionBuilderSupplier(StageNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DuplicatedStopRequestException")
                                .exceptionBuilderSupplier(DuplicatedStopRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTagsException")
                                .exceptionBuilderSupplier(InvalidTagsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ActionNotFoundException")
                                .exceptionBuilderSupplier(ActionNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidStageDeclarationException")
                                .exceptionBuilderSupplier(InvalidStageDeclarationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotLatestPipelineExecutionException")
                                .exceptionBuilderSupplier(NotLatestPipelineExecutionException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RequestFailedException")
                                .exceptionBuilderSupplier(RequestFailedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidClientTokenException")
                                .exceptionBuilderSupplier(InvalidClientTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidJobStateException")
                                .exceptionBuilderSupplier(InvalidJobStateException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PipelineVersionNotFoundException")
                                .exceptionBuilderSupplier(PipelineVersionNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidWebhookFilterPatternException")
                                .exceptionBuilderSupplier(InvalidWebhookFilterPatternException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OutputVariablesSizeExceededException")
                                .exceptionBuilderSupplier(OutputVariablesSizeExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("WebhookNotFoundException")
                                .exceptionBuilderSupplier(WebhookNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidApprovalTokenException")
                                .exceptionBuilderSupplier(InvalidApprovalTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ApprovalAlreadyCompletedException")
                                .exceptionBuilderSupplier(ApprovalAlreadyCompletedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PipelineNotFoundException")
                                .exceptionBuilderSupplier(PipelineNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("StageNotRetryableException")
                                .exceptionBuilderSupplier(StageNotRetryableException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTagsException")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidStructureException")
                                .exceptionBuilderSupplier(InvalidStructureException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidBlockerDeclarationException")
                                .exceptionBuilderSupplier(InvalidBlockerDeclarationException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNonceException")
                                .exceptionBuilderSupplier(InvalidNonceException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("JobNotFoundException")
                                .exceptionBuilderSupplier(JobNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PipelineExecutionNotStoppableException")
                                .exceptionBuilderSupplier(PipelineExecutionNotStoppableException::builder).httpStatusCode(400)
                                .build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        CodePipelineServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = CodePipelineServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
