// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.net.impls;

import static org.junit.Assert.*;

import java.time.Instant;
import java.util.Date;

import org.junit.Test;
import org.refcodes.data.Delimiter;
import org.refcodes.logger.RuntimeLogger;
import org.refcodes.logger.impls.RuntimeLoggerFactorySingleton;
import org.refcodes.net.HeaderField;
import org.refcodes.net.ResponseCookie;
import org.refcodes.net.ResponseHeaderFields;
import org.refcodes.net.impls.ResponseCookieImpl;
import org.refcodes.net.impls.ResponseHeaderFieldsImpl;
import org.refcodes.textual.impls.VerboseTextBuilderImpl;
import org.refcodes.time.DateFormat;

public class ResponseHeaderFieldsTest {

	private static RuntimeLogger LOGGER = RuntimeLoggerFactorySingleton.createRuntimeLogger();

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	@Test
	public void testHeaderFieldCookie() {
		ResponseHeaderFields theHeaderFields = new ResponseHeaderFieldsImpl();
		String theKey = "REFCODES";
		String theValue = "5161";
		theHeaderFields.addTo( HeaderField.SET_COOKIE.getName(), theKey + Delimiter.COOKIE_TUPEL_DELIMITER.getChar() + theValue );
		assertEquals( theHeaderFields.getAllCookies().size(), 1 );
		ResponseCookie theCookie = theHeaderFields.getAllCookies().get( 0 );

		LOGGER.info( new VerboseTextBuilderImpl().withElements( theHeaderFields.toHttpCookies() ).toString() );
		assertEquals( theKey, theCookie.getKey() );
		assertEquals( theValue, theCookie.getValue() );
	}

	@Test
	public void testHeaderFieldCookies() {
		ResponseHeaderFields theHeaderFields = new ResponseHeaderFieldsImpl();
		String theKey = "REFCODES";
		String theValue = "5161";
		theHeaderFields.addTo( HeaderField.SET_COOKIE.getName(), theKey + Delimiter.COOKIE_TUPEL_DELIMITER.getChar() + theValue );
		assertEquals( theHeaderFields.getAllCookies().size(), 1 );
		ResponseCookie theCookie = theHeaderFields.getAllCookies().get( 0 );

		LOGGER.info( new VerboseTextBuilderImpl().withElements( theHeaderFields.toHttpCookies() ).toString() );
		assertEquals( theKey, theCookie.getKey() );
		assertEquals( theValue, theCookie.getValue() );

		theKey = "COMCODES";
		theValue = "5162";
		theHeaderFields.addTo( HeaderField.SET_COOKIE.getName(), theKey + Delimiter.COOKIE_TUPEL_DELIMITER.getChar() + theValue );
		assertEquals( theHeaderFields.getAllCookies().size(), 2 );
		theCookie = theHeaderFields.getAllCookies().get( 1 );

		LOGGER.info( new VerboseTextBuilderImpl().withElements( theHeaderFields.toHttpCookies() ).toString() );
		assertEquals( theKey, theCookie.getKey() );
		assertEquals( theValue, theCookie.getValue() );
	}

	@Test
	public void testMixedCookies() {
		ResponseHeaderFields theHeaderFields = new ResponseHeaderFieldsImpl();
		String theKey = "REFCODES";
		String theValue = "5161";
		theHeaderFields.addTo( HeaderField.SET_COOKIE.getName(), theKey + Delimiter.COOKIE_TUPEL_DELIMITER.getChar() + theValue );
		assertEquals( theHeaderFields.getAllCookies().size(), 1 );
		ResponseCookie theCookie = theHeaderFields.getAllCookies().get( 0 );

		LOGGER.info( new VerboseTextBuilderImpl().withElements( theHeaderFields.toHttpCookies() ).toString() );
		assertEquals( theKey, theCookie.getKey() );
		assertEquals( theValue, theCookie.getValue() );

		theKey = "COMCODES";
		theValue = "5162";
		theHeaderFields.addTo( HeaderField.SET_COOKIE.getName(), theKey + Delimiter.COOKIE_TUPEL_DELIMITER.getChar() + theValue );
		assertEquals( theHeaderFields.getAllCookies().size(), 2 );
		theCookie = theHeaderFields.getAllCookies().get( 1 );

		LOGGER.info( new VerboseTextBuilderImpl().withElements( theHeaderFields.toHttpCookies() ).toString() );
		assertEquals( theKey, theCookie.getKey() );
		assertEquals( theValue, theCookie.getValue() );

		theKey = "FUNCODES";
		theValue = "5163";
		theHeaderFields.addCookie( theKey, theValue );
		LOGGER.info( new VerboseTextBuilderImpl().withElements( theHeaderFields.toHttpCookies() ).toString() );
		assertEquals( theHeaderFields.getAllCookies().size(), 3 );
		theCookie = theHeaderFields.getAllCookies().get( 2 );
		assertEquals( theKey, theCookie.getKey() );
		assertEquals( theValue, theCookie.getValue() );
	}

	@Test
	public void testCookieAttributes() {
		ResponseHeaderFields theHeaderFields = new ResponseHeaderFieldsImpl();
		String theKey = "REFCODES";
		String theValue = "5161";
		String theDomain = "refcodes.org";
		String thePath = "/rest";
		Date theExpiresDate = new Date();
		boolean isHttpOnly = true;
		boolean isSecure = true;
		theHeaderFields.addCookie( theKey, theValue ).withExpiresDate( theExpiresDate ).withHttpOnly( isHttpOnly ).withSecure( isSecure ).withDomain( theDomain ).withPath( thePath );
		System.out.println( new VerboseTextBuilderImpl().withElements( theHeaderFields.toHttpCookies() ).toString() );
		assertEquals( theHeaderFields.getAllCookies().size(), 1 );
		ResponseCookie theCookie = theHeaderFields.getAllCookies().get( 0 );
		assertEquals( theKey, theCookie.getKey() );
		assertEquals( theValue, theCookie.getValue() );
		assertEquals( theDomain, theCookie.getDomain() );
		assertEquals( thePath, theCookie.getPath() );
		assertEquals( theExpiresDate, theCookie.getExpiresDate() );
	}

	@Test
	public void testParseHttpCookie() {
		ResponseHeaderFields theHeaderFields = new ResponseHeaderFieldsImpl();
		String theKey = "REFCODES";
		String theValue = "5161";
		String theDomain = "refcodes.org";
		String thePath = "/rest";
		Date theExpiresDate = new Date();
		boolean isHttpOnly = false;
		boolean isSecure = true;
		theHeaderFields.addCookie( theKey, theValue ).withExpiresDate( theExpiresDate ).withHttpOnly( isHttpOnly ).withSecure( isSecure ).withDomain( theDomain ).withPath( thePath );
		System.out.println( new VerboseTextBuilderImpl().withElements( theHeaderFields.toHttpCookies() ).toString() );
		assertEquals( theHeaderFields.getAllCookies().size(), 1 );
		ResponseCookie theCookie = theHeaderFields.getAllCookies().get( 0 );

		ResponseCookie theClone = new ResponseCookieImpl( theCookie.toHttpCookie() );
		assertEquals( theKey, theClone.getKey() );
		assertEquals( theValue, theClone.getValue() );
		assertEquals( theDomain, theClone.getDomain() );
		assertEquals( DateFormat.NETSCAPE_COOKIE_DATE_FORMAT.getFormatter().format( Instant.ofEpochMilli( theExpiresDate.getTime() ) ), DateFormat.NETSCAPE_COOKIE_DATE_FORMAT.getFormatter().format( Instant.ofEpochMilli( theClone.getExpiresDate().getTime() ) ) );
		assertEquals( thePath, theClone.getPath() );
		assertEquals( isSecure, theClone.isSecure() );
		assertEquals( isHttpOnly, theClone.isHttpOnly() );
		assertEquals( theCookie.toHttpCookie(), theClone.toHttpCookie() );
	}

	@Test
	public void testMaxAge() {
		Date theDate = new Date();
		int theMaxAgeInSeconds = 60;
		int theMaxAgeInMillis = theMaxAgeInSeconds * 1000;
		ResponseCookie theCookie = new ResponseCookieImpl( "REFCODES=5161;MAX-AGE=" + theMaxAgeInSeconds + ";DOMAIN=refcodes.org;PATH=/rest;SECURE" );
		System.out.println( theCookie.toHttpCookie() );
		assertTrue( theDate.getTime() + theMaxAgeInMillis <= theCookie.getExpiresDate().getTime() );
		assertTrue( theDate.getTime() + theMaxAgeInMillis + 50 > theCookie.getExpiresDate().getTime() );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
