/**
 * OW2 Specifications
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DeploymentFactory.java 1520 2007-07-04 22:35:42Z sauthieg $
 * --------------------------------------------------------------------------
 */

package javax.enterprise.deploy.spi.factories;

import javax.enterprise.deploy.spi.DeploymentManager;
import javax.enterprise.deploy.spi.exceptions.DeploymentManagerCreationException;

/**
 * The DeploymentFactory interface is a deployment driver for a J2EE platform
 * product. It returns a DeploymentManager object which represents a connection
 * to a specific J2EE platform product.<br/>
 *
 * Each application server vendor must provide an implementation of this class
 * in order for the J2EE Deployment API to work with their product.<br/>
 *
 * The class implementing this interface should have a public no-argument
 * constructor, and it should be stateless (two instances of the class should
 * always behave the same). It is suggested but not required that the class
 * have a static initializer that registers an instance of the class with
 * the DeploymentFactoryManager class.<br/>
 *
 * A connected or disconnected DeploymentManager can be requested. A
 * DeploymentManager that runs connected to the platform can provide access
 * to J2EE resources. A DeploymentManager that runs disconnected only provides
 * module deployment configuration support.
 * @author Guillaume Sauthier
 */
public interface DeploymentFactory {

    /**
     * Tests whether this factory can create a DeploymentManager object based
     * on the specified URI. This does not indicate whether such an attempt will
     * be successful, only whether the factory can handle the uri.
     * @param uri The uri to check
     * @return true if the factory can handle the uri.
     */
    boolean handlesURI(String uri);

    /**
     * Return a connected DeploymentManager instance.
     * @param uri The URI that specifies the connection parameters
     * @param username An optional username (may be null if no authentication is
     *        required for this platform).
     * @param password An optional password (may be null if no authentication is
     *        required for this platform).
     * @return A ready DeploymentManager instance.
     * @throws DeploymentManagerCreationException occurs when a DeploymentManager
     *         could not be returned (server down, unable to authenticate, etc).
     */
    DeploymentManager getDeploymentManager(String uri,
                                           String username,
                                           String password)
                      throws DeploymentManagerCreationException;

    /**
     * Return a disconnected DeploymentManager instance.
     * @param uri the uri of the DeploymentManager to return.
     * @return A DeploymentManager disconnected instance.
     * @throws DeploymentManagerCreationException occurs if the DeploymentManager
     *         could not be created.
     */
    DeploymentManager getDisconnectedDeploymentManager(String uri)
                      throws DeploymentManagerCreationException;

    /**
     * Provide a string with the name of this vendor's DeploymentManager.
     * @return the name of the vendor's DeploymentManager.
     */
    String getDisplayName();

    /**
     * Provide a string identifying version of this vendor's DeploymentManager.
     * @return the name of the vendor's DeploymentManager.
     */
    String getProductVersion();
}
