/**
 * OW2 Specifications
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DConfigBean.java 1520 2007-07-04 22:35:42Z sauthieg $
 * --------------------------------------------------------------------------
 */

package javax.enterprise.deploy.spi;

import java.beans.PropertyChangeListener;

import javax.enterprise.deploy.model.DDBean;
import javax.enterprise.deploy.model.XpathEvent;
import javax.enterprise.deploy.spi.exceptions.BeanNotFoundException;
import javax.enterprise.deploy.spi.exceptions.ConfigurationException;

/**
 * The DConfigBean is a deployment configuration bean (DConfigBean) that is associated
 * with one or more deployment descriptor beans, (DDBean). A DConfigBean represents a
 * logical grouping of deployment configuration data to be presented to the Deployer. A
 * DConfigBean provides zero or more XPaths that identifies the XML information it
 * requires. A DConfigBean may contain other DConfigBeans and regular JavaBeans. The
 * top most DConfigBean is a DConfigBeanRoot object which represents a single XML
 * instance document.<br/>
 *
 * A DConfigBean is created by calling DConfigBean.getDConfigBean(DDBean) method, where
 * DConfigBean is the object that provided the XPath which the DDBean represents.<br/>
 *
 * A DConfigBean is a JavaBean component that presents the dynamic deployment configuration
 * information for a J2EE plugin to the deployer. It is a JavaBean. The JavaBean
 * architecture was chosen because of its versatility in providing both simple and complex
 * components. JavaBeans also enable the development of property sheets and property
 * editors, as well as sophisticated customization wizards.<br/>
 *
 * It is expected that a plugin vendor will provide a Property Editor for any complex
 * datatype in a DConfigBean that a deployer needs to edit through a property sheet. The
 * Property Editor should be implemented and made available to a tool according to the
 * guidelines defined in the JavaBeans API Specification version 1.01.
 * @author Guillaume Sauthier
 */
public interface DConfigBean {

    /**
     * Return the JavaBean containing the deployment descriptor XML text associated with
     * this DConfigBean.
     * @return The bean class containing the XML text for this DConfigBean.
     */
    DDBean getDDBean();

    /**
     * Return a list of XPaths designating the deployment descriptor information
     * this DConfigBean requires. A given server vendor will need to specify some
     * server-specific information. Each String returned by this method is an XPath
     * describing a certain portion of the standard deployment descriptor for which
     * there is corresponding server-specific configuration.
     * @return a list of XPath Strings representing XML data to be retrieved or 'null'
     *         if there are none.
     */
    String[] getXpaths();

    /**
     * Return the JavaBean containing the server-specific deployment configuration
     * information based upon the XML data provided by the DDBean.
     * @param bean The DDBean containing the XML data to be evaluated.
     * @return The DConfigBean to display the server-specific properties for the
     *         standard bean.
     * @throws ConfigurationException reports errors in generating a configuration
     *         bean. This DDBean is considered undeployable to this server until this
     *         exception is resolved. A suitably descriptive message is required so
     *         the user can diagnose the error.
     */
    DConfigBean getDConfigBean(DDBean bean) throws ConfigurationException;

    /**
     * Remove a child DConfigBean from this bean.
     * @param bean The child DConfigBean to be removed.
     * @throws BeanNotFoundExceptionthe bean provided is not in the child list of this
     *         bean.
     */
    void removeDConfigBean(DConfigBean bean) throws BeanNotFoundException;

    /**
     * A notification that the DDBean provided in the event has changed and this
     * bean or its child beans need to reevaluate themselves.
     * @param event an event containing a reference to the DDBean which has changed.
     */
    void notifyDDChange(XpathEvent event);

    /**
     * Register a property listener for this bean.
     * @param pcl PropertyChangeListener to add
     */
    void addPropertyChangeListener(PropertyChangeListener pcl);

    /**
     * Unregister a property listener for this bean.
     * @param pcl Listener to remove.
     */
    void removePropertyChangeListener(PropertyChangeListener pcl);
}
