/**
 * OW2 Specifications
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DeployableObject.java 1520 2007-07-04 22:35:42Z sauthieg $
 * --------------------------------------------------------------------------
 */

package javax.enterprise.deploy.model;

import java.io.FileNotFoundException;
import java.io.InputStream;
import java.util.Enumeration;

import javax.enterprise.deploy.model.exceptions.DDBeanCreateException;
import javax.enterprise.deploy.shared.ModuleType;

/**
 * The DeployableObject interface is an abstract representation of a J2EE deployable
 * module (JAR, WAR, RAR, EAR). A DeployableObject provides access to the module's
 * deployment descriptor and class files.
 * @author Guillaume Sauthier
 */
public interface DeployableObject {

    /**
     * Return the ModuleType of deployment descriptor (i.e., EAR, JAR, WAR, RAR)
     * this deployable object represents. Values are found in DeploymentManager.
     * @return The ModuleType of deployable object
     */
    ModuleType getType();

    /**
     * Return the top level standard bean representing the root of the deployment
     * descriptor.
     * @return A standard bean representing the deployment descriptor.
     */
    DDBeanRoot getDDBeanRoot();

    /**
     * Return an array of standard beans representing the XML content returned
     * based upon the XPath.
     * @param xpath An XPath string identifying the data to be extracted from
     *        the deployment descriptor.
     * @return a array of DDBeans or 'null' if no matching data found.
     */
    DDBean[] getChildBean(String xpath);

    /**
     * Return the XML content associated with the XPath from a deployment descriptor.
     * @param xpath An xpath string referring to a location in the deployment descriptor
     * @return a list XML content or 'null' if no matching data found.
     */
    String[] getText(String xpath);

    /**
     * Retrieve the specified class from this deployable module.<br/>
     *
     * One use: to get all finder methods from an EJB If the tool is attempting
     * to package an module and retrieve a class from the package, the class
     * request may fail. The class may not yet be available. The tool should
     * respect the manifest cross-path entries.
     * @param className Class to retrieve.
     * @return Class representation of the class
     */
    Class getClassFromScope(String className);

    /**
     * Returns the DTD version number given in the XML DOCTYPE text provided in
     * every standard J2EE module's deployment descriptor file.
     * @return
     * @deprecated As of version 1.1 replaced by DDBeanRoot.getDDBeanRootVersion()
     */
    String getModuleDTDVersion();

    /**
     * Returns a DDBeanRoot object for the XML instance document named. This method
     * should be used to return DDBeanRoot objects for non deployment descriptor XML
     * instance documents such as WSDL files.
     * @param filename
     * @return a DDBeanRoot object for the XML data.
     * @throws FileNotFoundException if the named file can not be found
     * @throws DDBeanCreateException if an error is encountered creating the DDBeanRoot
     *         object.
     */
    DDBeanRoot getDDBeanRoot(String filename) throws FileNotFoundException,
                                                     DDBeanCreateException;

    /**
     * Returns an enumeration of the module file entries. All elements in the enumeration
     * are of type String. Each String represents a file name relative to the root of the
     * module.
     * @return an enumeration of the archive file entries.
     */
    Enumeration entries();

    /**
     * Returns the InputStream for the given entry name The file name must be relative to
     * the root of the module.
     * @param name the file name relative to the root of the module.
     * @return the InputStream for the given entry name or null if not found.
     */
    InputStream getEntry(String name);
}
