/*
 * Copyright (c) 2001-2006, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: WhenExpression.java 3522 2006-10-30 00:56:28Z jmettraux $
 */

//
// WhenExpression.java
//
// john.mettraux@openwfe.org
//
// generated with 
// jtmpl 1.1.01 2004/05/19 (john.mettraux@openwfe.org)
//

package openwfe.org.engine.expressions;

import openwfe.org.OpenWfeException;
import openwfe.org.ApplicationContext;
import openwfe.org.time.Time;
import openwfe.org.time.Scheduler;
import openwfe.org.time.Schedulable;
import openwfe.org.engine.expool.ExpressionPool;
import openwfe.org.engine.launch.LaunchException;
import openwfe.org.engine.history.History;
import openwfe.org.engine.workitem.InFlowWorkItem;


/**
 * An 'if' expression that blocks until its condition realizes or until
 * it times out.
 * There is no else clause for a 'when' expression.
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Id: WhenExpression.java 3522 2006-10-30 00:56:28Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public class WhenExpression

    extends CleanCompositeFlowExpression

    //implements ExpressionWithTimeOut, Schedulable
    implements ExpressionWithTimeOut

{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(WhenExpression.class.getName());

    //
    // CONSTANTS & co

    //
    // FIELDS

    private InFlowWorkItem workitem = null;
    private boolean conditionTreated = false;

    private Long atId = null;

    private Long timeout = null;

    //
    // CONSTRUCTORS

    //
    // BEAN METHODS

    /**
     * When this expression gets applied, it 'traps' its workitem
     * until reply...
     */
    public InFlowWorkItem getWorkitem () { return this.workitem; }

    /**
     * Set to true when the condition got treated (before) the second and
     * last reply.
     */
    public boolean isConditionTreated () 
    { 
        //log.debug("isConditionTreated() returning "+this.conditionTreated);

        return this.conditionTreated; 
    }

    public void setWorkitem (final InFlowWorkItem wi) 
    { 
        if (wi == null) return;
        this.workitem = (InFlowWorkItem)wi.clone(); 
    }

    public void setConditionTreated (final boolean b) 
    { 
        //log.debug("setConditionTreated("+b+")");
        this.conditionTreated = b; 
    }

    /**
     * Returns the id of the scheduled 'at' job (used for cancelling it).
     */
    public Long getAtId ()
    {
        return this.atId;
    }

    /**
     * Returns the timeout set for this when expression.
     * Returns null if no timeout is desired.
     */
    public Long getTimeout()
    {
        return this.timeout;
    }

    /*
     * not necessary, at each 'rehydration', the expression is rescheduled
     * and receives a new job id.
     *
    public void setAtId (final Long l)
    {
        this.atId = l;
    }
     */
    
    /**
     * Sets the timeout for this expression.
     */
    public void setTimeout (final Long l)
    {
        this.timeout = l;
    }

    //
    // METHODS

    /**
     * The trigger method is called for rechecks, timeouts are
     * considered in the same run.
     */
    public void trigger (final Object[] params)
    {
        if (log.isDebugEnabled())
        {
            log.debug("trigger()");
            log.debug("trigger() timeout is "+this.timeout+" ms");
            log.debug("trigger() applyTime is "+this.getApplyTime());
        }

        //
        // should expression timeout ?

        if (this.timeout != null)
        {
            long t = applyTimeAsLong().longValue();
            t += this.timeout.longValue();

            if (t >= System.currentTimeMillis())
            {
                timeOutReply();
                return;
            }
        }

        //
        // no timeout, just a recheck

        recheck();
    }

    /**
     * This method is used to poll the conditional clause to
     * see wether it's possible to apply the child
     */
    protected void recheck()
    {
        log.debug("recheck()");

        final FlowExpressionId templateId = 
            (FlowExpressionId)getChildren().get(0);

        if (this.workitem == null)
        {
            log.debug("recheck() workitem is null. 'when' Skipped.");
            return;
        }

        if (isConditionTreated())
        {
            log.debug("trigger() condition already treated. Returning");
            return;
        }

        tag(this.workitem);

        log.debug("trigger() workitem got tagged");

        try
        {
            /*final FlowExpressionId rootExpressionId =*/ 
            getLauncher().launchSub
                (this.workitem, 
                 getId(), 
                 templateId,
                 new java.util.HashMap(0),  // empty var 'set'
                 true);  // async call (will return immediately)
            //
            // getId() indicates here that the parent id of the newly 
            // launched scope parent is this when expression...
            //
            // The last null stands for variables : none are passed for the
            // subprocess.
        }
        catch (final LaunchException e)
        {
            log.warn
                ("trigger() failed to launch condition checking logic !!!", e);
        }
    }

    /**
     * [re]schedules this expression.
     */
    public Long reschedule (final Scheduler s)
    {
        //
        // schedule for checks

        this.atId = getExpressionPool().schedule(this.getId());

        //
        // return recheck job id

        return this.atId;
    }

    /**
     * Returns the name of this expression as used
     * in the author field of the generated workitems
     */
    protected String getNameInHistory ()
    {
        return this.getClass().getName()+" $Revision: 3522 $";
    }

    //
    // METHODS from FlowExpression

    public void apply (final InFlowWorkItem wi) 
        throws ApplyException
    {
        log.debug("apply()");

        //
        // store 'when' workitem

        this.setWorkitem(wi);

        //
        // determine timeout

        this.determineTimeOut();

        //
        // store self

        this.storeItself();

        //
        // trigger a first time

        this.trigger(null);
    }

    /**
     * This override takes care of removing the conditional subexpression
     * before the flow leaves this when expression.
     */
    public void replyToParent (final InFlowWorkItem wi)
        throws ReplyException
    {
        unscheduleSelf();
            //
            // taking care of removing scheduling jobs used
            // by that expression

        if (getChildren() != null && getChildren().size() > 0)
        {
            if (log.isDebugEnabled())
            {
                log.debug
                    ("replyToParent() removing template "+
                     this.getChildren().get(0));
            }

            getExpressionPool()
                .removeExpression((FlowExpressionId)this.getChildren().get(0));
        }

        super.replyToParent(wi);
    }

    public void reply (final InFlowWorkItem wi)
        throws ReplyException
    {
        log.debug("reply()");

        if (isConditionTreated())
        {
            log.debug("reply() condition is treated - replying to parent");

            //ValueUtils.cleanBooleanResult(wi);

            replyToParent(wi);
            return;
        }

        if (ValueUtils.lookupBooleanResult(wi))
        {
            //
            // add history to workitem

            this.workitem.addHistoryItem
                (getNameInHistory(),
                 "triggered");

            //
            // log to history service

            historyLog
                (wi,
                 History.EVT_INFO, 
                 null,
                 getNameInHistory());
                    // participantName is set to null

            //
            // do the rest

            if (getChildren().size() < 2)
            {
                log.info
                    ("reply() condition realized but no child to apply");

                replyToParent(wi);
                return;
            }

            //
            // apply child

            log.debug("reply() setting conditionTreated to true");
            setConditionTreated(true);

            try
            {
                this.storeItself();

                getExpressionPool().apply
                    ((FlowExpressionId)getChildren().get(1), this.workitem);
            }
            catch (ApplyException ae)
            {
                throw new ReplyException("reply() failure", ae);
            }
        }
        else
        {
            log.info("reply() conditional clause returned with 'false'");

            //
            // register in scheduler for further evaluation

            this.reschedule(null);

            this.storeItself();
        }
    }

    /**
     * Cancels this expression by taking care of unscheduling its 'at job'.
     */
    public InFlowWorkItem cancel () 
        throws ApplyException
    {
        unscheduleSelf();

        return super.cancel();
    }

    protected void unscheduleSelf ()
    {
        log.debug("unscheduleSelf()");

        if (this.atId != null)
            getExpressionPool().getScheduler().unscheduleAt(this.atId);
    }

    //
    // METHODS from ExpressionWithTimeOut

    /**
     * Returns false if the expression should not be subject to a timeout.
     */
    public boolean isTimeOutActivated ()
    {
        return TimeoutUtils.isTimeOutActivated(this);
    }

    /**
     * The WhenExpression is subject to timeout, after a certain time, the
     * engine considers the condition will never evaluate to true and discards
     * the when, this method determines the size of this timeout.
     *
     * The priority list is :<br>
     * <ol>
     *   <li>when expression's 'timeout' attribute</li>
     *   <li>when expression's variable scope ('__timeout__')</li>
     *   <li>expressionPool 'expressionTimeout' param</li>
     * </ol>
     */
    public Long determineTimeOut ()
    {
        this.timeout = TimeoutUtils.determineTimeout(this, getWorkitem());

        log.debug("determineTimeOut() is "+this.timeout+" ms");

        return this.timeout;
    }

    /**
     * This method is called by the ExpressionStore when it notices that the
     * WhenExpression expired.
     */
    public void timeOutReply ()
        //throws ReplyException
    {
        this.getWorkitem().addHistoryItem
            (getNameInHistory(), 
             "timed out");
        
        //
        // log timeout in history

        historyLog
            (null, EVT_TIMED_OUT, null, getNameInHistory());
                // participantName is set to null

        //
        // resume flow
        
        log.debug("timeOutReply() resuming flow");

        try
        {
            replyToParent(this.getWorkitem());
        }
        catch (final ReplyException re)
        {
            log.warn("timeOutReply() failed to reply to parent exp", re);
        }
    }

    //
    // STATIC METHODS

}
