/*
 * Copyright (c) 2001-2006, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: RestoreExpression.java 3087 2006-08-30 07:08:10Z jmettraux $
 */

//
// RestoreExpression.java
//
// jmettraux@openwfe.org
//
// generated with 
// jtmpl 1.0.04 31.10.2002 John Mettraux (jmettraux@openwfe.org)
//

package openwfe.org.engine.expressions;

import openwfe.org.Utils;
import openwfe.org.engine.Definitions;
import openwfe.org.engine.workitem.WorkItemCoder;
import openwfe.org.engine.workitem.InFlowWorkItem;
import openwfe.org.engine.workitem.StringAttribute;
import openwfe.org.engine.expressions.sync.MergeUtils;


/**
 * Replaces the current workitem with a saved workitem (with the 'save'
 * expression)
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Date: 2006-08-30 09:08:10 +0200 (Wed, 30 Aug 2006) $
 * <br>$Id: RestoreExpression.java 3087 2006-08-30 07:08:10Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public class RestoreExpression

    extends ZeroChildExpression

{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(RestoreExpression.class.getName());

    //
    // CONSTANTS

    /**
     * the 'from-variable' attribute is used to indicate from which
     * variable the workitem should be restored
     */
    public final static String A_FROM_VARIABLE 
        = "from-variable";

    /**
     * the 'merge-lead' attribute is optional, if it is set, it has to take the
     * value 'restored' or 'current'.
     */
    public final static String A_MERGE_LEAD
        = "merge-lead";

    private final static String V_RESTORED = "restored";
    private final static String V_CURRENT = "current";

    /**
     * When the attribute 'to-field' is present, the restored workitem's 
     * attributes will be inserted in the current workitem as a field.
     * The name of the field is of course, the value given to this 'to-field'
     * attribute.
     */
    public final static String A_TO_FIELD 
        = "to-field";

    /**
     * The workitem is held in a file (XML encoded); This 'from-file'
     * attribute holds the path to it.
     */
    public final static String A_FROM_FILE
        = "from-file";

    /**
     * If this 'remove-file' attribute is set to 'yes' or 'true', the file
     * holding the workitem ('from-file') will be deleted after restoration.
     */
    public final static String A_REMOVE_FILE
        = "remove-file";
    
    //
    // FIELDS

    //
    // CONSTRUCTORS

    //
    // METHODS

    private InFlowWorkItem fetchWorkitemToRestore 
        //(final String variableName)
        (final Object workitemSource)
    {
        if (workitemSource instanceof String)
        {
            final Object var = lookupVariable((String)workitemSource);
            try
            {
                return (InFlowWorkItem)var;
            }
            catch (ClassCastException cce)
            {
                log.warn
                    ("fetchWorkitemToRestore() "+
                     "stuff to restore is not a workitem "+
                     "(it's of class "+var.getClass().getName()+")");
            }
        }
        else if (workitemSource instanceof java.io.File)
        {
            WorkItemCoder coder = Definitions
                .getWorkItemCoderLoader(this.context()).getXmlCoder();

            try
            {
                return (InFlowWorkItem)coder.decode
                    (workitemSource, this.context(), null);
            }
            catch (final Throwable t)
            {
                log.warn
                    ("fetchWorkitemToRestore() "+
                     "failed to retrieve wi from "+workitemSource, 
                     t);
            }
        }
        else
        {
            log.warn
                ("fetchWorkitemToRestore() "+
                 "cannot restore wi from source of class "+
                 workitemSource.getClass().getName());
        }

        //log.warn
        //    ("fetchWorkitemToRestore() "+
        //     "no workitem found in >"+workitemSource+"<. Resuming.");

        return null;
    }

    private InFlowWorkItem doMerge 
        (final InFlowWorkItem wi, 
         final Object workitemSource)
    {
        InFlowWorkItem newWorkitem = null;

        final String mergeLead = lookupAttribute(A_MERGE_LEAD, wi);

        final boolean theRestoredRules = V_RESTORED.equals(mergeLead);

        if (log.isDebugEnabled())
        {
            if (theRestoredRules)
                log.debug("apply() restored workitem has priority in merge");
            else
                log.debug("apply() current workitem has priority in merge");
        }

        //
        // fetch workitem to restore

        final InFlowWorkItem restoredWorkitem = 
            fetchWorkitemToRestore(workitemSource);

        if (restoredWorkitem == null)
            return wi;

        //
        // merge (or simply override)

        if (mergeLead == null)
            //
            // override
        {
            newWorkitem = restoredWorkitem;
        }
        else
            //
            // merge
        {
            if (theRestoredRules)
                newWorkitem = MergeUtils.merge(wi, restoredWorkitem);
            else
                newWorkitem = MergeUtils.merge(restoredWorkitem, wi);
        }

        return newWorkitem;
    }

    private InFlowWorkItem doInsert
        (final InFlowWorkItem wi, 
         final Object workitemSource,
         final String fieldName)
    {
        //
        // fetch workitem to restore

        final InFlowWorkItem restoredWorkitem = 
            fetchWorkitemToRestore(workitemSource);

        if (restoredWorkitem == null)
            return wi;

        //
        // do insert
        
        wi.getAttributes().put
            (new StringAttribute(fieldName), 
             restoredWorkitem.getAttributes());

        return wi;
    }

    public void apply (final InFlowWorkItem wi)
        throws ApplyException
    {
        InFlowWorkItem newWorkitem = wi;

        final String variableName = lookupAttribute(A_FROM_VARIABLE, wi);
        final String fileName = lookupAttribute(A_FROM_FILE, wi);

        Object workitemSource = null;

        if (fileName != null)
        {
            workitemSource = new java.io.File(fileName);
        }
        else if (variableName != null)
        {
            workitemSource = variableName;
        }

        if (workitemSource == null)
        {
            log.warn
                ("apply() no attribute '"+
                 A_FROM_VARIABLE+
                 "' or '"+A_FROM_FILE+
                 "' for expression, cannot restore anything.");
        }
        else
        {
            final String toField = lookupAttribute(A_TO_FIELD, wi);

            if (toField != null)
                newWorkitem = doInsert(wi, workitemSource, toField);
            else
                newWorkitem = doMerge(wi, workitemSource);
        }

        if (newWorkitem != wi && 
            fileName != null)
        {
            final boolean removeFile = 
                Utils.toBoolean(lookupAttribute(A_REMOVE_FILE, wi));

            if (removeFile)
            {
                final boolean b = ((java.io.File)workitemSource).delete();

                if (log.isDebugEnabled())
                {
                    log.debug
                        ("apply() "+
                         "removed file '"+workitemSource+
                         "'. Successful ? "+b);
                }
            }
        }

        //applyToParent(wi);
        applyToParent(newWorkitem);
    }
    
}
