/*
 * Decompiled with CFR 0.152.
 */
package com.mongodb.internal.connection;

import com.mongodb.assertions.Assertions;
import com.mongodb.internal.connection.BufferProvider;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.List;
import org.bson.BsonSerializationException;
import org.bson.ByteBuf;
import org.bson.io.OutputBuffer;

public class ByteBufferBsonOutput
extends OutputBuffer {
    private static final int MAX_SHIFT = 31;
    private static final int INITIAL_SHIFT = 10;
    public static final int INITIAL_BUFFER_SIZE = 1024;
    public static final int MAX_BUFFER_SIZE = 0x1000000;
    private final BufferProvider bufferProvider;
    private final List<ByteBuf> bufferList = new ArrayList<ByteBuf>();
    private int curBufferIndex = 0;
    private int position = 0;
    private boolean closed;
    private ByteBuf currentByteBuffer;

    public ByteBufferBsonOutput(BufferProvider bufferProvider) {
        this.bufferProvider = Assertions.notNull("bufferProvider", bufferProvider);
    }

    public Branch branch() {
        return new Branch(this);
    }

    public void writeBytes(byte[] bytes, int offset, int length) {
        this.ensureOpen();
        int currentOffset = offset;
        int remainingLen = length;
        while (remainingLen > 0) {
            ByteBuf buf = this.getCurrentByteBuffer();
            int bytesToPutInCurrentBuffer = Math.min(buf.remaining(), remainingLen);
            buf.put(bytes, currentOffset, bytesToPutInCurrentBuffer);
            remainingLen -= bytesToPutInCurrentBuffer;
            currentOffset += bytesToPutInCurrentBuffer;
        }
        this.position += length;
    }

    public void writeInt32(int value) {
        this.ensureOpen();
        ByteBuf buf = this.getCurrentByteBuffer();
        if (buf.remaining() >= 4) {
            buf.putInt(value);
            this.position += 4;
        } else {
            super.writeInt32(value);
        }
    }

    public void writeInt32(int absolutePosition, int value) {
        this.ensureOpen();
        if (absolutePosition < 0) {
            throw new IllegalArgumentException(String.format("position must be >= 0 but was %d", absolutePosition));
        }
        if (absolutePosition + 3 > this.position - 1) {
            throw new IllegalArgumentException(String.format("Cannot write 4 bytes starting at position %d: current size is %d bytes", this.position - 1, absolutePosition + 3));
        }
        BufferPositionPair bufferPositionPair = this.getBufferPositionPair(absolutePosition);
        ByteBuf byteBuffer = this.getByteBufferAtIndex(bufferPositionPair.bufferIndex);
        int capacity = byteBuffer.position() - bufferPositionPair.position;
        if (capacity >= 4) {
            byteBuffer.putInt(bufferPositionPair.position, value);
        } else {
            int valueToWrite = value;
            int pos = bufferPositionPair.position;
            int bufferIndex = bufferPositionPair.bufferIndex;
            for (int i = 0; i < 4; ++i) {
                byteBuffer.put(pos++, (byte)valueToWrite);
                valueToWrite >>= 8;
                if (--capacity != 0) continue;
                byteBuffer = this.getByteBufferAtIndex(++bufferIndex);
                pos = 0;
                capacity = byteBuffer.position();
            }
        }
    }

    public void writeDouble(double value) {
        this.ensureOpen();
        ByteBuf buf = this.getCurrentByteBuffer();
        if (buf.remaining() >= 8) {
            buf.putDouble(value);
            this.position += 8;
        } else {
            this.writeInt64(Double.doubleToRawLongBits(value));
        }
    }

    public void writeInt64(long value) {
        this.ensureOpen();
        ByteBuf buf = this.getCurrentByteBuffer();
        if (buf.remaining() >= 8) {
            buf.putLong(value);
            this.position += 8;
        } else {
            super.writeInt64(value);
        }
    }

    public void writeByte(int value) {
        this.ensureOpen();
        this.getCurrentByteBuffer().put((byte)value);
        ++this.position;
    }

    private ByteBuf getCurrentByteBuffer() {
        if (this.currentByteBuffer == null) {
            this.currentByteBuffer = this.getByteBufferAtIndex(this.curBufferIndex);
        }
        if (this.currentByteBuffer.hasRemaining()) {
            return this.currentByteBuffer;
        }
        ++this.curBufferIndex;
        this.currentByteBuffer = this.getByteBufferAtIndex(this.curBufferIndex);
        return this.currentByteBuffer;
    }

    private ByteBuf getByteBufferAtIndex(int index) {
        if (this.bufferList.size() < index + 1) {
            ByteBuf buffer = this.bufferProvider.getBuffer(index >= 21 ? 0x1000000 : Math.min(1024 << index, 0x1000000));
            this.bufferList.add(buffer);
        }
        return this.bufferList.get(index);
    }

    public int getPosition() {
        this.ensureOpen();
        return this.position;
    }

    public int getSize() {
        this.ensureOpen();
        return this.position;
    }

    protected void write(int absolutePosition, int value) {
        this.ensureOpen();
        if (absolutePosition < 0) {
            throw new IllegalArgumentException(String.format("position must be >= 0 but was %d", absolutePosition));
        }
        if (absolutePosition > this.position - 1) {
            throw new IllegalArgumentException(String.format("position must be <= %d but was %d", this.position - 1, absolutePosition));
        }
        BufferPositionPair bufferPositionPair = this.getBufferPositionPair(absolutePosition);
        ByteBuf byteBuffer = this.getByteBufferAtIndex(bufferPositionPair.bufferIndex);
        byteBuffer.put(bufferPositionPair.position++, (byte)value);
    }

    public List<ByteBuf> getByteBuffers() {
        this.ensureOpen();
        ArrayList<ByteBuf> buffers = new ArrayList<ByteBuf>(this.bufferList.size());
        for (ByteBuf cur : this.bufferList) {
            buffers.add(cur.duplicate().order(ByteOrder.LITTLE_ENDIAN).flip());
        }
        return buffers;
    }

    public List<ByteBuf> getDuplicateByteBuffers() {
        this.ensureOpen();
        ArrayList<ByteBuf> buffers = new ArrayList<ByteBuf>(this.bufferList.size());
        for (ByteBuf cur : this.bufferList) {
            buffers.add(cur.duplicate().order(ByteOrder.LITTLE_ENDIAN));
        }
        return buffers;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int pipe(OutputStream out) throws IOException {
        this.ensureOpen();
        byte[] tmp = new byte[1024];
        int total = 0;
        List<ByteBuf> byteBuffers = this.getByteBuffers();
        try {
            for (ByteBuf cur : byteBuffers) {
                while (cur.hasRemaining()) {
                    int numBytesToCopy = Math.min(cur.remaining(), tmp.length);
                    cur.get(tmp, 0, numBytesToCopy);
                    out.write(tmp, 0, numBytesToCopy);
                }
                total += cur.limit();
            }
        }
        finally {
            byteBuffers.forEach(ByteBuf::release);
        }
        return total;
    }

    public void truncateToPosition(int newPosition) {
        this.ensureOpen();
        if (newPosition == this.position) {
            return;
        }
        if (newPosition > this.position || newPosition < 0) {
            throw new IllegalArgumentException();
        }
        BufferPositionPair bufferPositionPair = this.getBufferPositionPair(newPosition);
        this.bufferList.get(bufferPositionPair.bufferIndex).position(bufferPositionPair.position);
        if (bufferPositionPair.bufferIndex + 1 < this.bufferList.size()) {
            this.currentByteBuffer = null;
        }
        while (this.bufferList.size() > bufferPositionPair.bufferIndex + 1) {
            ByteBuf buffer = this.bufferList.remove(this.bufferList.size() - 1);
            buffer.release();
        }
        this.curBufferIndex = bufferPositionPair.bufferIndex;
        this.position = newPosition;
    }

    public final void flush() throws IOException {
    }

    public void close() {
        if (this.isOpen()) {
            for (ByteBuf cur : this.bufferList) {
                cur.release();
            }
            this.currentByteBuffer = null;
            this.bufferList.clear();
            this.closed = true;
        }
    }

    private BufferPositionPair getBufferPositionPair(int absolutePosition) {
        int positionInBuffer = absolutePosition;
        int bufferIndex = 0;
        int bufferSize = this.bufferList.get(bufferIndex).position();
        int startPositionOfBuffer = 0;
        while (startPositionOfBuffer + bufferSize <= absolutePosition) {
            startPositionOfBuffer += bufferSize;
            positionInBuffer -= bufferSize;
            bufferSize = this.bufferList.get(++bufferIndex).position();
        }
        return new BufferPositionPair(bufferIndex, positionInBuffer);
    }

    private void ensureOpen() {
        if (!this.isOpen()) {
            throw new IllegalStateException("The output is closed");
        }
    }

    boolean isOpen() {
        return !this.closed;
    }

    private void merge(ByteBufferBsonOutput branch) {
        Assertions.assertTrue(branch instanceof Branch);
        branch.bufferList.forEach(ByteBuf::retain);
        this.bufferList.addAll(branch.bufferList);
        this.curBufferIndex += branch.curBufferIndex + 1;
        this.position += branch.position;
        this.currentByteBuffer = null;
    }

    protected int writeCharacters(String str, boolean checkNullTermination) {
        int stringLength = str.length();
        int sp = 0;
        int prevPos = this.position;
        ByteBuf curBuffer = this.getCurrentByteBuffer();
        int curBufferPos = curBuffer.position();
        int curBufferLimit = curBuffer.limit();
        int remaining = curBufferLimit - curBufferPos;
        if (curBuffer.isBackedByArray()) {
            byte[] dst = curBuffer.array();
            int arrayOffset = curBuffer.arrayOffset();
            if (remaining >= str.length() + 1) {
                sp = ByteBufferBsonOutput.writeOnArrayAscii(str, dst, arrayOffset + curBufferPos, checkNullTermination);
                curBufferPos += sp;
                if (sp == stringLength) {
                    dst[arrayOffset + curBufferPos++] = 0;
                    this.position += sp + 1;
                    curBuffer.position(curBufferPos);
                    return sp + 1;
                }
                this.position += sp;
                curBuffer.position(curBufferPos);
            }
        }
        return this.writeOnBuffers(str, checkNullTermination, sp, stringLength, curBufferLimit, curBufferPos, curBuffer, prevPos);
    }

    private int writeOnBuffers(String str, boolean checkNullTermination, int stringPointer, int stringLength, int bufferLimit, int bufferPos, ByteBuf buffer, int prevPos) {
        int c;
        int curBufferPos = bufferPos;
        int curBufferLimit = bufferLimit;
        ByteBuf curBuffer = buffer;
        for (int sp = stringPointer; sp < stringLength; sp += Character.charCount(c)) {
            int remaining = curBufferLimit - curBufferPos;
            c = str.charAt(sp);
            if (checkNullTermination && c == 0) {
                throw new BsonSerializationException(String.format("BSON cstring '%s' is not valid because it contains a null character at index %d", str, sp));
            }
            if (c < 128) {
                if (remaining == 0) {
                    curBuffer = this.getCurrentByteBuffer();
                    curBufferPos = 0;
                    curBufferLimit = curBuffer.limit();
                }
                curBuffer.put((byte)c);
                ++curBufferPos;
                ++this.position;
                continue;
            }
            if (c < 2048) {
                if (remaining < 2) {
                    this.write((byte)(192 + (c >> 6)));
                    this.write((byte)(128 + (c & 0x3F)));
                    curBuffer = this.getCurrentByteBuffer();
                    curBufferPos = curBuffer.position();
                    curBufferLimit = curBuffer.limit();
                    continue;
                }
                curBuffer.put((byte)(192 + (c >> 6)));
                curBuffer.put((byte)(128 + (c & 0x3F)));
                curBufferPos += 2;
                this.position += 2;
                continue;
            }
            c = Character.codePointAt(str, sp);
            if (c < 65536) {
                if (remaining < 3) {
                    this.write((byte)(224 + (c >> 12)));
                    this.write((byte)(128 + (c >> 6 & 0x3F)));
                    this.write((byte)(128 + (c & 0x3F)));
                    curBuffer = this.getCurrentByteBuffer();
                    curBufferPos = curBuffer.position();
                    curBufferLimit = curBuffer.limit();
                    continue;
                }
                curBuffer.put((byte)(224 + (c >> 12)));
                curBuffer.put((byte)(128 + (c >> 6 & 0x3F)));
                curBuffer.put((byte)(128 + (c & 0x3F)));
                curBufferPos += 3;
                this.position += 3;
                continue;
            }
            if (remaining < 4) {
                this.write((byte)(240 + (c >> 18)));
                this.write((byte)(128 + (c >> 12 & 0x3F)));
                this.write((byte)(128 + (c >> 6 & 0x3F)));
                this.write((byte)(128 + (c & 0x3F)));
                curBuffer = this.getCurrentByteBuffer();
                curBufferPos = curBuffer.position();
                curBufferLimit = curBuffer.limit();
                continue;
            }
            curBuffer.put((byte)(240 + (c >> 18)));
            curBuffer.put((byte)(128 + (c >> 12 & 0x3F)));
            curBuffer.put((byte)(128 + (c >> 6 & 0x3F)));
            curBuffer.put((byte)(128 + (c & 0x3F)));
            curBufferPos += 4;
            this.position += 4;
        }
        this.getCurrentByteBuffer().put((byte)0);
        ++this.position;
        return this.position - prevPos;
    }

    private static int writeOnArrayAscii(String str, byte[] dst, int arrayPosition, boolean checkNullTermination) {
        int pos = arrayPosition;
        int sp = 0;
        while (sp < str.length()) {
            char c = str.charAt(sp);
            if (checkNullTermination && c == '\u0000') {
                throw new BsonSerializationException(String.format("BSON cstring '%s' is not valid because it contains a null character at index %d", str, sp));
            }
            if (c >= '\u0080') break;
            dst[pos] = (byte)c;
            ++sp;
            ++pos;
        }
        return sp;
    }

    public static final class Branch
    extends ByteBufferBsonOutput {
        private final ByteBufferBsonOutput parent;

        private Branch(ByteBufferBsonOutput parent) {
            super(parent.bufferProvider);
            this.parent = parent;
        }

        @Override
        public void close() {
            if (this.isOpen()) {
                try {
                    Assertions.assertTrue(this.parent.isOpen());
                    this.parent.merge((ByteBufferBsonOutput)this);
                }
                finally {
                    super.close();
                }
            }
        }
    }

    private static final class BufferPositionPair {
        private final int bufferIndex;
        private int position;

        BufferPositionPair(int bufferIndex, int position) {
            this.bufferIndex = bufferIndex;
            this.position = position;
        }
    }
}

