/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.pvm.internal.el;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.Properties;

import javax.el.ELException;
import javax.el.ExpressionFactory;

/**
 * FactoryFinder based on javax.el.FactoryFinder
 */
class FactoryFinder {

  private FactoryFinder() {
  }

  /**
   * Create an ExpressionFactory instance.
   * 
   * @param properties Properties passed to the constructor of the implementation.
   * @return an instance of ExpressionFactory
   * @param className The name of the ExpressionFactory class.
   * @param classLoader The class loader to be used to load the class.
   * @return An instance of ExpressionFactory.
   * @throws ELException if the class could not be found or if it is not a subclass of
   *         ExpressionFactory or if the class could not be instantiated.
   */
  private static ExpressionFactory newInstance(Properties properties, String className,
    ClassLoader classLoader) {
    Class<?> clazz;
    try {
      clazz = classLoader.loadClass(className.trim());
    }
    catch (ClassNotFoundException e) {
      throw new ELException("expression factory class not found", e);
    }

    Class<? extends ExpressionFactory> factoryClass = clazz.asSubclass(ExpressionFactory.class);
    try {
      if (properties != null) {
        try {
          Constructor<? extends ExpressionFactory> constructor = factoryClass.getConstructor(Properties.class);
          try {
            return constructor.newInstance(properties);
          }
          catch (InvocationTargetException e) {
            throw new ELException(constructor + " threw exception", e.getCause());
          }
        }
        catch (NoSuchMethodException e) {
          // do nothing
        }
      }
      return factoryClass.newInstance();
    }
    catch (InstantiationException e) {
      throw new ELException("failed to instantiate: " + factoryClass, e);
    }
    catch (IllegalAccessException e) {
      throw new ELException(FactoryFinder.class + " has no access to " + factoryClass, e);
    }
  }

  /**
   * Finds and initializes service provider for service given as <code>serviceName</code> with
   * properties (if not null). Alternatively it will look up for property file from
   * <code>${java.home}/lib/propertyFileName</code> prior to initialize.
   * 
   * @param serviceName service name which provider should be found for
   * @param defaultServiceProvider default service provider that should be returned in case
   *        <code>serviceName</code> provider was not found
   * @param properties properties that will be passed to implementation class
   * @param propertyFileName name of the property file can be found in
   *        <code>${java.home}/lib</code>
   * @return new instance of implementation class found for given serviceName or if not found
   *         instance of defaultServiceName if given
   * @throws ELException in case of provider was not found
   */
  public static Object find(String serviceName, String defaultServiceProvider,
    Properties properties, String propertyFileName) {
    ClassLoader classLoader;
    try {
      classLoader = Thread.currentThread().getContextClassLoader();
    }
    catch (SecurityException e) {
      classLoader = ExpressionFactory.class.getClassLoader();
    }

    String className = getClassName(serviceName, defaultServiceProvider, propertyFileName, classLoader);
    return newInstance(properties, className, classLoader);
  }

  private static String getClassName(String serviceName, String defaultServiceProvider,
    String propertyFileName, ClassLoader classLoader) {
    InputStream serviceInput = classLoader.getResourceAsStream("META-INF/services/"
      + serviceName);
    if (serviceInput != null) {
      try {
        BufferedReader reader = new BufferedReader(new InputStreamReader(serviceInput, "UTF-8"));
        String className = reader.readLine();
        if (className != null) return className;
        reader.close();
      }
      catch (IOException e) {
        // do nothing
      }
      finally {
        try {
          serviceInput.close();
        }
        catch (IOException e) {
          // do nothing
        }
      }
    }

    try {
      String home = System.getProperty("java.home");
      if (home != null) {
        File propertyFile = new File(home + File.separator + "lib", propertyFileName);
        if (propertyFile.canRead()) {
          InputStream propertyInput = null;
          try {
            propertyInput = new FileInputStream(propertyFile);
            Properties props = new Properties();
            props.load(propertyInput);

            String className = props.getProperty(ExpressionFactory.class.getName());
            if (className != null) return className;
          }
          catch (IOException e) {
            // do nothing
          }
          finally {
            if (propertyInput != null) {
              try {
                propertyInput.close();
              }
              catch (IOException e) {
                // do nothing
              }
            }
          }
        }
      }
    }
    catch (SecurityException se) {
      // do nothing
    }

    try {
      String className = System.getProperty(serviceName);
      if (className != null) return className;
    }
    catch (SecurityException se) {
      // do nothing
    }

    return defaultServiceProvider;
  }
}
