/*
 * Decompiled with CFR 0.152.
 */
package org.graalvm.python.embedding;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.lang.invoke.VarHandle;
import java.net.URI;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.channels.NonWritableChannelException;
import java.nio.channels.SeekableByteChannel;
import java.nio.charset.Charset;
import java.nio.file.AccessMode;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryStream;
import java.nio.file.FileSystemException;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.NotDirectoryException;
import java.nio.file.NotLinkException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.nio.file.spi.FileSystemProvider;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.logging.ConsoleHandler;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;
import java.util.logging.SimpleFormatter;
import java.util.stream.Collectors;
import org.graalvm.polyglot.io.FileSystem;
import org.graalvm.python.embedding.VirtualFileSystem;

final class VirtualFileSystemImpl
implements FileSystem,
AutoCloseable {
    private static final Logger LOGGER = Logger.getLogger(VirtualFileSystem.class.getName());
    private static final String MULTI_VSF_CHECKS_AS_WARNING_PROP = "org.graalvm.python.vfs.multiple_vfs_checks_as_warning";
    public static final String MULTI_VFS_ALLOW_PROP = "org.graalvm.python.vfs.allow_multiple";
    public static final String MULTI_VFS_SINGLE_ROOT_URL_PROP = "org.graalvm.python.vfs.root_url";
    private static final String DEFAULT_VFS_ROOT = "org.graalvm.python.vfs";
    static final String VFS_VENV = "venv";
    static final String VFS_SRC = "src";
    private static final String FILES_LIST = "fileslist.txt";
    public static final String CONTENTS_FILE;
    private static final String INSTALLED_FILE;
    private static final String PROJ_DIR = "proj";
    private final String vfsRoot;
    private final VirtualFileSystem.HostIO allowHostIO;
    private final Map<String, BaseEntry> vfsEntries = new HashMap<String, BaseEntry>();
    private Class<?> resourceLoadingClass;
    static final String PLATFORM_SEPARATOR;
    private static final char RESOURCE_SEPARATOR_CHAR = '/';
    private static final String RESOURCE_SEPARATOR;
    final Path mountPoint;
    private final String mountPointLowerCase;
    private final Path extractDir;
    private boolean extractOnStartup = "true".equals(System.getProperty("org.graalvm.python.vfs.extractOnStartup")) || "true".equals(System.getProperty("graalpy.vfs.extractOnStartup"));
    private final Predicate<Path> extractFilter;
    private final boolean caseInsensitive;
    private final DeleteTempDir deleteTempDir;
    private final boolean allowMultipleLocations = Boolean.getBoolean("org.graalvm.python.vfs.allow_multiple");
    private final String vfsRootURL = System.getProperty("org.graalvm.python.vfs.root_url");
    private FileSystem extractedFilesFS = FileSystem.newReadOnlyFileSystem((FileSystem)FileSystem.newDefaultFileSystem());
    private boolean projWarning = false;
    private FileSystemProvider defaultFileSystemProvider;
    private static final Set<? extends OpenOption> READ_OPTIONS;

    private static String resourcePath(String ... components) {
        return String.join((CharSequence)"/", components);
    }

    private static String absoluteResourcePath(String ... components) {
        return "/" + String.join((CharSequence)"/", components);
    }

    VirtualFileSystemImpl(Predicate<Path> extractFilter, Path mountPoint, String resourceDirectory, VirtualFileSystem.HostIO allowHostIO, Class<?> resourceLoadingClass, boolean caseInsensitive) {
        this.vfsRoot = resourceDirectory == null ? DEFAULT_VFS_ROOT : resourceDirectory;
        this.resourceLoadingClass = resourceLoadingClass != null ? resourceLoadingClass : VirtualFileSystem.class;
        this.caseInsensitive = caseInsensitive;
        this.mountPoint = mountPoint;
        this.mountPointLowerCase = mountPoint.toString().toLowerCase(Locale.ROOT);
        VirtualFileSystemImpl.fine("VirtualFilesystem %s, allowHostIO: %s, resourceLoadingClass: %s, caseInsensitive: %s, extractOnStartup: %s%s", mountPoint, allowHostIO.toString(), this.resourceLoadingClass.getName(), caseInsensitive, this.extractOnStartup, extractFilter != null ? "" : ", extractFilter: null");
        this.extractFilter = extractFilter;
        if (extractFilter != null) {
            try {
                this.extractDir = Files.createTempDirectory("org.graalvm.python.vfsx", new FileAttribute[0]);
                this.deleteTempDir = new DeleteTempDir(this.extractDir);
                Runtime.getRuntime().addShutdownHook(this.deleteTempDir);
            }
            catch (IOException e) {
                throw new IllegalStateException(e);
            }
        } else {
            this.extractDir = null;
            this.deleteTempDir = null;
        }
        this.allowHostIO = allowHostIO;
        this.initEntries();
    }

    static FileSystem createDelegatingFileSystem(final VirtualFileSystemImpl vfs) {
        FileSystem d = switch (vfs.allowHostIO) {
            default -> throw new IncompatibleClassChangeError();
            case VirtualFileSystem.HostIO.NONE -> FileSystem.newDenyIOFileSystem();
            case VirtualFileSystem.HostIO.READ -> FileSystem.newReadOnlyFileSystem((FileSystem)FileSystem.newDefaultFileSystem());
            case VirtualFileSystem.HostIO.READ_WRITE -> FileSystem.newDefaultFileSystem();
        };
        FileSystem delegatingFS = FileSystem.newCompositeFileSystem((FileSystem)d, (FileSystem.Selector[])new FileSystem.Selector[]{new FileSystem.Selector(vfs){

            public boolean test(Path path) {
                Objects.requireNonNull(path);
                return vfs.pathIsInVfs(VirtualFileSystemImpl.toAbsoluteNormalizedPath(path));
            }
        }});
        if (vfs.allowHostIO == VirtualFileSystem.HostIO.NONE) {
            delegatingFS.setCurrentWorkingDirectory(vfs.mountPoint.resolve(VFS_SRC));
        }
        return delegatingFS;
    }

    @Override
    public void close() {
        if (this.deleteTempDir != null) {
            this.deleteTempDir.removeExtractDir();
        }
    }

    static boolean isWindows() {
        return System.getProperty("os.name").toLowerCase(Locale.ROOT).contains("windows");
    }

    String vfsSrcPath() {
        return this.resourcePathToPlatformPath(VirtualFileSystemImpl.absoluteResourcePath(this.vfsRoot, VFS_SRC));
    }

    String vfsVenvPath() {
        return this.resourcePathToPlatformPath(VirtualFileSystemImpl.absoluteResourcePath(this.vfsRoot, VFS_VENV));
    }

    private String resourcePathToPlatformPath(String resourcePath) {
        if (!resourcePath.startsWith("/")) {
            throw new IllegalArgumentException("Relative resource path");
        }
        String relative = resourcePath.substring(1);
        if (relative.length() <= this.vfsRoot.length() || !relative.startsWith(this.vfsRoot)) {
            String msg = "Resource path is expected to start with '/" + this.vfsRoot + "' but was '" + resourcePath + "'.\nPlease also ensure that your virtual file system resources root directory is '" + this.vfsRoot + "'";
            throw new IllegalArgumentException(msg);
        }
        String path = resourcePath.substring(this.vfsRoot.length() + 2);
        if (!PLATFORM_SEPARATOR.equals(RESOURCE_SEPARATOR)) {
            path = path.replace(RESOURCE_SEPARATOR, PLATFORM_SEPARATOR);
        }
        Object absolute = this.mountPoint.resolve(path).toString();
        if (resourcePath.endsWith(RESOURCE_SEPARATOR) && !((String)absolute).endsWith(PLATFORM_SEPARATOR)) {
            absolute = (String)absolute + PLATFORM_SEPARATOR;
        }
        return absolute;
    }

    private String platformPathToResourcePath(String inputPath) {
        String mountPointString = this.mountPoint.toString();
        Object path = inputPath;
        assert (((String)path).startsWith(mountPointString)) : String.format("path `%s` expected to start with `%s`", path, mountPointString);
        if (((String)path).startsWith(mountPointString)) {
            path = ((String)path).substring(mountPointString.length());
        }
        if (!PLATFORM_SEPARATOR.equals(RESOURCE_SEPARATOR)) {
            path = ((String)path).replace(PLATFORM_SEPARATOR, RESOURCE_SEPARATOR);
        }
        if (((String)path).endsWith(RESOURCE_SEPARATOR)) {
            path = ((String)path).substring(0, ((String)path).length() - RESOURCE_SEPARATOR.length());
        }
        path = "/" + this.vfsRoot + (String)path;
        return path;
    }

    private String toCaseComparable(String file) {
        return this.caseInsensitive ? file.toLowerCase(Locale.ROOT) : file;
    }

    private String multipleLocationsErrorMessage(String fmt, Object ... args) {
        String suffix = "";
        if (!this.allowMultipleLocations) {
            suffix = " This is an internal error. Please report it on https://github.com/oracle/graalpython.";
        }
        return String.format(fmt + suffix, args);
    }

    private IllegalStateException fileDirDuplicateMismatchError(String key) {
        throw new IllegalStateException(this.multipleLocationsErrorMessage("Entry %s is a file in one virtual filesystem location, but a directory in another.", key));
    }

    private void initEntries() {
        String filelistPath = VirtualFileSystemImpl.resourcePath(this.vfsRoot, FILES_LIST);
        String absFilelistPath = VirtualFileSystemImpl.absoluteResourcePath(this.vfsRoot, FILES_LIST);
        String srcPath = VirtualFileSystemImpl.absoluteResourcePath(this.vfsRoot, VFS_SRC);
        String venvPath = VirtualFileSystemImpl.absoluteResourcePath(this.vfsRoot, VFS_VENV);
        List<URL> filelistUrls = this.getFilelistURLs(filelistPath);
        for (URL url : filelistUrls) {
            try {
                InputStream stream = url.openStream();
                try {
                    String line;
                    if (stream == null) {
                        VirtualFileSystemImpl.warn("VFS.initEntries: could not read resource %s", filelistPath);
                        return;
                    }
                    BufferedReader br = new BufferedReader(new InputStreamReader(stream));
                    VirtualFileSystemImpl.finest("VFS entries:", new Object[0]);
                    while ((line = br.readLine()) != null) {
                        Path p;
                        if (line.isBlank()) continue;
                        String projPath = VirtualFileSystemImpl.absoluteResourcePath(this.vfsRoot, PROJ_DIR);
                        if (!this.projWarning && line.startsWith(projPath)) {
                            this.projWarning = true;
                            LOGGER.warning("");
                            LOGGER.warning(String.format("%s source root was deprecated, use %s instead.", projPath, srcPath));
                            LOGGER.warning("");
                        }
                        String platformPath = this.resourcePathToPlatformPath(line);
                        int i = this.mountPoint.toString().length();
                        DirEntry parent = null;
                        do {
                            DirEntry dirEntry;
                            String dir;
                            String dirKey;
                            BaseEntry genericEntry;
                            if ((genericEntry = this.vfsEntries.get(dirKey = this.toCaseComparable(dir = platformPath.substring(0, i)))) instanceof DirEntry) {
                                DirEntry de;
                                dirEntry = de = (DirEntry)genericEntry;
                            } else if (genericEntry == null) {
                                dirEntry = new DirEntry(dir);
                                this.vfsEntries.put(dirKey, dirEntry);
                                VirtualFileSystemImpl.finest("  %s", dirEntry.getResourcePath());
                                if (parent != null) {
                                    parent.entries.add(dirEntry);
                                }
                            } else {
                                throw this.fileDirDuplicateMismatchError(dirKey);
                            }
                            parent = dirEntry;
                            ++i;
                        } while ((i = platformPath.indexOf(PLATFORM_SEPARATOR, i)) != -1);
                        assert (parent != null);
                        if (platformPath.endsWith(PLATFORM_SEPARATOR)) continue;
                        FileEntry fileEntry = new FileEntry(platformPath);
                        BaseEntry previous = this.vfsEntries.put(this.toCaseComparable(platformPath), fileEntry);
                        if (previous != null) {
                            if (previous instanceof DirEntry) {
                                throw this.fileDirDuplicateMismatchError(platformPath);
                            }
                            if (filelistUrls.size() > 1 && !line.startsWith(venvPath) && !line.equals(absFilelistPath)) {
                                VirtualFileSystemImpl.reportFailedMultiVFSCheck(this.multipleLocationsErrorMessage("There are duplicate entries originating from different virtual filesystem instances. The duplicate entries path: %s.", line));
                            }
                            VirtualFileSystemImpl.fine(this.multipleLocationsErrorMessage("Duplicate entries virtual filesystem entries: " + line, new Object[0]), new Object[0]);
                        }
                        VirtualFileSystemImpl.finest("  %s", fileEntry.getResourcePath());
                        parent.entries.add(fileEntry);
                        if (!this.extractOnStartup || !this.shouldExtract(p = Paths.get(fileEntry.getPlatformPath(), new String[0]))) continue;
                        this.getExtractedPath(p);
                    }
                }
                finally {
                    if (stream == null) continue;
                    stream.close();
                }
            }
            catch (IOException ex) {
                throw new IllegalStateException(String.format("IO error during VirtualFileSystem initialization from location '%s'.", url), ex);
            }
        }
        if (this.vfsEntries.isEmpty()) {
            VirtualFileSystemImpl.warn("VFS.getEntry: no entries after init", new Object[0]);
        }
        if (filelistUrls.size() > 1) {
            this.validateMultipleVFSLocations(filelistUrls);
        }
    }

    private List<URL> getFilelistURLs(String filelistPath) {
        ArrayList<URL> filelistUrls;
        try {
            filelistUrls = Collections.list(this.resourceLoadingClass.getClassLoader().getResources(filelistPath));
        }
        catch (IOException e) {
            throw new IllegalStateException("IO error during reading the VirtualFileSystem metadata", e);
        }
        if (filelistUrls.isEmpty()) {
            throw new IllegalStateException(String.format("Could not find VirtualFileSystem metadata in Java resources. Resource not found: %s.", filelistPath));
        }
        if (filelistUrls.size() > 1 || this.vfsRootURL != null) {
            String locations = filelistUrls.stream().map(URL::toString).collect(Collectors.joining("\n"));
            if (this.vfsRootURL != null) {
                Optional<URL> filelist = filelistUrls.stream().filter(x -> x.toString().startsWith(this.vfsRootURL)).findFirst();
                if (filelist.isPresent()) {
                    LOGGER.fine(String.format("Found multiple virtual filesystem instances. Using '%s' as requested by System property '%s'.", this.vfsRootURL, MULTI_VFS_SINGLE_ROOT_URL_PROP));
                    return List.of(filelist.get());
                }
                throw new IllegalStateException(String.format("Could not find virtual filesystem with root '%s' as requested by System property '%s'. Found the following virtual filesystem locations:\n%s", this.vfsRootURL, MULTI_VFS_SINGLE_ROOT_URL_PROP, locations));
            }
            String message = String.format("Found multiple embedded virtual filesystem instances in the following locations:\n%s", locations);
            if (!this.allowMultipleLocations) {
                throw new IllegalStateException(String.format("%s\n\nIt is recommended to use virtual filesystem isolation. See the documentation of VirtualFilesystem$Builder#resourceDirectory(resourcePath) to learn about isolating multiple virtual filesystems in one application.\n\nUse experimental and unstable system property -D%s=URL to select only one virtual filesystem.The URL must point to the root of the desired virtual filesystem instance.\n\nUse experimental and unstable system property -D%s=true to merge the filesystems into one. In case of duplicate entries in multiple filesystems, one is chosen at random.", message, MULTI_VFS_SINGLE_ROOT_URL_PROP, MULTI_VFS_ALLOW_PROP));
            }
            LOGGER.fine(message + "\n\nThe virtual filesystems will be merged. ");
        }
        return filelistUrls;
    }

    private static void reportFailedMultiVFSCheck(String message) {
        if (!Boolean.getBoolean(MULTI_VSF_CHECKS_AS_WARNING_PROP)) {
            throw new IllegalStateException(message + String.format(" This error can be turned into a warning with -D%s=true", MULTI_VSF_CHECKS_AS_WARNING_PROP));
        }
        VirtualFileSystemImpl.warn(message, new Object[0]);
    }

    private void validateMultipleVFSLocations(List<URL> filelistUrls) {
        ArrayList<URL> contentsUrls;
        ArrayList<URL> installedUrls;
        try {
            installedUrls = Collections.list(this.resourceLoadingClass.getClassLoader().getResources(VirtualFileSystemImpl.resourcePath(this.vfsRoot, INSTALLED_FILE)));
        }
        catch (IOException e) {
            VirtualFileSystemImpl.warn("Cannot check compatibility of the merged virtual environments. Cannot read list of packages installed in the virtual environments. IOException: " + e.getMessage(), new Object[0]);
            return;
        }
        if (installedUrls.size() != filelistUrls.size()) {
            VirtualFileSystemImpl.warn("Could not read the list of installed packages for all virtual environments. Lists found:\n%s", installedUrls.stream().map(URL::toString).collect(Collectors.joining("\n")));
        }
        HashMap<String, String> pkgToVersion = new HashMap<String, String>();
        for (URL installedUrl : installedUrls) {
            try {
                InputStream stream = installedUrl.openStream();
                try {
                    String[] packages;
                    if (stream == null) {
                        throw new IOException("openStream() returned null");
                    }
                    for (String pkgAndVer : packages = new String(stream.readAllBytes()).split("(\\n|\\r\\n)")) {
                        if (pkgAndVer.isBlank() || pkgAndVer.trim().startsWith("#")) continue;
                        String[] parts = pkgAndVer.split("==");
                        if (parts.length != 2) {
                            VirtualFileSystemImpl.warn("Cannot parse package specification '%s' in %s. Ignoring it.", pkgAndVer, installedUrl);
                            continue;
                        }
                        String pkg = parts[0];
                        String version = parts[1];
                        String originalVer = pkgToVersion.put(pkg, version);
                        if (originalVer == null || originalVer.equals(version)) continue;
                        VirtualFileSystemImpl.reportFailedMultiVFSCheck(String.format("Package '%s' is installed in different versions ('%s' and '%s') in different virtual environments. This may result in disrupted functionality of the package or packages depending on it.", parts[0], originalVer, version));
                    }
                }
                finally {
                    if (stream == null) continue;
                    stream.close();
                }
            }
            catch (IOException e) {
                VirtualFileSystemImpl.warn("Cannot read list of installed packages in '%s'. Error: %s", installedUrl, e.getMessage());
            }
        }
        try {
            contentsUrls = Collections.list(this.resourceLoadingClass.getClassLoader().getResources(VirtualFileSystemImpl.resourcePath(this.vfsRoot, CONTENTS_FILE)));
        }
        catch (IOException e) {
            VirtualFileSystemImpl.warn("Cannot check compatibility of the merged virtual environments. Cannot read GraalPy version of the virtual environments. IOException: " + e.getMessage(), new Object[0]);
            return;
        }
        if (contentsUrls.size() != filelistUrls.size()) {
            VirtualFileSystemImpl.warn("Could not read the GraalPy version for all virtual environments. Version files found:\n%s", contentsUrls.stream().map(URL::toString).collect(Collectors.joining("\n")));
        }
        String graalPyVersion = null;
        URL graalPyVersionUrl = null;
        for (URL installedUrl : installedUrls) {
            try {
                InputStream stream = installedUrl.openStream();
                try {
                    if (stream == null) {
                        throw new IOException("openStream() returned null");
                    }
                    BufferedReader reader = new BufferedReader(new InputStreamReader(stream));
                    String ver = reader.readLine();
                    if (ver == null) continue;
                    ver = ver.trim();
                    if (graalPyVersion == null) {
                        graalPyVersion = ver;
                        graalPyVersionUrl = installedUrl;
                        continue;
                    }
                    if (graalPyVersion.equals(ver)) continue;
                    VirtualFileSystemImpl.reportFailedMultiVFSCheck("Following virtual environments appear to have been created by different GraalPy versions:\n" + String.valueOf(graalPyVersionUrl) + "\n" + String.valueOf(installedUrl));
                    break;
                }
                finally {
                    if (stream == null) continue;
                    stream.close();
                }
            }
            catch (IOException e) {
                VirtualFileSystemImpl.warn("Cannot read GraalPy version from '%s'. Error: %s", installedUrl, e.getMessage());
            }
        }
    }

    byte[] readResource(String path) throws IOException {
        List<URL> urls = Collections.list(this.resourceLoadingClass.getClassLoader().getResources(path.substring(1)));
        if (this.vfsRootURL != null) {
            urls = this.getURLInRoot(urls);
        }
        if (urls.isEmpty()) {
            throw new IllegalStateException("VFS.initEntries: could not find resource: " + path);
        }
        try (InputStream stream = urls.get(0).openStream();){
            int n;
            if (stream == null) {
                throw new IllegalStateException("VFS.initEntries: could not read resource: " + path);
            }
            ByteArrayOutputStream buffer = new ByteArrayOutputStream();
            byte[] data = new byte[4096];
            while ((n = stream.readNBytes(data, 0, data.length)) != 0) {
                buffer.write(data, 0, n);
            }
            buffer.flush();
            byte[] byArray = buffer.toByteArray();
            return byArray;
        }
    }

    private List<URL> getURLInRoot(List<URL> urls) {
        return urls.stream().filter(x -> x.toString().startsWith(this.vfsRootURL)).findFirst().map(List::of).orElseGet(List::of);
    }

    private BaseEntry getEntry(Path inputPath) {
        Path path = VirtualFileSystemImpl.toAbsoluteNormalizedPath(inputPath);
        return this.vfsEntries.get(this.toCaseComparable(path.toString()));
    }

    private boolean pathIsInVfs(Path path) {
        assert (VirtualFileSystemImpl.isNormalized(path));
        return this.toCaseComparable(path.toString()).startsWith(this.mountPointLowerCase);
    }

    private static boolean isNormalized(Path path) {
        for (Path name : path) {
            String strName = name.toString();
            if (!".".equals(strName) && !"..".equals(strName)) continue;
            return false;
        }
        return true;
    }

    private boolean shouldExtract(Path path) {
        boolean ret = this.extractFilter != null && this.extractFilter.test(path);
        VirtualFileSystemImpl.finest("VFS.shouldExtract '%s' %s", path, ret);
        return ret;
    }

    private static boolean followLinks(LinkOption ... linkOptions) {
        if (linkOptions != null) {
            for (LinkOption o : linkOptions) {
                if (Objects.requireNonNull(o) != LinkOption.NOFOLLOW_LINKS) continue;
                return false;
            }
        }
        return true;
    }

    private Path getExtractedPath(Path path) {
        assert (this.shouldExtract(path));
        return this.extractPath(path, true);
    }

    private Path extractPath(Path path, boolean extractLibsDir) {
        assert (this.extractDir != null);
        try {
            BaseEntry entry = this.getEntry(path);
            if (entry == null) {
                return null;
            }
            Path relPath = this.mountPoint.relativize(Paths.get(entry.getPlatformPath(), new String[0]));
            Path xPath = this.extractDir.resolve(relPath);
            if (!Files.exists(xPath, new LinkOption[0])) {
                if (entry instanceof FileEntry) {
                    Path pkgDir;
                    FileEntry fileEntry = (FileEntry)entry;
                    Path parent = xPath.getParent();
                    assert (parent == null || !Files.exists(parent, new LinkOption[0]) || Files.isDirectory(parent, new LinkOption[0]));
                    if (parent == null) {
                        throw new NullPointerException("Parent is null during extracting path.");
                    }
                    Files.createDirectories(parent, new FileAttribute[0]);
                    Files.write(xPath, fileEntry.getData(), new OpenOption[0]);
                    VirtualFileSystemImpl.finest("extracted '%s' -> '%s'", path, xPath);
                    if (extractLibsDir && (pkgDir = VirtualFileSystemImpl.getPythonPackageDir(path)) != null) {
                        Path libsDir = Paths.get(String.valueOf(pkgDir) + ".libs", new String[0]);
                        this.extract(libsDir);
                    }
                } else if (entry instanceof DirEntry) {
                    Files.createDirectories(xPath, new FileAttribute[0]);
                } else {
                    return path;
                }
            }
            return xPath;
        }
        catch (IOException e) {
            throw new RuntimeException(String.format("Error while extracting virtual filesystem path '%s' to the disk", path), e);
        }
    }

    private static Path getPythonPackageDir(Path path) {
        Path prev = null;
        Path p = path;
        while ((p = p.getParent()) != null) {
            Path fileName = p.getFileName();
            if (fileName != null && "site-packages".equals(fileName.toString())) {
                return prev;
            }
            prev = p;
        }
        return null;
    }

    private void extract(Path path) throws IOException {
        BaseEntry entry = this.getEntry(path);
        if (entry instanceof FileEntry) {
            this.extractPath(path, false);
        } else if (entry != null && ((DirEntry)entry).entries != null) {
            for (BaseEntry be : ((DirEntry)entry).entries) {
                this.extract(Path.of(be.getPlatformPath(), new String[0]));
            }
        }
    }

    void extractResources(Path externalResourceDirectory) throws IOException {
        VirtualFileSystemImpl.fine("VFS.extractResources '%s'", externalResourceDirectory);
        for (BaseEntry entry : this.vfsEntries.values()) {
            String resourcePath = entry.getResourcePath();
            assert (resourcePath.length() >= this.vfsRoot.length() + 1);
            if (resourcePath.length() == this.vfsRoot.length() + 1) continue;
            Path destFile = externalResourceDirectory.resolve(Path.of(resourcePath.substring(this.vfsRoot.length() + 2), new String[0]));
            if (entry instanceof DirEntry) {
                Files.createDirectories(destFile, new FileAttribute[0]);
                continue;
            }
            assert (entry instanceof FileEntry);
            Path parent = destFile.getParent();
            if (parent != null) {
                Files.createDirectories(parent, new FileAttribute[0]);
            }
            VirtualFileSystemImpl.finest("VFS.extractResources '%s' -> '%s'", resourcePath, destFile);
            Files.write(destFile, this.readResource(resourcePath), new OpenOption[0]);
        }
    }

    private synchronized FileSystemProvider getDefaultFileSystem() {
        if (this.defaultFileSystemProvider == null) {
            for (FileSystemProvider provider : FileSystemProvider.installedProviders()) {
                if (!"file".equals(provider.getScheme())) continue;
                this.defaultFileSystemProvider = provider;
            }
        }
        return this.defaultFileSystemProvider;
    }

    public Path parsePath(URI uri) {
        Objects.requireNonNull(uri);
        if (uri.getScheme().equals("file")) {
            Path path = this.getDefaultFileSystem().getPath(uri);
            VirtualFileSystemImpl.finest("VFS.parsePath '%s' -> '%s'", uri, path);
            return path;
        }
        String msg = "Unsupported URI scheme '%s'";
        VirtualFileSystemImpl.finer(msg, uri.getScheme());
        throw new UnsupportedOperationException(String.format(msg, uri.getScheme()));
    }

    public Path parsePath(String path) {
        Objects.requireNonNull(path);
        Path p = Paths.get(path, new String[0]);
        VirtualFileSystemImpl.finer("VFS.parsePath '%s' -> '%s'", path, p);
        return p;
    }

    public void checkAccess(Path p, Set<? extends AccessMode> modes, LinkOption ... linkOptions) throws IOException {
        Objects.requireNonNull(p);
        Objects.requireNonNull(modes);
        Path path = VirtualFileSystemImpl.toAbsoluteNormalizedPath(p);
        boolean extractable = this.shouldExtract(path);
        if (extractable && VirtualFileSystemImpl.followLinks(linkOptions)) {
            Path extractedPath = this.getExtractedPath(path);
            if (extractedPath != null) {
                this.extractedFilesFS.checkAccess(extractedPath, modes, linkOptions);
                return;
            }
            VirtualFileSystemImpl.finer("VFS.checkAccess could not extract path '%s'", p);
            throw new NoSuchFileException(String.format("no such file or directory: '%s'", path));
        }
        if (modes.contains((Object)AccessMode.WRITE)) {
            throw VirtualFileSystemImpl.securityException("VFS.checkAccess", String.format("read-only filesystem, write access not supported '%s'", path));
        }
        if (modes.contains((Object)AccessMode.EXECUTE)) {
            throw VirtualFileSystemImpl.securityException("VFS.checkAccess", String.format("execute access not supported for  '%s'", p));
        }
        if (this.getEntry(path) == null) {
            String msg = String.format("no such file or directory: '%s'", path);
            VirtualFileSystemImpl.finer("VFS.checkAccess %s", msg);
            throw new NoSuchFileException(msg);
        }
        VirtualFileSystemImpl.finer("VFS.checkAccess %s OK", path);
    }

    public void createDirectory(Path d, FileAttribute<?> ... attrs) throws IOException {
        throw VirtualFileSystemImpl.securityException("VFS.createDirectory", String.format("read-only filesystem, create directory not supported '%s'", d));
    }

    public void delete(Path p) throws IOException {
        throw VirtualFileSystemImpl.securityException("VFS.delete", String.format("read-only filesystem, delete not supported: '%s'", p));
    }

    public SeekableByteChannel newByteChannel(Path p, Set<? extends OpenOption> options, FileAttribute<?> ... attrs) throws IOException {
        Objects.requireNonNull(p);
        Objects.requireNonNull(options);
        for (OpenOption openOption : options) {
            Objects.requireNonNull(openOption);
        }
        Objects.requireNonNull(attrs);
        final Path path = VirtualFileSystemImpl.toAbsoluteNormalizedPath(p);
        boolean bl = this.shouldExtract(path);
        if (bl) {
            if (!options.contains(LinkOption.NOFOLLOW_LINKS)) {
                Path extractedPath = this.getExtractedPath(path);
                if (extractedPath != null) {
                    return this.extractedFilesFS.newByteChannel(extractedPath, options, new FileAttribute[0]);
                }
                VirtualFileSystemImpl.finer("VFS.newByteChannel could not extract path '%s'", p);
                throw new FileNotFoundException(String.format("no such file or directory: '%s'", p));
            }
            String msg = String.format("can't create byte channel for '%s' (NOFOLLOW_LINKS specified)", p);
            VirtualFileSystemImpl.finer("VFS.newByteChannel '%s'", msg);
            throw new IOException(msg);
        }
        VirtualFileSystemImpl.checkNoWriteOption(options, p);
        BaseEntry entry = this.getEntry(path);
        if (entry == null) {
            String msg = String.format("No such file or directory: '%s'", path);
            VirtualFileSystemImpl.finer("VFS.newByteChannel '%s'", path);
            throw new FileNotFoundException(msg);
        }
        if (!(entry instanceof FileEntry)) {
            VirtualFileSystemImpl.finer("VFS.newByteChannel Is a directory '%s'", path);
            throw new FileSystemException(path.toString(), null, "Is a directory");
        }
        final FileEntry fileEntry = (FileEntry)entry;
        return new SeekableByteChannel(){
            long position = 0L;
            final byte[] bytes = fileEntry.getData();
            final /* synthetic */ VirtualFileSystemImpl this$0;
            {
                this.this$0 = this$0;
            }

            @Override
            public int read(ByteBuffer dst) throws IOException {
                if (this.position > (long)this.bytes.length) {
                    return -1;
                }
                if (this.position == (long)this.bytes.length) {
                    return 0;
                }
                int length = Math.min(this.bytes.length - (int)this.position, dst.remaining());
                dst.put(this.bytes, (int)this.position, length);
                this.position += (long)length;
                if (dst.hasRemaining()) {
                    ++this.position;
                }
                return length;
            }

            @Override
            public int write(ByteBuffer src) throws IOException {
                VirtualFileSystemImpl.finer("VFS.newByteChannel '%s'", String.format("read-only filesystem: '%s'", path));
                throw new NonWritableChannelException();
            }

            @Override
            public long position() throws IOException {
                return this.position;
            }

            @Override
            public SeekableByteChannel position(long newPosition) throws IOException {
                this.position = Math.max(0L, newPosition);
                return this;
            }

            @Override
            public long size() throws IOException {
                return this.bytes.length;
            }

            @Override
            public SeekableByteChannel truncate(long size) throws IOException {
                VirtualFileSystemImpl.finer("VFS.newByteChannel '%s'", String.format("read-only filesystem: '%s'", path));
                throw new NonWritableChannelException();
            }

            @Override
            public boolean isOpen() {
                return true;
            }

            @Override
            public void close() throws IOException {
            }
        };
    }

    private static void checkNoWriteOption(Set<? extends OpenOption> options, Path path) {
        boolean write;
        HashSet<? extends OpenOption> writeOptions = new HashSet<OpenOption>(options);
        boolean read = writeOptions.contains(StandardOpenOption.READ);
        writeOptions.removeAll(READ_OPTIONS);
        if (read) {
            writeOptions.remove(StandardOpenOption.APPEND);
        }
        boolean bl = write = !writeOptions.isEmpty();
        if (write) {
            throw VirtualFileSystemImpl.securityException("VFS.newByteChannel", String.format("read-only filesystem, can't create byte channel for write access: '%s'", path));
        }
    }

    public DirectoryStream<Path> newDirectoryStream(Path d, final DirectoryStream.Filter<? super Path> filter) throws IOException {
        Objects.requireNonNull(d);
        final Path dir = VirtualFileSystemImpl.toAbsoluteNormalizedPath(d);
        Objects.requireNonNull(filter);
        BaseEntry entry = this.getEntry(dir);
        if (entry instanceof FileEntry) {
            VirtualFileSystemImpl.finer("VFS.newDirectoryStream not a directory %s", dir);
            throw new NotDirectoryException(dir.toString());
        }
        if (entry instanceof DirEntry) {
            final DirEntry dirEntry = (DirEntry)entry;
            return new DirectoryStream<Path>(this){
                final /* synthetic */ VirtualFileSystemImpl this$0;
                {
                    this.this$0 = this$0;
                }

                @Override
                public void close() throws IOException {
                }

                @Override
                public Iterator<Path> iterator() {
                    VirtualFileSystemImpl.finest("VFS.newDirectoryStream %s entries:", dir);
                    return dirEntry.entries.stream().filter(e -> {
                        boolean accept;
                        try {
                            accept = filter.accept(Path.of(e.platformPath, new String[0]));
                            VirtualFileSystemImpl.finest("VFS.newDirectoryStream entry %s accept: %s", e.platformPath, accept);
                        }
                        catch (IOException ex) {
                            LOGGER.log(Level.WARNING, ex, () -> String.format("Error when iterating entries of '%s'", dir));
                            return false;
                        }
                        return accept;
                    }).map(e -> Path.of(e.getPlatformPath(), new String[0])).iterator();
                }
            };
        }
        throw new NoSuchFileException(dir.toString());
    }

    private static Path toAbsoluteNormalizedPath(Path path) {
        if (path.isAbsolute()) {
            return path;
        }
        return path.toAbsolutePath().normalize();
    }

    public Path toAbsolutePath(Path path) {
        Objects.requireNonNull(path);
        if (path.isAbsolute()) {
            return path;
        }
        return path.toAbsolutePath();
    }

    public Path toRealPath(Path p, LinkOption ... linkOptions) throws IOException {
        Path path;
        Objects.requireNonNull(p);
        Path result = path = VirtualFileSystemImpl.toAbsoluteNormalizedPath(p);
        if (this.shouldExtract(path) && VirtualFileSystemImpl.followLinks(linkOptions) && (result = this.getExtractedPath(path)) == null) {
            VirtualFileSystemImpl.warn("no VFS entry for '%s'", p);
            throw new NoSuchFileException(String.format("no such file or directory: '%s'", p));
        }
        VirtualFileSystemImpl.finer("VFS.toRealPath '%s' -> '%s'", path, result);
        return result;
    }

    public Map<String, Object> readAttributes(Path p, String attributes, LinkOption ... options) throws IOException {
        int n;
        Objects.requireNonNull(p);
        Path path = VirtualFileSystemImpl.toAbsoluteNormalizedPath(p);
        boolean extractable = this.shouldExtract(path);
        if (extractable && VirtualFileSystemImpl.followLinks(options)) {
            Path extractedPath = this.getExtractedPath(path);
            if (extractedPath != null) {
                return this.extractedFilesFS.readAttributes(extractedPath, attributes, options);
            }
            VirtualFileSystemImpl.finer("VFS.readAttributes could not extract path '%s'", p);
            throw new NoSuchFileException(String.format("no such file or directory: '%s'", path));
        }
        BaseEntry entry = this.getEntry(path);
        if (entry == null) {
            String msg = String.format("no such file or directory: '%s'", path);
            VirtualFileSystemImpl.finer("VFS.readAttributes %s", msg);
            throw new NoSuchFileException(msg);
        }
        HashMap<String, Object> attrs = new HashMap<String, Object>();
        if (attributes.startsWith("unix:") || attributes.startsWith("posix:")) {
            VirtualFileSystemImpl.finer("VFS.readAttributes unsupported attributes '%s' %s", path, attributes);
            throw new UnsupportedOperationException();
        }
        attrs.put("creationTime", FileTime.fromMillis(0L));
        attrs.put("lastModifiedTime", FileTime.fromMillis(0L));
        attrs.put("lastAccessTime", FileTime.fromMillis(0L));
        attrs.put("isRegularFile", !extractable && entry instanceof FileEntry);
        attrs.put("isDirectory", entry instanceof DirEntry);
        attrs.put("isSymbolicLink", extractable);
        attrs.put("isOther", false);
        if (entry instanceof FileEntry) {
            FileEntry fileEntry = (FileEntry)entry;
            n = fileEntry.getData().length;
        } else {
            n = 0;
        }
        attrs.put("size", Long.valueOf(n));
        attrs.put("mode", 365);
        attrs.put("dev", 0L);
        attrs.put("nlink", 1);
        attrs.put("uid", 0);
        attrs.put("gid", 0);
        attrs.put("ctime", FileTime.fromMillis(0L));
        VirtualFileSystemImpl.finer("VFS.readAttributes '%s' %s", path, attrs);
        return attrs;
    }

    public void setCurrentWorkingDirectory(Path d) {
        throw new RuntimeException("should not reach here");
    }

    public void copy(Path s, Path t, CopyOption ... options) {
        throw new RuntimeException("should not reach here");
    }

    public void move(Path s, Path t, CopyOption ... options) {
        throw new RuntimeException("should not reach here");
    }

    public Charset getEncoding(Path p) {
        Objects.requireNonNull(p);
        return null;
    }

    public void createSymbolicLink(Path l, Path t, FileAttribute<?> ... attrs) throws IOException {
        Objects.requireNonNull(l);
        Objects.requireNonNull(t);
        throw VirtualFileSystemImpl.securityException("VFS.createSymbolicLink", String.format("read-only filesystem, can't create symbolic link from '%s' to '%s'", l, t));
    }

    public void createLink(Path l, Path e) throws IOException {
        Objects.requireNonNull(l);
        Objects.requireNonNull(e);
        throw VirtualFileSystemImpl.securityException("VFS.createLink", String.format("read-only filesystem, can't create link '%s' to '%s'", l, e));
    }

    public Path readSymbolicLink(Path link) throws IOException {
        Objects.requireNonNull(link);
        Path path = VirtualFileSystemImpl.toAbsoluteNormalizedPath(link);
        if (this.shouldExtract(path)) {
            Path result = this.getExtractedPath(path);
            if (result != null) {
                VirtualFileSystemImpl.finer("VFS.readSymbolicLink '%s' '%s'", link, result);
                return result;
            }
            VirtualFileSystemImpl.finer("VFS.readSymbolicLink could not extract path '%s'", link);
            throw new NoSuchFileException(String.format("no such file or directory: '%s'", path));
        }
        if (this.getEntry(path) == null) {
            VirtualFileSystemImpl.finer("VFS.readSymbolicLink no entry for path '%s'", link);
            throw new NoSuchFileException(String.format("no such file or directory: '%s'", path));
        }
        throw new NotLinkException(link.toString());
    }

    public void setAttribute(Path p, String attribute, Object value, LinkOption ... options) throws IOException {
        Objects.requireNonNull(p);
        throw VirtualFileSystemImpl.securityException("VFS.setAttribute", String.format("read-only filesystem, can't set attribute '%s' for '%s", attribute, p));
    }

    public String getMimeType(Path p) {
        Objects.requireNonNull(p);
        return null;
    }

    public Path getTempDirectory() {
        throw new RuntimeException("should not reach here");
    }

    private static void warn(String msgFormat, Object ... args) {
        if (LOGGER.isLoggable(Level.WARNING)) {
            LOGGER.log(Level.WARNING, String.format(msgFormat, args));
        }
    }

    private static void fine(String msgFormat, Object ... args) {
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.log(Level.FINE, String.format(msgFormat, args));
        }
    }

    private static void finer(String msgFormat, Object ... args) {
        if (LOGGER.isLoggable(Level.FINER)) {
            LOGGER.log(Level.FINER, String.format(msgFormat, args));
        }
    }

    private static void finest(String msgFormat, Object ... args) {
        if (LOGGER.isLoggable(Level.FINEST)) {
            LOGGER.log(Level.FINEST, String.format(msgFormat, args));
        }
    }

    private static SecurityException securityException(String from, String msg) {
        VirtualFileSystemImpl.finer("%s %s", from, msg);
        throw new SecurityException(msg);
    }

    static {
        LOGGER.setUseParentHandlers(false);
        ConsoleHandler consoleHandler = new ConsoleHandler();
        consoleHandler.setFormatter(new SimpleFormatter(){

            @Override
            public synchronized String format(LogRecord lr) {
                if (lr.getThrown() != null) {
                    StringWriter sw = new StringWriter();
                    PrintWriter pw = new PrintWriter(sw);
                    lr.getThrown().printStackTrace(pw);
                    pw.close();
                    return String.format("%s: %s\n%s", lr.getLevel().getName(), lr.getMessage(), sw.toString());
                }
                return String.format("%s: %s\n", lr.getLevel().getName(), lr.getMessage());
            }
        });
        LOGGER.addHandler(consoleHandler);
        CONTENTS_FILE = VirtualFileSystemImpl.resourcePath(VFS_VENV, "contents");
        INSTALLED_FILE = VirtualFileSystemImpl.resourcePath(VFS_VENV, "installed.txt");
        PLATFORM_SEPARATOR = Paths.get("", new String[0]).getFileSystem().getSeparator();
        RESOURCE_SEPARATOR = String.valueOf('/');
        READ_OPTIONS = Set.of(StandardOpenOption.READ, StandardOpenOption.DSYNC, StandardOpenOption.SPARSE, StandardOpenOption.SYNC, StandardOpenOption.TRUNCATE_EXISTING, LinkOption.NOFOLLOW_LINKS);
    }

    private static final class DeleteTempDir
    extends Thread {
        private final Path extractDir;
        private static final SimpleFileVisitor<Path> deleteVisitor = new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                Files.delete(file);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                Files.delete(dir);
                return FileVisitResult.CONTINUE;
            }
        };

        DeleteTempDir(Path extractDir) {
            this.extractDir = extractDir;
        }

        @Override
        public void run() {
            this.removeExtractDir();
        }

        private void removeExtractDir() {
            if (this.extractDir != null && Files.exists(this.extractDir, new LinkOption[0])) {
                try {
                    Files.walkFileTree(this.extractDir, deleteVisitor);
                }
                catch (IOException e) {
                    System.err.format("Could not delete temp directory '%s': %s", this.extractDir, e);
                }
            }
        }
    }

    private abstract class BaseEntry {
        final String platformPath;

        private BaseEntry(String platformPath) {
            this.platformPath = platformPath;
        }

        String getPlatformPath() {
            return this.platformPath;
        }

        String getResourcePath() {
            return VirtualFileSystemImpl.this.platformPathToResourcePath(this.platformPath);
        }
    }

    private final class DirEntry
    extends BaseEntry {
        List<BaseEntry> entries;

        DirEntry(String platformPath) {
            super(platformPath);
            this.entries = new ArrayList<BaseEntry>();
        }
    }

    private final class FileEntry
    extends BaseEntry {
        private byte[] data;

        public FileEntry(String path) {
            super(path);
        }

        private byte[] getData() throws IOException {
            if (this.data == null) {
                byte[] loaded = VirtualFileSystemImpl.this.readResource(this.getResourcePath());
                VarHandle.storeStoreFence();
                this.data = loaded;
            }
            return this.data;
        }
    }
}

