/*
 * Decompiled with CFR 0.152.
 */
package org.apache.linkis.storage.fs.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Paths;
import java.nio.file.attribute.GroupPrincipal;
import java.nio.file.attribute.PosixFileAttributeView;
import java.nio.file.attribute.PosixFileAttributes;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.nio.file.attribute.UserPrincipal;
import java.nio.file.attribute.UserPrincipalLookupService;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.linkis.common.io.FsPath;
import org.apache.linkis.common.utils.Utils;
import org.apache.linkis.storage.domain.FsPathListWithError;
import org.apache.linkis.storage.errorcode.LinkisStorageErrorCodeSummary;
import org.apache.linkis.storage.exception.StorageWarnException;
import org.apache.linkis.storage.fs.FileSystem;
import org.apache.linkis.storage.utils.StorageConfiguration;
import org.apache.linkis.storage.utils.StorageUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LocalFileSystem
extends FileSystem {
    private static final Logger LOG = LoggerFactory.getLogger(LocalFileSystem.class);
    private Map<String, String> properties;
    private String group;

    @Override
    public String listRoot() throws IOException {
        return "/";
    }

    @Override
    public long getTotalSpace(FsPath dest) throws IOException {
        String path = dest.getPath();
        LOG.info("Get total space with path:" + path);
        return new File(path).getTotalSpace();
    }

    @Override
    public long getFreeSpace(FsPath dest) throws IOException {
        String path = dest.getPath();
        LOG.info("Get free space with path:" + path);
        return new File(path).getFreeSpace();
    }

    @Override
    public long getUsableSpace(FsPath dest) throws IOException {
        String path = dest.getPath();
        return new File(path).getUsableSpace();
    }

    @Override
    public boolean canExecute(FsPath dest) throws IOException {
        return this.can(dest, PosixFilePermission.OWNER_EXECUTE, PosixFilePermission.GROUP_EXECUTE, PosixFilePermission.OTHERS_EXECUTE);
    }

    @Override
    public boolean setOwner(FsPath dest, String user, String group) throws IOException {
        if (!StorageUtils.isIOProxy()) {
            LOG.info("io not proxy, setOwner skip");
            return true;
        }
        if (user != null) {
            this.setOwner(dest, user);
        }
        if (group != null) {
            this.setGroup(dest, group);
        }
        this.setGroup(dest, (String)StorageConfiguration.STORAGE_USER_GROUP.getValue());
        return true;
    }

    @Override
    public boolean setOwner(FsPath dest, String user) throws IOException {
        LOG.info("Set owner with path:" + dest.getPath() + "and user:" + user);
        if (!StorageUtils.isIOProxy()) {
            LOG.info("io not proxy, setOwner skip");
            return true;
        }
        UserPrincipalLookupService lookupService = FileSystems.getDefault().getUserPrincipalLookupService();
        PosixFileAttributeView view = Files.getFileAttributeView(Paths.get(dest.getPath(), new String[0]), PosixFileAttributeView.class, LinkOption.NOFOLLOW_LINKS);
        UserPrincipal userPrincipal = lookupService.lookupPrincipalByName(user);
        view.setOwner(userPrincipal);
        return true;
    }

    @Override
    public boolean setGroup(FsPath dest, String group) throws IOException {
        LOG.info("Set group with path:" + dest.getPath() + "and group:" + this.user);
        if (!StorageUtils.isIOProxy()) {
            LOG.info("io not proxy, setGroup skip");
            return true;
        }
        UserPrincipalLookupService lookupService = FileSystems.getDefault().getUserPrincipalLookupService();
        PosixFileAttributeView view = Files.getFileAttributeView(Paths.get(dest.getPath(), new String[0]), PosixFileAttributeView.class, LinkOption.NOFOLLOW_LINKS);
        GroupPrincipal groupPrincipal = lookupService.lookupPrincipalByGroupName(group);
        view.setGroup(groupPrincipal);
        return true;
    }

    public boolean mkdir(FsPath dest) throws IOException {
        return this.mkdirs(dest);
    }

    public boolean mkdirs(FsPath dest) throws IOException {
        String path = dest.getPath();
        LOG.info("Try to mkdirs with path:" + path);
        File file = new File(path);
        Stack<File> dirsToMake = new Stack<File>();
        dirsToMake.push(file);
        File parent = file.getParentFile();
        while (!parent.exists()) {
            dirsToMake.push(parent);
            parent = parent.getParentFile();
        }
        if (!this.canMkdir(new FsPath(parent.getPath()))) {
            throw new IOException("no permission to  mkdir path " + path);
        }
        while (!dirsToMake.empty()) {
            File dirToMake = (File)dirsToMake.pop();
            if (dirToMake.mkdir()) {
                if (!this.user.equals(this.getOwner(dirToMake.getAbsolutePath()))) {
                    this.setOwner(new FsPath(dirToMake.getAbsolutePath()), this.user, null);
                }
                this.setPermission(new FsPath(dirToMake.getAbsolutePath()), this.getDefaultFolderPerm());
                continue;
            }
            return false;
        }
        return true;
    }

    public boolean canMkdir(FsPath destParentDir) throws IOException {
        LOG.info("Try to check if the directory can be created with path:" + destParentDir.getPath());
        if (!StorageUtils.isIOProxy()) {
            LOG.debug("io not proxy, not check owner, just check if have write permission ");
            return this.canWrite(destParentDir);
        }
        LOG.info("io proxy, check owner ");
        if (!this.isOwner(destParentDir.getPath())) {
            throw new IOException("current user:" + this.user + ", parentPath:" + destParentDir.getPath() + ", only owner can mkdir path " + destParentDir);
        }
        return true;
    }

    @Override
    public boolean copy(String origin, String dest) throws IOException {
        File file = new File(dest);
        LOG.info("Try to copy file from:" + origin + " to dest:" + dest);
        if (!this.isOwner(file.getParent())) {
            throw new IOException("you have on permission to create file " + dest);
        }
        FileUtils.copyFile((File)new File(origin), (File)file);
        try {
            this.setPermission(new FsPath(dest), this.getDefaultFilePerm());
            if (!this.user.equals(this.getOwner(dest))) {
                this.setOwner(new FsPath(dest), this.user, null);
            }
        }
        catch (Throwable e) {
            file.delete();
            if (e instanceof IOException) {
                throw (IOException)e;
            }
            throw new IOException(e);
        }
        return true;
    }

    @Override
    public boolean setPermission(FsPath dest, String permission) throws IOException {
        LOG.info("Try to set permission dest with path:" + dest.getPath());
        if (!StorageUtils.isIOProxy()) {
            LOG.info("io not proxy, setPermission as parent.");
            try {
                PosixFileAttributes attr = Files.readAttributes(Paths.get(dest.getParent().getPath(), new String[0]), PosixFileAttributes.class, new LinkOption[0]);
                LOG.debug("parent permissions: attr: " + attr);
                Files.setPosixFilePermissions(Paths.get(dest.getPath(), new String[0]), attr.permissions());
            }
            catch (NoSuchFileException e) {
                LOG.warn("File or folder does not exist or file name is garbled(\u6587\u4ef6\u6216\u8005\u6587\u4ef6\u5939\u4e0d\u5b58\u5728\u6216\u8005\u6587\u4ef6\u540d\u4e71\u7801)", (Throwable)e);
                throw new StorageWarnException(LinkisStorageErrorCodeSummary.TO_BE_UNKNOW.getErrorCode(), e.getMessage());
            }
            return true;
        }
        String path = dest.getPath();
        if (StringUtils.isNumeric((CharSequence)permission)) {
            permission = FsPath.permissionFormatted((String)permission);
        }
        Files.setPosixFilePermissions(Paths.get(path, new String[0]), PosixFilePermissions.fromString(permission));
        return true;
    }

    @Override
    public FsPathListWithError listPathWithError(FsPath path) throws IOException {
        File file = new File(path.getPath());
        File[] files = file.listFiles();
        LOG.info("Try to list path:" + path.getPath() + " with error msg");
        if (files != null) {
            ArrayList<FsPath> rtn = new ArrayList<FsPath>();
            String message = "";
            for (File f : files) {
                try {
                    rtn.add(this.get(f.getPath()));
                }
                catch (Throwable e) {
                    LOG.warn("Failed to list path:", e);
                    message = "The file name is garbled. Please go to the shared storage to delete it.(\u6587\u4ef6\u540d\u5b58\u5728\u4e71\u7801\uff0c\u8bf7\u624b\u52a8\u53bb\u5171\u4eab\u5b58\u50a8\u8fdb\u884c\u5220\u9664):" + e.getMessage();
                }
            }
            return new FsPathListWithError(rtn, message);
        }
        return null;
    }

    public void init(Map<String, String> properties) throws IOException {
        if (MapUtils.isNotEmpty(properties)) {
            this.properties = properties;
            if (properties.containsKey(StorageConfiguration.PROXY_USER.key())) {
                this.user = (String)StorageConfiguration.PROXY_USER.getValue(properties);
            }
            this.group = (String)StorageConfiguration.STORAGE_USER_GROUP.getValue(properties);
        } else {
            this.properties = new HashMap<String, String>();
        }
        if (FsPath.WINDOWS) {
            this.group = (String)StorageConfiguration.STORAGE_USER_GROUP.getValue(properties);
        }
        if (StringUtils.isEmpty((CharSequence)this.group)) {
            String groupInfo;
            try {
                groupInfo = Utils.exec((String[])new String[]{"id", this.user});
                LOG.info("Get groupinfo:" + groupInfo + "  with shell command: id " + this.user);
            }
            catch (RuntimeException e) {
                this.group = this.user;
                return;
            }
            String groups = groupInfo.substring(groupInfo.indexOf("groups=") + 7);
            this.group = groups.replaceAll("\\d+", "").replaceAll("\\(", "").replaceAll("\\)", "");
        }
    }

    public String fsName() {
        return "file";
    }

    public String rootUserName() {
        return (String)StorageConfiguration.LOCAL_ROOT_USER.getValue();
    }

    public FsPath get(String dest) throws IOException {
        FsPath fsPath = null;
        if (FsPath.WINDOWS) {
            fsPath = new FsPath("file://" + dest);
            return fsPath;
        }
        fsPath = new FsPath(dest);
        LOG.info("Try to get FsPath with  path:" + fsPath.getPath());
        PosixFileAttributes attr = null;
        try {
            attr = Files.readAttributes(Paths.get(fsPath.getPath(), new String[0]), PosixFileAttributes.class, new LinkOption[0]);
        }
        catch (NoSuchFileException e) {
            LOG.warn("File or folder does not exist or file name is garbled(\u6587\u4ef6\u6216\u8005\u6587\u4ef6\u5939\u4e0d\u5b58\u5728\u6216\u8005\u6587\u4ef6\u540d\u4e71\u7801)", (Throwable)e);
            throw new StorageWarnException(LinkisStorageErrorCodeSummary.TO_BE_UNKNOW.getErrorCode(), e.getMessage());
        }
        fsPath.setIsdir(attr.isDirectory());
        fsPath.setModification_time(attr.lastModifiedTime().toMillis());
        fsPath.setAccess_time(attr.lastAccessTime().toMillis());
        fsPath.setLength(attr.size());
        fsPath.setPermissions(attr.permissions());
        fsPath.setOwner(attr.owner().getName());
        fsPath.setGroup(attr.group().getName());
        return fsPath;
    }

    public InputStream read(FsPath dest) throws IOException {
        if (this.canRead(dest)) {
            return new FileInputStream(dest.getPath());
        }
        throw new IOException("you have no permission to read path " + dest.getPath());
    }

    public OutputStream write(FsPath dest, boolean overwrite) throws IOException {
        String path = dest.getPath();
        if (new File(path).isDirectory()) {
            throw new IOException("you cannot write a directory " + path);
        }
        if (this.exists(dest) && this.canWrite(dest)) {
            return new FileOutputStream(path, !overwrite);
        }
        if (this.canWrite(dest.getParent())) {
            return new FileOutputStream(path, !overwrite);
        }
        throw new IOException("you have no permission to write file " + path);
    }

    public boolean create(String dest) throws IOException {
        LOG.info("try to create file with path:" + dest);
        File file = new File(dest);
        if (!this.isOwner(file.getParent())) {
            throw new IOException("you have on permission to create file " + dest);
        }
        file.createNewFile();
        try {
            this.setPermission(new FsPath(dest), this.getDefaultFilePerm());
            if (!this.user.equals(this.getOwner(dest))) {
                this.setOwner(new FsPath(dest), this.user, null);
            }
        }
        catch (Throwable e) {
            file.delete();
            if (e instanceof IOException) {
                throw (IOException)e;
            }
            throw new IOException(e);
        }
        return true;
    }

    public List<FsPath> list(FsPath path) throws IOException {
        File file = new File(path.getPath());
        File[] files = file.listFiles();
        LOG.info("Try to get file list with path:" + path.getPath());
        if (files != null) {
            ArrayList<FsPath> rtn = new ArrayList<FsPath>();
            for (File f : files) {
                rtn.add(this.get(f.getPath()));
            }
            return rtn;
        }
        return null;
    }

    public boolean canRead(FsPath dest) throws IOException {
        return this.can(dest, PosixFilePermission.OWNER_READ, PosixFilePermission.GROUP_READ, PosixFilePermission.OTHERS_READ);
    }

    public boolean canWrite(FsPath dest) throws IOException {
        return this.can(dest, PosixFilePermission.OWNER_WRITE, PosixFilePermission.GROUP_WRITE, PosixFilePermission.OTHERS_WRITE);
    }

    public boolean exists(FsPath dest) throws IOException {
        return new File(dest.getPath()).exists();
    }

    public boolean delete(FsPath dest) throws IOException {
        String path = dest.getPath();
        if (this.isOwner(path)) {
            return new File(path).delete();
        }
        throw new IOException("only owner can delete file " + path);
    }

    public boolean renameTo(FsPath oldDest, FsPath newDest) throws IOException {
        String path = oldDest.getPath();
        if (this.isOwner(path)) {
            return new File(path).renameTo(new File(newDest.getPath()));
        }
        throw new IOException("only owner can rename path " + path);
    }

    public void close() throws IOException {
    }

    private boolean can(FsPath fsPath, PosixFilePermission userPermission, PosixFilePermission groupPermission, PosixFilePermission otherPermission) throws IOException {
        String path = fsPath.getPath();
        if (!this.exists(fsPath)) {
            throw new IOException("path " + path + " not exists.");
        }
        if (FsPath.WINDOWS) {
            return true;
        }
        PosixFileAttributes attr = Files.readAttributes(Paths.get(path, new String[0]), PosixFileAttributes.class, new LinkOption[0]);
        Set<PosixFilePermission> permissions = attr.permissions();
        if (attr.owner().getName().equals(this.user) && permissions.contains((Object)userPermission)) {
            return true;
        }
        String pathGroup = attr.group().getName();
        LOG.debug("pathGroup: {}, group: {}, permissions: {}", new Object[]{pathGroup, this.group, permissions});
        if ((pathGroup.equals(this.user) || this.group != null && this.group.contains(pathGroup)) && permissions.contains((Object)groupPermission)) {
            return true;
        }
        return permissions.contains((Object)otherPermission);
    }

    private String getOwner(String path) throws IOException {
        PosixFileAttributes attr = Files.readAttributes(Paths.get(path, new String[0]), PosixFileAttributes.class, new LinkOption[0]);
        return attr.owner().getName();
    }
}

