/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.utils;

import java.io.Closeable;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.StandardOpenOption;
import java.util.Arrays;
import java.util.Collections;
import java.util.Random;
import org.apache.kafka.common.utils.ByteBufferOutputStream;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.test.TestUtils;
import org.easymock.EasyMock;
import org.easymock.IAnswer;
import org.junit.Assert;
import org.junit.Test;

public class UtilsTest {
    @Test
    public void testGetHost() {
        Assert.assertEquals((Object)"127.0.0.1", (Object)Utils.getHost((String)"127.0.0.1:8000"));
        Assert.assertEquals((Object)"mydomain.com", (Object)Utils.getHost((String)"PLAINTEXT://mydomain.com:8080"));
        Assert.assertEquals((Object)"MyDomain.com", (Object)Utils.getHost((String)"PLAINTEXT://MyDomain.com:8080"));
        Assert.assertEquals((Object)"My_Domain.com", (Object)Utils.getHost((String)"PLAINTEXT://My_Domain.com:8080"));
        Assert.assertEquals((Object)"::1", (Object)Utils.getHost((String)"[::1]:1234"));
        Assert.assertEquals((Object)"2001:db8:85a3:8d3:1319:8a2e:370:7348", (Object)Utils.getHost((String)"PLAINTEXT://[2001:db8:85a3:8d3:1319:8a2e:370:7348]:5678"));
        Assert.assertEquals((Object)"2001:DB8:85A3:8D3:1319:8A2E:370:7348", (Object)Utils.getHost((String)"PLAINTEXT://[2001:DB8:85A3:8D3:1319:8A2E:370:7348]:5678"));
        Assert.assertEquals((Object)"fe80::b1da:69ca:57f7:63d8%3", (Object)Utils.getHost((String)"PLAINTEXT://[fe80::b1da:69ca:57f7:63d8%3]:5678"));
    }

    @Test
    public void testGetPort() {
        Assert.assertEquals((long)8000L, (long)Utils.getPort((String)"127.0.0.1:8000").intValue());
        Assert.assertEquals((long)8080L, (long)Utils.getPort((String)"mydomain.com:8080").intValue());
        Assert.assertEquals((long)8080L, (long)Utils.getPort((String)"MyDomain.com:8080").intValue());
        Assert.assertEquals((long)1234L, (long)Utils.getPort((String)"[::1]:1234").intValue());
        Assert.assertEquals((long)5678L, (long)Utils.getPort((String)"[2001:db8:85a3:8d3:1319:8a2e:370:7348]:5678").intValue());
        Assert.assertEquals((long)5678L, (long)Utils.getPort((String)"[2001:DB8:85A3:8D3:1319:8A2E:370:7348]:5678").intValue());
        Assert.assertEquals((long)5678L, (long)Utils.getPort((String)"[fe80::b1da:69ca:57f7:63d8%3]:5678").intValue());
    }

    @Test
    public void testFormatAddress() {
        Assert.assertEquals((Object)"127.0.0.1:8000", (Object)Utils.formatAddress((String)"127.0.0.1", (Integer)8000));
        Assert.assertEquals((Object)"mydomain.com:8080", (Object)Utils.formatAddress((String)"mydomain.com", (Integer)8080));
        Assert.assertEquals((Object)"[::1]:1234", (Object)Utils.formatAddress((String)"::1", (Integer)1234));
        Assert.assertEquals((Object)"[2001:db8:85a3:8d3:1319:8a2e:370:7348]:5678", (Object)Utils.formatAddress((String)"2001:db8:85a3:8d3:1319:8a2e:370:7348", (Integer)5678));
    }

    @Test
    public void testFormatBytes() {
        Assert.assertEquals((Object)"-1", (Object)Utils.formatBytes((long)-1L));
        Assert.assertEquals((Object)"1023 B", (Object)Utils.formatBytes((long)1023L));
        Assert.assertEquals((Object)"1 KB", (Object)Utils.formatBytes((long)1024L));
        Assert.assertEquals((Object)"1024 KB", (Object)Utils.formatBytes((long)1048575L));
        Assert.assertEquals((Object)"1 MB", (Object)Utils.formatBytes((long)0x100000L));
        Assert.assertEquals((Object)"1.1 MB", (Object)Utils.formatBytes((long)0x119999L));
        Assert.assertEquals((Object)"10 MB", (Object)Utils.formatBytes((long)0xA00000L));
    }

    @Test
    public void testJoin() {
        Assert.assertEquals((Object)"", (Object)Utils.join(Collections.emptyList(), (String)","));
        Assert.assertEquals((Object)"1", (Object)Utils.join(Arrays.asList("1"), (String)","));
        Assert.assertEquals((Object)"1,2,3", (Object)Utils.join(Arrays.asList(1, 2, 3), (String)","));
    }

    @Test
    public void testAbs() {
        Assert.assertEquals((long)0L, (long)Utils.abs((int)Integer.MIN_VALUE));
        Assert.assertEquals((long)10L, (long)Utils.abs((int)-10));
        Assert.assertEquals((long)10L, (long)Utils.abs((int)10));
        Assert.assertEquals((long)0L, (long)Utils.abs((int)0));
        Assert.assertEquals((long)1L, (long)Utils.abs((int)-1));
    }

    @Test
    public void writeToBuffer() throws IOException {
        byte[] input = new byte[]{0, 1, 2, 3, 4, 5};
        ByteBuffer source = ByteBuffer.wrap(input);
        this.doTestWriteToByteBuffer(source, ByteBuffer.allocate(input.length));
        this.doTestWriteToByteBuffer(source, ByteBuffer.allocateDirect(input.length));
        Assert.assertEquals((long)0L, (long)source.position());
        source.position(2);
        this.doTestWriteToByteBuffer(source, ByteBuffer.allocate(input.length));
        this.doTestWriteToByteBuffer(source, ByteBuffer.allocateDirect(input.length));
    }

    private void doTestWriteToByteBuffer(ByteBuffer source, ByteBuffer dest) throws IOException {
        int numBytes = source.remaining();
        int position = source.position();
        DataOutputStream out = new DataOutputStream((OutputStream)new ByteBufferOutputStream(dest));
        Utils.writeTo((DataOutput)out, (ByteBuffer)source, (int)source.remaining());
        dest.flip();
        Assert.assertEquals((long)numBytes, (long)dest.remaining());
        Assert.assertEquals((long)position, (long)source.position());
        Assert.assertEquals((Object)source, (Object)dest);
    }

    @Test
    public void toArray() {
        byte[] input = new byte[]{0, 1, 2, 3, 4};
        ByteBuffer buffer = ByteBuffer.wrap(input);
        Assert.assertArrayEquals((byte[])input, (byte[])Utils.toArray((ByteBuffer)buffer));
        Assert.assertEquals((long)0L, (long)buffer.position());
        Assert.assertArrayEquals((byte[])new byte[]{1, 2}, (byte[])Utils.toArray((ByteBuffer)buffer, (int)1, (int)2));
        Assert.assertEquals((long)0L, (long)buffer.position());
        buffer.position(2);
        Assert.assertArrayEquals((byte[])new byte[]{2, 3, 4}, (byte[])Utils.toArray((ByteBuffer)buffer));
        Assert.assertEquals((long)2L, (long)buffer.position());
    }

    @Test
    public void toArrayDirectByteBuffer() {
        byte[] input = new byte[]{0, 1, 2, 3, 4};
        ByteBuffer buffer = ByteBuffer.allocateDirect(5);
        buffer.put(input);
        buffer.rewind();
        Assert.assertArrayEquals((byte[])input, (byte[])Utils.toArray((ByteBuffer)buffer));
        Assert.assertEquals((long)0L, (long)buffer.position());
        Assert.assertArrayEquals((byte[])new byte[]{1, 2}, (byte[])Utils.toArray((ByteBuffer)buffer, (int)1, (int)2));
        Assert.assertEquals((long)0L, (long)buffer.position());
        buffer.position(2);
        Assert.assertArrayEquals((byte[])new byte[]{2, 3, 4}, (byte[])Utils.toArray((ByteBuffer)buffer));
        Assert.assertEquals((long)2L, (long)buffer.position());
    }

    @Test
    public void utf8ByteArraySerde() {
        String utf8String = "A\u00ea\u00f1\u00fcC";
        byte[] utf8Bytes = utf8String.getBytes(StandardCharsets.UTF_8);
        Assert.assertArrayEquals((byte[])utf8Bytes, (byte[])Utils.utf8((String)utf8String));
        Assert.assertEquals((long)utf8Bytes.length, (long)Utils.utf8Length((CharSequence)utf8String));
        Assert.assertEquals((Object)utf8String, (Object)Utils.utf8((byte[])utf8Bytes));
    }

    @Test
    public void utf8ByteBufferSerde() {
        this.doTestUtf8ByteBuffer(ByteBuffer.allocate(20));
        this.doTestUtf8ByteBuffer(ByteBuffer.allocateDirect(20));
    }

    private void doTestUtf8ByteBuffer(ByteBuffer utf8Buffer) {
        String utf8String = "A\u00ea\u00f1\u00fcC";
        byte[] utf8Bytes = utf8String.getBytes(StandardCharsets.UTF_8);
        utf8Buffer.position(4);
        utf8Buffer.put(utf8Bytes);
        utf8Buffer.position(4);
        Assert.assertEquals((Object)utf8String, (Object)Utils.utf8((ByteBuffer)utf8Buffer, (int)utf8Bytes.length));
        Assert.assertEquals((long)4L, (long)utf8Buffer.position());
        utf8Buffer.position(0);
        Assert.assertEquals((Object)utf8String, (Object)Utils.utf8((ByteBuffer)utf8Buffer, (int)4, (int)utf8Bytes.length));
        Assert.assertEquals((long)0L, (long)utf8Buffer.position());
    }

    private void subTest(ByteBuffer buffer) {
        Assert.assertEquals((long)65L, (long)Utils.readBytes((ByteBuffer)buffer, (int)0, (int)1)[0]);
        byte[] results = Utils.readBytes((ByteBuffer)buffer, (int)2, (int)3);
        Assert.assertEquals((long)121L, (long)results[0]);
        Assert.assertEquals((long)32L, (long)results[1]);
        Assert.assertEquals((long)83L, (long)results[2]);
        Assert.assertEquals((long)3L, (long)results.length);
        results = Utils.readBytes((ByteBuffer)buffer);
        Assert.assertEquals((long)65L, (long)results[0]);
        Assert.assertEquals((long)116L, (long)results[buffer.limit() - 1]);
        Assert.assertEquals((long)buffer.limit(), (long)results.length);
    }

    @Test
    public void testReadBytes() {
        byte[] myvar = "Any String you want".getBytes();
        ByteBuffer buffer = ByteBuffer.allocate(myvar.length);
        buffer.put(myvar);
        buffer.rewind();
        this.subTest(buffer);
        buffer = ByteBuffer.wrap(myvar).asReadOnlyBuffer();
        this.subTest(buffer);
    }

    @Test
    public void testMin() {
        Assert.assertEquals((long)1L, (long)Utils.min((long)1L, (long[])new long[0]));
        Assert.assertEquals((long)1L, (long)Utils.min((long)1L, (long[])new long[]{2L, 3L}));
        Assert.assertEquals((long)1L, (long)Utils.min((long)2L, (long[])new long[]{1L, 3L}));
        Assert.assertEquals((long)1L, (long)Utils.min((long)2L, (long[])new long[]{3L, 1L}));
    }

    @Test
    public void testCloseAll() {
        Closeable[] closeablesWithoutException = TestCloseable.createCloseables(false, false, false);
        try {
            Utils.closeAll((Closeable[])closeablesWithoutException);
            TestCloseable.checkClosed((TestCloseable[])closeablesWithoutException);
        }
        catch (IOException e) {
            Assert.fail((String)("Unexpected exception: " + e));
        }
        Closeable[] closeablesWithException = TestCloseable.createCloseables(true, true, true);
        try {
            Utils.closeAll((Closeable[])closeablesWithException);
            Assert.fail((String)"Expected exception not thrown");
        }
        catch (IOException e) {
            TestCloseable.checkClosed((TestCloseable[])closeablesWithException);
            TestCloseable.checkException(e, (TestCloseable[])closeablesWithException);
        }
        Closeable[] singleExceptionCloseables = TestCloseable.createCloseables(false, true, false);
        try {
            Utils.closeAll((Closeable[])singleExceptionCloseables);
            Assert.fail((String)"Expected exception not thrown");
        }
        catch (IOException e) {
            TestCloseable.checkClosed((TestCloseable[])singleExceptionCloseables);
            TestCloseable.checkException(e, new TestCloseable[]{singleExceptionCloseables[1]});
        }
        Closeable[] mixedCloseables = TestCloseable.createCloseables(false, true, false, true, true);
        try {
            Utils.closeAll((Closeable[])mixedCloseables);
            Assert.fail((String)"Expected exception not thrown");
        }
        catch (IOException e) {
            TestCloseable.checkClosed((TestCloseable[])mixedCloseables);
            TestCloseable.checkException(e, new TestCloseable[]{mixedCloseables[1], mixedCloseables[3], mixedCloseables[4]});
        }
    }

    @Test
    public void testReadFullyOrFailWithRealFile() throws IOException {
        try (FileChannel channel = FileChannel.open(TestUtils.tempFile().toPath(), StandardOpenOption.READ, StandardOpenOption.WRITE);){
            String msg = "hello, world";
            channel.write(ByteBuffer.wrap(msg.getBytes()), 0L);
            channel.force(true);
            Assert.assertEquals((String)"Message should be written to the file channel", (long)channel.size(), (long)msg.length());
            ByteBuffer perfectBuffer = ByteBuffer.allocate(msg.length());
            ByteBuffer smallBuffer = ByteBuffer.allocate(5);
            ByteBuffer largeBuffer = ByteBuffer.allocate(msg.length() + 1);
            Utils.readFullyOrFail((FileChannel)channel, (ByteBuffer)perfectBuffer, (long)0L, (String)"perfect");
            Assert.assertFalse((String)"Buffer should be filled up", (boolean)perfectBuffer.hasRemaining());
            Assert.assertEquals((String)"Buffer should be populated correctly", (Object)msg, (Object)new String(perfectBuffer.array()));
            Utils.readFullyOrFail((FileChannel)channel, (ByteBuffer)smallBuffer, (long)0L, (String)"small");
            Assert.assertFalse((String)"Buffer should be filled", (boolean)smallBuffer.hasRemaining());
            Assert.assertEquals((String)"Buffer should be populated correctly", (Object)"hello", (Object)new String(smallBuffer.array()));
            smallBuffer.clear();
            Utils.readFullyOrFail((FileChannel)channel, (ByteBuffer)smallBuffer, (long)7L, (String)"small");
            Assert.assertFalse((String)"Buffer should be filled", (boolean)smallBuffer.hasRemaining());
            Assert.assertEquals((String)"Buffer should be populated correctly", (Object)"world", (Object)new String(smallBuffer.array()));
            try {
                Utils.readFullyOrFail((FileChannel)channel, (ByteBuffer)largeBuffer, (long)0L, (String)"large");
                Assert.fail((String)"Expected EOFException to be raised");
            }
            catch (EOFException eOFException) {
                // empty catch block
            }
        }
    }

    @Test
    public void testReadFullyOrFailWithPartialFileChannelReads() throws IOException {
        FileChannel channelMock = (FileChannel)EasyMock.createMock(FileChannel.class);
        int bufferSize = 100;
        ByteBuffer buffer = ByteBuffer.allocate(100);
        StringBuilder expectedBufferContent = new StringBuilder();
        this.fileChannelMockExpectReadWithRandomBytes(channelMock, expectedBufferContent, 100);
        EasyMock.replay((Object[])new Object[]{channelMock});
        Utils.readFullyOrFail((FileChannel)channelMock, (ByteBuffer)buffer, (long)0L, (String)"test");
        Assert.assertEquals((String)"The buffer should be populated correctly", (Object)expectedBufferContent.toString(), (Object)new String(buffer.array()));
        Assert.assertFalse((String)"The buffer should be filled", (boolean)buffer.hasRemaining());
        EasyMock.verify((Object[])new Object[]{channelMock});
    }

    @Test
    public void testReadFullyWithPartialFileChannelReads() throws IOException {
        FileChannel channelMock = (FileChannel)EasyMock.createMock(FileChannel.class);
        int bufferSize = 100;
        StringBuilder expectedBufferContent = new StringBuilder();
        this.fileChannelMockExpectReadWithRandomBytes(channelMock, expectedBufferContent, 100);
        EasyMock.replay((Object[])new Object[]{channelMock});
        ByteBuffer buffer = ByteBuffer.allocate(100);
        Utils.readFully((FileChannel)channelMock, (ByteBuffer)buffer, (long)0L);
        Assert.assertEquals((String)"The buffer should be populated correctly.", (Object)expectedBufferContent.toString(), (Object)new String(buffer.array()));
        Assert.assertFalse((String)"The buffer should be filled", (boolean)buffer.hasRemaining());
        EasyMock.verify((Object[])new Object[]{channelMock});
    }

    @Test
    public void testReadFullyIfEofIsReached() throws IOException {
        FileChannel channelMock = (FileChannel)EasyMock.createMock(FileChannel.class);
        int bufferSize = 100;
        String fileChannelContent = "abcdefghkl";
        ByteBuffer buffer = ByteBuffer.allocate(100);
        EasyMock.expect((Object)channelMock.size()).andReturn((Object)"abcdefghkl".length());
        EasyMock.expect((Object)channelMock.read((ByteBuffer)EasyMock.anyObject(ByteBuffer.class), EasyMock.anyInt())).andAnswer((IAnswer)new IAnswer<Integer>(){

            public Integer answer() throws Throwable {
                ByteBuffer buffer = (ByteBuffer)EasyMock.getCurrentArguments()[0];
                buffer.put("abcdefghkl".getBytes());
                return -1;
            }
        });
        EasyMock.replay((Object[])new Object[]{channelMock});
        Utils.readFully((FileChannel)channelMock, (ByteBuffer)buffer, (long)0L);
        Assert.assertEquals((Object)"abcdefghkl", (Object)new String(buffer.array(), 0, buffer.position()));
        Assert.assertEquals((long)buffer.position(), (long)channelMock.size());
        Assert.assertTrue((boolean)buffer.hasRemaining());
        EasyMock.verify((Object[])new Object[]{channelMock});
    }

    private void fileChannelMockExpectReadWithRandomBytes(FileChannel channelMock, final StringBuilder expectedBufferContent, int bufferSize) throws IOException {
        int mockedBytesRead;
        int step = 20;
        Random random = new Random();
        for (int remainingBytes = bufferSize; remainingBytes > 0; remainingBytes -= mockedBytesRead) {
            mockedBytesRead = remainingBytes < 20 ? remainingBytes : random.nextInt(20);
            final StringBuilder sb = new StringBuilder();
            EasyMock.expect((Object)channelMock.read((ByteBuffer)EasyMock.anyObject(ByteBuffer.class), EasyMock.anyInt())).andAnswer((IAnswer)new IAnswer<Integer>(){

                public Integer answer() throws Throwable {
                    ByteBuffer buffer = (ByteBuffer)EasyMock.getCurrentArguments()[0];
                    for (int i = 0; i < mockedBytesRead; ++i) {
                        sb.append("a");
                    }
                    buffer.put(sb.toString().getBytes());
                    expectedBufferContent.append((CharSequence)sb);
                    return mockedBytesRead;
                }
            });
        }
    }

    @Test(timeout=120000L)
    public void testRecursiveDelete() throws IOException {
        Utils.delete(null);
        File tempFile = TestUtils.tempFile();
        Utils.delete((File)tempFile);
        Assert.assertFalse((boolean)Files.exists(tempFile.toPath(), new LinkOption[0]));
        File tempDir = TestUtils.tempDirectory();
        File tempDir2 = TestUtils.tempDirectory(tempDir.toPath(), "a");
        TestUtils.tempDirectory(tempDir.toPath(), "b");
        TestUtils.tempDirectory(tempDir2.toPath(), "c");
        Utils.delete((File)tempDir);
        Assert.assertFalse((boolean)Files.exists(tempDir.toPath(), new LinkOption[0]));
        Assert.assertFalse((boolean)Files.exists(tempDir2.toPath(), new LinkOption[0]));
        Utils.delete((File)tempDir);
        Assert.assertFalse((boolean)Files.exists(tempDir.toPath(), new LinkOption[0]));
    }

    private static class TestCloseable
    implements Closeable {
        private final int id;
        private final IOException closeException;
        private boolean closed;

        TestCloseable(int id, boolean exceptionOnClose) {
            this.id = id;
            this.closeException = exceptionOnClose ? new IOException("Test close exception " + id) : null;
        }

        @Override
        public void close() throws IOException {
            this.closed = true;
            if (this.closeException != null) {
                throw this.closeException;
            }
        }

        static TestCloseable[] createCloseables(boolean ... exceptionOnClose) {
            TestCloseable[] closeables = new TestCloseable[exceptionOnClose.length];
            for (int i = 0; i < closeables.length; ++i) {
                closeables[i] = new TestCloseable(i, exceptionOnClose[i]);
            }
            return closeables;
        }

        static void checkClosed(TestCloseable ... closeables) {
            for (TestCloseable closeable : closeables) {
                Assert.assertTrue((String)("Close not invoked for " + closeable.id), (boolean)closeable.closed);
            }
        }

        static void checkException(IOException e, TestCloseable ... closeablesWithException) {
            Assert.assertEquals((Object)closeablesWithException[0].closeException, (Object)e);
            Throwable[] suppressed = e.getSuppressed();
            Assert.assertEquals((long)(closeablesWithException.length - 1), (long)suppressed.length);
            for (int i = 1; i < closeablesWithException.length; ++i) {
                Assert.assertEquals((Object)closeablesWithException[i].closeException, (Object)suppressed[i - 1]);
            }
        }
    }
}

