/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.index;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentNavigableMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.Future;
import org.apache.cassandra.config.ColumnDefinition;
import org.apache.cassandra.db.Column;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.Row;
import org.apache.cassandra.db.SystemKeyspace;
import org.apache.cassandra.db.compaction.CompactionManager;
import org.apache.cassandra.db.filter.ExtendedFilter;
import org.apache.cassandra.db.index.AbstractSimplePerColumnSecondaryIndex;
import org.apache.cassandra.db.index.PerColumnSecondaryIndex;
import org.apache.cassandra.db.index.PerRowSecondaryIndex;
import org.apache.cassandra.db.index.SecondaryIndex;
import org.apache.cassandra.db.index.SecondaryIndexBuilder;
import org.apache.cassandra.db.index.SecondaryIndexSearcher;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.io.sstable.ReducingKeyIterator;
import org.apache.cassandra.io.sstable.SSTableReader;
import org.apache.cassandra.thrift.IndexExpression;
import org.apache.cassandra.thrift.IndexType;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SecondaryIndexManager {
    private static final Logger logger = LoggerFactory.getLogger(SecondaryIndexManager.class);
    public static final Updater nullUpdater = new Updater(){

        @Override
        public void insert(Column column) {
        }

        @Override
        public void update(Column oldColumn, Column column) {
        }

        @Override
        public void remove(Column current) {
        }

        @Override
        public void updateRowLevelIndexes() {
        }
    };
    private final ConcurrentNavigableMap<ByteBuffer, SecondaryIndex> indexesByColumn = new ConcurrentSkipListMap<ByteBuffer, SecondaryIndex>();
    private final Map<Class<? extends SecondaryIndex>, SecondaryIndex> rowLevelIndexMap = new HashMap<Class<? extends SecondaryIndex>, SecondaryIndex>();
    public final ColumnFamilyStore baseCfs;

    public SecondaryIndexManager(ColumnFamilyStore baseCfs) {
        this.baseCfs = baseCfs;
    }

    public void reload() {
        Set indexedColumnNames = this.indexesByColumn.keySet();
        for (ByteBuffer indexedColumn : indexedColumnNames) {
            ColumnDefinition def = this.baseCfs.metadata.getColumnDefinition(indexedColumn);
            if (def != null && def.getIndexType() != null) continue;
            this.removeIndexedColumn(indexedColumn);
        }
        for (ColumnDefinition cdef : this.baseCfs.metadata.allColumns()) {
            if (cdef.getIndexType() == null || indexedColumnNames.contains(cdef.name)) continue;
            this.addIndexedColumn(cdef);
        }
        Set reloadedIndexes = Collections.newSetFromMap(new IdentityHashMap());
        for (SecondaryIndex index : this.indexesByColumn.values()) {
            if (!reloadedIndexes.add(index)) continue;
            index.reload();
        }
    }

    public Set<String> allIndexesNames() {
        HashSet<String> names = new HashSet<String>(this.indexesByColumn.size());
        for (SecondaryIndex index : this.indexesByColumn.values()) {
            names.add(index.getIndexName());
        }
        return names;
    }

    public void maybeBuildSecondaryIndexes(Collection<SSTableReader> sstables, Set<String> idxNames) {
        if (idxNames.isEmpty()) {
            return;
        }
        logger.info(String.format("Submitting index build of %s for data in %s", idxNames, StringUtils.join(sstables, (String)", ")));
        SecondaryIndexBuilder builder = new SecondaryIndexBuilder(this.baseCfs, idxNames, new ReducingKeyIterator(sstables));
        Future<?> future = CompactionManager.instance.submitIndexBuild(builder);
        FBUtilities.waitOnFuture(future);
        this.flushIndexesBlocking();
        logger.info("Index build of " + idxNames + " complete");
    }

    public boolean indexes(ByteBuffer name, Collection<SecondaryIndex> indexes) {
        return !this.indexFor(name, indexes).isEmpty();
    }

    public List<SecondaryIndex> indexFor(ByteBuffer name, Collection<SecondaryIndex> indexes) {
        ArrayList<SecondaryIndex> matching = null;
        for (SecondaryIndex index : indexes) {
            if (!index.indexes(name)) continue;
            if (matching == null) {
                matching = new ArrayList<SecondaryIndex>();
            }
            matching.add(index);
        }
        return matching == null ? Collections.emptyList() : matching;
    }

    public boolean indexes(Column column) {
        return this.indexes(column.name());
    }

    public boolean indexes(ByteBuffer name) {
        return this.indexes(name, this.indexesByColumn.values());
    }

    public List<SecondaryIndex> indexFor(ByteBuffer name) {
        return this.indexFor(name, this.indexesByColumn.values());
    }

    public boolean hasIndexFor(List<IndexExpression> clause) {
        if (clause == null || clause.isEmpty()) {
            return false;
        }
        List<SecondaryIndexSearcher> searchers = this.getIndexSearchersForQuery(clause);
        if (searchers.isEmpty()) {
            return false;
        }
        for (SecondaryIndexSearcher searcher : searchers) {
            if (searcher.isIndexing(clause)) continue;
            return false;
        }
        return true;
    }

    public void removeIndexedColumn(ByteBuffer column) {
        SecondaryIndex index = (SecondaryIndex)this.indexesByColumn.remove(column);
        if (index == null) {
            return;
        }
        if (index instanceof PerRowSecondaryIndex) {
            index.removeColumnDef(column);
            if (index.getColumnDefs().isEmpty()) {
                this.rowLevelIndexMap.remove(index.getClass());
            }
        }
        index.removeIndex(column);
        SystemKeyspace.setIndexRemoved(this.baseCfs.metadata.ksName, index.getNameForSystemKeyspace(column));
    }

    public synchronized Future<?> addIndexedColumn(ColumnDefinition cdef) {
        SecondaryIndex index;
        if (this.indexesByColumn.containsKey(cdef.name)) {
            return null;
        }
        assert (cdef.getIndexType() != null);
        try {
            index = SecondaryIndex.createInstance(this.baseCfs, cdef);
        }
        catch (ConfigurationException e) {
            throw new RuntimeException(e);
        }
        if (index instanceof PerRowSecondaryIndex) {
            SecondaryIndex currentIndex = this.rowLevelIndexMap.get(index.getClass());
            if (currentIndex == null) {
                this.rowLevelIndexMap.put(index.getClass(), index);
                index.init();
            } else {
                index = currentIndex;
                index.addColumnDef(cdef);
                logger.info("Creating new index : {}", (Object)cdef);
            }
        } else {
            if (cdef.getIndexType() == IndexType.CUSTOM && index instanceof AbstractSimplePerColumnSecondaryIndex) {
                throw new RuntimeException("Cannot use a subclass of AbstractSimplePerColumnSecondaryIndex as a CUSTOM index, as they assume they are CFS backed");
            }
            index.init();
        }
        this.indexesByColumn.put(cdef.name, index);
        if (index.isIndexBuilt(cdef.name)) {
            return null;
        }
        return index.buildIndexAsync();
    }

    public SecondaryIndex getIndexForColumn(ByteBuffer column) {
        return (SecondaryIndex)this.indexesByColumn.get(column);
    }

    public void invalidate() {
        for (SecondaryIndex index : this.indexesByColumn.values()) {
            index.invalidate();
        }
    }

    public void flushIndexesBlocking() {
        for (SecondaryIndex index : this.indexesByColumn.values()) {
            index.forceBlockingFlush();
        }
    }

    public List<String> getBuiltIndexes() {
        ArrayList<String> indexList = new ArrayList<String>();
        for (Map.Entry entry : this.indexesByColumn.entrySet()) {
            SecondaryIndex index = (SecondaryIndex)entry.getValue();
            if (!index.isIndexBuilt((ByteBuffer)entry.getKey())) continue;
            indexList.add(((SecondaryIndex)entry.getValue()).getIndexName());
        }
        return indexList;
    }

    public Collection<ColumnFamilyStore> getIndexesBackedByCfs() {
        ArrayList<ColumnFamilyStore> cfsList = new ArrayList<ColumnFamilyStore>();
        for (SecondaryIndex index : this.indexesByColumn.values()) {
            ColumnFamilyStore cfs = index.getIndexCfs();
            if (cfs == null) continue;
            cfsList.add(cfs);
        }
        return cfsList;
    }

    public Collection<SecondaryIndex> getIndexesNotBackedByCfs() {
        Set<SecondaryIndex> indexes = Collections.newSetFromMap(new IdentityHashMap());
        for (SecondaryIndex index : this.indexesByColumn.values()) {
            if (index.getIndexCfs() != null) continue;
            indexes.add(index);
        }
        return indexes;
    }

    public Collection<SecondaryIndex> getIndexes() {
        Set<SecondaryIndex> indexes = Collections.newSetFromMap(new IdentityHashMap());
        indexes.addAll(this.indexesByColumn.values());
        return indexes;
    }

    public boolean hasIndexes() {
        return !this.indexesByColumn.isEmpty();
    }

    public long getTotalLiveSize() {
        long total = 0L;
        for (SecondaryIndex index : this.getIndexes()) {
            total += index.getLiveSize();
        }
        return total;
    }

    public void indexRow(ByteBuffer key, ColumnFamily cf) {
        HashSet appliedRowLevelIndexes = null;
        for (SecondaryIndex index : this.indexesByColumn.values()) {
            if (index instanceof PerRowSecondaryIndex) {
                if (appliedRowLevelIndexes == null) {
                    appliedRowLevelIndexes = new HashSet();
                }
                if (!appliedRowLevelIndexes.add(index.getClass())) continue;
                ((PerRowSecondaryIndex)index).index(key, cf);
                continue;
            }
            for (Column column : cf) {
                if (!index.indexes(column.name())) continue;
                ((PerColumnSecondaryIndex)index).insert(key, column);
            }
        }
    }

    public void deleteFromIndexes(DecoratedKey key, List<Column> indexedColumnsInRow) {
        HashSet cleanedRowLevelIndexes = null;
        for (Column column : indexedColumnsInRow) {
            SecondaryIndex index = (SecondaryIndex)this.indexesByColumn.get(column.name());
            if (index == null) continue;
            if (index instanceof PerRowSecondaryIndex) {
                if (cleanedRowLevelIndexes == null) {
                    cleanedRowLevelIndexes = new HashSet();
                }
                if (!cleanedRowLevelIndexes.add(index.getClass())) continue;
                ((PerRowSecondaryIndex)index).delete(key);
                continue;
            }
            ((PerColumnSecondaryIndex)index).delete(key.key, column);
        }
    }

    public Updater updaterFor(DecoratedKey key, ColumnFamily cf) {
        return this.indexesByColumn.isEmpty() && this.rowLevelIndexMap.isEmpty() ? nullUpdater : new StandardUpdater(key, cf);
    }

    public Updater updaterFor(DecoratedKey key) {
        return this.updaterFor(key, null);
    }

    private List<SecondaryIndexSearcher> getIndexSearchersForQuery(List<IndexExpression> clause) {
        HashMap<String, HashSet<ByteBuffer>> groupByIndexType = new HashMap<String, HashSet<ByteBuffer>>();
        for (IndexExpression ix : clause) {
            SecondaryIndex index = this.getIndexForColumn(ix.column_name);
            if (index == null) continue;
            HashSet<ByteBuffer> columns = (HashSet<ByteBuffer>)groupByIndexType.get(index.getClass().getCanonicalName());
            if (columns == null) {
                columns = new HashSet<ByteBuffer>();
                groupByIndexType.put(index.getClass().getCanonicalName(), columns);
            }
            columns.add(ix.column_name);
        }
        ArrayList<SecondaryIndexSearcher> indexSearchers = new ArrayList<SecondaryIndexSearcher>(groupByIndexType.size());
        for (Set column : groupByIndexType.values()) {
            indexSearchers.add(this.getIndexForColumn((ByteBuffer)column.iterator().next()).createSecondaryIndexSearcher(column));
        }
        return indexSearchers;
    }

    public List<Row> search(ExtendedFilter filter) {
        List<SecondaryIndexSearcher> indexSearchers = this.getIndexSearchersForQuery(filter.getClause());
        if (indexSearchers.isEmpty()) {
            return Collections.emptyList();
        }
        if (indexSearchers.size() > 1) {
            throw new RuntimeException("Unable to search across multiple secondary index types");
        }
        return indexSearchers.get(0).search(filter);
    }

    public Collection<SecondaryIndex> getIndexesByNames(Set<String> idxNames) {
        ArrayList<SecondaryIndex> result = new ArrayList<SecondaryIndex>();
        for (SecondaryIndex index : this.indexesByColumn.values()) {
            if (!idxNames.contains(index.getIndexName())) continue;
            result.add(index);
        }
        return result;
    }

    public void setIndexBuilt(Set<String> idxNames) {
        for (SecondaryIndex index : this.getIndexesByNames(idxNames)) {
            index.setIndexBuilt();
        }
    }

    public void setIndexRemoved(Set<String> idxNames) {
        for (SecondaryIndex index : this.getIndexesByNames(idxNames)) {
            index.setIndexRemoved();
        }
    }

    public boolean validate(Column column) {
        SecondaryIndex index = this.getIndexForColumn(column.name());
        return index == null || index.validate(column);
    }

    private class StandardUpdater
    implements Updater {
        private final DecoratedKey key;
        private final ColumnFamily cf;

        public StandardUpdater(DecoratedKey key, ColumnFamily cf) {
            this.key = key;
            this.cf = cf;
        }

        @Override
        public void insert(Column column) {
            if (column.isMarkedForDelete(System.currentTimeMillis())) {
                return;
            }
            for (SecondaryIndex index : SecondaryIndexManager.this.indexFor(column.name())) {
                if (!(index instanceof PerColumnSecondaryIndex)) continue;
                ((PerColumnSecondaryIndex)index).insert(this.key.key, column);
            }
        }

        @Override
        public void update(Column oldColumn, Column column) {
            if (oldColumn.equals(column)) {
                return;
            }
            for (SecondaryIndex index : SecondaryIndexManager.this.indexFor(column.name())) {
                if (!(index instanceof PerColumnSecondaryIndex)) continue;
                if (!column.isMarkedForDelete(System.currentTimeMillis())) {
                    ((PerColumnSecondaryIndex)index).insert(this.key.key, column);
                }
                ((PerColumnSecondaryIndex)index).delete(this.key.key, oldColumn);
            }
        }

        @Override
        public void remove(Column column) {
            if (column.isMarkedForDelete(System.currentTimeMillis())) {
                return;
            }
            for (SecondaryIndex index : SecondaryIndexManager.this.indexFor(column.name())) {
                if (!(index instanceof PerColumnSecondaryIndex)) continue;
                ((PerColumnSecondaryIndex)index).delete(this.key.key, column);
            }
        }

        @Override
        public void updateRowLevelIndexes() {
            for (SecondaryIndex index : SecondaryIndexManager.this.rowLevelIndexMap.values()) {
                ((PerRowSecondaryIndex)index).index(this.key.key, this.cf);
            }
        }
    }

    public static interface Updater {
        public void insert(Column var1);

        public void update(Column var1, Column var2);

        public void remove(Column var1);

        public void updateRowLevelIndexes();
    }
}

