/*
 * Decompiled with CFR 0.152.
 */
package io.trino.memory;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import io.airlift.stats.GcMonitor;
import io.airlift.stats.TestingGcMonitor;
import io.airlift.units.DataSize;
import io.trino.Session;
import io.trino.execution.StageId;
import io.trino.execution.TaskId;
import io.trino.execution.buffer.PagesSerdeFactory;
import io.trino.execution.buffer.TestingPagesSerdeFactory;
import io.trino.memory.MemoryPool;
import io.trino.memory.MemoryPoolListener;
import io.trino.memory.QueryContext;
import io.trino.memory.context.LocalMemoryContext;
import io.trino.operator.Driver;
import io.trino.operator.DriverContext;
import io.trino.operator.Operator;
import io.trino.operator.OperatorContext;
import io.trino.operator.OutputFactory;
import io.trino.operator.TableScanOperator;
import io.trino.operator.TaskContext;
import io.trino.plugin.tpch.TpchConnectorFactory;
import io.trino.spi.Page;
import io.trino.spi.QueryId;
import io.trino.spi.connector.ConnectorFactory;
import io.trino.spiller.SpillSpaceTracker;
import io.trino.sql.planner.plan.PlanNodeId;
import io.trino.testing.LocalQueryRunner;
import io.trino.testing.PageConsumerOperator;
import io.trino.testing.TestingSession;
import io.trino.testing.TestingTaskContext;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.testng.Assert;
import org.testng.annotations.AfterMethod;
import org.testng.annotations.Test;

@Test(singleThreaded=true)
public class TestMemoryPools {
    private static final DataSize TEN_MEGABYTES = DataSize.of((long)10L, (DataSize.Unit)DataSize.Unit.MEGABYTE);
    private static final DataSize TEN_MEGABYTES_WITHOUT_TWO_BYTES = DataSize.ofBytes((long)(TEN_MEGABYTES.toBytes() - 2L));
    private static final DataSize ONE_BYTE = DataSize.ofBytes((long)1L);
    private TaskId fakeTaskId;
    private LocalQueryRunner localQueryRunner;
    private MemoryPool userPool;
    private List<Driver> drivers;
    private TaskContext taskContext;

    private void setUp(Supplier<List<Driver>> driversSupplier) {
        Preconditions.checkState((this.localQueryRunner == null ? 1 : 0) != 0, (Object)"Already set up");
        Session session = TestingSession.testSessionBuilder().setCatalog("tpch").setSchema("tiny").setSystemProperty("task_default_concurrency", "1").build();
        this.localQueryRunner = LocalQueryRunner.builder((Session)session).withInitialTransaction().build();
        this.localQueryRunner.createCatalog("tpch", (ConnectorFactory)new TpchConnectorFactory(1), (Map)ImmutableMap.of());
        this.userPool = new MemoryPool(TEN_MEGABYTES);
        QueryId queryId = new QueryId("fake");
        this.fakeTaskId = new TaskId(new StageId(queryId, 0), 0, 0);
        SpillSpaceTracker spillSpaceTracker = new SpillSpaceTracker(DataSize.of((long)1L, (DataSize.Unit)DataSize.Unit.GIGABYTE));
        QueryContext queryContext = new QueryContext(new QueryId("query"), TEN_MEGABYTES, this.userPool, (GcMonitor)new TestingGcMonitor(), (Executor)this.localQueryRunner.getExecutor(), this.localQueryRunner.getScheduler(), TEN_MEGABYTES, spillSpaceTracker);
        this.taskContext = TestingTaskContext.createTaskContext((QueryContext)queryContext, (Executor)this.localQueryRunner.getExecutor(), (Session)this.localQueryRunner.getDefaultSession());
        this.drivers = driversSupplier.get();
    }

    private void setUpCountStarFromOrdersWithJoin() {
        this.setUp(() -> {
            PageConsumerOperator.PageConsumerOutputFactory outputFactory = new PageConsumerOperator.PageConsumerOutputFactory(types -> page -> {});
            return this.localQueryRunner.createDrivers("SELECT COUNT(*) FROM orders JOIN lineitem ON CAST(orders.orderkey AS VARCHAR) = CAST(lineitem.orderkey AS VARCHAR)", (OutputFactory)outputFactory, this.taskContext);
        });
    }

    private RevocableMemoryOperator setupConsumeRevocableMemory(DataSize reservedPerPage, long numberOfPages) {
        AtomicReference createOperator = new AtomicReference();
        this.setUp(() -> {
            DriverContext driverContext = this.taskContext.addPipelineContext(0, false, false, false).addDriverContext();
            OperatorContext revokableOperatorContext = driverContext.addOperatorContext(Integer.MAX_VALUE, new PlanNodeId("revokable_operator"), TableScanOperator.class.getSimpleName());
            PageConsumerOperator.PageConsumerOutputFactory outputFactory = new PageConsumerOperator.PageConsumerOutputFactory(types -> page -> {});
            Operator outputOperator = outputFactory.createOutputOperator(2, new PlanNodeId("output"), (List)ImmutableList.of(), Function.identity(), (PagesSerdeFactory)new TestingPagesSerdeFactory()).createOperator(driverContext);
            RevocableMemoryOperator revocableMemoryOperator = new RevocableMemoryOperator(revokableOperatorContext, reservedPerPage, numberOfPages);
            createOperator.set(revocableMemoryOperator);
            Driver driver = Driver.createDriver((DriverContext)driverContext, (Operator)revocableMemoryOperator, (Operator[])new Operator[]{outputOperator});
            return ImmutableList.of((Object)driver);
        });
        return (RevocableMemoryOperator)createOperator.get();
    }

    @AfterMethod(alwaysRun=true)
    public void tearDown() {
        if (this.localQueryRunner != null) {
            this.localQueryRunner.close();
            this.localQueryRunner = null;
        }
    }

    @Test
    public void testBlockingOnUserMemory() {
        this.setUpCountStarFromOrdersWithJoin();
        Assert.assertTrue((boolean)this.userPool.tryReserve(this.fakeTaskId, "test", TEN_MEGABYTES.toBytes()));
        this.runDriversUntilBlocked(this.waitingForUserMemory());
        Assert.assertTrue((this.userPool.getFreeBytes() <= 0L ? 1 : 0) != 0, (String)String.format("Expected empty pool but got [%d]", this.userPool.getFreeBytes()));
        this.userPool.free(this.fakeTaskId, "test", TEN_MEGABYTES.toBytes());
        this.assertDriversProgress(this.waitingForUserMemory());
    }

    @Test
    public void testNotifyListenerOnMemoryReserved() {
        this.setupConsumeRevocableMemory(ONE_BYTE, 10L);
        AtomicReference notifiedPool = new AtomicReference();
        AtomicLong notifiedBytes = new AtomicLong();
        this.userPool.addListener(MemoryPoolListener.onMemoryReserved(pool -> {
            notifiedPool.set(pool);
            notifiedBytes.set(pool.getReservedBytes());
        }));
        this.userPool.reserve(this.fakeTaskId, "test", 3L);
        Assert.assertEquals(notifiedPool.get(), (Object)this.userPool);
        Assert.assertEquals((long)notifiedBytes.get(), (long)3L);
    }

    @Test
    public void testMemoryFutureCancellation() {
        this.setUpCountStarFromOrdersWithJoin();
        ListenableFuture future = this.userPool.reserve(this.fakeTaskId, "test", TEN_MEGABYTES.toBytes());
        Assert.assertTrue((!future.isDone() ? 1 : 0) != 0);
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> future.cancel(true)).isInstanceOf(UnsupportedOperationException.class)).hasMessage("cancellation is not supported");
        this.userPool.free(this.fakeTaskId, "test", TEN_MEGABYTES.toBytes());
        Assert.assertTrue((boolean)future.isDone());
    }

    @Test
    public void testBlockingOnRevocableMemoryFreeUser() {
        this.setupConsumeRevocableMemory(ONE_BYTE, 10L);
        Assert.assertTrue((boolean)this.userPool.tryReserve(this.fakeTaskId, "test", TEN_MEGABYTES_WITHOUT_TWO_BYTES.toBytes()));
        Assert.assertEquals((long)this.runDriversUntilBlocked(this.waitingForRevocableMemory()), (long)2L);
        Assert.assertTrue((this.userPool.getFreeBytes() <= 0L ? 1 : 0) != 0, (String)String.format("Expected empty pool but got [%d]", this.userPool.getFreeBytes()));
        this.userPool.free(this.fakeTaskId, "test", 5L);
        Assert.assertEquals((long)this.runDriversUntilBlocked(this.waitingForRevocableMemory()), (long)5L);
        Assert.assertTrue((this.userPool.getFreeBytes() <= 0L ? 1 : 0) != 0, (String)String.format("Expected empty pool but got [%d]", this.userPool.getFreeBytes()));
        this.userPool.free(this.fakeTaskId, "test", 3L);
        this.assertDriversProgress(this.waitingForRevocableMemory());
        Assert.assertEquals((long)this.userPool.getFreeBytes(), (long)10L);
    }

    @Test
    public void testBlockingOnRevocableMemoryFreeViaRevoke() {
        RevocableMemoryOperator revocableMemoryOperator = this.setupConsumeRevocableMemory(ONE_BYTE, 5L);
        Assert.assertTrue((boolean)this.userPool.tryReserve(this.fakeTaskId, "test", TEN_MEGABYTES_WITHOUT_TWO_BYTES.toBytes()));
        Assert.assertEquals((long)this.runDriversUntilBlocked(this.waitingForRevocableMemory()), (long)2L);
        revocableMemoryOperator.getOperatorContext().requestMemoryRevoking();
        Assert.assertEquals((long)this.runDriversUntilBlocked(this.waitingForRevocableMemory()), (long)2L);
        revocableMemoryOperator.getOperatorContext().requestMemoryRevoking();
        this.assertDriversProgress(this.waitingForRevocableMemory());
        Assert.assertEquals((long)this.userPool.getFreeBytes(), (long)2L);
    }

    @Test
    public void testTaggedAllocations() {
        TaskId testTask = new TaskId(new StageId(new QueryId("test_query"), 0), 0, 0);
        MemoryPool testPool = new MemoryPool(DataSize.ofBytes((long)1000L));
        testPool.reserve(testTask, "test_tag", 10L);
        Map allocations = (Map)testPool.getTaggedMemoryAllocations().get(new QueryId("test_query"));
        Assert.assertEquals((Map)allocations, (Map)ImmutableMap.of((Object)"test_tag", (Object)10L));
        testPool.free(testTask, "test_tag", 5L);
        Assert.assertEquals((Map)allocations, (Map)ImmutableMap.of((Object)"test_tag", (Object)5L));
        testPool.reserve(testTask, "test_tag2", 20L);
        Assert.assertEquals((Map)allocations, (Map)ImmutableMap.of((Object)"test_tag", (Object)5L, (Object)"test_tag2", (Object)20L));
        testPool.free(testTask, "test_tag", 5L);
        Assert.assertEquals((Map)allocations, (Map)ImmutableMap.of((Object)"test_tag2", (Object)20L));
        testPool.free(testTask, "test_tag2", 20L);
        Assert.assertEquals((int)testPool.getTaggedMemoryAllocations().size(), (int)0);
    }

    @Test
    public void testPerTaskAllocations() {
        QueryId query1 = new QueryId("test_query1");
        TaskId q1task1 = new TaskId(new StageId(query1, 0), 0, 0);
        TaskId q1task2 = new TaskId(new StageId(query1, 0), 1, 0);
        QueryId query2 = new QueryId("test_query2");
        TaskId q2task1 = new TaskId(new StageId(query2, 0), 0, 0);
        MemoryPool testPool = new MemoryPool(DataSize.ofBytes((long)1000L));
        testPool.reserve(q1task1, "tag", 10L);
        Assertions.assertThat(testPool.getQueryMemoryReservations().keySet()).hasSize(1);
        Assertions.assertThat((long)testPool.getQueryMemoryReservation(query1)).isEqualTo(10L);
        Assertions.assertThat(testPool.getTaskMemoryReservations().keySet()).hasSize(1);
        Assertions.assertThat((long)testPool.getTaskMemoryReservation(q1task1)).isEqualTo(10L);
        testPool.reserve(q1task2, "tag", 7L);
        Assertions.assertThat(testPool.getQueryMemoryReservations().keySet()).hasSize(1);
        Assertions.assertThat((long)testPool.getQueryMemoryReservation(query1)).isEqualTo(17L);
        Assertions.assertThat(testPool.getTaskMemoryReservations().keySet()).hasSize(2);
        Assertions.assertThat((long)testPool.getTaskMemoryReservation(q1task1)).isEqualTo(10L);
        Assertions.assertThat((long)testPool.getTaskMemoryReservation(q1task2)).isEqualTo(7L);
        testPool.reserve(q2task1, "tag", 9L);
        Assertions.assertThat(testPool.getQueryMemoryReservations().keySet()).hasSize(2);
        Assertions.assertThat((long)testPool.getQueryMemoryReservation(query1)).isEqualTo(17L);
        Assertions.assertThat((long)testPool.getQueryMemoryReservation(query2)).isEqualTo(9L);
        Assertions.assertThat((long)testPool.getTaskMemoryReservation(q1task1)).isEqualTo(10L);
        Assertions.assertThat(testPool.getTaskMemoryReservations().keySet()).hasSize(3);
        Assertions.assertThat((long)testPool.getTaskMemoryReservation(q1task2)).isEqualTo(7L);
        Assertions.assertThat((long)testPool.getTaskMemoryReservation(q2task1)).isEqualTo(9L);
        testPool.reserve(q1task1, "tag", 3L);
        Assertions.assertThat((long)testPool.getQueryMemoryReservation(query1)).isEqualTo(20L);
        Assertions.assertThat((long)testPool.getQueryMemoryReservation(query2)).isEqualTo(9L);
        Assertions.assertThat(testPool.getTaskMemoryReservations().keySet()).hasSize(3);
        Assertions.assertThat((long)testPool.getTaskMemoryReservation(q1task1)).isEqualTo(13L);
        Assertions.assertThat((long)testPool.getTaskMemoryReservation(q1task2)).isEqualTo(7L);
        Assertions.assertThat((long)testPool.getTaskMemoryReservation(q2task1)).isEqualTo(9L);
        testPool.free(q1task1, "tag", 5L);
        Assertions.assertThat(testPool.getQueryMemoryReservations().keySet()).hasSize(2);
        Assertions.assertThat((long)testPool.getQueryMemoryReservation(query1)).isEqualTo(15L);
        Assertions.assertThat((long)testPool.getQueryMemoryReservation(query2)).isEqualTo(9L);
        Assertions.assertThat(testPool.getTaskMemoryReservations().keySet()).hasSize(3);
        Assertions.assertThat((long)testPool.getTaskMemoryReservation(q1task1)).isEqualTo(8L);
        Assertions.assertThat((long)testPool.getTaskMemoryReservation(q1task2)).isEqualTo(7L);
        Assertions.assertThat((long)testPool.getTaskMemoryReservation(q2task1)).isEqualTo(9L);
        Assertions.assertThatThrownBy(() -> testPool.free(q1task1, "tag", 9L)).hasMessage("tried to free more memory than is reserved by task");
        Assertions.assertThat(testPool.getQueryMemoryReservations().keySet()).hasSize(2);
        Assertions.assertThat((long)testPool.getQueryMemoryReservation(query1)).isEqualTo(15L);
        Assertions.assertThat((long)testPool.getQueryMemoryReservation(query2)).isEqualTo(9L);
        Assertions.assertThat(testPool.getTaskMemoryReservations().keySet()).hasSize(3);
        Assertions.assertThat((long)testPool.getTaskMemoryReservation(q1task1)).isEqualTo(8L);
        Assertions.assertThat((long)testPool.getTaskMemoryReservation(q1task2)).isEqualTo(7L);
        Assertions.assertThat((long)testPool.getTaskMemoryReservation(q2task1)).isEqualTo(9L);
        testPool.free(q1task1, "tag", 8L);
        Assertions.assertThat(testPool.getQueryMemoryReservations().keySet()).hasSize(2);
        Assertions.assertThat((long)testPool.getQueryMemoryReservation(query1)).isEqualTo(7L);
        Assertions.assertThat((long)testPool.getQueryMemoryReservation(query2)).isEqualTo(9L);
        Assertions.assertThat(testPool.getTaskMemoryReservations().keySet()).hasSize(2);
        Assertions.assertThat((long)testPool.getTaskMemoryReservation(q1task1)).isEqualTo(0L);
        Assertions.assertThat((long)testPool.getTaskMemoryReservation(q1task2)).isEqualTo(7L);
        Assertions.assertThat((long)testPool.getTaskMemoryReservation(q2task1)).isEqualTo(9L);
        testPool.free(q1task2, "tag", 7L);
        Assertions.assertThat(testPool.getQueryMemoryReservations().keySet()).hasSize(1);
        Assertions.assertThat((long)testPool.getQueryMemoryReservation(query1)).isEqualTo(0L);
        Assertions.assertThat((long)testPool.getQueryMemoryReservation(query2)).isEqualTo(9L);
        Assertions.assertThat(testPool.getTaskMemoryReservations().keySet()).hasSize(1);
        Assertions.assertThat((long)testPool.getTaskMemoryReservation(q1task1)).isEqualTo(0L);
        Assertions.assertThat((long)testPool.getTaskMemoryReservation(q1task2)).isEqualTo(0L);
        Assertions.assertThat((long)testPool.getTaskMemoryReservation(q2task1)).isEqualTo(9L);
    }

    @Test
    public void testGlobalRevocableAllocations() {
        MemoryPool testPool = new MemoryPool(DataSize.ofBytes((long)1000L));
        Assertions.assertThat((boolean)testPool.tryReserveRevocable(999L)).isTrue();
        Assertions.assertThat((boolean)testPool.tryReserveRevocable(2L)).isFalse();
        Assertions.assertThat((long)testPool.getReservedBytes()).isEqualTo(0L);
        Assertions.assertThat((long)testPool.getReservedRevocableBytes()).isEqualTo(999L);
        Assertions.assertThat((Map)testPool.getTaskMemoryReservations()).isEmpty();
        Assertions.assertThat((Map)testPool.getQueryMemoryReservations()).isEmpty();
        Assertions.assertThat((Map)testPool.getTaggedMemoryAllocations()).isEmpty();
        QueryId query = new QueryId("test_query1");
        TaskId task = new TaskId(new StageId(query, 0), 0, 0);
        ListenableFuture memoryFuture = testPool.reserve(task, "tag", 2L);
        Assertions.assertThat((Future)memoryFuture).isNotDone();
        testPool.freeRevocable(999L);
        Assertions.assertThat((Future)memoryFuture).isDone();
        Assertions.assertThat((long)testPool.getReservedBytes()).isEqualTo(2L);
        Assertions.assertThat((long)testPool.getReservedRevocableBytes()).isEqualTo(0L);
    }

    @Test
    public void testPerTaskRevocableAllocations() {
        QueryId query1 = new QueryId("test_query1");
        TaskId q1task1 = new TaskId(new StageId(query1, 0), 0, 0);
        TaskId q1task2 = new TaskId(new StageId(query1, 0), 1, 0);
        QueryId query2 = new QueryId("test_query2");
        TaskId q2task1 = new TaskId(new StageId(query2, 0), 0, 0);
        MemoryPool testPool = new MemoryPool(DataSize.ofBytes((long)1000L));
        testPool.reserveRevocable(q1task1, 10L);
        Assertions.assertThat(testPool.getQueryRevocableMemoryReservations().keySet()).hasSize(1);
        Assertions.assertThat((long)testPool.getQueryRevocableMemoryReservation(query1)).isEqualTo(10L);
        Assertions.assertThat(testPool.getTaskRevocableMemoryReservations().keySet()).hasSize(1);
        Assertions.assertThat((long)testPool.getTaskRevocableMemoryReservation(q1task1)).isEqualTo(10L);
        testPool.reserveRevocable(q1task2, 7L);
        Assertions.assertThat(testPool.getQueryRevocableMemoryReservations().keySet()).hasSize(1);
        Assertions.assertThat((long)testPool.getQueryRevocableMemoryReservation(query1)).isEqualTo(17L);
        Assertions.assertThat(testPool.getTaskRevocableMemoryReservations().keySet()).hasSize(2);
        Assertions.assertThat((long)testPool.getTaskRevocableMemoryReservation(q1task1)).isEqualTo(10L);
        Assertions.assertThat((long)testPool.getTaskRevocableMemoryReservation(q1task2)).isEqualTo(7L);
        testPool.reserveRevocable(q2task1, 9L);
        Assertions.assertThat(testPool.getQueryRevocableMemoryReservations().keySet()).hasSize(2);
        Assertions.assertThat((long)testPool.getQueryRevocableMemoryReservation(query1)).isEqualTo(17L);
        Assertions.assertThat((long)testPool.getQueryRevocableMemoryReservation(query2)).isEqualTo(9L);
        Assertions.assertThat((long)testPool.getTaskRevocableMemoryReservation(q1task1)).isEqualTo(10L);
        Assertions.assertThat(testPool.getTaskRevocableMemoryReservations().keySet()).hasSize(3);
        Assertions.assertThat((long)testPool.getTaskRevocableMemoryReservation(q1task2)).isEqualTo(7L);
        Assertions.assertThat((long)testPool.getTaskRevocableMemoryReservation(q2task1)).isEqualTo(9L);
        testPool.reserveRevocable(q1task1, 3L);
        Assertions.assertThat((long)testPool.getQueryRevocableMemoryReservation(query1)).isEqualTo(20L);
        Assertions.assertThat((long)testPool.getQueryRevocableMemoryReservation(query2)).isEqualTo(9L);
        Assertions.assertThat(testPool.getTaskRevocableMemoryReservations().keySet()).hasSize(3);
        Assertions.assertThat((long)testPool.getTaskRevocableMemoryReservation(q1task1)).isEqualTo(13L);
        Assertions.assertThat((long)testPool.getTaskRevocableMemoryReservation(q1task2)).isEqualTo(7L);
        Assertions.assertThat((long)testPool.getTaskRevocableMemoryReservation(q2task1)).isEqualTo(9L);
        testPool.freeRevocable(q1task1, 5L);
        Assertions.assertThat(testPool.getQueryRevocableMemoryReservations().keySet()).hasSize(2);
        Assertions.assertThat((long)testPool.getQueryRevocableMemoryReservation(query1)).isEqualTo(15L);
        Assertions.assertThat((long)testPool.getQueryRevocableMemoryReservation(query2)).isEqualTo(9L);
        Assertions.assertThat(testPool.getTaskRevocableMemoryReservations().keySet()).hasSize(3);
        Assertions.assertThat((long)testPool.getTaskRevocableMemoryReservation(q1task1)).isEqualTo(8L);
        Assertions.assertThat((long)testPool.getTaskRevocableMemoryReservation(q1task2)).isEqualTo(7L);
        Assertions.assertThat((long)testPool.getTaskRevocableMemoryReservation(q2task1)).isEqualTo(9L);
        Assertions.assertThatThrownBy(() -> testPool.freeRevocable(q1task1, 9L)).hasMessage("tried to free more revocable memory than is reserved by task");
        Assertions.assertThat(testPool.getQueryRevocableMemoryReservations().keySet()).hasSize(2);
        Assertions.assertThat((long)testPool.getQueryRevocableMemoryReservation(query1)).isEqualTo(15L);
        Assertions.assertThat((long)testPool.getQueryRevocableMemoryReservation(query2)).isEqualTo(9L);
        Assertions.assertThat(testPool.getTaskRevocableMemoryReservations().keySet()).hasSize(3);
        Assertions.assertThat((long)testPool.getTaskRevocableMemoryReservation(q1task1)).isEqualTo(8L);
        Assertions.assertThat((long)testPool.getTaskRevocableMemoryReservation(q1task2)).isEqualTo(7L);
        Assertions.assertThat((long)testPool.getTaskRevocableMemoryReservation(q2task1)).isEqualTo(9L);
        testPool.freeRevocable(q1task1, 8L);
        Assertions.assertThat(testPool.getQueryRevocableMemoryReservations().keySet()).hasSize(2);
        Assertions.assertThat((long)testPool.getQueryRevocableMemoryReservation(query1)).isEqualTo(7L);
        Assertions.assertThat((long)testPool.getQueryRevocableMemoryReservation(query2)).isEqualTo(9L);
        Assertions.assertThat(testPool.getTaskRevocableMemoryReservations().keySet()).hasSize(2);
        Assertions.assertThat((long)testPool.getTaskRevocableMemoryReservation(q1task1)).isEqualTo(0L);
        Assertions.assertThat((long)testPool.getTaskRevocableMemoryReservation(q1task2)).isEqualTo(7L);
        Assertions.assertThat((long)testPool.getTaskRevocableMemoryReservation(q2task1)).isEqualTo(9L);
        testPool.freeRevocable(q1task2, 7L);
        Assertions.assertThat(testPool.getQueryRevocableMemoryReservations().keySet()).hasSize(1);
        Assertions.assertThat((long)testPool.getQueryRevocableMemoryReservation(query1)).isEqualTo(0L);
        Assertions.assertThat((long)testPool.getQueryRevocableMemoryReservation(query2)).isEqualTo(9L);
        Assertions.assertThat(testPool.getTaskRevocableMemoryReservations().keySet()).hasSize(1);
        Assertions.assertThat((long)testPool.getTaskRevocableMemoryReservation(q1task1)).isEqualTo(0L);
        Assertions.assertThat((long)testPool.getTaskRevocableMemoryReservation(q1task2)).isEqualTo(0L);
        Assertions.assertThat((long)testPool.getTaskRevocableMemoryReservation(q2task1)).isEqualTo(9L);
    }

    private long runDriversUntilBlocked(Predicate<OperatorContext> reason) {
        long iterationsCount = 0L;
        while (!TestMemoryPools.isOperatorBlocked(this.drivers, reason)) {
            for (Driver driver : this.drivers) {
                driver.processForNumberOfIterations(1);
            }
            ++iterationsCount;
        }
        for (Driver driver : this.drivers) {
            Assert.assertFalse((boolean)driver.isFinished());
        }
        return iterationsCount;
    }

    private void assertDriversProgress(Predicate<OperatorContext> reason) {
        do {
            Assert.assertFalse((boolean)TestMemoryPools.isOperatorBlocked(this.drivers, reason));
            boolean progress = false;
            for (Driver driver : this.drivers) {
                ListenableFuture blocked = driver.processUntilBlocked();
                progress |= blocked.isDone();
            }
            Assert.assertTrue((boolean)progress);
        } while (!this.drivers.stream().allMatch(Driver::isFinished));
    }

    private Predicate<OperatorContext> waitingForUserMemory() {
        return operatorContext -> !operatorContext.isWaitingForMemory().isDone();
    }

    private Predicate<OperatorContext> waitingForRevocableMemory() {
        return operatorContext -> !operatorContext.isWaitingForRevocableMemory().isDone() && !operatorContext.isMemoryRevokingRequested();
    }

    private static boolean isOperatorBlocked(List<Driver> drivers, Predicate<OperatorContext> reason) {
        for (Driver driver : drivers) {
            for (OperatorContext operatorContext : driver.getDriverContext().getOperatorContexts()) {
                if (!reason.test(operatorContext)) continue;
                return true;
            }
        }
        return false;
    }

    private static class RevocableMemoryOperator
    implements Operator {
        private final DataSize reservedPerPage;
        private final long numberOfPages;
        private final OperatorContext operatorContext;
        private long producedPagesCount;
        private final LocalMemoryContext revocableMemoryContext;

        public RevocableMemoryOperator(OperatorContext operatorContext, DataSize reservedPerPage, long numberOfPages) {
            this.operatorContext = operatorContext;
            this.reservedPerPage = reservedPerPage;
            this.numberOfPages = numberOfPages;
            this.revocableMemoryContext = operatorContext.localRevocableMemoryContext();
        }

        public ListenableFuture<Void> startMemoryRevoke() {
            return Futures.immediateVoidFuture();
        }

        public void finishMemoryRevoke() {
            this.revocableMemoryContext.setBytes(0L);
        }

        public OperatorContext getOperatorContext() {
            return this.operatorContext;
        }

        public void finish() {
            this.revocableMemoryContext.setBytes(0L);
        }

        public boolean isFinished() {
            return this.producedPagesCount >= this.numberOfPages;
        }

        public boolean needsInput() {
            return false;
        }

        public void addInput(Page page) {
            throw new UnsupportedOperationException();
        }

        public Page getOutput() {
            this.revocableMemoryContext.setBytes(this.revocableMemoryContext.getBytes() + this.reservedPerPage.toBytes());
            ++this.producedPagesCount;
            if (this.producedPagesCount == this.numberOfPages) {
                this.finish();
            }
            return new Page(10);
        }
    }
}

