/*
 * Decompiled with CFR 0.152.
 */
package io.airlift.log;

import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableSet;
import com.google.common.io.ByteStreams;
import com.google.common.io.MoreFiles;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.errorprone.annotations.concurrent.GuardedBy;
import io.airlift.log.LogFileName;
import io.airlift.log.LogHistoryManager;
import io.airlift.log.MessageOutput;
import io.airlift.units.DataSize;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.logging.ErrorManager;
import java.util.logging.Formatter;
import java.util.zip.GZIPOutputStream;

final class RollingFileMessageOutput
implements MessageOutput {
    private static final int MAX_OPEN_NEW_LOG_ATTEMPTS = 100;
    private static final int MAX_BATCH_BYTES = Math.toIntExact(new DataSize(1.0, DataSize.Unit.MEGABYTE).toBytes());
    private static final String TEMP_PREFIX = ".tmp.";
    private static final String DELETED_PREFIX = ".deleted.";
    private static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("-yyyyMMdd.HHmmss");
    private final Path symlink;
    private final long maxFileSize;
    private final CompressionType compressionType;
    private final Formatter formatter;
    @GuardedBy(value="this")
    private Path currentOutputFile;
    @GuardedBy(value="this")
    private LogFileName currentOutputFileName;
    @GuardedBy(value="this")
    private long currentFileSize;
    @GuardedBy(value="this")
    private OutputStream currentOutputStream;
    private final LogHistoryManager historyManager;
    private final ExecutorService compressionExecutor;
    private static final String TEMP_FILE_EXTENSION = ".tmp";
    private static final String LOG_FILE_EXTENSION = ".log";

    RollingFileMessageOutput(String filename, DataSize maxFileSize, DataSize maxTotalSize, CompressionType compressionType, Formatter formatter) {
        Objects.requireNonNull(filename, "filename is null");
        Objects.requireNonNull(maxFileSize, "maxFileSize is null");
        Objects.requireNonNull(maxTotalSize, "maxTotalSize is null");
        Objects.requireNonNull(compressionType, "compressionType is null");
        Objects.requireNonNull(formatter, "formatter is null");
        this.maxFileSize = maxFileSize.toBytes();
        this.compressionType = compressionType;
        this.formatter = formatter;
        this.symlink = Paths.get(filename, new String[0]);
        try {
            MoreFiles.createParentDirectories((Path)this.symlink, (FileAttribute[])new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        try {
            RollingFileMessageOutput.recoverLegacyTempFiles(filename);
        }
        catch (IOException e) {
            new ErrorManager().error("Unable to recover legacy logging temp files", e, 0);
        }
        if (Files.exists(this.symlink, LinkOption.NOFOLLOW_LINKS)) {
            try {
                if (Files.isDirectory(this.symlink, new LinkOption[0])) {
                    throw new IllegalArgumentException("Log file is an existing directory: " + filename);
                }
                if (!Files.isSymbolicLink(this.symlink)) {
                    BasicFileAttributes attributes = Files.readAttributes(this.symlink, BasicFileAttributes.class, new LinkOption[0]);
                    LocalDateTime createTime = LocalDateTime.ofInstant(attributes.creationTime().toInstant(), ZoneId.systemDefault()).withNano(0);
                    Path logFile = this.symlink.resolveSibling(String.valueOf(this.symlink.getFileName()) + DATE_TIME_FORMATTER.format(createTime) + "--" + String.valueOf(UUID.randomUUID()));
                    Files.move(this.symlink, logFile, StandardCopyOption.ATOMIC_MOVE);
                }
            }
            catch (IOException e) {
                throw new UncheckedIOException("Unable to update move legacy log file to a new file", e);
            }
        }
        RollingFileMessageOutput.tryCleanupTempFiles(this.symlink);
        this.historyManager = new LogHistoryManager(this.symlink, maxTotalSize);
        try {
            this.rollFile();
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        this.compressionExecutor = compressionType != CompressionType.NONE ? Executors.newSingleThreadExecutor(new ThreadFactoryBuilder().setDaemon(true).setNameFormat("log-compression-%d").build()) : null;
    }

    @Override
    public synchronized void flush() throws IOException {
        if (this.currentOutputStream != null) {
            this.currentOutputStream.flush();
        }
    }

    @Override
    public synchronized void close() throws IOException {
        IOException exception = new IOException("Exception thrown attempting to close the file output.");
        if (this.currentOutputStream != null) {
            try {
                this.currentOutputStream.flush();
            }
            catch (IOException e) {
                exception.addSuppressed(e);
            }
            try {
                this.currentOutputStream.close();
            }
            catch (IOException e) {
                exception.addSuppressed(e);
            }
        }
        if (this.compressionExecutor != null) {
            this.compressionExecutor.shutdown();
            try {
                this.compressionExecutor.awaitTermination(1L, TimeUnit.MINUTES);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
        this.currentOutputStream = null;
        this.currentOutputFile = null;
        this.currentOutputFileName = null;
        this.currentFileSize = 0L;
        if (exception.getSuppressed().length > 0) {
            throw exception;
        }
    }

    @Override
    public synchronized void writeMessage(byte[] message) throws IOException {
        if (this.currentFileSize > 0L && this.currentFileSize + (long)message.length > this.maxFileSize) {
            try {
                this.rollFile();
            }
            catch (IOException e) {
                this.currentFileSize = 0L;
                throw new IOException("Error rolling log file", e);
            }
        }
        this.historyManager.pruneLogFilesIfNecessary(this.currentFileSize + (long)message.length);
        this.currentFileSize += (long)message.length;
        this.currentOutputStream.write(message);
    }

    private synchronized void rollFile() throws IOException {
        LogFileName newFileName = null;
        Path newFile = null;
        BufferedOutputStream newOutputStream = null;
        for (int i = 0; i < 100; ++i) {
            try {
                newFileName = LogFileName.generateNextLogFileName(this.symlink, this.compressionType.getExtension());
                newFile = this.symlink.resolveSibling(newFileName.getFileName());
                newOutputStream = new BufferedOutputStream(Files.newOutputStream(newFile, StandardOpenOption.CREATE_NEW), MAX_BATCH_BYTES);
                break;
            }
            catch (FileAlreadyExistsException fileAlreadyExistsException) {
                continue;
            }
        }
        if (newOutputStream == null) {
            throw new IOException("Could not create new a unique log file: " + String.valueOf(newFile));
        }
        IOException exception = new IOException(String.format("Unable to %s log file", this.currentOutputStream == null ? "setup initial" : "roll"));
        if (this.currentOutputStream != null) {
            try {
                this.currentOutputStream.close();
            }
            catch (IOException e) {
                exception.addSuppressed(new IOException("Unable to close old output stream: " + String.valueOf(this.currentOutputFile), e));
            }
            this.historyManager.addFile(this.currentOutputFile, this.currentOutputFileName, this.currentFileSize);
            if (this.compressionExecutor != null) {
                Path originalFile = this.currentOutputFile;
                LogFileName originalLogFileName = this.currentOutputFileName;
                long originalFileSize = this.currentFileSize;
                this.compressionExecutor.submit(() -> {
                    try {
                        this.compressInternal(originalFile, originalLogFileName, originalFileSize);
                    }
                    catch (IOException e) {
                        exception.addSuppressed(e);
                    }
                });
            }
        }
        this.currentOutputFile = newFile;
        this.currentOutputFileName = newFileName;
        this.currentOutputStream = newOutputStream;
        this.currentFileSize = 0L;
        try {
            Files.deleteIfExists(this.symlink);
            Files.createSymbolicLink(this.symlink, newFile.getFileName(), new FileAttribute[0]);
        }
        catch (IOException e) {
            exception.addSuppressed(new IOException(String.format("Unable to update symlink %s to %s", this.symlink, newFile), e));
        }
        if (exception.getSuppressed().length > 0) {
            throw exception;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void compressInternal(Path originalFile, LogFileName originalLogFileName, long originalFileSize) throws IOException {
        long compressedSize;
        RollingFileMessageOutput.tryCleanupTempFiles(this.symlink);
        String compressionExtension = this.compressionType.getExtension().orElseThrow(IllegalStateException::new);
        Path tempFile = originalFile.resolveSibling(TEMP_PREFIX + String.valueOf(originalFile.getFileName()) + compressionExtension);
        try (InputStream input = Files.newInputStream(originalFile, new OpenOption[0]);
             GZIPOutputStream gzipOutputStream = new GZIPOutputStream(Files.newOutputStream(tempFile, new OpenOption[0]));){
            ByteStreams.copy((InputStream)input, (OutputStream)gzipOutputStream);
        }
        catch (IOException e) {
            throw new IOException("Unable to compress log file", e);
        }
        try {
            compressedSize = Files.size(tempFile);
        }
        catch (IOException e) {
            throw new IOException("Unable to get size of compress log file", e);
        }
        RollingFileMessageOutput rollingFileMessageOutput = this;
        synchronized (rollingFileMessageOutput) {
            if (!this.historyManager.removeFile(originalFile)) {
                try {
                    Files.deleteIfExists(tempFile);
                }
                catch (IOException e) {
                    throw new IOException("Unable to delete compress log file", e);
                }
                return;
            }
            Path compressedFile = originalFile.resolveSibling(String.valueOf(originalFile.getFileName()) + compressionExtension);
            LogFileName compressedFileName = originalLogFileName.withCompression(compressedFile);
            try {
                Files.move(tempFile, compressedFile, StandardCopyOption.ATOMIC_MOVE);
            }
            catch (IOException e) {
                this.historyManager.addFile(originalFile, originalLogFileName, originalFileSize);
                try {
                    Files.deleteIfExists(tempFile);
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            }
            this.historyManager.addFile(compressedFile, compressedFileName, compressedSize);
            try {
                Files.deleteIfExists(originalFile);
            }
            catch (IOException deleteException) {
                try {
                    Files.move(originalFile, originalFile.resolveSibling(DELETED_PREFIX + String.valueOf(originalFile.getFileName())), StandardCopyOption.ATOMIC_MOVE);
                }
                catch (IOException ignored) {
                    throw new IOException("Unable to delete original file after compression", deleteException);
                }
            }
        }
    }

    private static void tryCleanupTempFiles(Path masterLogFile) {
        try {
            for (Path file : MoreFiles.listFiles((Path)masterLogFile.getParent())) {
                String fileNameWithoutPrefix;
                String fileName = file.getFileName().toString();
                if (fileName.startsWith(TEMP_PREFIX)) {
                    fileNameWithoutPrefix = fileName.substring(TEMP_PREFIX.length());
                } else {
                    if (!fileName.startsWith(DELETED_PREFIX)) continue;
                    fileNameWithoutPrefix = fileName.substring(DELETED_PREFIX.length());
                }
                if (!LogFileName.parseHistoryLogFileName(masterLogFile.getFileName().toString(), fileNameWithoutPrefix).isPresent()) continue;
                try {
                    Files.deleteIfExists(file);
                }
                catch (IOException iOException) {}
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    public synchronized Set<LogFileName> getFiles() {
        ImmutableSet.Builder files = ImmutableSet.builder().addAll(this.historyManager.getFiles());
        if (this.currentOutputFileName != null) {
            files.add((Object)this.currentOutputFileName);
        }
        return files.build();
    }

    static void recoverLegacyTempFiles(String logPath) throws IOException {
        File logPathFile = new File(logPath).getParentFile();
        File[] tempFiles = logPathFile.listFiles((dir, name) -> name.endsWith(TEMP_FILE_EXTENSION));
        if (tempFiles == null) {
            return;
        }
        ArrayList<String> errorMessages = new ArrayList<String>();
        for (File tempFile : tempFiles) {
            String newName = tempFile.getName().substring(0, tempFile.getName().length() - TEMP_FILE_EXTENSION.length());
            File newFile = new File(tempFile.getParent(), newName + LOG_FILE_EXTENSION);
            if (tempFile.renameTo(newFile)) continue;
            errorMessages.add(String.format("Could not rename temp file [%s] to [%s]", tempFile, newFile));
        }
        if (!errorMessages.isEmpty()) {
            throw new IOException("Error recovering temp files\n" + Joiner.on((String)"\n").join(errorMessages));
        }
    }

    public static enum CompressionType {
        NONE(Optional.empty()),
        GZIP(Optional.of(".gz"));

        private final Optional<String> extension;

        private CompressionType(Optional<String> extension) {
            this.extension = Objects.requireNonNull(extension, "extension is null");
        }

        public Optional<String> getExtension() {
            return this.extension;
        }
    }
}

