/*
 * Decompiled with CFR 0.152.
 */
package io.trino.hive.formats.line.sequence;

import com.google.common.base.Preconditions;
import com.google.common.io.Closer;
import com.google.common.io.CountingOutputStream;
import io.airlift.slice.DynamicSliceOutput;
import io.airlift.slice.SizeOf;
import io.airlift.slice.Slice;
import io.airlift.slice.SliceOutput;
import io.airlift.slice.Slices;
import io.trino.hive.formats.DataOutputStream;
import io.trino.hive.formats.ReadWriteUtils;
import io.trino.hive.formats.compression.Codec;
import io.trino.hive.formats.compression.CompressionKind;
import io.trino.hive.formats.compression.MemoryCompressedSliceOutput;
import io.trino.hive.formats.compression.ValueCompressor;
import io.trino.hive.formats.line.LineWriter;
import java.io.Closeable;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ThreadLocalRandom;
import java.util.function.LongSupplier;

public class SequenceFileWriter
implements LineWriter {
    private static final int INSTANCE_SIZE = SizeOf.instanceSize(SequenceFileWriter.class);
    private static final Slice SEQUENCE_FILE_MAGIC = Slices.utf8Slice((String)"SEQ");
    private static final byte SEQUENCE_FILE_VERSION = 6;
    static final String TRINO_SEQUENCE_FILE_WRITER_VERSION_METADATA_KEY = "trino.writer.version";
    static final String TRINO_SEQUENCE_FILE_WRITER_VERSION;
    private final ValueWriter valueWriter;
    private final LongSupplier writtenBytes;

    public SequenceFileWriter(OutputStream rawOutput, Optional<CompressionKind> compressionKind, boolean blockCompression, Map<String, String> metadata) throws IOException {
        try {
            Objects.requireNonNull(rawOutput, "raw output is null");
            Objects.requireNonNull(compressionKind, "compressionKind is null");
            Preconditions.checkArgument((!blockCompression || compressionKind.isPresent() ? 1 : 0) != 0, (Object)"Block compression can only be enabled when a compression codec is provided");
            Preconditions.checkArgument((!compressionKind.equals(Optional.of(CompressionKind.LZOP)) ? 1 : 0) != 0, (Object)"LZOP cannot be used with SequenceFile. LZO compression can be used, but LZ4 is preferred.");
            CountingOutputStream countingOutputStream = new CountingOutputStream(rawOutput);
            this.writtenBytes = () -> ((CountingOutputStream)countingOutputStream).getCount();
            DataOutputStream output = new DataOutputStream((OutputStream)countingOutputStream);
            output.write(SEQUENCE_FILE_MAGIC);
            output.write(6);
            ReadWriteUtils.writeLengthPrefixedString(output, Slices.utf8Slice((String)"org.apache.hadoop.io.BytesWritable"));
            ReadWriteUtils.writeLengthPrefixedString(output, Slices.utf8Slice((String)"org.apache.hadoop.io.Text"));
            output.writeBoolean(compressionKind.isPresent());
            output.writeBoolean(blockCompression);
            if (compressionKind.isPresent()) {
                ReadWriteUtils.writeLengthPrefixedString(output, Slices.utf8Slice((String)compressionKind.get().getHadoopClassName()));
            }
            output.writeInt(Integer.reverseBytes(metadata.size() + 1));
            SequenceFileWriter.writeMetadataProperty(output, TRINO_SEQUENCE_FILE_WRITER_VERSION_METADATA_KEY, TRINO_SEQUENCE_FILE_WRITER_VERSION);
            for (Map.Entry<String, String> entry : metadata.entrySet()) {
                SequenceFileWriter.writeMetadataProperty(output, entry.getKey(), entry.getValue());
            }
            long syncFirst = ThreadLocalRandom.current().nextLong();
            long syncSecond = ThreadLocalRandom.current().nextLong();
            output.writeLong(syncFirst);
            output.writeLong(syncSecond);
            Optional<Codec> codec = compressionKind.map(CompressionKind::createCodec);
            this.valueWriter = blockCompression ? new BlockCompressionValueWriter(output, codec.orElseThrow(), syncFirst, syncSecond) : new SingleValueWriter(output, codec, syncFirst, syncSecond);
        }
        catch (Throwable e) {
            OutputStream outputStream = rawOutput;
            try {
                throw e;
            }
            catch (Throwable throwable) {
                if (outputStream != null) {
                    try {
                        outputStream.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                }
                throw throwable;
            }
        }
    }

    private static void writeMetadataProperty(DataOutputStream output, String key, String value) throws IOException {
        ReadWriteUtils.writeLengthPrefixedString(output, Slices.utf8Slice((String)key));
        ReadWriteUtils.writeLengthPrefixedString(output, Slices.utf8Slice((String)value));
    }

    @Override
    public long getWrittenBytes() {
        return this.writtenBytes.getAsLong();
    }

    @Override
    public long getRetainedSizeInBytes() {
        return (long)INSTANCE_SIZE + this.valueWriter.getRetainedSizeInBytes();
    }

    @Override
    public void write(Slice value) throws IOException {
        this.valueWriter.write(value);
    }

    @Override
    public void close() throws IOException {
        this.valueWriter.close();
    }

    static {
        String version = SequenceFileWriter.class.getPackage().getImplementationVersion();
        TRINO_SEQUENCE_FILE_WRITER_VERSION = version == null ? "UNKNOWN" : version;
    }

    private static class BlockCompressionValueWriter
    implements ValueWriter {
        private static final int INSTANCE_SIZE = SizeOf.instanceSize(BlockCompressionValueWriter.class);
        private static final int MAX_ROWS = 10000000;
        private static final int TARGET_BLOCK_SIZE = 0x100000;
        private static final int OTHER_MIN_BUFFER_SIZE = 4096;
        private static final int OTHER_MAX_BUFFER_SIZE = 16384;
        private static final int VALUE_MIN_BUFFER_SIZE = 4096;
        private static final int VALUE_MAX_BUFFER_SIZE = 0x100000;
        private final DataOutputStream output;
        private final long syncFirst;
        private final long syncSecond;
        private MemoryCompressedSliceOutput keyLengthOutput;
        private MemoryCompressedSliceOutput keyOutput;
        private MemoryCompressedSliceOutput valueLengthOutput;
        private MemoryCompressedSliceOutput valueOutput;
        private int valueCount;
        private final Closer closer = Closer.create();

        public BlockCompressionValueWriter(DataOutputStream output, Codec trinoCodec, long syncFirst, long syncSecond) throws IOException {
            try {
                this.output = (DataOutputStream)this.closer.register((Closeable)Objects.requireNonNull(output, "output is null"));
                Objects.requireNonNull(trinoCodec, "trinoCodec is null");
                this.syncFirst = syncFirst;
                this.syncSecond = syncSecond;
                this.keyLengthOutput = trinoCodec.createMemoryCompressedSliceOutput(4096, 16384);
                this.closer.register(this.keyLengthOutput::destroy);
                this.keyOutput = trinoCodec.createMemoryCompressedSliceOutput(4096, 16384);
                this.closer.register(this.keyOutput::destroy);
                this.valueLengthOutput = trinoCodec.createMemoryCompressedSliceOutput(4096, 16384);
                this.closer.register(this.valueLengthOutput::destroy);
                this.valueOutput = trinoCodec.createMemoryCompressedSliceOutput(4096, 0x100000);
                this.closer.register(this.valueOutput::destroy);
            }
            catch (Throwable e) {
                Closer closer = this.closer;
                try {
                    throw e;
                }
                catch (Throwable throwable) {
                    if (closer != null) {
                        try {
                            closer.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
            }
        }

        @Override
        public long getRetainedSizeInBytes() {
            return (long)INSTANCE_SIZE + this.output.getRetainedSize() + this.keyLengthOutput.getRetainedSize() + this.keyOutput.getRetainedSize() + this.valueLengthOutput.getRetainedSize() + this.valueOutput.getRetainedSize();
        }

        @Override
        public void close() throws IOException {
            this.flushBlock();
            this.closer.close();
        }

        @Override
        public void write(Slice value) throws IOException {
            try {
                this.keyLengthOutput.writeInt(4);
                this.keyOutput.writeInt(0);
                ReadWriteUtils.writeVInt(this.valueLengthOutput, value.length() + ReadWriteUtils.computeVIntLength(value.length()));
                ReadWriteUtils.writeVInt(this.valueOutput, value.length());
                this.valueOutput.writeBytes(value);
                ++this.valueCount;
                if (this.valueCount >= 10000000 || this.getBufferedSize() > 0x100000) {
                    this.flushBlock();
                }
            }
            catch (Throwable e) {
                Closer closer = this.closer;
                try {
                    throw e;
                }
                catch (Throwable throwable) {
                    if (closer != null) {
                        try {
                            closer.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
            }
        }

        private int getBufferedSize() {
            return this.keyLengthOutput.size() + this.keyOutput.size() + this.valueLengthOutput.size() + this.valueOutput.size();
        }

        private void flushBlock() throws IOException {
            if (this.valueCount == 0) {
                return;
            }
            this.output.writeInt(-1);
            this.output.writeLong(this.syncFirst);
            this.output.writeLong(this.syncSecond);
            ReadWriteUtils.writeVInt(this.output, this.valueCount);
            this.keyLengthOutput = this.writeBlockAndRecycleBuffer(this.keyLengthOutput);
            this.keyOutput = this.writeBlockAndRecycleBuffer(this.keyOutput);
            this.valueLengthOutput = this.writeBlockAndRecycleBuffer(this.valueLengthOutput);
            this.valueOutput = this.writeBlockAndRecycleBuffer(this.valueOutput);
            this.valueCount = 0;
        }

        private MemoryCompressedSliceOutput writeBlockAndRecycleBuffer(MemoryCompressedSliceOutput compressedBlock) throws IOException {
            compressedBlock.close();
            ReadWriteUtils.writeVInt(this.output, compressedBlock.getCompressedSize());
            for (Slice slice : compressedBlock.getCompressedSlices()) {
                this.output.write(slice);
            }
            return compressedBlock.createRecycledCompressedSliceOutput();
        }
    }

    private static interface ValueWriter
    extends Closeable {
        public void write(Slice var1) throws IOException;

        public long getRetainedSizeInBytes();
    }

    private static class SingleValueWriter
    implements ValueWriter {
        private static final int INSTANCE_SIZE = SizeOf.instanceSize(SingleValueWriter.class);
        private static final int SYNC_INTERVAL = 10240;
        private final DataOutputStream output;
        private final long syncFirst;
        private final long syncSecond;
        private final ValueCompressor valueCompressor;
        private final DynamicSliceOutput uncompressedBuffer = new DynamicSliceOutput(0);
        private long currentPosition;
        private long lastSyncPosition;
        private final Closer closer = Closer.create();

        public SingleValueWriter(DataOutputStream output, Optional<Codec> codec, long syncFirst, long syncSecond) throws IOException {
            try {
                this.output = (DataOutputStream)this.closer.register((Closeable)Objects.requireNonNull(output, "output is null"));
                Objects.requireNonNull(codec, "codec is null");
                this.valueCompressor = codec.isPresent() ? codec.get().createValueCompressor() : null;
                this.syncFirst = syncFirst;
                this.syncSecond = syncSecond;
                this.currentPosition += output.longSize();
            }
            catch (Throwable e) {
                Closer closer = this.closer;
                try {
                    throw e;
                }
                catch (Throwable throwable) {
                    if (closer != null) {
                        try {
                            closer.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
            }
        }

        @Override
        public void write(Slice value) throws IOException {
            try {
                this.uncompressedBuffer.reset();
                ReadWriteUtils.writeVInt((SliceOutput)this.uncompressedBuffer, value.length());
                this.uncompressedBuffer.writeBytes(value);
                value = this.uncompressedBuffer.slice();
                Slice compressedValue = this.valueCompressor == null ? value : this.valueCompressor.compress(value);
                this.output.writeInt(Integer.reverseBytes(4 + compressedValue.length()));
                this.output.writeInt(Integer.reverseBytes(4));
                this.output.writeInt(0);
                this.output.write(compressedValue);
                this.currentPosition += (long)(12 + compressedValue.length());
                if (this.output.longSize() - this.lastSyncPosition > 10240L) {
                    this.output.writeInt(-1);
                    this.output.writeLong(this.syncFirst);
                    this.output.writeLong(this.syncSecond);
                    this.currentPosition += 20L;
                    this.lastSyncPosition = this.currentPosition;
                }
            }
            catch (Throwable e) {
                Closer closer = this.closer;
                try {
                    throw e;
                }
                catch (Throwable throwable) {
                    if (closer != null) {
                        try {
                            closer.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
            }
        }

        @Override
        public long getRetainedSizeInBytes() {
            return (long)INSTANCE_SIZE + this.output.getRetainedSize();
        }

        @Override
        public void close() throws IOException {
            this.closer.close();
        }
    }
}

