/*
 * Decompiled with CFR 0.152.
 */
package io.trino.filesystem.gcs;

import com.google.api.client.util.Preconditions;
import com.google.api.gax.paging.Page;
import com.google.cloud.storage.Blob;
import com.google.cloud.storage.Storage;
import com.google.cloud.storage.StorageBatch;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import com.google.common.collect.UnmodifiableIterator;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import io.airlift.concurrent.MoreFutures;
import io.trino.filesystem.FileIterator;
import io.trino.filesystem.Location;
import io.trino.filesystem.TrinoFileSystem;
import io.trino.filesystem.TrinoInputFile;
import io.trino.filesystem.TrinoOutputFile;
import io.trino.filesystem.gcs.GcsFileIterator;
import io.trino.filesystem.gcs.GcsInputFile;
import io.trino.filesystem.gcs.GcsLocation;
import io.trino.filesystem.gcs.GcsOutputFile;
import io.trino.filesystem.gcs.GcsUtils;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Set;
import java.util.concurrent.Future;

public class GcsFileSystem
implements TrinoFileSystem {
    private final ListeningExecutorService executorService;
    private final Storage storage;
    private final int readBlockSizeBytes;
    private final long writeBlockSizeBytes;
    private final int pageSize;
    private final int batchSize;

    public GcsFileSystem(ListeningExecutorService executorService, Storage storage, int readBlockSizeBytes, long writeBlockSizeBytes, int pageSize, int batchSize) {
        this.executorService = Objects.requireNonNull(executorService, "executorService is null");
        this.storage = Objects.requireNonNull(storage, "storage is null");
        this.readBlockSizeBytes = readBlockSizeBytes;
        this.writeBlockSizeBytes = writeBlockSizeBytes;
        this.pageSize = pageSize;
        this.batchSize = batchSize;
    }

    public TrinoInputFile newInputFile(Location location) {
        GcsLocation gcsLocation = new GcsLocation(location);
        GcsFileSystem.checkIsValidFile(gcsLocation);
        return new GcsInputFile(gcsLocation, this.storage, this.readBlockSizeBytes, OptionalLong.empty());
    }

    public TrinoInputFile newInputFile(Location location, long length) {
        GcsLocation gcsLocation = new GcsLocation(location);
        GcsFileSystem.checkIsValidFile(gcsLocation);
        return new GcsInputFile(gcsLocation, this.storage, this.readBlockSizeBytes, OptionalLong.of(length));
    }

    public TrinoOutputFile newOutputFile(Location location) {
        GcsLocation gcsLocation = new GcsLocation(location);
        GcsFileSystem.checkIsValidFile(gcsLocation);
        return new GcsOutputFile(gcsLocation, this.storage, this.writeBlockSizeBytes);
    }

    public void deleteFile(Location location) throws IOException {
        GcsLocation gcsLocation = new GcsLocation(location);
        GcsFileSystem.checkIsValidFile(gcsLocation);
        Blob blob = GcsUtils.getBlobOrThrow(this.storage, gcsLocation, new Storage.BlobGetOption[0]);
        blob.delete(new Blob.BlobSourceOption[0]);
    }

    public void deleteFiles(Collection<Location> locations) throws IOException {
        ArrayList<ListenableFuture> batchFutures = new ArrayList<ListenableFuture>();
        try {
            for (List locationBatch : Iterables.partition(locations, (int)this.batchSize)) {
                StorageBatch batch = this.storage.batch();
                for (Location location : locationBatch) {
                    batch.delete(GcsUtils.getBlobOrThrow(this.storage, new GcsLocation(location), new Storage.BlobGetOption[0]).getBlobId(), new Storage.BlobSourceOption[0]);
                }
                batchFutures.add(this.executorService.submit(() -> ((StorageBatch)batch).submit()));
            }
            MoreFutures.getFutureValue((Future)Futures.allAsList(batchFutures));
        }
        catch (RuntimeException e) {
            throw GcsUtils.handleGcsException(e, "delete files", locations);
        }
    }

    public void deleteDirectory(Location location) throws IOException {
        GcsLocation gcsLocation = new GcsLocation(GcsFileSystem.normalizeToDirectory(location));
        try {
            ArrayList<ListenableFuture> batchFutures = new ArrayList<ListenableFuture>();
            for (List blobBatch : Iterables.partition((Iterable)this.getPage(gcsLocation, new Storage.BlobListOption[0]).iterateAll(), (int)this.batchSize)) {
                StorageBatch batch = this.storage.batch();
                for (Blob blob : blobBatch) {
                    batch.delete(blob.getBlobId(), new Storage.BlobSourceOption[0]);
                }
                batchFutures.add(this.executorService.submit(() -> ((StorageBatch)batch).submit()));
            }
            MoreFutures.getFutureValue((Future)Futures.allAsList(batchFutures));
        }
        catch (RuntimeException e) {
            throw GcsUtils.handleGcsException(e, "deleting directory", gcsLocation);
        }
    }

    public void renameFile(Location source, Location target) throws IOException {
        throw new IOException("GCS does not support renames");
    }

    public FileIterator listFiles(Location location) throws IOException {
        GcsLocation gcsLocation = new GcsLocation(GcsFileSystem.normalizeToDirectory(location));
        try {
            return new GcsFileIterator(gcsLocation, this.getPage(gcsLocation, new Storage.BlobListOption[0]));
        }
        catch (RuntimeException e) {
            throw GcsUtils.handleGcsException(e, "listing files", gcsLocation);
        }
    }

    private static Location normalizeToDirectory(Location location) {
        String path = location.path();
        if (!path.isEmpty() && !path.endsWith("/")) {
            return location.appendSuffix("/");
        }
        return location;
    }

    private static void checkIsValidFile(GcsLocation gcsLocation) {
        Preconditions.checkState((!gcsLocation.path().isEmpty() ? 1 : 0) != 0, (String)"Location path is empty: %s", (Object[])new Object[]{gcsLocation});
        Preconditions.checkState((!gcsLocation.path().endsWith("/") ? 1 : 0) != 0, (String)"Location path ends with a slash: %s", (Object[])new Object[]{gcsLocation});
    }

    private Page<Blob> getPage(GcsLocation location, Storage.BlobListOption ... blobListOptions) {
        ArrayList<Storage.BlobListOption> optionsBuilder = new ArrayList<Storage.BlobListOption>();
        if (!location.path().isEmpty()) {
            optionsBuilder.add(Storage.BlobListOption.prefix((String)location.path()));
        }
        Arrays.stream(blobListOptions).forEach(optionsBuilder::add);
        optionsBuilder.add(Storage.BlobListOption.pageSize((long)this.pageSize));
        return this.storage.list(location.bucket(), (Storage.BlobListOption[])optionsBuilder.toArray(Storage.BlobListOption[]::new));
    }

    public Optional<Boolean> directoryExists(Location location) throws IOException {
        GcsLocation gcsLocation = new GcsLocation(location);
        if (gcsLocation.path().isEmpty()) {
            return Optional.of(this.bucketExists(gcsLocation.bucket()));
        }
        if (this.listFiles(location).hasNext()) {
            return Optional.of(true);
        }
        return Optional.empty();
    }

    private boolean bucketExists(String bucket) {
        return this.storage.get(bucket, new Storage.BucketGetOption[0]) != null;
    }

    public void createDirectory(Location location) throws IOException {
        GcsFileSystem.validateGcsLocation(location);
    }

    public void renameDirectory(Location source, Location target) throws IOException {
        throw new IOException("GCS does not support directory renames");
    }

    public Set<Location> listDirectories(Location location) throws IOException {
        GcsLocation gcsLocation = new GcsLocation(GcsFileSystem.normalizeToDirectory(location));
        try {
            Page<Blob> page = this.getPage(gcsLocation, Storage.BlobListOption.currentDirectory(), Storage.BlobListOption.matchGlob((String)(gcsLocation.path() + "*/")));
            UnmodifiableIterator blobIterator = Iterators.filter(page.iterateAll().iterator(), blob -> blob.isDirectory());
            ImmutableSet.Builder locationBuilder = ImmutableSet.builder();
            while (blobIterator.hasNext()) {
                locationBuilder.add((Object)Location.of((String)(gcsLocation.getBase() + ((Blob)blobIterator.next()).getName())));
            }
            return locationBuilder.build();
        }
        catch (RuntimeException e) {
            throw GcsUtils.handleGcsException(e, "listing directories", gcsLocation);
        }
    }

    public Optional<Location> createTemporaryDirectory(Location targetPath, String temporaryPrefix, String relativePrefix) {
        GcsFileSystem.validateGcsLocation(targetPath);
        return Optional.empty();
    }

    private static void validateGcsLocation(Location location) {
        new GcsLocation(location);
    }
}

