/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.hive.metastore.thrift;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.base.Ticker;
import com.google.common.collect.ImmutableList;
import com.google.common.net.HostAndPort;
import io.airlift.units.Duration;
import io.trino.plugin.hive.metastore.thrift.FailureAwareThriftMetastoreClient;
import io.trino.plugin.hive.metastore.thrift.MetastoreLocator;
import io.trino.plugin.hive.metastore.thrift.StaticMetastoreConfig;
import io.trino.plugin.hive.metastore.thrift.ThriftMetastoreAuthenticationConfig;
import io.trino.plugin.hive.metastore.thrift.ThriftMetastoreClient;
import io.trino.plugin.hive.metastore.thrift.ThriftMetastoreClientFactory;
import java.net.URI;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nullable;
import javax.inject.Inject;
import org.apache.thrift.TException;

public class StaticMetastoreLocator
implements MetastoreLocator {
    private final List<Backoff> backoffs;
    private final ThriftMetastoreClientFactory clientFactory;
    private final String metastoreUsername;

    @Inject
    public StaticMetastoreLocator(StaticMetastoreConfig config, ThriftMetastoreAuthenticationConfig authenticationConfig, ThriftMetastoreClientFactory clientFactory) {
        this(config, authenticationConfig, clientFactory, Ticker.systemTicker());
    }

    @VisibleForTesting
    StaticMetastoreLocator(StaticMetastoreConfig config, ThriftMetastoreAuthenticationConfig authenticationConfig, ThriftMetastoreClientFactory clientFactory, Ticker ticker) {
        this(config.getMetastoreUris(), config.getMetastoreUsername(), clientFactory, ticker);
        Preconditions.checkArgument((Strings.isNullOrEmpty((String)this.metastoreUsername) || authenticationConfig.getAuthenticationType() == ThriftMetastoreAuthenticationConfig.ThriftMetastoreAuthenticationType.NONE ? 1 : 0) != 0, (String)"%s cannot be used together with %s authentication", (Object)"hive.metastore.username", (Object)((Object)authenticationConfig.getAuthenticationType()));
    }

    public StaticMetastoreLocator(List<URI> metastoreUris, @Nullable String metastoreUsername, ThriftMetastoreClientFactory clientFactory) {
        this(metastoreUris, metastoreUsername, clientFactory, Ticker.systemTicker());
    }

    private StaticMetastoreLocator(List<URI> metastoreUris, @Nullable String metastoreUsername, ThriftMetastoreClientFactory clientFactory, Ticker ticker) {
        Objects.requireNonNull(metastoreUris, "metastoreUris is null");
        Preconditions.checkArgument((!metastoreUris.isEmpty() ? 1 : 0) != 0, (Object)"metastoreUris must specify at least one URI");
        this.backoffs = (List)metastoreUris.stream().map(StaticMetastoreLocator::checkMetastoreUri).map(uri -> HostAndPort.fromParts((String)uri.getHost(), (int)uri.getPort())).map(address -> new Backoff((HostAndPort)address, ticker)).collect(ImmutableList.toImmutableList());
        this.metastoreUsername = metastoreUsername;
        this.clientFactory = Objects.requireNonNull(clientFactory, "clientFactory is null");
    }

    @Override
    public ThriftMetastoreClient createMetastoreClient(Optional<String> delegationToken) throws TException {
        Comparator<Backoff> comparator = Comparator.comparingLong(Backoff::getBackoffDuration).thenComparingLong(Backoff::getLastFailureTimestamp);
        List backoffsSorted = (List)this.backoffs.stream().sorted(comparator).collect(ImmutableList.toImmutableList());
        TException lastException = null;
        for (Backoff backoff : backoffsSorted) {
            try {
                return this.getClient(backoff.getAddress(), backoff, delegationToken);
            }
            catch (TException e) {
                lastException = e;
            }
        }
        List addresses = (List)backoffsSorted.stream().map(Backoff::getAddress).collect(ImmutableList.toImmutableList());
        throw new TException("Failed connecting to Hive metastore: " + addresses, (Throwable)lastException);
    }

    private ThriftMetastoreClient getClient(HostAndPort address, final Backoff backoff, Optional<String> delegationToken) throws TException {
        FailureAwareThriftMetastoreClient client = new FailureAwareThriftMetastoreClient(this.clientFactory.create(address, delegationToken), new FailureAwareThriftMetastoreClient.Callback(){

            @Override
            public void success() {
                backoff.success();
            }

            @Override
            public void failed(TException e) {
                backoff.fail();
            }
        });
        if (!Strings.isNullOrEmpty((String)this.metastoreUsername)) {
            client.setUGI(this.metastoreUsername);
        }
        return client;
    }

    private static URI checkMetastoreUri(URI uri) {
        Objects.requireNonNull(uri, "uri is null");
        String scheme = uri.getScheme();
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)scheme) ? 1 : 0) != 0, (String)"metastoreUri scheme is missing: %s", (Object)uri);
        Preconditions.checkArgument((boolean)scheme.equals("thrift"), (String)"metastoreUri scheme must be thrift: %s", (Object)uri);
        Preconditions.checkArgument((uri.getHost() != null ? 1 : 0) != 0, (String)"metastoreUri host is missing: %s", (Object)uri);
        Preconditions.checkArgument((uri.getPort() != -1 ? 1 : 0) != 0, (String)"metastoreUri port is missing: %s", (Object)uri);
        return uri;
    }

    @VisibleForTesting
    static class Backoff {
        static final long MIN_BACKOFF = new Duration(50.0, TimeUnit.MILLISECONDS).roundTo(TimeUnit.NANOSECONDS);
        static final long MAX_BACKOFF = new Duration(60.0, TimeUnit.SECONDS).roundTo(TimeUnit.NANOSECONDS);
        private final HostAndPort address;
        private final Ticker ticker;
        private long backoffDuration = MIN_BACKOFF;
        private OptionalLong lastFailureTimestamp = OptionalLong.empty();

        Backoff(HostAndPort address, Ticker ticker) {
            this.address = Objects.requireNonNull(address, "address is null");
            this.ticker = Objects.requireNonNull(ticker, "ticker is null");
        }

        public HostAndPort getAddress() {
            return this.address;
        }

        synchronized void fail() {
            this.lastFailureTimestamp = OptionalLong.of(this.ticker.read());
            this.backoffDuration = Math.min(this.backoffDuration * 2L, MAX_BACKOFF);
        }

        synchronized void success() {
            this.lastFailureTimestamp = OptionalLong.empty();
            this.backoffDuration = MIN_BACKOFF;
        }

        synchronized long getLastFailureTimestamp() {
            return this.lastFailureTimestamp.orElse(Long.MIN_VALUE);
        }

        synchronized long getBackoffDuration() {
            if (this.lastFailureTimestamp.isEmpty()) {
                return 0L;
            }
            long timeSinceLastFail = this.ticker.read() - this.lastFailureTimestamp.getAsLong();
            return Math.max(this.backoffDuration - timeSinceLastFail, 0L);
        }
    }
}

