/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.azurebfs.services;

import io.trino.hadoop.$internal.org.slf4j.Logger;
import io.trino.hadoop.$internal.org.slf4j.LoggerFactory;
import java.util.Collection;
import java.util.LinkedList;
import java.util.Queue;
import java.util.Stack;
import java.util.concurrent.CountDownLatch;
import org.apache.hadoop.fs.azurebfs.contracts.services.ReadBufferStatus;
import org.apache.hadoop.fs.azurebfs.services.AbfsInputStream;
import org.apache.hadoop.fs.azurebfs.services.ReadBuffer;
import org.apache.hadoop.fs.azurebfs.services.ReadBufferWorker;

final class ReadBufferManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(ReadBufferManager.class);
    private static final int NUM_BUFFERS = 16;
    private static final int BLOCK_SIZE = 0x400000;
    private static final int NUM_THREADS = 8;
    private static final int THRESHOLD_AGE_MILLISECONDS = 3000;
    private Thread[] threads = new Thread[8];
    private byte[][] buffers;
    private Stack<Integer> freeList = new Stack();
    private Queue<ReadBuffer> readAheadQueue = new LinkedList<ReadBuffer>();
    private LinkedList<ReadBuffer> inProgressList = new LinkedList();
    private LinkedList<ReadBuffer> completedReadList = new LinkedList();
    private static final ReadBufferManager BUFFER_MANAGER = new ReadBufferManager();

    static ReadBufferManager getBufferManager() {
        return BUFFER_MANAGER;
    }

    private void init() {
        int i;
        this.buffers = new byte[16][];
        for (i = 0; i < 16; ++i) {
            this.buffers[i] = new byte[0x400000];
            this.freeList.add(i);
        }
        for (i = 0; i < 8; ++i) {
            Thread t = new Thread(new ReadBufferWorker(i));
            t.setDaemon(true);
            this.threads[i] = t;
            t.setName("ABFS-prefetch-" + i);
            t.start();
        }
        ReadBufferWorker.UNLEASH_WORKERS.countDown();
    }

    private ReadBufferManager() {
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void queueReadAhead(AbfsInputStream stream, long requestedOffset, int requestedLength) {
        ReadBuffer buffer;
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("Start Queueing readAhead for {} offset {} length {}", stream.getPath(), requestedOffset, requestedLength);
        }
        ReadBufferManager readBufferManager = this;
        synchronized (readBufferManager) {
            if (this.isAlreadyQueued(stream, requestedOffset)) {
                return;
            }
            if (this.freeList.isEmpty() && !this.tryEvict()) {
                return;
            }
            buffer = new ReadBuffer();
            buffer.setStream(stream);
            buffer.setOffset(requestedOffset);
            buffer.setLength(0);
            buffer.setRequestedLength(requestedLength);
            buffer.setStatus(ReadBufferStatus.NOT_AVAILABLE);
            buffer.setLatch(new CountDownLatch(1));
            Integer bufferIndex = this.freeList.pop();
            buffer.setBuffer(this.buffers[bufferIndex]);
            buffer.setBufferindex(bufferIndex);
            this.readAheadQueue.add(buffer);
            this.notifyAll();
        }
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("Done q-ing readAhead for file {} offset {} buffer idx {}", stream.getPath(), requestedOffset, buffer.getBufferindex());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    int getBlock(AbfsInputStream stream, long position, int length, byte[] buffer) {
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("getBlock for file {}  position {}  thread {}", stream.getPath(), position, Thread.currentThread().getName());
        }
        this.waitForProcess(stream, position);
        int bytesRead = 0;
        ReadBufferManager readBufferManager = this;
        synchronized (readBufferManager) {
            bytesRead = this.getBlockFromCompletedQueue(stream, position, length, buffer);
        }
        if (bytesRead > 0) {
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("Done read from Cache for {} position {} length {}", stream.getPath(), position, bytesRead);
            }
            return bytesRead;
        }
        return 0;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void waitForProcess(AbfsInputStream stream, long position) {
        ReadBuffer readBuf;
        ReadBufferManager readBufferManager = this;
        synchronized (readBufferManager) {
            this.clearFromReadAheadQueue(stream, position);
            readBuf = this.getFromList(this.inProgressList, stream, position);
        }
        if (readBuf != null) {
            try {
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.trace("got a relevant read buffer for file {} offset {} buffer idx {}", stream.getPath(), readBuf.getOffset(), readBuf.getBufferindex());
                }
                readBuf.getLatch().await();
            }
            catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
            }
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("latch done for file {} buffer idx {} length {}", stream.getPath(), readBuf.getBufferindex(), readBuf.getLength());
            }
        }
    }

    private synchronized boolean tryEvict() {
        ReadBuffer nodeToEvict = null;
        if (this.completedReadList.size() <= 0) {
            return false;
        }
        for (ReadBuffer buf : this.completedReadList) {
            if (!buf.isFirstByteConsumed() || !buf.isLastByteConsumed()) continue;
            nodeToEvict = buf;
            break;
        }
        if (nodeToEvict != null) {
            return this.evict(nodeToEvict);
        }
        for (ReadBuffer buf : this.completedReadList) {
            if (!buf.isAnyByteConsumed()) continue;
            nodeToEvict = buf;
            break;
        }
        if (nodeToEvict != null) {
            return this.evict(nodeToEvict);
        }
        long earliestBirthday = Long.MAX_VALUE;
        for (ReadBuffer buf : this.completedReadList) {
            if (buf.getTimeStamp() >= earliestBirthday) continue;
            nodeToEvict = buf;
            earliestBirthday = buf.getTimeStamp();
        }
        if (this.currentTimeMillis() - earliestBirthday > 3000L && nodeToEvict != null) {
            return this.evict(nodeToEvict);
        }
        return false;
    }

    private boolean evict(ReadBuffer buf) {
        this.freeList.push(buf.getBufferindex());
        this.completedReadList.remove(buf);
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("Evicting buffer idx {}; was used for file {} offset {} length {}", buf.getBufferindex(), buf.getStream().getPath(), buf.getOffset(), buf.getLength());
        }
        return true;
    }

    private boolean isAlreadyQueued(AbfsInputStream stream, long requestedOffset) {
        return this.isInList(this.readAheadQueue, stream, requestedOffset) || this.isInList(this.inProgressList, stream, requestedOffset) || this.isInList(this.completedReadList, stream, requestedOffset);
    }

    private boolean isInList(Collection<ReadBuffer> list, AbfsInputStream stream, long requestedOffset) {
        return this.getFromList(list, stream, requestedOffset) != null;
    }

    private ReadBuffer getFromList(Collection<ReadBuffer> list, AbfsInputStream stream, long requestedOffset) {
        for (ReadBuffer buffer : list) {
            if (buffer.getStream() != stream) continue;
            if (buffer.getStatus() == ReadBufferStatus.AVAILABLE && requestedOffset >= buffer.getOffset() && requestedOffset < buffer.getOffset() + (long)buffer.getLength()) {
                return buffer;
            }
            if (requestedOffset < buffer.getOffset() || requestedOffset >= buffer.getOffset() + (long)buffer.getRequestedLength()) continue;
            return buffer;
        }
        return null;
    }

    private void clearFromReadAheadQueue(AbfsInputStream stream, long requestedOffset) {
        ReadBuffer buffer = this.getFromList(this.readAheadQueue, stream, requestedOffset);
        if (buffer != null) {
            this.readAheadQueue.remove(buffer);
            this.notifyAll();
            this.freeList.push(buffer.getBufferindex());
        }
    }

    private int getBlockFromCompletedQueue(AbfsInputStream stream, long position, int length, byte[] buffer) {
        ReadBuffer buf = this.getFromList(this.completedReadList, stream, position);
        if (buf == null || position >= buf.getOffset() + (long)buf.getLength()) {
            return 0;
        }
        int cursor = (int)(position - buf.getOffset());
        int availableLengthInBuffer = buf.getLength() - cursor;
        int lengthToCopy = Math.min(length, availableLengthInBuffer);
        System.arraycopy(buf.getBuffer(), cursor, buffer, 0, lengthToCopy);
        if (cursor == 0) {
            buf.setFirstByteConsumed(true);
        }
        if (cursor + lengthToCopy == buf.getLength()) {
            buf.setLastByteConsumed(true);
        }
        buf.setAnyByteConsumed(true);
        return lengthToCopy;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    ReadBuffer getNextBlockToRead() throws InterruptedException {
        ReadBuffer buffer = null;
        ReadBufferManager readBufferManager = this;
        synchronized (readBufferManager) {
            while (this.readAheadQueue.size() == 0) {
                this.wait();
            }
            buffer = this.readAheadQueue.remove();
            this.notifyAll();
            if (buffer == null) {
                return null;
            }
            buffer.setStatus(ReadBufferStatus.READING_IN_PROGRESS);
            this.inProgressList.add(buffer);
        }
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("ReadBufferWorker picked file {} for offset {}", (Object)buffer.getStream().getPath(), (Object)buffer.getOffset());
        }
        return buffer;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void doneReading(ReadBuffer buffer, ReadBufferStatus result, int bytesActuallyRead) {
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("ReadBufferWorker completed file {} for offset {} bytes {}", buffer.getStream().getPath(), buffer.getOffset(), bytesActuallyRead);
        }
        ReadBufferManager readBufferManager = this;
        synchronized (readBufferManager) {
            this.inProgressList.remove(buffer);
            if (result == ReadBufferStatus.AVAILABLE && bytesActuallyRead > 0) {
                buffer.setStatus(ReadBufferStatus.AVAILABLE);
                buffer.setTimeStamp(this.currentTimeMillis());
                buffer.setLength(bytesActuallyRead);
                this.completedReadList.add(buffer);
            } else {
                this.freeList.push(buffer.getBufferindex());
            }
        }
        buffer.getLatch().countDown();
    }

    private long currentTimeMillis() {
        return System.nanoTime() / 1000L / 1000L;
    }

    static {
        BUFFER_MANAGER.init();
    }
}

