/*
 * Decompiled with CFR 0.152.
 */
package io.trino.filesystem.gcs;

import com.google.cloud.ReadChannel;
import com.google.cloud.storage.Blob;
import com.google.common.primitives.Ints;
import com.google.common.primitives.Longs;
import io.trino.filesystem.TrinoInputStream;
import io.trino.filesystem.gcs.GcsLocation;
import io.trino.filesystem.gcs.GcsUtils;
import java.io.EOFException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Objects;
import java.util.OptionalLong;

public class GcsInputStream
extends TrinoInputStream {
    private final GcsLocation location;
    private final Blob blob;
    private final int readBlockSizeBytes;
    private final long fileSize;
    private final OptionalLong predeclaredLength;
    private ReadChannel readChannel;
    private ByteBuffer readBuffer = ByteBuffer.allocate(1);
    private long currentPosition;
    private long nextPosition;
    private boolean closed;

    public GcsInputStream(GcsLocation location, Blob blob, int readBlockSizeBytes, OptionalLong predeclaredLength) throws IOException {
        this.location = Objects.requireNonNull(location, "location is null");
        this.blob = Objects.requireNonNull(blob, "blob is null");
        this.readBlockSizeBytes = readBlockSizeBytes;
        this.predeclaredLength = Objects.requireNonNull(predeclaredLength, "predeclaredLength is null");
        this.fileSize = predeclaredLength.orElse(blob.getSize());
        this.openStream();
    }

    public int available() throws IOException {
        this.ensureOpen();
        this.repositionStream();
        return Ints.saturatedCast((long)(this.fileSize - this.currentPosition));
    }

    public long getPosition() throws IOException {
        return this.nextPosition;
    }

    public void seek(long newPosition) throws IOException {
        this.ensureOpen();
        if (newPosition < 0L) {
            throw new IOException("Negative seek offset");
        }
        if (newPosition > this.fileSize) {
            throw new IOException("Cannot seek to %s. File size is %s: %s".formatted(newPosition, this.fileSize, this.location));
        }
        this.nextPosition = newPosition;
    }

    public int read() throws IOException {
        this.ensureOpen();
        this.repositionStream();
        try {
            this.readBuffer.position(0);
            int bytesRead = this.readChannel.read(this.readBuffer);
            if (bytesRead == 1) {
                ++this.currentPosition;
                ++this.nextPosition;
                return this.readBuffer.get(0) & 0xFF;
            }
            return -1;
        }
        catch (IOException e) {
            throw new IOException("Error reading file: " + this.location, e);
        }
    }

    public int read(byte[] buffer, int offset, int length) throws IOException {
        Objects.checkFromIndexSize(offset, length, buffer.length);
        this.ensureOpen();
        this.repositionStream();
        ByteBuffer wrappedBuffer = ByteBuffer.wrap(buffer, offset, length);
        try {
            int readSize = this.readChannel.read(wrappedBuffer);
            if (readSize > 0) {
                this.currentPosition += (long)readSize;
                this.nextPosition += (long)readSize;
            }
            return readSize;
        }
        catch (IOException e) {
            throw new IOException("Error reading file: " + this.location, e);
        }
    }

    public long skip(long n) throws IOException {
        this.ensureOpen();
        long skipSize = Longs.constrainToRange((long)n, (long)0L, (long)(this.fileSize - this.nextPosition));
        this.nextPosition += skipSize;
        return skipSize;
    }

    public void skipNBytes(long n) throws IOException {
        this.ensureOpen();
        if (n <= 0L) {
            return;
        }
        long position = this.nextPosition + n;
        if (position < 0L || position > this.fileSize) {
            throw new EOFException("Unable to skip %s bytes (position=%s, fileSize=%s): %s".formatted(n, this.nextPosition, this.fileSize, this.location));
        }
        this.nextPosition = position;
    }

    public void close() throws IOException {
        if (!this.closed) {
            this.closed = true;
            try {
                this.readChannel.close();
            }
            catch (RuntimeException e) {
                throw GcsUtils.handleGcsException(e, "closing file", this.location);
            }
        }
    }

    private void ensureOpen() throws IOException {
        if (this.closed) {
            throw new IOException("Output stream closed: " + this.location);
        }
    }

    private void openStream() throws IOException {
        try {
            this.readChannel = GcsUtils.getReadChannel(this.blob, this.location, 0L, this.readBlockSizeBytes, this.predeclaredLength);
        }
        catch (RuntimeException e) {
            throw GcsUtils.handleGcsException(e, "read file", this.location);
        }
    }

    private void repositionStream() throws IOException {
        if (this.nextPosition == this.currentPosition) {
            return;
        }
        this.readChannel.seek(this.nextPosition);
        this.currentPosition = this.nextPosition;
    }
}

