/*
 * Decompiled with CFR 0.152.
 */
package restx.stats;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.github.kevinsawicki.http.HttpRequest;
import com.google.common.base.Charsets;
import com.google.common.base.Optional;
import com.google.common.base.Stopwatch;
import com.google.common.hash.Hashing;
import com.google.common.io.FileWriteMode;
import com.google.common.io.Files;
import java.io.File;
import java.nio.charset.Charset;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import javax.inject.Named;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import restx.AppSettings;
import restx.RestxRequest;
import restx.RestxResponse;
import restx.common.UUIDGenerator;
import restx.common.Version;
import restx.factory.AutoStartable;
import restx.factory.Component;
import restx.stats.RestxStats;
import restx.stats.RestxStatsSettings;

@Component
public class RestxStatsCollector
implements AutoStartable,
AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(RestxStatsCollector.class);
    private final UUIDGenerator uuidGenerator;
    private final ObjectMapper objectMapper;
    private final RestxStats stats;
    private final long startupTime;
    private final long previousTotalUptime;
    private final boolean storageEnabled;
    private final long storagePeriod;
    private final File storageStatsDir;
    private final boolean shareEnabled;
    private final long sharePeriod;
    private final String shareURL;
    private volatile long lastTouchTime;
    private volatile long lastStorageTime;
    private volatile long lastShareTime;

    public RestxStatsCollector(@Named(value="app.name") Optional<String> appName, @Named(value="restx.server.type") Optional<String> serverType, @Named(value="restx.server.port") Optional<String> serverPort, AppSettings appSettings, RestxStatsSettings statsSettings, ObjectMapper objectMapper) {
        this.objectMapper = objectMapper;
        this.uuidGenerator = new UUIDGenerator.DefaultUUIDGenerator();
        this.storageEnabled = statsSettings.storageEnable();
        this.storageStatsDir = new File((String)statsSettings.storageDir().or((Object)(System.getProperty("user.home") + "/.restx/stats")));
        this.storagePeriod = statsSettings.storagePeriod();
        this.shareEnabled = statsSettings.shareEnable();
        this.shareURL = statsSettings.shareURL();
        this.sharePeriod = statsSettings.sharePeriod();
        if (this.storageEnabled && !this.storageStatsDir.exists()) {
            this.storageStatsDir.mkdirs();
        }
        this.stats = this.loadPreviousStatsIfAvailable(new RestxStats().setAppNameHash(Hashing.md5().hashString((CharSequence)appName.or((Object)"DEFAULT"), Charsets.UTF_8).toString()).setMachineId(this.getMachineId()).setRestxMode(appSettings.mode()).setPort(Integer.parseInt((String)serverPort.or((Object)"0"))));
        this.startupTime = System.currentTimeMillis();
        this.previousTotalUptime = this.stats.getTotalUptime();
        this.stats.setServer((String)serverType.or((Object)"unknown")).setOs(this.getOs()).setJava(this.getJava()).setRestxVersion(Version.getVersion((String)"io.restx", (String)"restx-core")).setDataAccessInfo(this.guessDataAccessInfo());
        this.fillPerHttpMethodRequestStats();
        this.updateHeapSize();
        this.touch();
    }

    public RestxStats getStats() {
        this.updateUptime();
        this.updateHeapSize();
        this.touch();
        return this.stats;
    }

    public void start() {
        if (this.shareEnabled) {
            logger.info("collecting and sharing stats enabled - see http://restx.io/stats.html for details.");
        }
        logger.debug("stats collection started - current stats {}", (Object)this.stats);
    }

    final void notifyRequest(RestxRequest req, RestxResponse resp, Stopwatch stop) {
        RestxStats.RequestStats requestStats = this.stats.getRequestStats().get(req.getHttpMethod());
        if (requestStats != null) {
            long maxDuration;
            long minDuration;
            requestStats.getRequestsCount().incrementAndGet();
            long duration = stop.elapsed(TimeUnit.MICROSECONDS);
            requestStats.getTotalDuration().addAndGet(duration);
            while ((minDuration = requestStats.getMinDuration().get()) > duration && !requestStats.getMinDuration().compareAndSet(minDuration, duration)) {
            }
            while ((maxDuration = requestStats.getMaxDuration().get()) < duration && !requestStats.getMaxDuration().compareAndSet(maxDuration, duration)) {
            }
        }
        this.touch();
    }

    private void touch() {
        long now = System.currentTimeMillis();
        if (now - this.lastTouchTime > 100L) {
            this.stats.setTimestamp(new DateTime(now));
            this.lastTouchTime = now;
            this.maybeStoreStats(now);
            this.maybeShareStats(now);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void maybeStoreStats(long now) {
        if (this.storageEnabled && now - this.lastStorageTime > this.storagePeriod) {
            boolean shouldUpdate = false;
            RestxStatsCollector restxStatsCollector = this;
            synchronized (restxStatsCollector) {
                if (now - this.lastStorageTime > this.storagePeriod) {
                    shouldUpdate = true;
                    this.lastStorageTime = now;
                }
            }
            if (shouldUpdate) {
                this.updateHeapSize();
                this.updateUptime();
                this.storeStats();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void maybeShareStats(long now) {
        if (this.shareEnabled && now - this.lastShareTime > this.sharePeriod) {
            boolean shouldUpdate = false;
            RestxStatsCollector restxStatsCollector = this;
            synchronized (restxStatsCollector) {
                if (now - this.lastShareTime > this.sharePeriod) {
                    shouldUpdate = true;
                    this.lastShareTime = now;
                }
            }
            if (shouldUpdate) {
                this.updateHeapSize();
                this.updateUptime();
                this.shareStats();
            }
        }
    }

    private void updateUptime() {
        long currentUptime = System.currentTimeMillis() - this.startupTime;
        this.stats.setCurrentUptime(currentUptime);
        this.stats.setTotalUptime(this.previousTotalUptime + currentUptime);
    }

    private void updateHeapSize() {
        this.stats.setHeapSize(Runtime.getRuntime().totalMemory());
    }

    private synchronized void storeStats() {
        File statsFile = this.getStatsFile(this.stats.getStatsId());
        try {
            this.objectMapper.writer().writeValue(statsFile, (Object)this.stats);
        }
        catch (Exception e) {
            logger.info("saving stats to {} failed. Exception: {}", (Object)statsFile, (Object)e.getMessage());
        }
    }

    private void shareStats() {
        try {
            int code = HttpRequest.post((CharSequence)this.shareURL).connectTimeout(5000).readTimeout(5000).send(this.objectMapper.writer().writeValueAsString((Object)this.stats).getBytes(Charsets.UTF_8)).code();
            if (code >= 400) {
                logger.info("sharing stats on {} failed. Response code: {}", (Object)this.shareURL, (Object)code);
            }
        }
        catch (Exception e) {
            logger.info("sharing stats on {} failed. Exception: {}", (Object)this.shareURL, (Object)e.getMessage());
        }
    }

    private RestxStats loadPreviousStatsIfAvailable(RestxStats stats) {
        if (!this.storageEnabled) {
            return stats;
        }
        try {
            File statsFile = this.getStatsFile(stats.getStatsId());
            if (statsFile.exists()) {
                stats = (RestxStats)this.objectMapper.reader(RestxStats.class).readValue(statsFile);
                this.lastStorageTime = statsFile.lastModified();
            }
            return stats;
        }
        catch (Exception e) {
            return stats;
        }
    }

    private String guessDataAccessInfo() {
        return "unknown";
    }

    private String getOs() {
        return System.getProperty("os.name") + ", " + System.getProperty("os.version") + ", " + System.getProperty("os.arch");
    }

    private String getJava() {
        return "VM: " + System.getProperty("java.vm.name") + ", " + System.getProperty("java.vm.version") + "; Version: " + System.getProperty("java.version") + ", " + System.getProperty("java.runtime.version");
    }

    private synchronized String getMachineId() {
        if (!this.storageEnabled) {
            return this.uuidGenerator.doGenerate();
        }
        File machineIdFile = new File(this.storageStatsDir, "machineId");
        if (machineIdFile.exists()) {
            try {
                return Files.asCharSource((File)machineIdFile, (Charset)Charsets.UTF_8).read();
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
        String machineId = this.uuidGenerator.doGenerate();
        try {
            Files.asCharSink((File)machineIdFile, (Charset)Charsets.UTF_8, (FileWriteMode[])new FileWriteMode[0]).write((CharSequence)machineId);
        }
        catch (Exception exception) {
            // empty catch block
        }
        return machineId;
    }

    private void fillPerHttpMethodRequestStats() {
        Map<String, RestxStats.RequestStats> requestStats = this.stats.getRequestStats();
        for (String httpMethod : new String[]{"GET", "POST", "PUT", "DELETE", "HEAD", "OPTIONS", "TRACE", "CONNECT"}) {
            if (requestStats.containsKey(httpMethod)) continue;
            requestStats.put(httpMethod, new RestxStats.RequestStats().setHttpMethod(httpMethod));
        }
    }

    private File getStatsFile(String statsId) {
        return new File(this.storageStatsDir, "restx-stats-" + statsId + ".json");
    }

    @Override
    public void close() throws Exception {
        if (this.storageEnabled) {
            this.stats.setTimestamp(DateTime.now());
            this.updateHeapSize();
            this.updateUptime();
            this.storeStats();
        }
        if (this.shareEnabled) {
            this.stats.setTimestamp(DateTime.now());
            this.updateHeapSize();
            this.updateUptime();
            this.shareStats();
        }
    }
}

