package io.quarkus.kubernetes.spi;

import java.util.Collections;
import java.util.Map;

import io.quarkus.builder.item.MultiBuildItem;

/**
 * Produce this build item to request the Kubernetes extension to generate
 * a Kubernetes {@code RoleBinding} resource. The configuration here is limited;
 * in particular, you can't specify subjects of the role binding. The role will always
 * be bound to the application's service account.
 * <p>
 * Note that this can't be used to generate a {@code ClusterRoleBinding}.
 */
public final class KubernetesRoleBindingBuildItem extends MultiBuildItem {
    /**
     * Name of the generated {@code RoleBinding} resource.
     * Can be {@code null}, in which case the resource name is autogenerated.
     */
    private final String name;
    /**
     * RoleRef configuration.
     */
    private final RoleRef roleRef;
    /**
     * The target manifest that should include this role.
     */
    private final String target;
    /**
     * The target subjects.
     */
    private final Subject[] subjects;

    /**
     * The labels of the cluster role resource.
     */
    private final Map<String, String> labels;

    public KubernetesRoleBindingBuildItem(String role, boolean clusterWide) {
        this(null, role, clusterWide, null);
    }

    public KubernetesRoleBindingBuildItem(String name, String role, boolean clusterWide) {
        this(name, role, clusterWide, null);
    }

    public KubernetesRoleBindingBuildItem(String name, String role, boolean clusterWide, String target) {
        this(name, target, Collections.emptyMap(),
                new RoleRef(role, clusterWide),
                new Subject("", "ServiceAccount", name, null));
    }

    public KubernetesRoleBindingBuildItem(String name, String target, Map<String, String> labels, RoleRef roleRef,
            Subject... subjects) {
        this.name = name;
        this.target = target;
        this.labels = labels;
        this.roleRef = roleRef;
        this.subjects = subjects;
    }

    public String getName() {
        return this.name;
    }

    public String getTarget() {
        return target;
    }

    public Map<String, String> getLabels() {
        return labels;
    }

    public RoleRef getRoleRef() {
        return roleRef;
    }

    public Subject[] getSubjects() {
        return subjects;
    }
}
