/*
 * Copyright 2024-2025 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.github.a2ap.core.model;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;

import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * Represents an artifact produced by a task during agent execution.
 * 
 * Artifacts are structured outputs generated by agents as they process tasks.
 * They represent discrete pieces of work or content that can be created, modified,
 * or referenced throughout the task lifecycle. Artifacts provide a way to track
 * and manage the tangible outputs of agent processing.
 * 
 * Key characteristics:
 * - Each artifact has a unique identifier within the task context
 * - Artifacts can contain multiple parts of different types (text, files, data)
 * - They can be incrementally updated through TaskArtifactUpdateEvent objects
 * - Metadata provides additional context for processing or display purposes
 * 
 * Common artifact types include:
 * - Generated documents or reports
 * - Code files or scripts
 * - Data analysis results
 * - Configuration files
 * - Media content
 * 
 * Artifacts implement the TaskUpdate interface, allowing them to be used in
 * task update operations and event streaming scenarios.
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
public class Artifact implements TaskUpdate {

    /**
     * artifact id
     */
    @JsonProperty("artifactId")
    private String artifactId;

    /**
     * The name of the artifact.
     */
    @JsonProperty("name")
    private String name;

    /**
     * An optional description of the artifact.
     */
    @JsonProperty("description")
    private String description;

    /**
     * The parts that make up the artifact content. Required field.
     */
    @JsonProperty("parts")
    private List<Part> parts;

    /**
     * Optional metadata associated with the artifact.
     */
    @JsonProperty("metadata")
    private Map<String, Object> metadata;

    public Artifact() {
    }

    public Artifact(String artifactId, String name, String description, List<Part> parts,
                    Map<String, Object> metadata) {
        this.artifactId = artifactId;
        this.name = name;
        this.description = description;
        this.parts = parts;
        this.metadata = metadata;
    }

    public static Builder builder() {
        return new Builder();
    }

    public String getArtifactId() {
        return artifactId;
    }

    public void setArtifactId(String artifactId) {
        this.artifactId = artifactId;
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public String getDescription() {
        return description;
    }

    public void setDescription(String description) {
        this.description = description;
    }

    public List<Part> getParts() {
        return parts;
    }

    public void setParts(List<Part> parts) {
        this.parts = parts;
    }

    public Map<String, Object> getMetadata() {
        return metadata;
    }

    public void setMetadata(Map<String, Object> metadata) {
        this.metadata = metadata;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o)
            return true;
        if (o == null || getClass() != o.getClass())
            return false;
        Artifact artifact = (Artifact) o;
        return Objects.equals(artifactId, artifact.artifactId) && Objects.equals(name, artifact.name)
                && Objects.equals(description, artifact.description) && Objects.equals(parts, artifact.parts)
                && Objects.equals(metadata, artifact.metadata);
    }

    @Override
    public int hashCode() {
        return Objects.hash(artifactId, name, description, parts, metadata);
    }

    @Override
    public String toString() {
        return "Artifact{" + "artifactId='" + artifactId + '\'' + ", name='" + name + '\'' + ", description='"
                + description + '\'' + ", parts=" + parts + ", metadata=" + metadata + '}';
    }

    /**
     * Builder for creating instances of {@link Artifact}.
     */
    public static class Builder {

        private String artifactId;

        private String name;

        private String description;

        private List<Part> parts;

        private Map<String, Object> metadata;

        private Builder() {
        }

        public Builder artifactId(String artifactId) {
            this.artifactId = artifactId;
            return this;
        }

        public Builder name(String name) {
            this.name = name;
            return this;
        }

        public Builder description(String description) {
            this.description = description;
            return this;
        }

        public Builder parts(List<Part> parts) {
            this.parts = parts;
            return this;
        }

        public Builder metadata(Map<String, Object> metadata) {
            this.metadata = metadata;
            return this;
        }

        public Artifact build() {
            return new Artifact(artifactId, name, description, parts, metadata);
        }

    }

}
