/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.ksql.util;

import io.confluent.ksql.schema.ksql.types.SqlBaseType;
import io.confluent.ksql.schema.ksql.types.SqlDecimal;
import io.confluent.ksql.schema.ksql.types.SqlType;
import io.confluent.ksql.schema.ksql.types.SqlTypes;
import io.confluent.ksql.util.KsqlException;
import io.confluent.ksql.util.KsqlPreconditions;
import java.math.BigDecimal;
import java.math.MathContext;
import java.math.RoundingMode;
import org.apache.kafka.connect.data.Decimal;
import org.apache.kafka.connect.data.Schema;
import org.apache.kafka.connect.data.SchemaBuilder;

public final class DecimalUtil {
    public static final String PRECISION_FIELD = "connect.decimal.precision";
    public static final int PRECISION_DEFAULT = 64;

    private DecimalUtil() {
    }

    public static boolean isDecimal(Schema schema) {
        return schema.type() == Schema.Type.BYTES && "org.apache.kafka.connect.data.Decimal".equals(schema.name());
    }

    public static void requireDecimal(Schema schema) {
        KsqlPreconditions.checkArgument(DecimalUtil.isDecimal(schema), String.format("Expected schema of type DECIMAL but got a schema of type %s and name %s", schema.type(), schema.name()));
    }

    public static SchemaBuilder builder(Schema schema) {
        DecimalUtil.requireDecimal(schema);
        return DecimalUtil.builder(DecimalUtil.precision(schema), DecimalUtil.scale(schema));
    }

    public static SchemaBuilder builder(int precision, int scale) {
        SqlDecimal.validateParameters((int)precision, (int)scale);
        return Decimal.builder((int)scale).optional().parameter(PRECISION_FIELD, Integer.toString(precision));
    }

    public static int scale(Schema schema) {
        DecimalUtil.requireDecimal(schema);
        String scaleString = (String)schema.parameters().get("scale");
        if (scaleString == null) {
            throw new KsqlException("Invalid Decimal schema: scale parameter not found.");
        }
        try {
            return Integer.parseInt(scaleString);
        }
        catch (NumberFormatException e) {
            throw new KsqlException("Invalid scale parameter found in Decimal schema: ", e);
        }
    }

    public static int precision(Schema schema) {
        DecimalUtil.requireDecimal(schema);
        String precisionString = (String)schema.parameters().get(PRECISION_FIELD);
        if (precisionString == null) {
            return 64;
        }
        try {
            return Integer.parseInt(precisionString);
        }
        catch (NumberFormatException e) {
            throw new KsqlException("Invalid precision parameter found in Decimal schema: ", e);
        }
    }

    public static BigDecimal ensureFit(BigDecimal value, Schema schema) {
        return DecimalUtil.ensureFit(value, DecimalUtil.precision(schema), DecimalUtil.scale(schema));
    }

    public static BigDecimal ensureFit(BigDecimal value, SqlDecimal schema) {
        return DecimalUtil.ensureFit(value, schema.getPrecision(), schema.getScale());
    }

    public static BigDecimal ensureFit(BigDecimal value, int precision, int scale) {
        if (value == null) {
            return null;
        }
        SqlDecimal.validateParameters((int)precision, (int)scale);
        DecimalUtil.ensureMax(value, precision, scale);
        try {
            return value.setScale(scale, RoundingMode.UNNECESSARY);
        }
        catch (ArithmeticException e) {
            throw new KsqlException(String.format("Cannot fit decimal '%s' into DECIMAL(%d, %d) without rounding. (Requires %d,%d)", value.toPlainString(), precision, scale, value.precision(), value.scale()));
        }
    }

    public static SqlDecimal toSqlDecimal(SqlType type) {
        switch (type.baseType()) {
            case DECIMAL: {
                return (SqlDecimal)type;
            }
            case INTEGER: {
                return SqlTypes.INT_UPCAST_TO_DECIMAL;
            }
            case BIGINT: {
                return SqlTypes.BIGINT_UPCAST_TO_DECIMAL;
            }
        }
        throw new KsqlException("Cannot convert " + type.baseType() + " to " + SqlBaseType.DECIMAL + ".");
    }

    public static boolean canImplicitlyCast(SqlDecimal s1, SqlDecimal s2) {
        return s1.getScale() <= s2.getScale() && s1.getPrecision() - s1.getScale() <= s2.getPrecision() - s2.getScale();
    }

    public static BigDecimal cast(Integer value, int precision, int scale) {
        if (value == null) {
            return null;
        }
        return DecimalUtil.cast(value.longValue(), precision, scale);
    }

    public static BigDecimal cast(Long value, int precision, int scale) {
        if (value == null) {
            return null;
        }
        return DecimalUtil.cast((long)value, precision, scale);
    }

    public static BigDecimal cast(long value, int precision, int scale) {
        SqlDecimal.validateParameters((int)precision, (int)scale);
        BigDecimal decimal = new BigDecimal(value, new MathContext(precision));
        DecimalUtil.ensureMax(decimal, precision, scale);
        return decimal.setScale(scale, RoundingMode.UNNECESSARY);
    }

    public static BigDecimal cast(Double value, int precision, int scale) {
        if (value == null) {
            return null;
        }
        return DecimalUtil.cast((double)value, precision, scale);
    }

    public static BigDecimal cast(double value, int precision, int scale) {
        SqlDecimal.validateParameters((int)precision, (int)scale);
        BigDecimal decimal = new BigDecimal(value, new MathContext(precision));
        DecimalUtil.ensureMax(decimal, precision, scale);
        return decimal.setScale(scale, RoundingMode.HALF_UP);
    }

    public static BigDecimal cast(BigDecimal value, int precision, int scale) {
        if (value == null) {
            return null;
        }
        if (precision == value.precision() && scale == value.scale()) {
            return value;
        }
        SqlDecimal.validateParameters((int)precision, (int)scale);
        DecimalUtil.ensureMax(value, precision, scale);
        return value.setScale(scale, RoundingMode.HALF_UP);
    }

    public static BigDecimal cast(String value, int precision, int scale) {
        if (value == null) {
            return null;
        }
        SqlDecimal.validateParameters((int)precision, (int)scale);
        BigDecimal decimal = new BigDecimal(value.trim());
        DecimalUtil.ensureMax(decimal, precision, scale);
        return decimal.setScale(scale, RoundingMode.HALF_UP);
    }

    private static void ensureMax(BigDecimal value, int precision, int scale) {
        int digits = precision - scale;
        BigDecimal maxValue = BigDecimal.valueOf(Math.pow(10.0, digits));
        if (maxValue.compareTo(value.abs()) < 1) {
            throw new ArithmeticException(String.format("Numeric field overflow: A field with precision %d and scale %d must round to an absolute value less than 10^%d. Got %s", precision, scale, digits, value.toPlainString()));
        }
    }

    public static SqlType fromValue(BigDecimal value) {
        BigDecimal decimal;
        BigDecimal bigDecimal = decimal = value.scale() < 0 ? value.setScale(0, 7) : value;
        if (decimal.compareTo(BigDecimal.ZERO) == 0) {
            return SqlTypes.decimal((int)(decimal.scale() + 1), (int)decimal.scale());
        }
        return SqlTypes.decimal((int)decimal.precision(), (int)Math.max(decimal.scale(), 0));
    }

    public static SqlDecimal widen(SqlType t0, SqlType t1) {
        SqlDecimal lDecimal = DecimalUtil.toSqlDecimal(t0);
        SqlDecimal rDecimal = DecimalUtil.toSqlDecimal(t1);
        int wholePrecision = Math.max(lDecimal.getPrecision() - lDecimal.getScale(), rDecimal.getPrecision() - rDecimal.getScale());
        int scale = Math.max(lDecimal.getScale(), rDecimal.getScale());
        return SqlTypes.decimal((int)(wholePrecision + scale), (int)scale);
    }
}

