/*
 * Decompiled with CFR 0.152.
 */
package io.agroal.pool;

import io.agroal.api.AgroalDataSource;
import io.agroal.api.AgroalDataSourceListener;
import io.agroal.api.configuration.AgroalConnectionPoolConfiguration;
import io.agroal.api.transaction.TransactionAware;
import io.agroal.api.transaction.TransactionIntegration;
import io.agroal.pool.ConnectionFactory;
import io.agroal.pool.ConnectionHandler;
import io.agroal.pool.DefaultMetricsRepository;
import io.agroal.pool.MetricsRepository;
import io.agroal.pool.Pool;
import io.agroal.pool.util.AgroalSynchronizer;
import io.agroal.pool.util.ListenerHelper;
import io.agroal.pool.util.PriorityScheduledExecutor;
import io.agroal.pool.util.StampedCopyOnWriteArrayList;
import io.agroal.pool.util.UncheckedArrayList;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.LongAccumulator;
import java.util.concurrent.atomic.LongAdder;

public final class ConnectionPool
implements Pool {
    private final AgroalConnectionPoolConfiguration configuration;
    private final AgroalDataSourceListener[] listeners;
    private final StampedCopyOnWriteArrayList<ConnectionHandler> allConnections;
    private final AgroalSynchronizer synchronizer;
    private final ConnectionFactory connectionFactory;
    private final PriorityScheduledExecutor housekeepingExecutor;
    private final TransactionIntegration transactionIntegration;
    private final boolean idleValidationEnabled;
    private final boolean leakEnabled;
    private final boolean validationEnabled;
    private final boolean reapEnabled;
    private final LongAccumulator maxUsed = new LongAccumulator(Math::max, Long.MIN_VALUE);
    private final LongAdder activeCount = new LongAdder();
    private MetricsRepository metricsRepository;
    private ThreadLocal<List<ConnectionHandler>> localCache;

    public ConnectionPool(AgroalConnectionPoolConfiguration configuration, AgroalDataSourceListener ... listeners) {
        this.configuration = configuration;
        this.listeners = listeners;
        this.allConnections = new StampedCopyOnWriteArrayList<ConnectionHandler>(ConnectionHandler.class);
        this.localCache = new ConnectionHandlerThreadLocal();
        this.synchronizer = new AgroalSynchronizer();
        this.connectionFactory = new ConnectionFactory(configuration.connectionFactoryConfiguration(), listeners);
        this.housekeepingExecutor = new PriorityScheduledExecutor(1, "Agroal_" + System.identityHashCode(this));
        this.transactionIntegration = configuration.transactionIntegration();
        this.idleValidationEnabled = !configuration.idleValidationTimeout().isZero();
        this.leakEnabled = !configuration.leakTimeout().isZero();
        this.validationEnabled = !configuration.validationTimeout().isZero();
        this.reapEnabled = !configuration.reapTimeout().isZero();
    }

    @Override
    public void init() {
        if (this.leakEnabled) {
            this.housekeepingExecutor.schedule(new LeakTask(), this.configuration.leakTimeout().toNanos(), TimeUnit.NANOSECONDS);
        }
        if (this.validationEnabled) {
            this.housekeepingExecutor.schedule(new ValidationTask(), this.configuration.validationTimeout().toNanos(), TimeUnit.NANOSECONDS);
        }
        if (this.reapEnabled) {
            this.housekeepingExecutor.schedule(new ReapTask(), this.configuration.reapTimeout().toNanos(), TimeUnit.NANOSECONDS);
        }
        this.transactionIntegration.addResourceRecoveryFactory((TransactionIntegration.ResourceRecoveryFactory)this.connectionFactory);
        if (this.configuration.initialSize() < this.configuration.minSize()) {
            ListenerHelper.fireOnInfo(this.listeners, "Initial size smaller than min. Connections will be created when necessary");
        } else if (this.configuration.initialSize() > this.configuration.maxSize()) {
            ListenerHelper.fireOnInfo(this.listeners, "Initial size bigger than max. Connections will be destroyed as soon as they return to the pool");
        }
        for (int n = this.configuration.initialSize(); n > 0; --n) {
            this.housekeepingExecutor.executeNow(new CreateConnectionTask().initial());
        }
    }

    @Override
    public AgroalConnectionPoolConfiguration getConfiguration() {
        return this.configuration;
    }

    @Override
    public AgroalDataSourceListener[] getListeners() {
        return this.listeners;
    }

    @Override
    public void flushPool(AgroalDataSource.FlushMode mode) {
        this.housekeepingExecutor.execute(new FlushTask(mode));
    }

    @Override
    public void close() {
        this.transactionIntegration.removeResourceRecoveryFactory((TransactionIntegration.ResourceRecoveryFactory)this.connectionFactory);
        for (Runnable task : this.housekeepingExecutor.shutdownNow()) {
            if (!(task instanceof DestroyConnectionTask)) continue;
            task.run();
        }
        for (ConnectionHandler handler : this.allConnections) {
            handler.setState(ConnectionHandler.State.FLUSH);
            new DestroyConnectionTask(handler).run();
        }
        this.allConnections.clear();
        this.activeCount.reset();
        this.synchronizer.release(this.synchronizer.getQueueLength());
    }

    @Override
    public Connection getConnection() throws SQLException {
        ListenerHelper.fireBeforeConnectionAcquire(this.listeners);
        long metricsStamp = this.metricsRepository.beforeConnectionAcquire();
        if (this.housekeepingExecutor.isShutdown()) {
            throw new SQLException("This pool is closed and does not handle any more connections!");
        }
        ConnectionHandler checkedOutHandler = this.handlerFromTransaction();
        if (checkedOutHandler == null) {
            do {
                if ((checkedOutHandler = this.handlerFromLocalCache()) != null) continue;
                checkedOutHandler = this.handlerFromSharedCache();
            } while (this.idleValidationEnabled && !this.idleValidation(checkedOutHandler));
            this.activeCount.increment();
        }
        this.metricsRepository.afterConnectionAcquire(metricsStamp);
        ListenerHelper.fireOnConnectionAcquired(this.listeners, checkedOutHandler);
        if (this.leakEnabled || this.reapEnabled) {
            checkedOutHandler.setLastAccess(System.nanoTime());
        }
        if (this.leakEnabled) {
            if (checkedOutHandler.getHoldingThread() != null) {
                Throwable warn = new Throwable("Shared connection between threads '" + checkedOutHandler.getHoldingThread().getName() + "' and '" + Thread.currentThread().getName() + "'");
                warn.setStackTrace(checkedOutHandler.getHoldingThread().getStackTrace());
                ListenerHelper.fireOnWarning(this.listeners, warn);
            }
            checkedOutHandler.setHoldingThread(Thread.currentThread());
        }
        this.transactionIntegration.associate((TransactionAware)checkedOutHandler, checkedOutHandler.getXaResource());
        return checkedOutHandler.newConnectionWrapper();
    }

    private ConnectionHandler handlerFromTransaction() throws SQLException {
        return (ConnectionHandler)this.transactionIntegration.getTransactionAware();
    }

    private ConnectionHandler handlerFromLocalCache() {
        List<ConnectionHandler> cachedConnections = this.localCache.get();
        while (!cachedConnections.isEmpty()) {
            ConnectionHandler handler = cachedConnections.remove(cachedConnections.size() - 1);
            if (!handler.setState(ConnectionHandler.State.CHECKED_IN, ConnectionHandler.State.CHECKED_OUT)) continue;
            return handler;
        }
        return null;
    }

    private ConnectionHandler handlerFromSharedCache() throws SQLException {
        long remaining = this.configuration.acquisitionTimeout().toNanos();
        remaining = remaining > 0L ? remaining : Long.MAX_VALUE;
        try {
            while (true) {
                if (this.allConnections.size() < this.configuration.minSize()) {
                    ConnectionHandler handler = this.housekeepingExecutor.executeNow(new CreateConnectionTask()).get();
                    if (handler == null || !handler.setState(ConnectionHandler.State.CHECKED_IN, ConnectionHandler.State.CHECKED_OUT)) continue;
                    return handler;
                }
                for (ConnectionHandler handler : this.allConnections) {
                    if (!handler.setState(ConnectionHandler.State.CHECKED_IN, ConnectionHandler.State.CHECKED_OUT)) continue;
                    return handler;
                }
                if (this.allConnections.size() < this.configuration.maxSize()) {
                    ConnectionHandler handler = this.housekeepingExecutor.executeNow(new CreateConnectionTask()).get();
                    if (handler == null || !handler.setState(ConnectionHandler.State.CHECKED_IN, ConnectionHandler.State.CHECKED_OUT)) continue;
                    return handler;
                }
                long synchronizationStamp = this.synchronizer.getStamp();
                long start = System.nanoTime();
                if (remaining < 0L || !this.synchronizer.tryAcquireNanos(synchronizationStamp, remaining)) {
                    throw new SQLException("Sorry, acquisition timeout!");
                }
                remaining -= System.nanoTime() - start;
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new SQLException("Interrupted while acquiring");
        }
        catch (ExecutionException e) {
            try {
                throw e.getCause();
            }
            catch (Error | RuntimeException | SQLException e2) {
                throw e2;
            }
            catch (Throwable t) {
                throw new SQLException("Exception while creating new connection", t);
            }
        }
        catch (CancellationException | RejectedExecutionException e) {
            throw new SQLException("Can't create new connection as the pool is shutting down", e);
        }
    }

    private boolean idleValidation(ConnectionHandler handler) {
        if (System.nanoTime() - handler.getLastAccess() < this.configuration.idleValidationTimeout().toNanos()) {
            return true;
        }
        ListenerHelper.fireBeforeConnectionValidation(this.listeners, handler);
        if (handler.setState(ConnectionHandler.State.CHECKED_OUT, ConnectionHandler.State.VALIDATION)) {
            if (this.configuration.connectionValidator().isValid(handler.getConnection())) {
                handler.setState(ConnectionHandler.State.CHECKED_OUT);
                ListenerHelper.fireOnConnectionValid(this.listeners, handler);
                return true;
            }
            handler.setState(ConnectionHandler.State.FLUSH);
            this.allConnections.remove(handler);
            this.metricsRepository.afterConnectionInvalid();
            ListenerHelper.fireOnConnectionInvalid(this.listeners, handler);
            this.housekeepingExecutor.execute(new DestroyConnectionTask(handler));
        }
        return false;
    }

    @Override
    public void returnConnectionHandler(ConnectionHandler handler) throws SQLException {
        ListenerHelper.fireBeforeConnectionReturn(this.listeners, handler);
        if (this.leakEnabled) {
            handler.setHoldingThread(null);
        }
        if (this.idleValidationEnabled || this.reapEnabled) {
            handler.setLastAccess(System.nanoTime());
        }
        if (this.transactionIntegration.disassociate((TransactionAware)handler)) {
            this.activeCount.decrement();
            int currentSize = this.allConnections.size();
            if (currentSize > this.configuration.maxSize() || this.configuration.flushOnClose() && currentSize > this.configuration.minSize()) {
                handler.setState(ConnectionHandler.State.FLUSH);
                this.allConnections.remove(handler);
                this.synchronizer.releaseConditional();
                this.housekeepingExecutor.execute(new FlushTask(AgroalDataSource.FlushMode.ALL, handler));
                return;
            }
            handler.resetConnection();
            this.localCache.get().add(handler);
            if (handler.setState(ConnectionHandler.State.CHECKED_OUT, ConnectionHandler.State.CHECKED_IN)) {
                this.synchronizer.releaseConditional();
                this.metricsRepository.afterConnectionReturn();
                ListenerHelper.fireOnConnectionReturn(this.listeners, handler);
            } else {
                this.allConnections.remove(handler);
                this.housekeepingExecutor.execute(new FlushTask(AgroalDataSource.FlushMode.ALL, handler));
                this.housekeepingExecutor.execute(new FillTask());
            }
        }
    }

    public void onMetricsEnabled(boolean metricsEnabled) {
        this.setMetricsRepository(metricsEnabled ? new DefaultMetricsRepository(this) : new MetricsRepository.EmptyMetricsRepository());
    }

    @Override
    public MetricsRepository getMetrics() {
        return this.metricsRepository;
    }

    public void setMetricsRepository(MetricsRepository metricsRepository) {
        this.metricsRepository = metricsRepository;
    }

    @Override
    public long activeCount() {
        return this.activeCount.sum();
    }

    @Override
    public long availableCount() {
        return (long)this.allConnections.size() - this.activeCount.sum();
    }

    @Override
    public long maxUsedCount() {
        return this.maxUsed.get();
    }

    @Override
    public void resetMaxUsedCount() {
        this.maxUsed.reset();
    }

    @Override
    public long awaitingCount() {
        return this.synchronizer.getQueueLength();
    }

    private final class DestroyConnectionTask
    implements Runnable {
        private final ConnectionHandler handler;

        public DestroyConnectionTask(ConnectionHandler handler) {
            this.handler = handler;
        }

        @Override
        public void run() {
            ListenerHelper.fireBeforeConnectionDestroy(ConnectionPool.this.listeners, this.handler);
            try {
                this.handler.closeConnection();
            }
            catch (SQLException e) {
                ListenerHelper.fireOnWarning(ConnectionPool.this.listeners, e);
            }
            this.handler.setState(ConnectionHandler.State.DESTROYED);
            ConnectionPool.this.metricsRepository.afterConnectionDestroy();
            ListenerHelper.fireOnConnectionDestroy(ConnectionPool.this.listeners, this.handler);
        }
    }

    private final class ReapTask
    implements Runnable {
        private ReapTask() {
        }

        @Override
        public void run() {
            ConnectionPool.this.housekeepingExecutor.schedule(this, ConnectionPool.this.configuration.reapTimeout().toNanos(), TimeUnit.NANOSECONDS);
            ConnectionPool.this.localCache = new ConnectionHandlerThreadLocal();
            for (ConnectionHandler handler : ConnectionPool.this.allConnections) {
                ConnectionPool.this.housekeepingExecutor.execute(new ReapConnectionTask(handler));
            }
        }

        private class ReapConnectionTask
        implements Runnable {
            private final ConnectionHandler handler;

            public ReapConnectionTask(ConnectionHandler handler) {
                this.handler = handler;
            }

            @Override
            public void run() {
                ListenerHelper.fireBeforeConnectionReap(ConnectionPool.this.listeners, this.handler);
                if (ConnectionPool.this.allConnections.size() > ConnectionPool.this.configuration.minSize() && this.handler.setState(ConnectionHandler.State.CHECKED_IN, ConnectionHandler.State.FLUSH)) {
                    if (System.nanoTime() - this.handler.getLastAccess() > ConnectionPool.this.configuration.reapTimeout().toNanos()) {
                        ConnectionPool.this.allConnections.remove(this.handler);
                        ConnectionPool.this.metricsRepository.afterConnectionReap();
                        ListenerHelper.fireOnConnectionReap(ConnectionPool.this.listeners, this.handler);
                        ConnectionPool.this.housekeepingExecutor.execute(new DestroyConnectionTask(this.handler));
                    } else {
                        this.handler.setState(ConnectionHandler.State.CHECKED_IN);
                    }
                }
            }
        }
    }

    private final class ValidationTask
    implements Runnable {
        private ValidationTask() {
        }

        @Override
        public void run() {
            ConnectionPool.this.housekeepingExecutor.schedule(this, ConnectionPool.this.configuration.validationTimeout().toNanos(), TimeUnit.NANOSECONDS);
            for (ConnectionHandler handler : ConnectionPool.this.allConnections) {
                ConnectionPool.this.housekeepingExecutor.execute(new ValidateConnectionTask(handler));
            }
        }

        private class ValidateConnectionTask
        implements Runnable {
            private final ConnectionHandler handler;

            public ValidateConnectionTask(ConnectionHandler handler) {
                this.handler = handler;
            }

            @Override
            public void run() {
                ListenerHelper.fireBeforeConnectionValidation(ConnectionPool.this.listeners, this.handler);
                if (this.handler.setState(ConnectionHandler.State.CHECKED_IN, ConnectionHandler.State.VALIDATION)) {
                    if (ConnectionPool.this.configuration.connectionValidator().isValid(this.handler.getConnection())) {
                        this.handler.setState(ConnectionHandler.State.CHECKED_IN);
                        ListenerHelper.fireOnConnectionValid(ConnectionPool.this.listeners, this.handler);
                    } else {
                        this.handler.setState(ConnectionHandler.State.FLUSH);
                        ConnectionPool.this.allConnections.remove(this.handler);
                        ConnectionPool.this.metricsRepository.afterConnectionInvalid();
                        ListenerHelper.fireOnConnectionInvalid(ConnectionPool.this.listeners, this.handler);
                        ConnectionPool.this.housekeepingExecutor.execute(new DestroyConnectionTask(this.handler));
                    }
                }
            }
        }
    }

    private final class LeakTask
    implements Runnable {
        private LeakTask() {
        }

        @Override
        public void run() {
            ConnectionPool.this.housekeepingExecutor.schedule(this, ConnectionPool.this.configuration.leakTimeout().toNanos(), TimeUnit.NANOSECONDS);
            for (ConnectionHandler handler : ConnectionPool.this.allConnections) {
                ConnectionPool.this.housekeepingExecutor.execute(new LeakConnectionTask(handler));
            }
        }

        private class LeakConnectionTask
        implements Runnable {
            private final ConnectionHandler handler;

            public LeakConnectionTask(ConnectionHandler handler) {
                this.handler = handler;
            }

            @Override
            public void run() {
                ListenerHelper.fireBeforeConnectionLeak(ConnectionPool.this.listeners, this.handler);
                Thread thread = this.handler.getHoldingThread();
                if (thread != null && System.nanoTime() - this.handler.getLastAccess() > ConnectionPool.this.configuration.leakTimeout().toNanos()) {
                    ConnectionPool.this.metricsRepository.afterLeakDetection();
                    ListenerHelper.fireOnConnectionLeak(ConnectionPool.this.listeners, this.handler);
                }
            }
        }
    }

    private final class FillTask
    implements Runnable {
        private FillTask() {
        }

        @Override
        public void run() {
            for (int n = ConnectionPool.this.configuration.minSize() - ConnectionPool.this.allConnections.size(); n > 0; --n) {
                ConnectionPool.this.housekeepingExecutor.executeNow(new CreateConnectionTask());
            }
        }
    }

    private final class FlushTask
    implements Runnable {
        private final AgroalDataSource.FlushMode mode;
        private final ConnectionHandler handler;

        public FlushTask(AgroalDataSource.FlushMode mode) {
            this.mode = mode;
            this.handler = null;
        }

        public FlushTask(AgroalDataSource.FlushMode mode, ConnectionHandler handler) {
            this.mode = mode;
            this.handler = handler;
        }

        @Override
        public void run() {
            if (this.handler != null) {
                ListenerHelper.fireBeforeConnectionFlush(ConnectionPool.this.listeners, this.handler);
                this.flush(this.mode, this.handler);
            } else {
                for (ConnectionHandler handler : ConnectionPool.this.allConnections) {
                    ListenerHelper.fireBeforeConnectionFlush(ConnectionPool.this.listeners, handler);
                    this.flush(this.mode, handler);
                }
                this.afterFlush(this.mode);
            }
        }

        private void flush(AgroalDataSource.FlushMode mode, ConnectionHandler handler) {
            switch (mode) {
                case ALL: {
                    handler.setState(ConnectionHandler.State.FLUSH);
                    this.flushHandler(handler);
                    break;
                }
                case GRACEFUL: {
                    if (handler.setState(ConnectionHandler.State.CHECKED_IN, ConnectionHandler.State.FLUSH)) {
                        this.flushHandler(handler);
                        break;
                    }
                    handler.setState(ConnectionHandler.State.FLUSH);
                    break;
                }
                case IDLE: {
                    if (ConnectionPool.this.allConnections.size() <= ConnectionPool.this.configuration.minSize() || !handler.setState(ConnectionHandler.State.CHECKED_IN, ConnectionHandler.State.FLUSH)) break;
                    this.flushHandler(handler);
                    break;
                }
                case INVALID: {
                    ListenerHelper.fireBeforeConnectionValidation(ConnectionPool.this.listeners, handler);
                    if (!handler.setState(ConnectionHandler.State.CHECKED_IN, ConnectionHandler.State.VALIDATION)) break;
                    if (ConnectionPool.this.configuration.connectionValidator().isValid(handler.getConnection())) {
                        ListenerHelper.fireOnConnectionValid(ConnectionPool.this.listeners, handler);
                        handler.setState(ConnectionHandler.State.CHECKED_IN);
                        break;
                    }
                    ListenerHelper.fireOnConnectionInvalid(ConnectionPool.this.listeners, handler);
                    this.flushHandler(handler);
                    break;
                }
            }
        }

        public void flushHandler(ConnectionHandler handler) {
            ConnectionPool.this.allConnections.remove(handler);
            ConnectionPool.this.metricsRepository.afterConnectionFlush();
            ListenerHelper.fireOnConnectionFlush(ConnectionPool.this.listeners, handler);
            ConnectionPool.this.housekeepingExecutor.execute(new DestroyConnectionTask(handler));
        }

        private void afterFlush(AgroalDataSource.FlushMode mode) {
            switch (mode) {
                case ALL: 
                case GRACEFUL: 
                case INVALID: 
                case FILL: {
                    ConnectionPool.this.housekeepingExecutor.execute(new FillTask());
                    break;
                }
                case IDLE: {
                    break;
                }
                default: {
                    ListenerHelper.fireOnWarning(ConnectionPool.this.listeners, "Unsupported Flush mode " + mode);
                }
            }
        }
    }

    private final class CreateConnectionTask
    implements Callable<ConnectionHandler> {
        private boolean initial;

        private CreateConnectionTask() {
        }

        private CreateConnectionTask initial() {
            this.initial = true;
            return this;
        }

        @Override
        public ConnectionHandler call() throws SQLException {
            if (!this.initial && ConnectionPool.this.allConnections.size() >= ConnectionPool.this.configuration.maxSize()) {
                return null;
            }
            ListenerHelper.fireBeforeConnectionCreation(ConnectionPool.this.listeners);
            long metricsStamp = ConnectionPool.this.metricsRepository.beforeConnectionCreation();
            try {
                ConnectionHandler handler = new ConnectionHandler(ConnectionPool.this.connectionFactory.createConnection(), ConnectionPool.this);
                if (!ConnectionPool.this.configuration.maxLifetime().isZero()) {
                    handler.setMaxLifetimeTask(ConnectionPool.this.housekeepingExecutor.schedule(new FlushTask(AgroalDataSource.FlushMode.GRACEFUL, handler), ConnectionPool.this.configuration.maxLifetime().toNanos(), TimeUnit.NANOSECONDS));
                }
                ListenerHelper.fireOnConnectionCreation(ConnectionPool.this.listeners, handler);
                ConnectionPool.this.metricsRepository.afterConnectionCreation(metricsStamp);
                handler.setState(ConnectionHandler.State.CHECKED_IN);
                ConnectionPool.this.allConnections.add(handler);
                ConnectionPool.this.maxUsed.accumulate(ConnectionPool.this.allConnections.size());
                ListenerHelper.fireOnConnectionPooled(ConnectionPool.this.listeners, handler);
                ConnectionHandler connectionHandler = handler;
                return connectionHandler;
            }
            catch (SQLException e) {
                ListenerHelper.fireOnWarning(ConnectionPool.this.listeners, e);
                throw e;
            }
            finally {
                ConnectionPool.this.synchronizer.releaseConditional();
            }
        }
    }

    private final class ConnectionHandlerThreadLocal
    extends ThreadLocal<List<ConnectionHandler>> {
        private ConnectionHandlerThreadLocal() {
        }

        @Override
        protected List<ConnectionHandler> initialValue() {
            return new UncheckedArrayList<ConnectionHandler>(ConnectionHandler.class);
        }
    }
}

