/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.esig.dss.spi.x509;

import eu.europa.esig.dss.enumerations.CertificateOrigin;
import eu.europa.esig.dss.enumerations.CertificateRefOrigin;
import eu.europa.esig.dss.enumerations.DigestAlgorithm;
import eu.europa.esig.dss.model.Digest;
import eu.europa.esig.dss.model.x509.CertificateToken;
import eu.europa.esig.dss.spi.DSSASN1Utils;
import eu.europa.esig.dss.spi.DSSUtils;
import eu.europa.esig.dss.spi.OID;
import eu.europa.esig.dss.spi.SignatureCertificateSource;
import eu.europa.esig.dss.spi.x509.CandidatesForSigningCertificate;
import eu.europa.esig.dss.spi.x509.CertificateRef;
import eu.europa.esig.dss.spi.x509.CertificateSource;
import eu.europa.esig.dss.spi.x509.CertificateValidity;
import eu.europa.esig.dss.spi.x509.SignerIdentifier;
import eu.europa.esig.dss.utils.Utils;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.ASN1Set;
import org.bouncycastle.asn1.cms.Attribute;
import org.bouncycastle.asn1.cms.AttributeTable;
import org.bouncycastle.asn1.ess.ESSCertID;
import org.bouncycastle.asn1.ess.ESSCertIDv2;
import org.bouncycastle.asn1.ess.OtherCertID;
import org.bouncycastle.asn1.ess.SigningCertificate;
import org.bouncycastle.asn1.ess.SigningCertificateV2;
import org.bouncycastle.asn1.pkcs.PKCSObjectIdentifiers;
import org.bouncycastle.asn1.x509.Certificate;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cms.CMSSignedData;
import org.bouncycastle.cms.SignerInformation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class CMSCertificateSource
extends SignatureCertificateSource {
    private static final Logger LOG = LoggerFactory.getLogger(CMSCertificateSource.class);
    private final transient CMSSignedData cmsSignedData;
    private final transient SignerInformation currentSignerInformation;

    protected CMSCertificateSource(CMSSignedData cmsSignedData, SignerInformation currentSignerInformation) {
        Objects.requireNonNull(cmsSignedData, "CMS SignedData is null, it must be provided!");
        Objects.requireNonNull(currentSignerInformation, "currentSignerInformation is null, it must be provided!");
        this.cmsSignedData = cmsSignedData;
        this.currentSignerInformation = currentSignerInformation;
        this.extractCertificateIdentifiers();
        this.extractSignedCertificates();
        this.extractSigningCertificateReferences();
        this.extractCertificateValues();
        this.extractCertificateRefsFromUnsignedAttribute(PKCSObjectIdentifiers.id_aa_ets_certificateRefs, CertificateRefOrigin.COMPLETE_CERTIFICATE_REFS);
        this.extractCertificateRefsFromUnsignedAttribute(OID.attributeCertificateRefsOid, CertificateRefOrigin.ATTRIBUTE_CERTIFICATE_REFS);
    }

    private void extractCertificateIdentifiers() {
        SignerIdentifier currentSignerIdentifier = DSSASN1Utils.toSignerIdentifier(this.currentSignerInformation.getSID());
        boolean found = false;
        Collection signers = this.cmsSignedData.getSignerInfos().getSigners();
        for (SignerInformation signerInformation : signers) {
            SignerIdentifier signerIdentifier = DSSASN1Utils.toSignerIdentifier(signerInformation.getSID());
            if (signerIdentifier.isEquivalent(currentSignerIdentifier)) {
                signerIdentifier.setCurrent(true);
                found = true;
            }
            this.addCertificateIdentifier(signerIdentifier, CertificateOrigin.SIGNED_DATA);
        }
        if (!found) {
            LOG.warn("SID not found in SignerInfos");
            currentSignerIdentifier.setCurrent(true);
            this.addCertificateIdentifier(currentSignerIdentifier, CertificateOrigin.SIGNED_DATA);
        }
    }

    private void extractSignedCertificates() {
        try {
            Collection x509CertificateHolders = this.cmsSignedData.getCertificates().getMatches(null);
            for (X509CertificateHolder x509CertificateHolder : x509CertificateHolders) {
                this.addCertificate(DSSASN1Utils.getCertificate(x509CertificateHolder), CertificateOrigin.SIGNED_DATA);
            }
        }
        catch (Exception e) {
            LOG.warn("Cannot extract certificates from CMS Signed Data : {}", (Object)e.getMessage());
        }
    }

    private void extractSigningCertificateReferences() {
        AttributeTable signedAttributes = this.currentSignerInformation.getSignedAttributes();
        if (signedAttributes != null && signedAttributes.size() > 0) {
            Attribute signingCertificateAttributeV2;
            Attribute signingCertificateAttributeV1 = signedAttributes.get(PKCSObjectIdentifiers.id_aa_signingCertificate);
            if (signingCertificateAttributeV1 != null) {
                this.extractSigningCertificateV1(signingCertificateAttributeV1);
            }
            if ((signingCertificateAttributeV2 = signedAttributes.get(PKCSObjectIdentifiers.id_aa_signingCertificateV2)) != null) {
                this.extractSigningCertificateV2(signingCertificateAttributeV2);
            }
        }
    }

    private void extractSigningCertificateV1(Attribute attribute) {
        ASN1Set attrValues = attribute.getAttrValues();
        for (int ii = 0; ii < attrValues.size(); ++ii) {
            ASN1Encodable asn1Encodable = attrValues.getObjectAt(ii);
            try {
                SigningCertificate signingCertificate = SigningCertificate.getInstance((Object)asn1Encodable);
                if (signingCertificate != null) {
                    this.extractESSCertIDs(signingCertificate.getCerts(), CertificateRefOrigin.SIGNING_CERTIFICATE);
                    continue;
                }
                LOG.warn("SigningCertificate attribute is null");
                continue;
            }
            catch (Exception e) {
                LOG.warn("SigningCertificate attribute '{}' is not well defined!", (Object)Utils.toBase64((byte[])DSSASN1Utils.getDEREncoded(asn1Encodable)));
            }
        }
    }

    private void extractESSCertIDs(ESSCertID[] essCertIDs, CertificateRefOrigin origin) {
        for (ESSCertID essCertID : essCertIDs) {
            CertificateRef certRef = new CertificateRef();
            byte[] certHash = essCertID.getCertHash();
            if (Utils.isArrayNotEmpty((byte[])certHash)) {
                certRef.setCertDigest(new Digest(DigestAlgorithm.SHA1, certHash));
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Found Certificate Hash in signingCertificateAttributeV1 {} with algorithm {}", (Object)Utils.toHex((byte[])certHash), (Object)DigestAlgorithm.SHA1);
                }
            }
            certRef.setCertificateIdentifier(DSSASN1Utils.toSignerIdentifier(essCertID.getIssuerSerial()));
            this.addCertificateRef(certRef, origin);
        }
    }

    private void extractSigningCertificateV2(Attribute attribute) {
        ASN1Set attrValues = attribute.getAttrValues();
        for (int ii = 0; ii < attrValues.size(); ++ii) {
            ASN1Encodable asn1Encodable = attrValues.getObjectAt(ii);
            try {
                SigningCertificateV2 signingCertificate = SigningCertificateV2.getInstance((Object)asn1Encodable);
                if (signingCertificate != null) {
                    this.extractESSCertIDv2s(signingCertificate.getCerts(), CertificateRefOrigin.SIGNING_CERTIFICATE);
                    continue;
                }
                LOG.warn("SigningCertificateV2 attribute is null");
                continue;
            }
            catch (Exception e) {
                LOG.warn("SigningCertificateV2 attribute '{}' is not well defined!", (Object)Utils.toBase64((byte[])DSSASN1Utils.getDEREncoded(asn1Encodable)));
            }
        }
    }

    private void extractESSCertIDv2s(ESSCertIDv2[] essCertIDv2s, CertificateRefOrigin origin) {
        for (ESSCertIDv2 essCertIDv2 : essCertIDv2s) {
            CertificateRef certRef = new CertificateRef();
            DigestAlgorithm digestAlgorithm = DigestAlgorithm.forOID((String)essCertIDv2.getHashAlgorithm().getAlgorithm().getId());
            byte[] certHash = essCertIDv2.getCertHash();
            certRef.setCertDigest(new Digest(digestAlgorithm, certHash));
            if (LOG.isDebugEnabled()) {
                LOG.debug("Found Certificate Hash in SigningCertificateV2 {} with algorithm {}", (Object)Utils.toHex((byte[])certHash), (Object)digestAlgorithm);
            }
            certRef.setCertificateIdentifier(DSSASN1Utils.toSignerIdentifier(essCertIDv2.getIssuerSerial()));
            this.addCertificateRef(certRef, origin);
        }
    }

    private void extractCertificateValues() {
        Attribute attribute;
        AttributeTable unsignedAttributes = this.currentSignerInformation.getUnsignedAttributes();
        if (unsignedAttributes != null && (attribute = unsignedAttributes.get(PKCSObjectIdentifiers.id_aa_ets_certValues)) != null) {
            ASN1Sequence seq = (ASN1Sequence)attribute.getAttrValues().getObjectAt(0);
            for (int ii = 0; ii < seq.size(); ++ii) {
                try {
                    Certificate cs = Certificate.getInstance((Object)seq.getObjectAt(ii));
                    this.addCertificate(DSSUtils.loadCertificate(cs.getEncoded()), CertificateOrigin.CERTIFICATE_VALUES);
                    continue;
                }
                catch (Exception e) {
                    LOG.warn("Unable to parse encapsulated certificate : {}", (Object)e.getMessage());
                }
            }
        }
    }

    private void extractCertificateRefsFromUnsignedAttribute(ASN1ObjectIdentifier attributeOid, CertificateRefOrigin origin) {
        Attribute attribute;
        AttributeTable unsignedAttributes = this.currentSignerInformation.getUnsignedAttributes();
        if (unsignedAttributes != null && (attribute = unsignedAttributes.get(attributeOid)) != null) {
            ASN1Sequence seq = (ASN1Sequence)attribute.getAttrValues().getObjectAt(0);
            for (int ii = 0; ii < seq.size(); ++ii) {
                try {
                    OtherCertID otherCertId = OtherCertID.getInstance((Object)seq.getObjectAt(ii));
                    CertificateRef certRef = DSSASN1Utils.getCertificateRef(otherCertId);
                    this.addCertificateRef(certRef, origin);
                    continue;
                }
                catch (Exception e) {
                    LOG.warn("Unable to parse encapsulated OtherCertID : {}", (Object)e.getMessage());
                }
            }
        }
    }

    @Override
    protected CandidatesForSigningCertificate extractCandidatesForSigningCertificate(CertificateSource signingCertificateSource) {
        CandidatesForSigningCertificate candidates = new CandidatesForSigningCertificate();
        SignerIdentifier currentSignerIdentifier = this.getCurrentCertificateIdentifier();
        if (currentSignerIdentifier != null && !currentSignerIdentifier.isEmpty()) {
            Set<CertificateToken> foundTokens;
            CertificateToken certificate = this.getCertificateToken(currentSignerIdentifier);
            if (certificate == null && signingCertificateSource != null && Utils.isCollectionNotEmpty(foundTokens = signingCertificateSource.getBySignerIdentifier(currentSignerIdentifier))) {
                LOG.debug("Resolved signing certificate by certificate identifier");
                certificate = foundTokens.iterator().next();
            }
            CertificateValidity certificateValidity = certificate != null ? new CertificateValidity(certificate) : new CertificateValidity(currentSignerIdentifier);
            List<CertificateRef> signingCertRefs = this.getSigningCertificateRefs();
            if (Utils.isCollectionNotEmpty(signingCertRefs)) {
                SignerIdentifier sigCertIdentifier;
                CertificateRef signingCertRef = signingCertRefs.iterator().next();
                Digest certDigest = signingCertRef.getCertDigest();
                certificateValidity.setDigestPresent(certDigest != null);
                if (certificate != null) {
                    byte[] certificateDigest = certificate.getDigest(certDigest.getAlgorithm());
                    certificateValidity.setDigestEqual(Arrays.equals(certificateDigest, certDigest.getValue()));
                }
                certificateValidity.setIssuerSerialPresent((sigCertIdentifier = signingCertRef.getCertificateIdentifier()) != null);
                if (sigCertIdentifier != null) {
                    if (certificate != null) {
                        certificateValidity.setSerialNumberEqual(certificate.getSerialNumber().equals(sigCertIdentifier.getSerialNumber()));
                        certificateValidity.setDistinguishedNameEqual(DSSASN1Utils.x500PrincipalAreEquals(certificate.getIssuerX500Principal(), sigCertIdentifier.getIssuerName()));
                    } else {
                        certificateValidity.setSerialNumberEqual(currentSignerIdentifier.getSerialNumber().equals(sigCertIdentifier.getSerialNumber()));
                        certificateValidity.setDistinguishedNameEqual(DSSASN1Utils.x500PrincipalAreEquals(currentSignerIdentifier.getIssuerName(), sigCertIdentifier.getIssuerName()));
                    }
                    certificateValidity.setSignerIdMatch(currentSignerIdentifier.isEquivalent(sigCertIdentifier));
                }
            }
            candidates.add(certificateValidity);
            candidates.setTheCertificateValidity(certificateValidity);
        } else if (signingCertificateSource != null) {
            List<CertificateToken> certificates = signingCertificateSource.getCertificates();
            LOG.debug("No signing certificate reference found. Resolve all {} certificates from the provided certificate source as signing candidates.", (Object)certificates.size());
            for (CertificateToken certCandidate : certificates) {
                candidates.add(new CertificateValidity(certCandidate));
            }
        }
        return candidates;
    }
}

