/*
 * Decompiled with CFR 0.152.
 */
package dev.scheibelhofer.crypto.provider;

import dev.scheibelhofer.crypto.provider.Pem;
import dev.scheibelhofer.crypto.provider.PemKeystoreException;
import java.security.Key;
import java.security.KeyStoreException;
import java.security.KeyStoreSpi;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.security.interfaces.ECPrivateKey;
import java.security.interfaces.ECPublicKey;
import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.security.auth.x500.X500Principal;

public abstract class PemKeystore
extends KeyStoreSpi {
    final Map<String, Pem.PrivateKeyEntry> privateKeys = new LinkedHashMap<String, Pem.PrivateKeyEntry>();
    final Map<String, Pem.EncryptedPrivateKeyEntry> encryptedPrivateKeys = new LinkedHashMap<String, Pem.EncryptedPrivateKeyEntry>();
    final Map<String, List<Pem.CertificateEntry>> certificateChains = new LinkedHashMap<String, List<Pem.CertificateEntry>>();
    final Map<String, Pem.CertificateEntry> certificates = new LinkedHashMap<String, Pem.CertificateEntry>();
    final Date creationDate = new Date();
    static final String SUBJECT_KEY_ID = "2.5.29.14";
    static final String AUTHORITY_KEY_ID = "2.5.29.35";

    void clearKeystore() {
        this.privateKeys.clear();
        this.encryptedPrivateKeys.clear();
        this.certificateChains.clear();
        this.certificates.clear();
    }

    @Override
    public Key engineGetKey(String alias, char[] password) throws NoSuchAlgorithmException, UnrecoverableKeyException {
        Pem.PrivateKeyEntry privateKeyEntry = this.privateKeys.get(alias);
        if (privateKeyEntry != null) {
            return privateKeyEntry.privateKey;
        }
        Pem.EncryptedPrivateKeyEntry encryptedPrivateKeyEntry = this.encryptedPrivateKeys.get(alias);
        if (encryptedPrivateKeyEntry == null) {
            return null;
        }
        try {
            encryptedPrivateKeyEntry.decryptPrivateKey(password);
            return encryptedPrivateKeyEntry.privateKey;
        }
        catch (PemKeystoreException e) {
            throw new NoSuchAlgorithmException("failed decrypting encrypted private key", e);
        }
    }

    @Override
    public Certificate[] engineGetCertificateChain(String alias) {
        List<Pem.CertificateEntry> certEntries = this.certificateChains.get(alias);
        if (certEntries == null) {
            return null;
        }
        return certEntries.stream().map(ce -> ce.certificate).collect(Collectors.toList()).toArray(new Certificate[certEntries.size()]);
    }

    @Override
    public Certificate engineGetCertificate(String alias) {
        Pem.CertificateEntry ce = this.certificates.get(alias);
        if (ce == null) {
            return null;
        }
        return ce.certificate;
    }

    @Override
    public Date engineGetCreationDate(String alias) {
        return this.creationDate;
    }

    @Override
    public void engineSetKeyEntry(String alias, Key key, char[] password, Certificate[] chain) throws KeyStoreException {
        if (key instanceof PrivateKey && password == null) {
            Pem.PrivateKeyEntry keyEntry = new Pem.PrivateKeyEntry(alias, (PrivateKey)key);
            this.privateKeys.put(alias, keyEntry);
            List certificateChain = Stream.of(chain).filter(X509Certificate.class::isInstance).map(X509Certificate.class::cast).map(c -> new Pem.CertificateEntry(alias, (X509Certificate)c)).collect(Collectors.toList());
            this.certificateChains.put(alias, certificateChain);
        }
    }

    @Override
    public void engineSetKeyEntry(String alias, byte[] encryptedKey, Certificate[] chain) throws KeyStoreException {
        Pem.EncryptedPrivateKeyEntry encryptedKeyEntry = new Pem.EncryptedPrivateKeyEntry(alias);
        encryptedKeyEntry.initFromEncoding(encryptedKey);
        this.encryptedPrivateKeys.put(alias, encryptedKeyEntry);
        List certificateChain = Stream.of(chain).filter(X509Certificate.class::isInstance).map(X509Certificate.class::cast).map(c -> new Pem.CertificateEntry(alias, (X509Certificate)c)).collect(Collectors.toList());
        this.certificateChains.put(alias, certificateChain);
    }

    @Override
    public void engineSetCertificateEntry(String alias, Certificate cert) throws KeyStoreException {
        if (!(cert instanceof X509Certificate)) {
            throw new KeyStoreException("certificate entry must be of type java.security.cert.X509Certificate, but is " + cert.getClass());
        }
        X509Certificate x509Cert = (X509Certificate)cert;
        Pem.CertificateEntry certEntry = new Pem.CertificateEntry(alias, x509Cert);
        this.certificates.put(alias, certEntry);
    }

    @Override
    public void engineDeleteEntry(String alias) throws KeyStoreException {
        this.encryptedPrivateKeys.remove(alias);
        this.privateKeys.remove(alias);
        this.certificateChains.remove(alias);
        this.certificates.remove(alias);
    }

    @Override
    public Enumeration<String> engineAliases() {
        HashSet<String> aliases = new HashSet<String>();
        aliases.addAll(this.certificates.keySet());
        aliases.addAll(this.privateKeys.keySet());
        aliases.addAll(this.encryptedPrivateKeys.keySet());
        return Collections.enumeration(aliases);
    }

    @Override
    public boolean engineContainsAlias(String alias) {
        return this.certificates.containsKey(alias) || this.privateKeys.containsKey(alias) || this.encryptedPrivateKeys.containsKey(alias);
    }

    @Override
    public int engineSize() {
        return this.certificates.size() + this.privateKeys.size() + this.encryptedPrivateKeys.size();
    }

    @Override
    public boolean engineIsKeyEntry(String alias) {
        return this.privateKeys.containsKey(alias) || this.encryptedPrivateKeys.containsKey(alias);
    }

    @Override
    public boolean engineIsCertificateEntry(String alias) {
        return this.certificates.containsKey(alias);
    }

    @Override
    public String engineGetCertificateAlias(Certificate cert) {
        if (cert == null) {
            return null;
        }
        for (Map.Entry<String, Pem.CertificateEntry> entry : this.certificates.entrySet()) {
            if (!cert.equals(entry.getValue().certificate)) continue;
            return entry.getKey();
        }
        for (Map.Entry<String, Object> entry : this.certificateChains.entrySet()) {
            List chain = (List)entry.getValue();
            if (chain.size() == 0) continue;
            Pem.CertificateEntry firstCertChainEntry = (Pem.CertificateEntry)chain.get(0);
            if (!cert.equals(firstCertChainEntry.certificate)) continue;
            return entry.getKey();
        }
        return null;
    }

    static boolean matching(PublicKey publicKey, PrivateKey privateKey) {
        if (publicKey instanceof RSAPublicKey && privateKey instanceof RSAPrivateKey) {
            return PemKeystore.matching((RSAPublicKey)publicKey, (RSAPrivateKey)privateKey);
        }
        if (publicKey instanceof ECPublicKey && privateKey instanceof ECPrivateKey) {
            return PemKeystore.matching((ECPublicKey)publicKey, (ECPrivateKey)privateKey);
        }
        return false;
    }

    static boolean matching(ECPublicKey publicKey, ECPrivateKey privateKey) {
        try {
            byte[] data = new byte[32];
            Signature s = Signature.getInstance("SHA256withECDSA");
            s.initSign(privateKey);
            s.update(data);
            byte[] sig = s.sign();
            s.initVerify(publicKey);
            s.update(data);
            return s.verify(sig);
        }
        catch (Exception e) {
            return false;
        }
    }

    static boolean matching(RSAPublicKey publicKey, RSAPrivateKey privateKey) {
        return publicKey.getModulus().equals(privateKey.getModulus());
    }

    void buildCertChains(List<Pem.CertificateEntry> certList) {
        if (certList.isEmpty()) {
            return;
        }
        HashSet<Pem.CertificateEntry> usedCertificates = new HashSet<Pem.CertificateEntry>();
        for (String alias : this.privateKeys.keySet()) {
            String newAlias;
            Pem.PrivateKeyEntry privateKeyEntry = this.privateKeys.get(alias);
            List<Pem.CertificateEntry> certChain = this.buildChainFor(privateKeyEntry, certList);
            if (certChain.size() <= 0) continue;
            if (privateKeyEntry.alias != null) {
                newAlias = alias;
            } else {
                newAlias = this.makeAlias(certChain.get(0));
                this.privateKeys.remove(alias);
                this.privateKeys.put(newAlias, privateKeyEntry);
            }
            this.certificateChains.put(newAlias, certChain);
            usedCertificates.addAll(certChain);
        }
        certList.removeAll(usedCertificates);
    }

    String makeAlias(Pem.CertificateEntry certificateEntry) {
        X500Principal subject = certificateEntry.certificate.getSubjectX500Principal();
        return subject.getName();
    }

    List<Pem.CertificateEntry> buildChainFor(Pem.PrivateKeyEntry privateKeyEntry, List<Pem.CertificateEntry> certList) {
        Optional<Pem.CertificateEntry> privateKeyCertificate = certList.stream().filter(c -> PemKeystore.matching(c.certificate.getPublicKey(), privateKeyEntry.privateKey)).findFirst();
        ArrayList<Pem.CertificateEntry> certChain = new ArrayList<Pem.CertificateEntry>(4);
        Pem.CertificateEntry cert = privateKeyCertificate.orElse(null);
        while (cert != null) {
            Pem.CertificateEntry currentCertEntry = cert;
            byte[] authorityKeyID = currentCertEntry.certificate.getExtensionValue(AUTHORITY_KEY_ID);
            certChain.add(currentCertEntry);
            cert = certList.stream().filter(ce -> !ce.equals(currentCertEntry)).filter(ce -> ce.certificate.getSubjectX500Principal().equals(currentCertEntry.certificate.getIssuerX500Principal())).filter(ce -> PemKeystore.matchingKeyIDs(authorityKeyID, ce.certificate)).findFirst().orElse(null);
        }
        return certChain;
    }

    static boolean matchingKeyIDs(byte[] authorityKeyID, X509Certificate c) {
        if (authorityKeyID == null) {
            return true;
        }
        byte[] certSubjectKeyId = c.getExtensionValue(SUBJECT_KEY_ID);
        if (certSubjectKeyId == null) {
            return true;
        }
        return Arrays.equals(authorityKeyID, authorityKeyID.length - 20, authorityKeyID.length, certSubjectKeyId, certSubjectKeyId.length - 20, certSubjectKeyId.length);
    }

    String makeUniqueAlias(Set<String> existingAliases, String suggestedAlias) {
        Object alias = suggestedAlias;
        int i = 2;
        while (existingAliases.contains(alias)) {
            alias = suggestedAlias + "-" + i;
            ++i;
        }
        return alias;
    }

    String makeUniqueAlias(Set<String> aliasSet, Pem.Entry entry) {
        if (entry.alias != null) {
            return this.makeUniqueAlias(aliasSet, entry.alias);
        }
        if (entry instanceof Pem.PrivateKeyEntry) {
            return this.makeUniqueAlias(aliasSet, "private-key");
        }
        if (entry instanceof Pem.EncryptedPrivateKeyEntry) {
            return this.makeUniqueAlias(aliasSet, "encrypted-private-key");
        }
        return this.makeUniqueAlias(aliasSet, "entry");
    }
}

