/*
 * Decompiled with CFR 0.152.
 */
package dev.langchain4j.internal;

import dev.langchain4j.internal.CustomMimeTypesFileTypeDetector;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Path;
import java.util.HashMap;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.AssertionsForClassTypes;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.ValueSource;

class CustomMimeTypesFileTypeDetectorTest {
    CustomMimeTypesFileTypeDetectorTest() {
    }

    @Test
    void should_return_a_mime_type_from_default_mapping_from_path() {
        CustomMimeTypesFileTypeDetector detector = new CustomMimeTypesFileTypeDetector();
        Path path = Path.of("/foo/bar/index.html", new String[0]);
        String mimeType = detector.probeContentType(path);
        Assertions.assertThat((String)mimeType).isEqualTo("text/html");
    }

    @Test
    void should_return_a_mime_type_from_default_mapping_from_uri() {
        CustomMimeTypesFileTypeDetector detector = new CustomMimeTypesFileTypeDetector();
        URI uri = URI.create("https://foo.com/bar/style.css");
        String mimeType = detector.probeContentType(uri);
        Assertions.assertThat((String)mimeType).isEqualTo("text/css");
    }

    @Test
    void should_return_a_mime_type_from_default_mapping_from_string() {
        CustomMimeTypesFileTypeDetector detector = new CustomMimeTypesFileTypeDetector();
        String mimeType = detector.probeContentType("README.md");
        Assertions.assertThat((String)mimeType).isEqualTo("text/x-markdown");
    }

    @Test
    void can_provide_custom_mapping() {
        HashMap<String, String> mapping = new HashMap<String, String>();
        mapping.put("abcd", "text/abcd");
        CustomMimeTypesFileTypeDetector detector = new CustomMimeTypesFileTypeDetector(mapping);
        String mimeType = detector.probeContentType("foo.abcd");
        Assertions.assertThat((String)mimeType).isEqualTo("text/abcd");
    }

    @Test
    void should_return_null_when_no_mapping_exist_from_path() {
        CustomMimeTypesFileTypeDetector detector = new CustomMimeTypesFileTypeDetector();
        String mimeType = detector.probeContentType("foo.banana");
        Assertions.assertThat((String)mimeType).isNull();
    }

    @Test
    void should_return_null_when_no_mapping_exist_from_uri() {
        CustomMimeTypesFileTypeDetector detector = new CustomMimeTypesFileTypeDetector();
        String mimeType = detector.probeContentType(URI.create("https://foo.bar.com/baz.yui"));
        Assertions.assertThat((String)mimeType).isNull();
    }

    @Test
    void should_return_mime_type_from_website_from_uri() {
        CustomMimeTypesFileTypeDetector detector = new CustomMimeTypesFileTypeDetector();
        String mimeType = detector.probeContentType(URI.create("https://docs.langchain4j.dev/logo.svg"));
        Assertions.assertThat((String)mimeType).isEqualTo("image/svg+xml");
    }

    @ParameterizedTest
    @ValueSource(strings={"http://example.org/cat", "http://example.org/cat.banana", "http://example.org/some.path/cat", "http://example.org/cat?query=dog.png"})
    void should_fail_to_detect_mime_type(String url) throws MalformedURLException, URISyntaxException {
        CustomMimeTypesFileTypeDetector detector = new CustomMimeTypesFileTypeDetector();
        String mimeType = detector.probeContentType(new URL(url).toURI());
        Assertions.assertThat((String)mimeType).isNull();
    }

    @Test
    void should_handle_null_path_input() {
        CustomMimeTypesFileTypeDetector detector = new CustomMimeTypesFileTypeDetector();
        AssertionsForClassTypes.assertThatThrownBy(() -> detector.probeContentType((Path)null)).isInstanceOf(NullPointerException.class);
    }

    @Test
    void should_handle_null_uri_input() {
        CustomMimeTypesFileTypeDetector detector = new CustomMimeTypesFileTypeDetector();
        AssertionsForClassTypes.assertThatThrownBy(() -> detector.probeContentType((URI)null)).isInstanceOf(NullPointerException.class);
    }

    @Test
    void should_handle_null_string_input() {
        CustomMimeTypesFileTypeDetector detector = new CustomMimeTypesFileTypeDetector();
        AssertionsForClassTypes.assertThatThrownBy(() -> detector.probeContentType((String)null)).isInstanceOf(NullPointerException.class);
    }

    @Test
    void should_handle_empty_string_input() {
        CustomMimeTypesFileTypeDetector detector = new CustomMimeTypesFileTypeDetector();
        String mimeType = detector.probeContentType("");
        Assertions.assertThat((String)mimeType).isNull();
    }

    @Test
    void should_handle_file_without_extension() {
        CustomMimeTypesFileTypeDetector detector = new CustomMimeTypesFileTypeDetector();
        String mimeType = detector.probeContentType("README");
        Assertions.assertThat((String)mimeType).isNull();
    }

    @Test
    void should_be_case_insensitive_for_extensions() {
        CustomMimeTypesFileTypeDetector detector = new CustomMimeTypesFileTypeDetector();
        String lowerCaseMimeType = detector.probeContentType("document.pdf");
        String upperCaseMimeType = detector.probeContentType("document.PDF");
        String mixedCaseMimeType = detector.probeContentType("document.PdF");
        Assertions.assertThat((String)lowerCaseMimeType).isEqualTo("application/pdf");
        Assertions.assertThat((String)upperCaseMimeType).isEqualTo("application/pdf");
        Assertions.assertThat((String)mixedCaseMimeType).isEqualTo("application/pdf");
    }

    @Test
    void should_handle_path_with_no_filename() {
        CustomMimeTypesFileTypeDetector detector = new CustomMimeTypesFileTypeDetector();
        String mimeType = detector.probeContentType(Path.of("/foo/bar/", new String[0]));
        Assertions.assertThat((String)mimeType).isNull();
    }

    @Test
    void should_handle_relative_paths() {
        CustomMimeTypesFileTypeDetector detector = new CustomMimeTypesFileTypeDetector();
        String mimeType = detector.probeContentType(Path.of("./docs/readme.txt", new String[0]));
        Assertions.assertThat((String)mimeType).isEqualTo("text/plain");
    }

    @Test
    void should_handle_complex_uri_with_query_parameters() {
        CustomMimeTypesFileTypeDetector detector = new CustomMimeTypesFileTypeDetector();
        String mimeType = detector.probeContentType(URI.create("https://example.com/api/download/image.jpg?version=1&format=high"));
        Assertions.assertThat((String)mimeType).isEqualTo("image/jpeg");
    }

    @Test
    void should_handle_uri_with_fragment() {
        CustomMimeTypesFileTypeDetector detector = new CustomMimeTypesFileTypeDetector();
        String mimeType = detector.probeContentType(URI.create("https://example.com/docs/guide.html#section1"));
        Assertions.assertThat((String)mimeType).isEqualTo("text/html");
    }

    @Test
    void should_override_default_mapping_with_custom() {
        HashMap<String, String> customMapping = new HashMap<String, String>();
        customMapping.put("html", "application/custom-html");
        CustomMimeTypesFileTypeDetector detector = new CustomMimeTypesFileTypeDetector(customMapping);
        String mimeType = detector.probeContentType("index.html");
        Assertions.assertThat((String)mimeType).isEqualTo("application/custom-html");
    }

    @Test
    void should_fallback_to_default_when_custom_mapping_missing() {
        HashMap<String, String> customMapping = new HashMap<String, String>();
        customMapping.put("custom", "application/custom");
        CustomMimeTypesFileTypeDetector detector = new CustomMimeTypesFileTypeDetector(customMapping);
        String mimeType = detector.probeContentType("style.css");
        Assertions.assertThat((String)mimeType).isEqualTo("text/css");
    }

    @Test
    void should_handle_empty_custom_mapping() {
        HashMap emptyMapping = new HashMap();
        CustomMimeTypesFileTypeDetector detector = new CustomMimeTypesFileTypeDetector(emptyMapping);
        String mimeType = detector.probeContentType("document.pdf");
        Assertions.assertThat((String)mimeType).isEqualTo("application/pdf");
    }

    @Test
    void should_handle_null_custom_mapping() {
        AssertionsForClassTypes.assertThatThrownBy(() -> new CustomMimeTypesFileTypeDetector(null)).isInstanceOf(NullPointerException.class);
    }

    @ParameterizedTest
    @ValueSource(strings={"document.pdf", "spreadsheet.xlsx", "presentation.pptx", "document.docx", "archive.zip", "compressed.rar", "data.json", "config.xml"})
    void should_detect_common_application_formats(String filename) {
        CustomMimeTypesFileTypeDetector detector = new CustomMimeTypesFileTypeDetector();
        String mimeType = detector.probeContentType(filename);
        Assertions.assertThat((String)mimeType).startsWith((CharSequence)"application/");
    }

    @Test
    void should_handle_file_with_leading_dot() {
        CustomMimeTypesFileTypeDetector detector = new CustomMimeTypesFileTypeDetector();
        String mimeType = detector.probeContentType(".hidden.txt");
        Assertions.assertThat((String)mimeType).isEqualTo("text/plain");
    }

    @Test
    void should_handle_file_with_only_extension() {
        CustomMimeTypesFileTypeDetector detector = new CustomMimeTypesFileTypeDetector();
        String mimeType = detector.probeContentType(".css");
        Assertions.assertThat((String)mimeType).isEqualTo("text/css");
    }
}

