// Copyright Vespa.ai. Licensed under the terms of the Apache 2.0 license. See LICENSE in the project root.
package com.yahoo.document;

/**
 * A bucket id contains bit used for various purposes. In most use cases, these can use the default
 * settings, but the number of bits used for the different purposes is configurable, to allow for
 * special uses.
 *
 * Because of this, bucket ids cannot be generated without knowing how the bucket id is configured to
 * be put together, so all bucket ids must be generated by this factory class.
 *
 * For more information about what the sub parts of a bucket id actually is, read the bucket splitting documentation.
 *
 * @author Håkon Humberset
 */
public class BucketIdFactory {

    private final int gidBits;
    private final int locationBits;
    private final int countBits;

    private final long initialCount;
    private final long locationMask;
    private final long gidMask;

    /** Create a factory, using the default configuration. */
    public BucketIdFactory() {
        this(32, 26, 6);
    }

    /**
     * Create a factory, using the provided configuration.
     * @param gidBits How many bits that are used to specify gidbits.
     */
    public BucketIdFactory(int locationBits, int gidBits, int countBits) {
        this.locationBits = locationBits;
        this.gidBits = gidBits;
        this.countBits = countBits;
        initialCount = 58l << (64 - countBits);
        locationMask = 0xFFFFFFFFFFFFFFFFl >>> (64 - getLocationBitCount());
        gidMask = ((0xFFFFFFFFFFFFFFFFl >>> getLocationBitCount()) << (getLocationBitCount() + BucketId.COUNT_BITS)) >>> BucketId.COUNT_BITS;

    }

    /**
     * Create a factory, with parameters gotten from configuration.
     * TODO: Not implemented yet
     * @param configId The config id from where to get config.
     */
    public BucketIdFactory(String configId) {
        this(32, 26, 6);
    }

    /** @return Get number of bits used for storing of LSB part of location.*/
    public int getLocationBitCount() { return locationBits; }

    /** @return Get number of bits used to specify gid. */
    public int getGidBitCount() { return gidBits; }

    /** @return Get number of bits used to store bit count used. */
    public int getCountBitCount() { return countBits; }

    /**
     * Get the gid bit contribution in the bucket id, shifted to the correct
     * position in the id.
     *
     * @param gid The gid we need to calculate contribution from.
     * @return A mask to or with the bucket id to get the bit set.
     */
    private long getGidContribution(byte[] gid) {
        long gidbits = 0;
        for (int i=4; i<12; ++i) {
            gidbits <<= 8;
            long tall = gid[15 - i] & 0xFFl;
            assert(tall >= 0 && tall <= 255);
            gidbits |= tall;
        }
        return gidbits & gidMask;
    }

    /**
     * Get the bucket id for a given document.
     *
     * @param doc The doc.
     * @return The bucket id.
     */
    public BucketId getBucketId(DocumentId doc) {
        long location = doc.getScheme().getLocation();
        byte[] gid = doc.getGlobalId();
        long gidContribution = getGidContribution(gid);

        return new BucketId(64 - BucketId.COUNT_BITS, initialCount | (gidMask & gidContribution) | (locationMask & location));
    }

}
