/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.search.dispatch;

import com.yahoo.search.dispatch.RequestDuration;
import com.yahoo.search.dispatch.searchcluster.Group;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Random;
import java.util.Set;
import java.util.logging.Logger;

public class LoadBalancer {
    private static final Logger log = Logger.getLogger(LoadBalancer.class.getName());
    private static final long DEFAULT_LATENCY_DECAY_RATE = 1000L;
    private static final long MIN_LATENCY_DECAY_RATE = 42L;
    private static final double LATENCY_DECAY_TIME = (double)Duration.ofSeconds(5L).toMillis() / 1000.0;
    private static final Duration INITIAL_QUERY_TIME = Duration.ofMillis(1L);
    private static final double MIN_QUERY_TIME = (double)Duration.ofMillis(1L).toMillis() / 1000.0;
    private final Map<Integer, GroupStatus> scoreboard = new HashMap<Integer, GroupStatus>();
    private final GroupScheduler scheduler;

    public LoadBalancer(Collection<Group> groups, Policy policy) {
        for (Group group : groups) {
            this.scoreboard.put(group.id(), new GroupStatus(group));
        }
        if (this.scoreboard.size() == 1) {
            policy = Policy.ROUNDROBIN;
        }
        this.scheduler = switch (policy) {
            default -> throw new IncompatibleClassChangeError();
            case Policy.ROUNDROBIN -> new RoundRobinScheduler(this.scoreboard);
            case Policy.BEST_OF_RANDOM_2 -> new BestOfRandom2(new Random(), this.scoreboard);
            case Policy.LATENCY_AMORTIZED_OVER_REQUESTS -> new AdaptiveScheduler(AdaptiveScheduler.Type.REQUESTS, new Random(), this.scoreboard);
            case Policy.LATENCY_AMORTIZED_OVER_TIME -> new AdaptiveScheduler(AdaptiveScheduler.Type.TIME, new Random(), this.scoreboard);
        };
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Optional<Group> takeGroup(Set<Integer> rejectedGroups) {
        LoadBalancer loadBalancer = this;
        synchronized (loadBalancer) {
            Optional<GroupStatus> best = this.scheduler.takeNextGroup(rejectedGroups);
            if (best.isPresent()) {
                GroupStatus gs = best.get();
                gs.allocate();
                Group ret = gs.group;
                log.fine(() -> "Offering <" + String.valueOf(ret) + "> for query connection");
                return Optional.of(ret);
            }
            return Optional.empty();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void releaseGroup(Group group, boolean success, RequestDuration searchTime) {
        LoadBalancer loadBalancer = this;
        synchronized (loadBalancer) {
            GroupStatus sched = this.scoreboard.get(group.id());
            sched.release(success, searchTime);
        }
    }

    static class GroupStatus {
        private final Group group;
        private int allocations = 0;
        private Decayer decayer;

        GroupStatus(Group group) {
            this.group = group;
            this.decayer = new NoDecay();
        }

        void setDecayer(Decayer decayer) {
            this.decayer = decayer;
        }

        void allocate() {
            ++this.allocations;
        }

        void release(boolean success, RequestDuration searchTime) {
            --this.allocations;
            if (this.allocations < 0) {
                log.warning("Double free of query target group detected");
                this.allocations = 0;
            }
            if (success) {
                this.decayer.decay(searchTime);
            }
        }

        double weight() {
            return 1.0 / this.decayer.averageCost();
        }

        int groupId() {
            return this.group.id();
        }

        static class NoDecay
        implements Decayer {
            NoDecay() {
            }

            @Override
            public void decay(RequestDuration duration) {
            }

            @Override
            public double averageCost() {
                return MIN_QUERY_TIME;
            }
        }

        static interface Decayer {
            public void decay(RequestDuration var1);

            public double averageCost();
        }
    }

    public static enum Policy {
        ROUNDROBIN,
        LATENCY_AMORTIZED_OVER_REQUESTS,
        LATENCY_AMORTIZED_OVER_TIME,
        BEST_OF_RANDOM_2;

    }

    private static class RoundRobinScheduler
    implements GroupScheduler {
        private int needle = 0;
        private final Map<Integer, GroupStatus> scoreboard;

        public RoundRobinScheduler(Map<Integer, GroupStatus> scoreboard) {
            this.scoreboard = scoreboard;
        }

        @Override
        public Optional<GroupStatus> takeNextGroup(Set<Integer> rejectedGroups) {
            GroupStatus bestCandidate = null;
            int groupId = this.needle;
            for (int i = 0; i < this.scoreboard.size(); ++i) {
                GroupStatus better;
                GroupStatus candidate = this.scoreboard.get(groupId);
                if (!(rejectedGroups != null && rejectedGroups.contains(candidate.groupId()) || (better = RoundRobinScheduler.betterGroup(bestCandidate, candidate)) != candidate)) {
                    bestCandidate = candidate;
                }
                groupId = this.nextScoreboardIndex(groupId);
            }
            this.needle = this.nextScoreboardIndex(bestCandidate.groupId());
            return Optional.of(bestCandidate);
        }

        private static GroupStatus betterGroup(GroupStatus first, GroupStatus second) {
            if (second == null) {
                return first;
            }
            if (first == null) {
                return second;
            }
            if (first.group.hasSufficientCoverage() != second.group.hasSufficientCoverage()) {
                return first.group.hasSufficientCoverage() ? first : second;
            }
            return first;
        }

        private int nextScoreboardIndex(int current) {
            int next = current + 1;
            if (next >= this.scoreboard.size()) {
                next %= this.scoreboard.size();
            }
            return next;
        }
    }

    static class BestOfRandom2
    implements GroupScheduler {
        private final Random random;
        private final Map<Integer, GroupStatus> scoreboard;

        public BestOfRandom2(Random random, Map<Integer, GroupStatus> scoreboard) {
            this.random = random;
            this.scoreboard = scoreboard;
        }

        @Override
        public Optional<GroupStatus> takeNextGroup(Set<Integer> rejectedGroups) {
            GroupStatus gs = this.selectBestOf2(rejectedGroups, true);
            return gs != null ? Optional.of(gs) : Optional.ofNullable(this.selectBestOf2(rejectedGroups, false));
        }

        private GroupStatus selectBestOf2(Set<Integer> rejectedGroups, boolean requireCoverage) {
            ArrayList<Integer> candidates = new ArrayList<Integer>(this.scoreboard.size());
            for (GroupStatus gs : this.scoreboard.values()) {
                if (rejectedGroups != null && rejectedGroups.contains(gs.group.id()) || requireCoverage && !gs.group.hasSufficientCoverage()) continue;
                candidates.add(gs.groupId());
            }
            GroupStatus candA = this.selectRandom(candidates);
            GroupStatus candB = this.selectRandom(candidates);
            if (candA == null) {
                return candB;
            }
            if (candB == null) {
                return candA;
            }
            if (candB.allocations < candA.allocations) {
                return candB;
            }
            return candA;
        }

        private GroupStatus selectRandom(List<Integer> candidates) {
            if (!candidates.isEmpty()) {
                int index = this.random.nextInt(candidates.size());
                return this.scoreboard.get(candidates.remove(index));
            }
            return null;
        }
    }

    static class AdaptiveScheduler
    implements GroupScheduler {
        private final Random random;
        private final Map<Integer, GroupStatus> scoreboard;

        private static double toDouble(Duration duration) {
            return (double)duration.toNanos() / 1.0E9;
        }

        private static Duration fromDouble(double seconds) {
            return Duration.ofNanos((long)(seconds * 1.0E9));
        }

        public AdaptiveScheduler(Type type, Random random, Map<Integer, GroupStatus> scoreboard) {
            this.random = random;
            this.scoreboard = scoreboard;
            scoreboard.forEach((id, gs) -> gs.setDecayer(type == Type.REQUESTS ? new DecayByRequests() : new DecayByTime()));
        }

        private Optional<GroupStatus> selectGroup(double needle, boolean requireCoverage, Set<Integer> rejected) {
            double sum = 0.0;
            int n = 0;
            for (GroupStatus gs : this.scoreboard.values()) {
                if (rejected != null && rejected.contains(gs.group.id()) || requireCoverage && !gs.group.hasSufficientCoverage()) continue;
                sum += gs.weight();
                ++n;
            }
            if (n == 0) {
                return Optional.empty();
            }
            double accum = 0.0;
            for (GroupStatus gs : this.scoreboard.values()) {
                if (rejected != null && rejected.contains(gs.group.id()) || requireCoverage && !gs.group.hasSufficientCoverage() || !(needle < (accum += gs.weight()) / sum)) continue;
                return Optional.of(gs);
            }
            return Optional.empty();
        }

        @Override
        public Optional<GroupStatus> takeNextGroup(Set<Integer> rejectedGroups) {
            double needle = this.random.nextDouble();
            Optional<GroupStatus> gs = this.selectGroup(needle, true, rejectedGroups);
            if (gs.isPresent()) {
                return gs;
            }
            return this.selectGroup(needle, false, rejectedGroups);
        }

        static enum Type {
            TIME,
            REQUESTS;

        }

        static class DecayByRequests
        implements GroupStatus.Decayer {
            private long queries;
            private double averageSearchTime;

            DecayByRequests() {
                this(0L, INITIAL_QUERY_TIME);
            }

            DecayByRequests(long initialQueries, Duration initialSearchTime) {
                this.queries = initialQueries;
                this.averageSearchTime = AdaptiveScheduler.toDouble(initialSearchTime);
            }

            @Override
            public void decay(RequestDuration duration) {
                double searchTime = Math.max(AdaptiveScheduler.toDouble(duration.duration()), MIN_QUERY_TIME);
                double decayRate = Math.min(this.queries + 42L, 1000L);
                ++this.queries;
                this.averageSearchTime = (searchTime + (decayRate - 1.0) * this.averageSearchTime) / decayRate;
            }

            @Override
            public double averageCost() {
                return this.averageSearchTime;
            }

            Duration averageSearchTime() {
                return AdaptiveScheduler.fromDouble(this.averageSearchTime);
            }
        }

        static class DecayByTime
        implements GroupStatus.Decayer {
            private double averageSearchTime;
            private RequestDuration prev;

            DecayByTime() {
                this(INITIAL_QUERY_TIME, RequestDuration.of(Duration.ZERO));
            }

            DecayByTime(Duration initialSearchTime, RequestDuration start) {
                this.averageSearchTime = AdaptiveScheduler.toDouble(initialSearchTime);
                this.prev = start;
            }

            @Override
            public void decay(RequestDuration duration) {
                double searchTime = Math.max(AdaptiveScheduler.toDouble(duration.duration()), MIN_QUERY_TIME);
                double sampleWeight = AdaptiveScheduler.toDouble(duration.difference(this.prev));
                this.averageSearchTime = (sampleWeight * searchTime + LATENCY_DECAY_TIME * this.averageSearchTime) / (LATENCY_DECAY_TIME + sampleWeight);
                this.prev = duration;
            }

            @Override
            public double averageCost() {
                return this.averageSearchTime;
            }

            Duration averageSearchTime() {
                return AdaptiveScheduler.fromDouble(this.averageSearchTime);
            }
        }
    }

    private static interface GroupScheduler {
        public Optional<GroupStatus> takeNextGroup(Set<Integer> var1);
    }
}

